#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
高级控件库模块
包含所有高级类控件的定义和实现
提供表格、树形控件、图形视图、Web视图等高级控件
"""

# 导入类型提示
from typing import Dict, Any, Type, List, Optional, Callable
# 导入PySide6高级控件组件
from PySide6.QtWidgets import (
    QTableWidget,  # 表格控件（二维表格）
    QTableWidgetItem,  # 表格项
    QTreeWidget,  # 树形控件（层次结构数据）
    QTreeWidgetItem,  # 树形控件项
    QCalendarWidget,  # 日历控件
    QDockWidget,  # 停靠窗口
    QMdiArea,  # MDI区域（多文档界面）
    QMdiSubWindow,  # MDI子窗口
    QSplitter,  # 分割器
    QToolBox,  # 工具箱
    QWizard,  # 向导对话框
    QWizardPage,  # 向导页面
    QUndoView,  # 撤销视图
    QWebEngineView,  # Web引擎视图（显示网页）
    QGraphicsScene,  # 图形场景
    QGraphicsView,  # 图形视图
    QGraphicsItem,  # 图形项基类
    QWidget,  # 基础窗口部件
    QPushButton,  # 按钮
    QLabel,  # 标签
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QHeaderView,  # 表头视图
    QAbstractItemView,  # 抽象项视图基类
    QSizePolicy  # 大小策略
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    Signal,  # 信号定义
    QObject,  # Qt对象基类
    QUrl,  # URL对象
    QRectF,  # 浮点矩形
    QPointF  # 浮点点坐标
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QPixmap,  # 像素图
    QIcon,  # 图标
    QStandardItemModel,  # 标准项模型
    QStandardItem  # 标准项
)

# 导入控件基类和注册表
from ..control_base import BaseControl, control_registry


class TableWidgetControl(BaseControl):
    """表格控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "TableWidget"
        self._category = "高级"
        self._description = "用于显示和编辑表格数据的控件"
        self._size = (300, 200)
        self._properties = {
            "rowCount": 5,
            "columnCount": 3,
            "horizontalHeaderLabels": ["列1", "列2", "列3"],
            "verticalHeaderLabels": ["行1", "行2", "行3", "行4", "行5"],
            "selectionMode": QAbstractItemView.SelectionMode.SingleSelection,
            "selectionBehavior": QAbstractItemView.SelectionBehavior.SelectRows,
            "sortingEnabled": False,
            "alternatingRowColors": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QTableWidget
    
    def create_instance(self, parent=None) -> QWidget:
        table = QTableWidget(parent)
        
        # 设置行列数
        table.setRowCount(self.get_property("rowCount", 5))
        table.setColumnCount(self.get_property("columnCount", 3))
        
        # 设置表头
        table.setHorizontalHeaderLabels(self.get_property("horizontalHeaderLabels", ["列1", "列2", "列3"]))
        table.setVerticalHeaderLabels(self.get_property("verticalHeaderLabels", ["行1", "行2", "行3", "行4", "行5"]))
        
        # 设置选择模式
        table.setSelectionMode(self.get_property("selectionMode", QAbstractItemView.SelectionMode.SingleSelection))
        table.setSelectionBehavior(self.get_property("selectionBehavior", QAbstractItemView.SelectionBehavior.SelectRows))
        
        # 设置排序
        table.setSortingEnabled(self.get_property("sortingEnabled", False))
        
        # 设置交替行颜色
        table.setAlternatingRowColors(self.get_property("alternatingRowColors", True))
        
        # 填充一些示例数据
        for row in range(table.rowCount()):
            for col in range(table.columnCount()):
                item = QTableWidgetItem(f"数据 {row+1}-{col+1}")
                table.setItem(row, col, item)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            table.resize(width, height)
        
        return table


class TreeWidgetControl(BaseControl):
    """树形控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "TreeWidget"
        self._category = "高级"
        self._description = "用于显示树形结构的控件"
        self._size = (300, 200)
        self._properties = {
            "headerLabels": ["名称", "类型", "大小"],
            "columnCount": 3,
            "selectionMode": QAbstractItemView.SelectionMode.SingleSelection,
            "alternatingRowColors": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QTreeWidget
    
    def create_instance(self, parent=None) -> QWidget:
        tree = QTreeWidget(parent)
        
        # 设置列数和表头
        tree.setColumnCount(self.get_property("columnCount", 3))
        tree.setHeaderLabels(self.get_property("headerLabels", ["名称", "类型", "大小"]))
        
        # 设置选择模式
        tree.setSelectionMode(self.get_property("selectionMode", QAbstractItemView.SelectionMode.SingleSelection))
        
        # 设置交替行颜色
        tree.setAlternatingRowColors(self.get_property("alternatingRowColors", True))
        
        # 添加一些示例数据
        # 根节点
        root_item = QTreeWidgetItem(["根节点", "文件夹", ""])
        tree.addTopLevelItem(root_item)
        
        # 子节点
        child1 = QTreeWidgetItem(["子节点1", "文件", "10KB"])
        child2 = QTreeWidgetItem(["子节点2", "文件", "20KB"])
        root_item.addChild(child1)
        root_item.addChild(child2)
        
        # 另一个根节点
        root2 = QTreeWidgetItem(["另一个根节点", "文件夹", ""])
        tree.addTopLevelItem(root2)
        
        child3 = QTreeWidgetItem(["子节点3", "文件", "15KB"])
        root2.addChild(child3)
        
        # 展开所有项
        tree.expandAll()
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            tree.resize(width, height)
        
        return tree


class CalendarWidgetControl(BaseControl):
    """日历控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "CalendarWidget"
        self._category = "高级"
        self._description = "用于显示日历的控件"
        self._size = (300, 200)
        self._properties = {
            "gridVisible": True,
            "navigationBarVisible": True,
            "selectionMode": QCalendarWidget.SelectionMode.SingleSelection,
            "horizontalHeaderFormat": QCalendarWidget.HorizontalHeaderFormat.SingleLetterDayNames,
            "verticalHeaderFormat": QCalendarWidget.VerticalHeaderFormat.ISOWeekNumbers
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QCalendarWidget
    
    def create_instance(self, parent=None) -> QWidget:
        calendar = QCalendarWidget(parent)
        
        # 设置网格可见
        calendar.setGridVisible(self.get_property("gridVisible", True))
        
        # 设置导航栏可见
        calendar.setNavigationBarVisible(self.get_property("navigationBarVisible", True))
        
        # 设置选择模式
        calendar.setSelectionMode(self.get_property("selectionMode", QCalendarWidget.SelectionMode.SingleSelection))
        
        # 设置水平表头格式
        calendar.setHorizontalHeaderFormat(self.get_property("horizontalHeaderFormat", QCalendarWidget.HorizontalHeaderFormat.SingleLetterDayNames))
        
        # 设置垂直表头格式
        calendar.setVerticalHeaderFormat(self.get_property("verticalHeaderFormat", QCalendarWidget.VerticalHeaderFormat.ISOWeekNumbers))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            calendar.resize(width, height)
        
        return calendar


class DockWidgetControl(BaseControl):
    """停靠窗口控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "DockWidget"
        self._category = "高级"
        self._description = "可停靠的窗口控件"
        self._size = (200, 150)
        self._properties = {
            "title": "停靠窗口",
            "allowedAreas": Qt.DockWidgetArea.AllDockWidgetAreas,
            "floating": False,
            "features": QDockWidget.DockWidgetFeature.AllDockWidgetFeatures,
            "widget": None  # 内部控件
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QDockWidget
    
    def create_instance(self, parent=None) -> QWidget:
        dock = QDockWidget(self.get_property("title", "停靠窗口"), parent)
        
        # 设置允许停靠的区域
        dock.setAllowedAreas(self.get_property("allowedAreas", Qt.DockWidgetArea.AllDockWidgetAreas))
        
        # 设置浮动状态
        dock.setFloating(self.get_property("floating", False))
        
        # 设置停靠特性
        dock.setFeatures(self.get_property("features", QDockWidget.DockWidgetFeature.AllDockWidgetFeatures))
        
        # 创建内部控件
        widget = QWidget()
        layout = QVBoxLayout()
        
        label = QLabel("这是停靠窗口的内容")
        layout.addWidget(label)
        
        widget.setLayout(layout)
        dock.setWidget(widget)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            dock.resize(width, height)
        
        return dock


class MdiAreaControl(BaseControl):
    """多文档区域控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "MdiArea"
        self._category = "高级"
        self._description = "多文档界面区域控件"
        self._size = (400, 300)
        self._properties = {
            "viewMode": QMdiArea.ViewMode.SubWindowView,
            "tabPosition": QTabWidget.TabPosition.North,
            "tabShape": QTabWidget.TabShape.Rounded,
            "documentMode": False,
            "tabsClosable": False,
            "tabsMovable": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QMdiArea
    
    def create_instance(self, parent=None) -> QWidget:
        mdi_area = QMdiArea(parent)
        
        # 设置视图模式
        mdi_area.setViewMode(self.get_property("viewMode", QMdiArea.ViewMode.SubWindowView))
        
        # 设置标签页位置
        mdi_area.setTabPosition(self.get_property("tabPosition", QTabWidget.TabPosition.North))
        
        # 设置标签页形状
        mdi_area.setTabShape(self.get_property("tabShape", QTabWidget.TabShape.Rounded))
        
        # 设置文档模式
        mdi_area.setDocumentMode(self.get_property("documentMode", False))
        
        # 设置标签页可关闭
        mdi_area.setTabsClosable(self.get_property("tabsClosable", False))
        
        # 设置标签页可移动
        mdi_area.setTabsMovable(self.get_property("tabsMovable", False))
        
        # 添加一些示例子窗口
        for i in range(3):
            sub_window = QMdiSubWindow()
            sub_window.setWindowTitle(f"子窗口 {i+1}")
            sub_window.resize(200, 150)
            
            content = QWidget()
            layout = QVBoxLayout()
            layout.addWidget(QLabel(f"这是子窗口 {i+1} 的内容"))
            content.setLayout(layout)
            
            sub_window.setWidget(content)
            mdi_area.addSubWindow(sub_window)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            mdi_area.resize(width, height)
        
        return mdi_area


class SplitterControl(BaseControl):
    """分割器控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "Splitter"
        self._category = "高级"
        self._description = "用于分割窗口区域的控件"
        self._size = (300, 200)
        self._properties = {
            "orientation": Qt.Orientation.Horizontal,
            "childrenCollapsible": True,
            "opaqueResize": False,
            "handleWidth": 6,
            "widgets": []  # 子控件列表
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QSplitter
    
    def create_instance(self, parent=None) -> QWidget:
        splitter = QSplitter(parent)
        
        # 设置方向
        splitter.setOrientation(self.get_property("orientation", Qt.Orientation.Horizontal))
        
        # 设置子控件可折叠
        splitter.setChildrenCollapsible(self.get_property("childrenCollapsible", True))
        
        # 设置不透明调整大小
        splitter.setOpaqueResize(self.get_property("opaqueResize", False))
        
        # 设置手柄宽度
        splitter.setHandleWidth(self.get_property("handleWidth", 6))
        
        # 添加一些示例子控件
        widget1 = QWidget()
        widget1.setStyleSheet("background-color: lightblue;")
        layout1 = QVBoxLayout()
        layout1.addWidget(QLabel("左侧/上方区域"))
        widget1.setLayout(layout1)
        
        widget2 = QWidget()
        widget2.setStyleSheet("background-color: lightgreen;")
        layout2 = QVBoxLayout()
        layout2.addWidget(QLabel("右侧/下方区域"))
        widget2.setLayout(layout2)
        
        splitter.addWidget(widget1)
        splitter.addWidget(widget2)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            splitter.resize(width, height)
        
        return splitter


class ToolBoxControl(BaseControl):
    """工具箱控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ToolBox"
        self._category = "高级"
        self._description = "工具箱控件，类似选项卡但垂直排列"
        self._size = (300, 200)
        self._properties = {
            "currentIndex": 0,
            "tabs": ["页面1", "页面2", "页面3"]
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QToolBox
    
    def create_instance(self, parent=None) -> QWidget:
        toolbox = QToolBox(parent)
        
        # 设置当前索引
        toolbox.setCurrentIndex(self.get_property("currentIndex", 0))
        
        # 添加一些示例页面
        tabs = self.get_property("tabs", ["页面1", "页面2", "页面3"])
        
        for i, tab_name in enumerate(tabs):
            widget = QWidget()
            layout = QVBoxLayout()
            layout.addWidget(QLabel(f"这是 {tab_name} 的内容"))
            widget.setLayout(layout)
            
            toolbox.addItem(widget, tab_name)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            toolbox.resize(width, height)
        
        return toolbox


class WizardControl(BaseControl):
    """向导控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "Wizard"
        self._category = "高级"
        self._description = "向导对话框控件"
        self._size = (400, 300)
        self._properties = {
            "title": "向导",
            "options": QWizard.WizardOption.HaveHelpButton | QWizard.WizardOption.HaveFinishButtonOnEarlyPages,
            "wizardStyle": QWizard.WizardStyle.ModernStyle,
            "pages": ["欢迎", "设置", "完成"]
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("显示向导", parent)
        
        # 连接点击事件
        def show_wizard():
            wizard = QWizard(parent)
            wizard.setWindowTitle(self.get_property("title", "向导"))
            wizard.setOptions(self.get_property("options", QWizard.WizardOption.HaveHelpButton | QWizard.WizardOption.HaveFinishButtonOnEarlyPages))
            wizard.setWizardStyle(self.get_property("wizardStyle", QWizard.WizardStyle.ModernStyle))
            
            pages = self.get_property("pages", ["欢迎", "设置", "完成"])
            
            # 添加欢迎页面
            welcome_page = QWizardPage()
            welcome_page.setTitle(pages[0])
            welcome_layout = QVBoxLayout()
            welcome_layout.addWidget(QLabel("欢迎使用向导！"))
            welcome_page.setLayout(welcome_layout)
            wizard.addPage(welcome_page)
            
            # 添加设置页面
            settings_page = QWizardPage()
            settings_page.setTitle(pages[1])
            settings_layout = QVBoxLayout()
            settings_layout.addWidget(QLabel("请进行设置"))
            settings_page.setLayout(settings_layout)
            wizard.addPage(settings_page)
            
            # 添加完成页面
            finish_page = QWizardPage()
            finish_page.setTitle(pages[2])
            finish_layout = QVBoxLayout()
            finish_layout.addWidget(QLabel("向导完成！"))
            finish_page.setLayout(finish_layout)
            wizard.addPage(finish_page)
            
            wizard.exec()
        
        button.clicked.connect(show_wizard)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class UndoViewControl(BaseControl):
    """撤销视图控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "UndoView"
        self._category = "高级"
        self._description = "撤销/重做操作历史视图控件"
        self._size = (200, 150)
        self._properties = {
            "cleanIcon": None,
            "groupLabel": "组"
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("显示撤销视图", parent)
        
        # 连接点击事件
        def show_undo_view():
            from PySide6.QtWidgets import QDialog
            from PySide6.QtGui import QUndoStack, QUndoCommand
            
            dialog = QDialog(parent)
            dialog.setWindowTitle("撤销/重做历史")
            dialog.resize(300, 200)
            
            layout = QVBoxLayout()
            
            # 创建撤销栈
            undo_stack = QUndoStack()
            
            # 添加一些示例操作
            for i in range(5):
                undo_stack.push(QUndoCommand(f"操作 {i+1}"))
            
            # 创建撤销视图
            undo_view = QUndoView(undo_stack)
            undo_view.setCleanIcon(self.get_property("cleanIcon", None))
            undo_view.setGroupLabel(self.get_property("groupLabel", "组"))
            
            layout.addWidget(undo_view)
            
            # 添加操作按钮
            button_layout = QHBoxLayout()
            
            undo_button = QPushButton("撤销")
            undo_button.clicked.connect(undo_stack.undo)
            button_layout.addWidget(undo_button)
            
            redo_button = QPushButton("重做")
            redo_button.clicked.connect(undo_stack.redo)
            button_layout.addWidget(redo_button)
            
            layout.addLayout(button_layout)
            
            dialog.setLayout(layout)
            dialog.exec()
        
        button.clicked.connect(show_undo_view)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class WebViewControl(BaseControl):
    def __init__(self):
        super().__init__()
        self._name = "WebView"
        self._category = "高级"
        self._description = "网页视图控件"
        self._size = (400, 300)
        self._properties = {
            "url": "https://www.qt.io",
            "html": ""
        }

    def get_control_class(self) -> Type[QWidget]:
        return QWebEngineView

    def create_instance(self, parent=None) -> QWidget:
        view = QWebEngineView(parent)
        html = self.get_property("html", "")
        url = self.get_property("url", "")
        if html:
            view.setHtml(html)
        elif url:
            view.setUrl(QUrl(url))
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            view.resize(width, height)
        return view


# 注册所有高级控件
def register_advanced_controls():
    """注册所有高级控件到控件注册表"""
    control_registry.register_control(TableWidgetControl())
    control_registry.register_control(TreeWidgetControl())
    control_registry.register_control(CalendarWidgetControl())
    control_registry.register_control(DockWidgetControl())
    control_registry.register_control(MdiAreaControl())
    control_registry.register_control(SplitterControl())
    control_registry.register_control(ToolBoxControl())
    control_registry.register_control(WizardControl())
    control_registry.register_control(UndoViewControl())
    control_registry.register_control(WebViewControl())


# 自动注册
register_advanced_controls()