#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
控件数据结构和操作模块
定义控件的基本属性、方法和操作
"""

"""
控件数据结构和操作模块
定义控件的基本属性、方法和操作
用于在可视化编程工具中表示和管理GUI控件
"""

# 导入标准库模块
import json  # JSON数据处理，用于控件的序列化和反序列化
import uuid  # 通用唯一标识符生成，用于为每个控件生成唯一ID
# 导入类型提示
from typing import Dict, List, Any, Optional, Tuple
# 导入PySide6相关模块
from PySide6.QtWidgets import QWidget  # Qt基础窗口部件类
from PySide6.QtCore import QRect, QPoint  # Qt矩形和点坐标类


class Control:
    """
    控件基类
    定义控件的基本属性和操作，表示画布上的一个可视化控件
    每个控件都有位置、大小、属性等信息
    """
    
    def __init__(self, control_type: str, name: str = None, x: int = 50, y: int = 50, 
                 width: int = 100, height: int = 30, parent_id: str = None):
        """
        初始化控件
        
        Args:
            control_type: 控件类型（如"Button"、"Label"、"LineEdit"等）
            name: 控件名称，如果为None则自动生成（格式：控件类型_前8位ID）
            x: X坐标，控件在画布上的水平位置（像素）
            y: Y坐标，控件在画布上的垂直位置（像素）
            width: 控件宽度（像素）
            height: 控件高度（像素）
            parent_id: 父控件ID，用于容器控件（如GroupBox、TabWidget等）的子控件
        """
        # 生成唯一标识符（UUID的字符串形式）
        self.id = str(uuid.uuid4())
        # 保存控件类型
        self.control_type = control_type
        # 设置控件名称：如果未提供则自动生成（控件类型+ID前8位）
        self.name = name or f"{control_type}_{self.id[:8]}"
        # 设置控件位置坐标
        self.x = x
        self.y = y
        # 设置控件尺寸
        self.width = width
        self.height = height
        # 保存父控件ID（用于表示控件层级关系）
        self.parent_id = parent_id
        # 控件特定属性字典（如文本、颜色、字体等）
        self.properties = {}
        # 事件处理字典（存储事件处理函数或代码）
        self.events = {}
        # 样式属性字典（如边框、背景色等）
        self.style = {}
        # 子控件ID列表（用于容器控件，如GroupBox、TabWidget等）
        self.children = []
        # Z轴顺序：用于控制控件的显示层级（数值越大越靠前）
        self.z_order = 0
        
        # 设置控件类型的默认属性（如按钮的默认文本、标签的默认内容等）
        self._set_default_properties()
    
    def _set_default_properties(self):
        """
        设置控件类型的默认属性
        根据控件类型，为控件设置相应的默认属性值
        这些默认值会在控件创建时自动应用
        """
        # ========== 定义各控件类型的默认属性字典 ==========
        default_props = {
            # Label标签控件的默认属性
            "Label": {
                "text": "标签",  # 默认文本内容
                "alignment": "center"  # 默认文本对齐方式（居中）
            },
            # Button按钮控件的默认属性
            "Button": {
                "text": "按钮"  # 默认按钮文本
            },
            # LineEdit单行文本输入框的默认属性
            "LineEdit": {
                "placeholder_text": "请输入文本",  # 默认占位符文本（提示用户输入）
                "text": ""  # 默认文本内容（为空）
            },
            # TextEdit多行文本编辑器的默认属性
            "TextEdit": {
                "placeholder_text": "请输入多行文本",  # 默认占位符文本
                "plain_text": ""  # 默认纯文本内容（为空）
            },
            # CheckBox复选框控件的默认属性
            "CheckBox": {
                "text": "复选框",  # 默认复选框标签文本
                "checked": False  # 默认选中状态（未选中）
            },
            # RadioButton单选按钮控件的默认属性
            "RadioButton": {
                "text": "单选框",  # 默认单选按钮标签文本
                "checked": False  # 默认选中状态（未选中）
            },
            # ComboBox下拉选择框的默认属性
            "ComboBox": {
                "items": ["选项1", "选项2", "选项3"],  # 默认选项列表
                "current_index": 0  # 默认选中索引（第一个选项）
            },
            # Slider滑动条的默认属性
            "Slider": {
                "minimum": 0,  # 默认最小值
                "maximum": 100,  # 默认最大值
                "value": 50,  # 默认当前值（中间值）
                "orientation": "horizontal"  # 默认方向（水平）
            },
            # ProgressBar进度条的默认属性
            "ProgressBar": {
                "minimum": 0,  # 默认最小值
                "maximum": 100,  # 默认最大值
                "value": 30  # 默认当前值（30%）
            },
            # Frame框架控件的默认属性
            "Frame": {
                "frame_shape": "box",  # 默认框架形状（盒子形状）
                "frame_shadow": "raised"  # 默认框架阴影（凸起）
            },
            # GroupBox分组框控件的默认属性
            "GroupBox": {
                "title": "组框",  # 默认标题文本
                "flat": False  # 默认是否扁平（不平扁）
            },
            # TabWidget选项卡控件的默认属性
            "TabWidget": {
                "tab_position": "north"  # 默认选项卡位置（上方）
            },
            # ListWidget列表控件的默认属性
            "ListWidget": {
                "items": ["项目1", "项目2", "项目3"]  # 默认列表项
            },
            # SpinBox整数旋转框的默认属性
            "SpinBox": {
                "minimum": 0,  # 默认最小值
                "maximum": 99,  # 默认最大值
                "value": 0  # 默认当前值
            },
            # DoubleSpinBox浮点数旋转框的默认属性
            "DoubleSpinBox": {
                "minimum": 0.0,  # 默认最小值（浮点数）
                "maximum": 99.99,  # 默认最大值（浮点数）
                "value": 0.0,  # 默认当前值（浮点数）
                "decimals": 2  # 默认小数位数（2位）
            },
            # TreeWidget树形控件的默认属性
            "TreeWidget": {
                "header_labels": ["名称", "类型"],  # 默认表头标签
                "items": []  # 默认树项列表（为空）
            },
            # DateEdit日期编辑框的默认属性
            "DateEdit": {
                "calendar_popup": True,  # 默认是否显示日历弹出框（是）
                "display_format": "yyyy-MM-dd"  # 默认日期显示格式（年-月-日）
            },
            # TimeEdit时间编辑框的默认属性
            "TimeEdit": {
                "display_format": "HH:mm:ss"  # 默认时间显示格式（时:分:秒）
            },
            # DateTimeEdit日期时间编辑框的默认属性
            "DateTimeEdit": {
                "display_format": "yyyy-MM-dd HH:mm:ss"  # 默认日期时间显示格式
            }
        }
        
        # ========== 应用默认属性 ==========
        # 检查当前控件类型是否在默认属性字典中
        if self.control_type in default_props:
            # 如果控件类型存在默认属性，更新控件的属性字典
            # update方法会将默认属性添加到properties字典中（如果属性已存在则更新，不存在则添加）
            self.properties.update(default_props[self.control_type])
    
    def get_rect(self) -> QRect:
        """获取控件的矩形区域"""
        return QRect(self.x, self.y, self.width, self.height)
    
    def set_position(self, x: int, y: int):
        """设置控件位置"""
        self.x = x
        self.y = y
    
    def set_size(self, width: int, height: int):
        """设置控件大小"""
        self.width = max(width, 20)  # 最小宽度
        self.height = max(height, 20)  # 最小高度
    
    def move_by(self, dx: int, dy: int):
        """移动控件"""
        self.x += dx
        self.y += dy
    
    def resize_by(self, dw: int, dh: int):
        """调整控件大小"""
        self.set_size(self.width + dw, self.height + dh)
    
    def contains_point(self, x: int, y: int) -> bool:
        """检查点是否在控件内"""
        return (self.x <= x <= self.x + self.width and 
                self.y <= y <= self.y + self.height)
    
    def get_resize_handle(self, x: int, y: int) -> Optional[str]:
        """
        获取调整大小的句柄位置
        根据鼠标点击的坐标，判断是否点击在控件的调整大小句柄上
        句柄位于控件的边缘和角落，用于调整控件大小
        
        Args:
            x: 鼠标点击的X坐标（相对于画布）
            y: 鼠标点击的Y坐标（相对于画布）
            
        Returns:
            Optional[str]: 
                None: 不在任何句柄上
                "top_left": 左上角句柄
                "top": 上边缘句柄
                "top_right": 右上角句柄
                "left": 左边缘句柄
                "right": 右边缘句柄
                "bottom_left": 左下角句柄
                "bottom": 下边缘句柄
                "bottom_right": 右下角句柄
        """
        # ========== 定义句柄大小 ==========
        # 句柄大小：8像素，表示句柄的可点击区域半径（点击点到句柄中心的最大距离）
        handle_size = 8
        
        # ========== 检查左上角句柄 ==========
        # 检查点击位置是否在左上角句柄范围内
        # abs(x - self.x) <= handle_size: X坐标到控件左上角X的距离是否在句柄大小内
        # abs(y - self.y) <= handle_size: Y坐标到控件左上角Y的距离是否在句柄大小内
        if (abs(x - self.x) <= handle_size and 
            abs(y - self.y) <= handle_size):
            # 如果两个条件都满足，说明点击在左上角句柄上
            return "top_left"
        
        # ========== 检查上边缘句柄 ==========
        # 检查点击位置是否在上边缘句柄范围内（不包括角落）
        # self.x + handle_size <= x <= self.x + self.width - handle_size: X坐标在左边缘和右边缘之间
        # abs(y - self.y) <= handle_size: Y坐标在上边缘的句柄范围内
        if (self.x + handle_size <= x <= self.x + self.width - handle_size and 
            abs(y - self.y) <= handle_size):
            # 如果满足条件，说明点击在上边缘句柄上
            return "top"
        
        # ========== 检查右上角句柄 ==========
        # 检查点击位置是否在右上角句柄范围内
        # abs(x - (self.x + self.width)) <= handle_size: X坐标到控件右上角X的距离
        # abs(y - self.y) <= handle_size: Y坐标到控件右上角Y的距离
        if (abs(x - (self.x + self.width)) <= handle_size and 
            abs(y - self.y) <= handle_size):
            # 如果满足条件，说明点击在右上角句柄上
            return "top_right"
        
        # ========== 检查左边缘句柄 ==========
        # 检查点击位置是否在左边缘句柄范围内（不包括角落）
        # abs(x - self.x) <= handle_size: X坐标在左边缘的句柄范围内
        # self.y + handle_size <= y <= self.y + self.height - handle_size: Y坐标在上边缘和下边缘之间
        if (abs(x - self.x) <= handle_size and 
            self.y + handle_size <= y <= self.y + self.height - handle_size):
            # 如果满足条件，说明点击在左边缘句柄上
            return "left"
        
        # ========== 检查右边缘句柄 ==========
        # 检查点击位置是否在右边缘句柄范围内（不包括角落）
        # abs(x - (self.x + self.width)) <= handle_size: X坐标在右边缘的句柄范围内
        # self.y + handle_size <= y <= self.y + self.height - handle_size: Y坐标在上边缘和下边缘之间
        if (abs(x - (self.x + self.width)) <= handle_size and 
            self.y + handle_size <= y <= self.y + self.height - handle_size):
            # 如果满足条件，说明点击在右边缘句柄上
            return "right"
        
        # ========== 检查左下角句柄 ==========
        # 检查点击位置是否在左下角句柄范围内
        # abs(x - self.x) <= handle_size: X坐标到控件左下角X的距离
        # abs(y - (self.y + self.height)) <= handle_size: Y坐标到控件左下角Y的距离
        if (abs(x - self.x) <= handle_size and 
            abs(y - (self.y + self.height)) <= handle_size):
            # 如果满足条件，说明点击在左下角句柄上
            return "bottom_left"
        
        # ========== 检查下边缘句柄 ==========
        # 检查点击位置是否在下边缘句柄范围内（不包括角落）
        # self.x + handle_size <= x <= self.x + self.width - handle_size: X坐标在左边缘和右边缘之间
        # abs(y - (self.y + self.height)) <= handle_size: Y坐标在下边缘的句柄范围内
        if (self.x + handle_size <= x <= self.x + self.width - handle_size and 
            abs(y - (self.y + self.height)) <= handle_size):
            # 如果满足条件，说明点击在下边缘句柄上
            return "bottom"
        
        # ========== 检查右下角句柄 ==========
        if (abs(x - (self.x + self.width)) <= handle_size and 
            abs(y - (self.y + self.height)) <= handle_size):
            return "bottom_right"
        
        return None
    
    def to_dict(self) -> Dict[str, Any]:
        """将控件转换为字典格式，用于保存和加载"""
        return {
            "id": self.id,
            "control_type": self.control_type,
            "name": self.name,
            "x": self.x,
            "y": self.y,
            "width": self.width,
            "height": self.height,
            "parent_id": self.parent_id,
            "properties": self.properties,
            "events": self.events,
            "style": self.style,
            "children": self.children,
            "z_order": self.z_order
        }
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'Control':
        """从字典创建控件对象"""
        control = cls(
            control_type=data["control_type"],
            name=data.get("name"),
            x=data.get("x", 50),
            y=data.get("y", 50),
            width=data.get("width", 100),
            height=data.get("height", 30),
            parent_id=data.get("parent_id")
        )
        
        control.id = data.get("id", control.id)
        control.properties = data.get("properties", {})
        control.events = data.get("events", {})
        control.style = data.get("style", {})
        control.children = data.get("children", [])
        control.z_order = data.get("z_order", 0)
        
        return control
    
    def clone(self) -> 'Control':
        """克隆控件"""
        cloned = Control(
            control_type=self.control_type,
            name=f"{self.name}_copy",
            x=self.x + 20,  # 偏移位置
            y=self.y + 20,
            width=self.width,
            height=self.height,
            parent_id=self.parent_id
        )
        
        cloned.properties = self.properties.copy()
        cloned.events = self.events.copy()
        cloned.style = self.style.copy()
        cloned.children = self.children.copy()
        
        return cloned


class ControlManager:
    """控件管理器，负责管理所有控件的创建、删除和操作"""
    
    def __init__(self):
        self.controls: Dict[str, Control] = {}
        self.selected_control_id: Optional[str] = None
        self.selected_control_ids: List[str] = []
        self.next_z_order = 1  # 下一个Z轴顺序
        self.type_counts: Dict[str, int] = {}
    
    def add_control(self, control_type: str, x: int = 50, y: int = 50, 
                   width: int = None, height: int = None, parent_id: str = None, 
                   base_name: str = None, control_id: str = None) -> Control:
        """
        添加新控件
        
        Args:
            control_type: 控件类型
            x: X坐标
            y: Y坐标
            width: 宽度，如果为None则使用默认值
            height: 高度，如果为None则使用默认值
            parent_id: 父控件ID
            base_name: 基础名称，用于复制时生成序号递增的名称
            control_id: 指定控件ID，如果为None则自动生成
            
        Returns:
            新创建的控件对象
        """
        # 设置默认尺寸
        default_sizes = {
            "Label": (100, 30),
            "Button": (80, 30),
            "LineEdit": (150, 30),
            "TextEdit": (200, 100),
            "CheckBox": (120, 30),
            "RadioButton": (120, 30),
            "ComboBox": (150, 30),
            "Slider": (150, 30),
            "ProgressBar": (150, 20),
            "Frame": (200, 150),
            "GroupBox": (200, 150),
            "TabWidget": (300, 200),
            "ListWidget": (150, 100),
            "SpinBox": (100, 30),
            "DoubleSpinBox": (100, 30),
            "TreeWidget": (200, 150),
            "DateEdit": (140, 30),
            "TimeEdit": (140, 30),
            "DateTimeEdit": (180, 30)
        }
        
        if width is None or height is None:
            default_width, default_height = default_sizes.get(control_type, (100, 30))
            width = width if width is not None else default_width
            height = height if height is not None else default_height
        
        
        # 创建控件
        control = Control(
            control_type=control_type,
            x=x, y=y,
            width=width, height=height,
            parent_id=parent_id
        )
        
        if control_id:
            control.id = control_id
        
        # 使用智能命名系统
        try:
            if base_name is not None:
                # 如果提供了基础名称（例如复制时），使用它作为基础
                display_name = base_name
            else:
                # 否则使用控件类型的显示名称
                from .control_library_pyside import ControlLibrary
                info = ControlLibrary.get_control_info(control_type)
                display_name = info.get("display_name", control_type)
            
            # 使用智能命名生成下一个可用名称
            next_name = self._generate_next_name(display_name, control_type)
            control.name = next_name
            control.properties["name"] = next_name
            
            # 更新计数器（为了向后兼容）
            self.type_counts[control_type] = self.type_counts.get(control_type, 0) + 1
            
        except Exception:
            # 如果无法获取智能名称，使用原始逻辑
            display_name = control_type
            count = self.type_counts.get(control_type, 0) + 1
            self.type_counts[control_type] = count
            readable_name = f"{display_name}{count}"
            control.name = readable_name
            control.properties["name"] = readable_name
        
        # 设置Z轴顺序
        control.z_order = self.next_z_order
        self.next_z_order += 1
        
        # 添加到管理器
        self.controls[control.id] = control
        
        # 如果是容器控件，添加到父控件的子控件列表
        if parent_id and parent_id in self.controls:
            self.controls[parent_id].children.append(control.id)
        
        # 选中新创建的控件
        self.select_control(control.id)
        
        return control
    
    def delete_control(self, control_id: str) -> bool:
        """
        删除控件
        
        Args:
            control_id: 要删除的控件ID
            
        Returns:
            是否成功删除
        """
        if control_id not in self.controls:
            return False
        
        # 获取要删除的控件
        control = self.controls[control_id]
        
        # 递归删除所有子控件
        for child_id in control.children[:]:  # 使用副本避免修改列表时的问题
            self.delete_control(child_id)
        
        # 从父控件的子控件列表中移除
        if control.parent_id and control.parent_id in self.controls:
            parent = self.controls[control.parent_id]
            if control_id in parent.children:
                parent.children.remove(control_id)
        
        # 删除控件
        del self.controls[control_id]
        
        # 如果删除的是当前选中的控件，取消选中
        if self.selected_control_id == control_id:
            self.selected_control_id = None
        
        return True
    
    def delete_multiple_controls(self, control_ids: List[str]) -> int:
        """
        批量删除控件
        
        Args:
            control_ids: 要删除的控件ID列表
            
        Returns:
            成功删除的控件数量
        """
        deleted_count = 0
        for control_id in control_ids:
            if self.delete_control(control_id):
                deleted_count += 1
        
        # 如果有选中的控件被删除，清空选择
        if deleted_count > 0:
            self.clear_selection()
        
        return deleted_count
    
    def clone_multiple_controls(self, control_ids: List[str]) -> Dict[str, str]:
        """
        批量克隆控件
        
        Args:
            control_ids: 要克隆的控件ID列表
            
        Returns:
            原ID到新ID的映射字典
        """
        id_mapping = {}
        
        # 按Z轴顺序排序，确保子控件在父控件之后克隆
        sorted_controls = sorted(
            [self.controls[cid] for cid in control_ids if cid in self.controls],
            key=lambda c: c.z_order
        )
        
        for control in sorted_controls:
            # 克隆控件，偏移位置
            new_x = control.x + 20
            new_y = control.y + 20
            new_control = self.add_control(
                control_type=control.control_type,
                x=new_x,
                y=new_y,
                width=control.width,
                height=control.height,
                parent_id=control.parent_id
            )
            
            # 复制属性（除了name，因为add_control已经使用了智能命名）
            new_properties = control.properties.copy()
            new_properties["name"] = new_control.name  # 保持智能生成的名称
            new_control.properties = new_properties
            
            id_mapping[control.id] = new_control.id
        
        return id_mapping
    
    def paste_multiple_controls(self, id_mapping: Dict[str, str]) -> List[str]:
        """
        批量粘贴控件（基于克隆映射）
        
        Args:
            id_mapping: 原ID到新ID的映射字典
            
        Returns:
            新创建的控件ID列表
        """
        new_ids = []
        
        for original_id, new_id in id_mapping.items():
            if new_id in self.controls:
                new_ids.append(new_id)
        
        # 选中所有粘贴的控件
        if new_ids:
            self.set_selection(new_ids)
        
        return new_ids
    
    def delete_multiple_controls(self, control_ids: List[str]) -> int:
        """
        批量删除控件
        
        Args:
            control_ids: 要删除的控件ID列表
            
        Returns:
            成功删除的控件数量
        """
        deleted_count = 0
        for control_id in control_ids:
            if self.delete_control(control_id):
                deleted_count += 1
        
        # 如果有选中的控件被删除，清空选择
        if deleted_count > 0:
            self.clear_selection()
        
        return deleted_count
    
    def clone_multiple_controls(self, control_ids: List[str]) -> Dict[str, str]:
        """
        批量克隆控件
        
        Args:
            control_ids: 要克隆的控件ID列表
            
        Returns:
            原ID到新ID的映射字典
        """
        id_mapping = {}
        
        # 按Z轴顺序排序，确保子控件在父控件之后克隆
        sorted_controls = sorted(
            [self.controls[cid] for cid in control_ids if cid in self.controls],
            key=lambda c: c.z_order
        )
        
        for control in sorted_controls:
            # 克隆控件，偏移位置
            new_x = control.x + 20
            new_y = control.y + 20
            new_control = self.add_control(
                control_type=control.control_type,
                x=new_x,
                y=new_y,
                width=control.width,
                height=control.height,
                parent_id=control.parent_id
            )
            
            # 复制属性
            new_control.properties.update(control.properties)
            # 使用智能命名方法
            new_control.name = self._generate_next_name(control.name, control.control_type)
            new_control.properties["name"] = new_control.name
            
            id_mapping[control.id] = new_control.id
        
        return id_mapping
    
    def paste_multiple_controls(self, id_mapping: Dict[str, str]) -> List[str]:
        """
        批量粘贴控件（基于克隆映射）
        
        Args:
            id_mapping: 原ID到新ID的映射字典
            
        Returns:
            新创建的控件ID列表
        """
        new_ids = []
        
        for original_id, new_id in id_mapping.items():
            if new_id in self.controls:
                new_ids.append(new_id)
        
        # 选中所有粘贴的控件
        if new_ids:
            self.set_selection(new_ids)
        
        return new_ids
    
    def copy_selected_controls(self) -> Optional[Dict[str, str]]:
        """
        复制选中的控件
        
        Returns:
            复制映射字典，如果无选中控件则返回None
        """
        selected_ids = self.get_selected_ids()
        if not selected_ids:
            return None
        
        # 克隆选中的控件
        mapping = self.clone_multiple_controls(selected_ids)
        
        # 保存复制状态
        self._copied_controls_mapping = mapping
        
        return mapping
    
    def get_control(self, control_id: str) -> Optional[Control]:
        """获取控件对象"""
        return self.controls.get(control_id)

    def get_all_control_ids(self) -> List[str]:
        """获取所有控件ID列表"""
        return list(self.controls.keys())
    
    def get_control_at(self, x: int, y: int) -> Optional[Control]:
        """
        获取指定位置的控件（从上到下搜索）
        
        Args:
            x: X坐标
            y: Y坐标
            
        Returns:
            指定位置的控件，如果没有则返回None
        """
        # 按Z轴顺序从高到低排序
        sorted_controls = sorted(
            self.controls.values(),
            key=lambda c: c.z_order,
            reverse=True
        )
        
        for control in sorted_controls:
            if control.contains_point(x, y):
                return control
        
        return None
    
    def select_control(self, control_id: Optional[str]):
        if control_id is None or control_id in self.controls:
            self.selected_control_id = control_id
            self.selected_control_ids = [] if control_id is None else [control_id]
    
    def get_selected_control(self) -> Optional[Control]:
        if self.selected_control_ids:
            cid = self.selected_control_ids[-1]
            return self.controls.get(cid)
        return None

    def add_to_selection(self, control_id: str):
        if control_id in self.controls:
            if control_id not in self.selected_control_ids:
                self.selected_control_ids.append(control_id)
            self.selected_control_id = control_id

    def remove_from_selection(self, control_id: str):
        if control_id in self.selected_control_ids:
            self.selected_control_ids = [cid for cid in self.selected_control_ids if cid != control_id]
            if self.selected_control_id == control_id:
                self.selected_control_id = self.selected_control_ids[-1] if self.selected_control_ids else None

    def clear_selection(self):
        self.selected_control_ids = []
        self.selected_control_id = None
    
    def get_selected_ids(self) -> List[str]:
        return list(self.selected_control_ids)

    def set_selection(self, ids: List[str]):
        self.selected_control_ids = [cid for cid in ids if cid in self.controls]
        self.selected_control_id = self.selected_control_ids[-1] if self.selected_control_ids else None
    
    def move_control(self, control_id: str, dx: int, dy: int) -> bool:
        """
        移动控件
        
        Args:
            control_id: 控件ID
            dx: X方向偏移
            dy: Y方向偏移
            
        Returns:
            是否成功移动
        """
        if control_id not in self.controls:
            return False
        
        control = self.controls[control_id]
        control.move_by(dx, dy)
        
        # 如果有父控件，检查是否仍在父控件范围内
        if control.parent_id and control.parent_id in self.controls:
            parent = self.controls[control.parent_id]
            # 确保子控件不会移出父控件
            control.x = max(0, min(control.x, parent.width - control.width))
            control.y = max(0, min(control.y, parent.height - control.height))
        
        return True
    
    def resize_control(self, control_id: str, handle: str, dx: int, dy: int) -> bool:
        """
        调整控件大小
        
        Args:
            control_id: 控件ID
            handle: 调整句柄位置
            dx: X方向偏移
            dy: Y方向偏移
            
        Returns:
            是否成功调整
        """
        if control_id not in self.controls:
            return False
        
        control = self.controls[control_id]
        min_width, min_height = 20, 20  # 最小尺寸
        
        if handle == "top_left":
            new_x = control.x + dx
            new_y = control.y + dy
            new_width = control.width - dx
            new_height = control.height - dy
            
            if new_width >= min_width and new_height >= min_height:
                control.x = new_x
                control.y = new_y
                control.width = new_width
                control.height = new_height
                
        elif handle == "top":
            new_y = control.y + dy
            new_height = control.height - dy
            
            if new_height >= min_height:
                control.y = new_y
                control.height = new_height
                
        elif handle == "top_right":
            new_y = control.y + dy
            new_width = control.width + dx
            new_height = control.height - dy
            
            if new_width >= min_width and new_height >= min_height:
                control.y = new_y
                control.width = new_width
                control.height = new_height
                
        elif handle == "left":
            new_x = control.x + dx
            new_width = control.width - dx
            
            if new_width >= min_width:
                control.x = new_x
                control.width = new_width
                
        elif handle == "right":
            new_width = control.width + dx
            if new_width >= min_width:
                control.width = new_width
                
        elif handle == "bottom_left":
            new_x = control.x + dx
            new_width = control.width - dx
            new_height = control.height + dy
            
            if new_width >= min_width and new_height >= min_height:
                control.x = new_x
                control.width = new_width
                control.height = new_height
                
        elif handle == "bottom":
            new_height = control.height + dy
            if new_height >= min_height:
                control.height = new_height
                
        elif handle == "bottom_right":
            new_width = control.width + dx
            new_height = control.height + dy
            
            if new_width >= min_width and new_height >= min_height:
                control.width = new_width
                control.height = new_height
        
        # 如果有父控件，确保不会超出父控件范围
        if control.parent_id and control.parent_id in self.controls:
            parent = self.controls[control.parent_id]
            control.x = max(0, min(control.x, parent.width - control.width))
            control.y = max(0, min(control.y, parent.height - control.height))
            control.width = min(control.width, parent.width - control.x)
            control.height = min(control.height, parent.height - control.y)
        
        return True
    
    def bring_to_front(self, control_id: str) -> bool:
        """
        将控件置于顶层
        
        Args:
            control_id: 控件ID
            
        Returns:
            是否成功
        """
        if control_id not in self.controls:
            return False
        
        control = self.controls[control_id]
        control.z_order = self.next_z_order
        self.next_z_order += 1
        
        return True
    
    def send_to_back(self, control_id: str) -> bool:
        """
        将控件置于底层
        
        Args:
            control_id: 控件ID
            
        Returns:
            是否成功
        """
        if control_id not in self.controls:
            return False
        
        # 找到当前最小的Z轴顺序
        min_z_order = min(c.z_order for c in self.controls.values())
        
        control = self.controls[control_id]
        if control.z_order > min_z_order:
            control.z_order = min_z_order - 1
        
        return True
    
    def clear_all(self):
        """清除所有控件"""
        self.controls.clear()
        self.selected_control_id = None
        self.next_z_order = 1
    
    def to_dict(self) -> Dict[str, Any]:
        """将管理器转换为字典格式，用于保存"""
        return {
            "controls": {cid: control.to_dict() for cid, control in self.controls.items()},
            "selected_control_id": self.selected_control_id,
            "next_z_order": self.next_z_order
        }
    
    def from_dict(self, data: Dict[str, Any]):
        """从字典加载管理器状态"""
        self.controls.clear()
        
        # 加载所有控件
        for cid, control_data in data.get("controls", {}).items():
            control = Control.from_dict(control_data)
            self.controls[cid] = control
        
        # 恢复选中状态
        self.selected_control_id = data.get("selected_control_id")
        
        # 恢复Z轴顺序计数器
        max_z_order = 0
        for control in self.controls.values():
            if control.z_order > max_z_order:
                max_z_order = control.z_order
        self.next_z_order = max_z_order + 1
    
    def delete_multiple_controls(self, control_ids: List[str]) -> int:
        """
        批量删除控件
        
        Args:
            control_ids: 要删除的控件ID列表
            
        Returns:
            成功删除的控件数量
        """
        deleted_count = 0
        for control_id in control_ids:
            if self.delete_control(control_id):
                deleted_count += 1
        
        # 如果有选中的控件被删除，清空选择
        if deleted_count > 0:
            self.clear_selection()
        
        return deleted_count
    
    def clone_multiple_controls(self, control_ids: List[str]) -> Dict[str, str]:
        """
        批量克隆控件
        
        Args:
            control_ids: 要克隆的控件ID列表
            
        Returns:
            原ID到新ID的映射字典
        """
        id_mapping = {}
        
        # 按Z轴顺序排序，确保子控件在父控件之后克隆
        sorted_controls = sorted(
            [self.controls[cid] for cid in control_ids if cid in self.controls],
            key=lambda c: c.z_order
        )
        
        for control in sorted_controls:
            # 克隆控件，偏移位置
            new_x = control.x + 20
            new_y = control.y + 20
            new_control = self.add_control(
                control_type=control.control_type,
                x=new_x,
                y=new_y,
                width=control.width,
                height=control.height,
                parent_id=control.parent_id
            )
            
            # 复制属性（除了name，因为add_control已经使用了智能命名）
            new_properties = control.properties.copy()
            new_properties["name"] = new_control.name
            new_control.properties = new_properties
            
            id_mapping[control.id] = new_control.id
        
        return id_mapping
    
    def copy_selected_controls(self) -> Optional[Dict[str, str]]:
        selected_ids = self.get_selected_ids()
        if not selected_ids:
            return None
        # 复制操作仅保存当前选中控件到剪贴板，不在画布上创建克隆
        self.clipboard_controls = [self.controls[cid] for cid in selected_ids if cid in self.controls]
        return {cid: cid for cid in selected_ids}
    
    def paste_controls(self) -> List[str]:
        """
        粘贴剪贴板中的控件
        
        Returns:
            新创建的控件ID列表
        """
        if not hasattr(self, 'clipboard_controls') or not self.clipboard_controls:
            return []
        
        new_ids = []
        
        # 复制控件并应用偏移
        for control in self.clipboard_controls:
            # 直接从原始控件创建新控件，确保使用正确的名称作为基础
            new_control = self.add_control(
                control_type=control.control_type,
                x=control.x + 20,  # 偏移20像素
                y=control.y + 20,
                width=control.width,
                height=control.height,
                parent_id=control.parent_id,
                base_name=control.name  # 重要：使用原始控件名称作为基础
            )
            
            # 复制属性（保留add_control生成的name）
            for key, value in control.properties.items():
                if key != "name":  # 不要覆盖生成的名称
                    if isinstance(value, dict):
                        new_control.properties[key] = value.copy()
                    else:
                        new_control.properties[key] = value
            
            new_ids.append(new_control.id)
        
        # 选中新创建的控件
        if new_ids:
            self.set_selection(new_ids)
        
        return new_ids
    
    def has_copied_controls(self) -> bool:
        """检查是否有复制的控件"""
        return hasattr(self, 'clipboard_controls') and bool(self.clipboard_controls)
    
    def _generate_next_name(self, base_name: str, control_type: str) -> str:
        """
        生成下一个可用的名称
        例如：按钮1 → 按钮2 → 按钮3
             我的按钮10 → 我的按钮11
             标签5 → 标签6
        
        Args:
            base_name: 基础名称
            control_type: 控件类型
            
        Returns:
            下一个可用的名称
        """
        import re
        
        # 简单直接的命名逻辑
        # 1. 检查基础名称是否包含数字后缀
        match = re.match(r'^(.+?)(\d+)$', base_name)
        
        if match:
            # 有数字后缀的情况 (如: 按钮1, 我的按钮10)
            prefix = match.group(1)
            current_num = int(match.group(2))
            
            # 找出所有以相同前缀开头且格式为"前缀+数字"的名称
            # 并找出最大的数字
            max_num = current_num
            
            for name in [c.name for c in self.controls.values()]:
                # 只匹配完全符合"前缀+数字"格式的名称
                name_match = re.fullmatch(r'^' + re.escape(prefix) + r'(\d+)$', name)
                if name_match:
                    num = int(name_match.group(1))
                    if num > max_num:
                        max_num = num
            
            # 生成下一个序号
            return f"{prefix}{max_num + 1}"
        else:
            # 没有数字后缀的情况 (如: 按钮, 自定义按钮)
            # 检查是否存在完全相同的名称
            if any(c.name == base_name for c in self.controls.values()):
                # 查找以基础名称开头且后面跟着数字的控件
                max_num = 0
                
                for name in [c.name for c in self.controls.values()]:
                    # 只匹配完全符合"基础名称+数字"格式的名称
                    name_match = re.fullmatch(r'^' + re.escape(base_name) + r'(\d+)$', name)
                    if name_match:
                        num = int(name_match.group(1))
                        if num > max_num:
                            max_num = num
                
                return f"{base_name}{max_num + 1}"
            else:
                # 如果不存在相同名称，检查是否有该名称+数字的控件
                max_num = 0
                
                for name in [c.name for c in self.controls.values()]:
                    name_match = re.fullmatch(r'^' + re.escape(base_name) + r'(\d+)$', name)
                    if name_match:
                        num = int(name_match.group(1))
                        if num > max_num:
                            max_num = num
                
                # 如果有该名称+数字的控件，返回下一个序号
                if max_num > 0:
                    return f"{base_name}{max_num + 1}"
                else:
                    # 否则返回基础名称+1
                    return f"{base_name}1"
