#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
控件面板UI组件
包含控件列表和拖拽功能
"""

"""
控件面板UI组件模块
包含控件列表和拖拽功能
提供可视化的控件库界面，用户可以从这里拖拽控件到设计画布
"""

# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QWidget,  # 基础窗口部件
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QLabel,  # 标签
    QTabWidget  # 选项卡控件，用于分类显示不同类型的控件
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    Signal,  # 信号定义
    QMimeData  # MIME数据，用于拖放操作
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QDrag,  # 拖放操作
    QPixmap  # 像素图
)


class ControlPaletteWidget(QWidget):
    """
    控件面板组件
    显示可用的控件列表，支持按分类组织
    用户可以从这里拖拽控件到设计画布
    """
    
    # ========== 信号定义 ==========
    # 控件被拖拽时发出信号，参数为控件类型
    control_dragged = Signal(str)
    
    def __init__(self, parent=None):
        """
        初始化控件面板
        
        Args:
            parent: 父窗口部件
        """
        # 调用父类构造函数
        super().__init__(parent)
        # 初始化用户界面
        self._init_ui()
    
    def _init_ui(self):
        """
        初始化用户界面
        创建标题、选项卡等UI组件
        """
        # 创建垂直布局作为主布局
        layout = QVBoxLayout(self)
        # 设置布局边距：左、上、右、下各5像素
        layout.setContentsMargins(5, 5, 5, 5)
        
        # ========== 创建标题 ==========
        self.title_label = QLabel("控件库")
        # 设置标题样式：加粗、字号14px、内边距5px（颜色由主题控制）
        self.title_label.setStyleSheet("font-weight: bold; font-size: 14px; padding: 5px;")
        # 将标题添加到布局
        layout.addWidget(self.title_label)
        
        # ========== 创建选项卡控件 ==========
        # 创建选项卡控件，用于分类显示不同类型的控件
        self.tab_widget = QTabWidget()
        # 将选项卡控件添加到布局
        layout.addWidget(self.tab_widget)
        
        # 添加基础控件选项卡（如Label、Button等）
        self._add_basic_controls_tab()
        
        # 添加容器控件选项卡（如Frame、GroupBox、TabWidget等）
        self._add_container_controls_tab()
        
        # 添加输入控件选项卡（如LineEdit、TextEdit、CheckBox等）
        self._add_input_controls_tab()
        
        # 添加显示控件选项卡（如ProgressBar、LCDNumber等）
        self._add_display_controls_tab()

        # 添加数据控件选项卡（如ListWidget、TableWidget、TreeWidget等）
        self._add_data_controls_tab()
    
    def _add_basic_controls_tab(self):
        """添加基础控件选项卡"""
        tab = QWidget()
        layout = QVBoxLayout(tab)
        
        # 基础控件列表
        basic_controls = [
            ("Label", "标签"),
            ("Button", "按钮")
        ]
        
        # ========== 遍历基础控件列表并创建控件项 ==========
        # 遍历基础控件列表中的每个控件（控件类型和显示名称）
        for control_type, control_name in basic_controls:
            # 为每个控件创建控件项组件（可拖拽的控件图标）
            control_widget = self._create_control_widget(control_type, control_name)
            # 将控件项添加到布局中
            layout.addWidget(control_widget)
        
        # ========== 添加弹性空间 ==========
        # 在布局底部添加弹性空间，使控件项靠顶部对齐
        layout.addStretch()
        
        # ========== 添加选项卡 ==========
        # 将创建的选项卡添加到选项卡控件中，标签为"基础"
        self.tab_widget.addTab(tab, "基础")
    
    def _add_container_controls_tab(self):
        """添加容器控件选项卡"""
        tab = QWidget()
        layout = QVBoxLayout(tab)
        
        # 容器控件列表
        container_controls = [
            ("Frame", "框架"),
            ("GroupBox", "组框"),
            ("TabWidget", "标签页"),
            ("StackedWidget", "堆叠页")
        ]
        
        # ========== 遍历容器控件列表并创建控件项 ==========
        # 遍历容器控件列表中的每个控件（控件类型和显示名称）
        for control_type, control_name in container_controls:
            # 为每个控件创建控件项组件（可拖拽的控件图标）
            control_widget = self._create_control_widget(control_type, control_name)
            # 将控件项添加到布局中
            layout.addWidget(control_widget)
        
        # ========== 添加弹性空间 ==========
        # 在布局底部添加弹性空间，使控件项靠顶部对齐
        layout.addStretch()
        
        # ========== 添加选项卡 ==========
        # 将创建的选项卡添加到选项卡控件中，标签为"容器"
        self.tab_widget.addTab(tab, "容器")
    
    def _add_input_controls_tab(self):
        """添加输入控件选项卡"""
        tab = QWidget()
        layout = QVBoxLayout(tab)
        
        # 输入控件列表
        input_controls = [
            ("LineEdit", "输入框"),
            ("TextEdit", "文本框"),
            ("CheckBox", "复选框"),
            ("RadioButton", "单选框"),
            ("ComboBox", "下拉框"),
            ("SpinBox", "整数输入"),
            ("DoubleSpinBox", "浮点输入"),
            ("DateEdit", "日期选择"),
            ("TimeEdit", "时间选择"),
            ("DateTimeEdit", "日期时间")
        ]
        
        # ========== 遍历输入控件列表并创建控件项 ==========
        # 遍历输入控件列表中的每个控件（控件类型和显示名称）
        for control_type, control_name in input_controls:
            # 为每个控件创建控件项组件（可拖拽的控件图标）
            control_widget = self._create_control_widget(control_type, control_name)
            # 将控件项添加到布局中
            layout.addWidget(control_widget)
        
        # ========== 添加弹性空间 ==========
        # 在布局底部添加弹性空间，使控件项靠顶部对齐
        layout.addStretch()
        
        # ========== 添加选项卡 ==========
        # 将创建的选项卡添加到选项卡控件中，标签为"输入"
        self.tab_widget.addTab(tab, "输入")
    
    def _add_display_controls_tab(self):
        """添加显示控件选项卡"""
        tab = QWidget()
        layout = QVBoxLayout(tab)
        
        # 显示控件列表
        display_controls = [
            ("Slider", "滑块"),
            ("ProgressBar", "进度条"),
            ("ListWidget", "列表"),
            ("ImageBox", "图片框")
        ]
        
        # ========== 遍历显示控件列表并创建控件项 ==========
        # 遍历显示控件列表中的每个控件（控件类型和显示名称）
        for control_type, control_name in display_controls:
            # 为每个控件创建控件项组件（可拖拽的控件图标）
            control_widget = self._create_control_widget(control_type, control_name)
            # 将控件项添加到布局中
            layout.addWidget(control_widget)
        
        # ========== 添加弹性空间 ==========
        # 在布局底部添加弹性空间，使控件项靠顶部对齐
        layout.addStretch()
        
        # ========== 添加选项卡 ==========
        # 将创建的选项卡添加到选项卡控件中，标签为"显示"
        self.tab_widget.addTab(tab, "显示")

    def _add_data_controls_tab(self):
        """添加数据控件选项卡"""
        tab = QWidget()
        layout = QVBoxLayout(tab)
        data_controls = [
            ("TreeWidget", "树形视图"),
            ("TableWidget", "表格")
        ]
        # ========== 遍历数据控件列表并创建控件项 ==========
        # 遍历数据控件列表中的每个控件（控件类型和显示名称）
        for control_type, control_name in data_controls:
            # 为每个控件创建控件项组件（可拖拽的控件图标）
            control_widget = self._create_control_widget(control_type, control_name)
            # 将控件项添加到布局中
            layout.addWidget(control_widget)
        
        # ========== 添加弹性空间 ==========
        # 在布局底部添加弹性空间，使控件项靠顶部对齐
        layout.addStretch()
        
        # ========== 添加选项卡 ==========
        # 将创建的选项卡添加到选项卡控件中，标签为"数据"
        self.tab_widget.addTab(tab, "数据")
    
    def _create_control_widget(self, control_type: str, control_name: str) -> QWidget:
        """
        创建控件项窗口部件
        为指定的控件类型创建一个可拖拽的控件项，包含图标和名称
        
        Args:
            control_type: 控件类型（如"Label"、"Button"等）
            control_name: 控件的显示名称（中文名称）
            
        Returns:
            QWidget: 创建的控件项窗口部件
        """
        # ========== 创建控件项窗口部件 ==========
        # 创建基础窗口部件作为控件项的容器
        widget = QWidget()
        widget.setObjectName("ControlPaletteItem")  # 设置唯一标识，用于主题样式
        # 设置控件项的固定大小：宽度120像素，高度40像素
        widget.setFixedSize(120, 40)
        
        # ========== 创建布局 ==========
        # 创建水平布局，用于排列图标和名称（从左到右）
        layout = QHBoxLayout(widget)
        # 设置布局边距：左、上、右、下各5像素
        layout.setContentsMargins(5, 5, 5, 5)
        
        # ========== 创建控件图标标签 ==========
        # 创建标签控件用于显示控件图标（emoji图标）
        icon_label = QLabel()
        # 设置图标标签的固定大小：24x24像素（正方形）
        icon_label.setFixedSize(24, 24)
        # 设置图标标签的对齐方式为居中对齐（图标在标签中心显示）
        icon_label.setAlignment(Qt.AlignCenter)
        
        # ========== 根据控件类型设置不同的图标 ==========
        # 根据控件类型，为控件项设置对应的emoji图标，用于在控件面板中显示
        
        # ========== 基础控件图标 ==========
        # 检查是否为Label（标签）控件
        if control_type == "Label":
            # 设置标签图标为📝（记事本emoji，表示文本标签）
            icon_label.setText("📝")
        # 检查是否为Button（按钮）控件
        elif control_type == "Button":
            # 设置按钮图标为🔘（圆形按钮emoji）
            icon_label.setText("🔘")
        
        # ========== 容器控件图标 ==========
        # 检查是否为Frame（框架）控件
        elif control_type == "Frame":
            # 设置框架图标为🖼️（画框emoji，表示容器框架）
            icon_label.setText("🖼️")
        # 检查是否为GroupBox（组框）控件
        elif control_type == "GroupBox":
            # 设置组框图标为📦（包裹emoji，表示分组容器）
            icon_label.setText("📦")
        # 检查是否为TabWidget（标签页）控件
        elif control_type == "TabWidget":
            # 设置标签页图标为📑（书签emoji，表示多页标签）
            icon_label.setText("📑")
        
        # ========== 输入控件图标 ==========
        # 检查是否为LineEdit（单行文本输入框）控件
        elif control_type == "LineEdit":
            # 设置输入框图标为📝（记事本emoji）
            icon_label.setText("📝")
        # 检查是否为TextEdit（多行文本编辑器）控件
        elif control_type == "TextEdit":
            # 设置文本编辑器图标为📄（文档emoji）
            icon_label.setText("📄")
        # 检查是否为CheckBox（复选框）控件
        elif control_type == "CheckBox":
            # 设置复选框图标为☑️（带勾选框emoji）
            icon_label.setText("☑️")
        # 检查是否为RadioButton（单选按钮）控件
        elif control_type == "RadioButton":
            # 设置单选按钮图标为🔘（圆形按钮emoji）
            icon_label.setText("🔘")
        # 检查是否为ComboBox（下拉选择框）控件
        elif control_type == "ComboBox":
            # 设置下拉框图标为📋（剪贴板emoji）
            icon_label.setText("📋")
        # 检查是否为SpinBox（整数旋转框）控件
        elif control_type == "SpinBox":
            # 设置整数输入框图标为🔢（数字emoji）
            icon_label.setText("🔢")
        # 检查是否为DoubleSpinBox（浮点数旋转框）控件
        elif control_type == "DoubleSpinBox":
            # 设置浮点数输入框图标为🔢（数字emoji）
            icon_label.setText("🔢")
        
        # ========== 显示控件图标 ==========
        # 检查是否为Slider（滑动条）控件
        elif control_type == "Slider":
            # 设置滑动条图标为🎚️（滑块emoji）
            icon_label.setText("🎚️")
        # 检查是否为ProgressBar（进度条）控件
        elif control_type == "ProgressBar":
            # 设置进度条图标为📊（条形图emoji）
            icon_label.setText("📊")
        # 检查是否为ListWidget（列表控件）控件
        elif control_type == "ListWidget":
            # 设置列表控件图标为📝（记事本emoji）
            icon_label.setText("📝")
        
        # ========== 数据控件图标 ==========
        # 检查是否为TreeWidget（树形控件）控件
        elif control_type == "TreeWidget":
            # 设置树形控件图标为🌳（树emoji）
            icon_label.setText("🌳")
        # 检查是否为TableWidget（表格控件）控件
        elif control_type == "TableWidget":
            # 设置表格控件图标为📊（图表emoji）
            icon_label.setText("📊")
        # 检查是否为ImageBox（图片框）控件
        elif control_type == "ImageBox":
            # 设置图片框图标为🖼️（画框emoji）
            icon_label.setText("🖼️")
        
        # ========== 日期时间控件图标 ==========
        # 检查是否为DateEdit（日期编辑框）控件
        elif control_type == "DateEdit":
            # 设置日期编辑框图标为📅（日历emoji）
            icon_label.setText("📅")
        # 检查是否为TimeEdit（时间编辑框）控件
        elif control_type == "TimeEdit":
            # 设置时间编辑框图标为⏰（闹钟emoji）
            icon_label.setText("⏰")
        # 检查是否为DateTimeEdit（日期时间编辑框）控件
        elif control_type == "DateTimeEdit":
            # 设置日期时间编辑框图标为📆（日历emoji）
            icon_label.setText("📆")
        # ========== 处理其他控件类型 ==========
        # 检查是否为StackedWidget（堆叠控件）
        elif control_type == "StackedWidget":
            # 设置堆叠控件图标为🪜（梯子emoji，表示多层级）
            icon_label.setText("🪜")
        else:
            # ========== 默认图标 ==========
            # 如果控件类型不匹配任何已知类型，设置默认图标为❓（问号emoji）
            icon_label.setText("❓")
        
        # ========== 将图标添加到布局 ==========
        # 将图标标签添加到水平布局中（图标在左侧）
        layout.addWidget(icon_label)
        
        # ========== 创建控件名称标签 ==========
        # 创建控件名称标签，显示控件的显示名称（中文名称）
        name_label = QLabel(control_name)
        # 将名称标签添加到布局中（名称在图标右侧）
        layout.addWidget(name_label)
        
        # ========== 设置控件项样式 ==========
        # 不设置固定样式，让组件项使用组件库的主题样式
        # 组件项的样式由组件库的样式表控制，会根据主题自动变化
        
        # ========== 设置控件属性 ==========
        # 设置控件属性：关闭时自动删除（用于内存管理）
        widget.setAttribute(Qt.WA_DeleteOnClose)
        
        # ========== 存储控件类型 ==========
        # 将控件类型作为属性存储到控件对象中，用于拖拽时识别控件类型
        widget.control_type = control_type
        
        # ========== 绑定鼠标事件处理 ==========
        # 为控件项添加鼠标按下事件处理函数，用于启动拖拽操作
        # 使用lambda函数将事件和控件对象传递给_start_drag方法
        widget.mousePressEvent = lambda event: self._start_drag(event, widget)
        
        # ========== 返回创建的控件项 ==========
        # 返回创建的控件项窗口部件
        return widget
    
    def _start_drag(self, event, widget):
        """
        开始拖拽操作
        当用户在控件面板上按下鼠标左键时调用，启动拖拽流程
        
        Args:
            event: 鼠标事件对象，包含鼠标位置、按钮等信息
            widget: 被拖拽的控件项窗口部件
        """
        # ========== 检查拖拽条件 ==========
        # 检查是否为鼠标左键按下，且控件项包含控件类型属性
        if event.button() == Qt.LeftButton and hasattr(widget, 'control_type'):
            # ========== 创建拖拽对象 ==========
            # 创建拖拽操作对象（self为拖拽的源窗口部件）
            drag = QDrag(self)
            
            # ========== 创建MIME数据 ==========
            # 创建MIME数据对象，用于在拖放操作中传递数据
            mime_data = QMimeData()
            # 设置MIME数据为控件类型字符串（设计画布可以通过此字符串识别控件类型）
            mime_data.setText(widget.control_type)
            # 将MIME数据设置到拖拽对象
            drag.setMimeData(mime_data)
            
            # ========== 创建拖拽图标 ==========
            # 创建像素图，大小与控件项相同
            pixmap = QPixmap(widget.size())
            # 使用透明色填充像素图（初始为透明）
            pixmap.fill(Qt.transparent)
            # 将控件项的渲染内容绘制到像素图上（创建控件项的截图）
            widget.render(pixmap)
            # 将像素图设置为拖拽时的视觉图标（用户拖拽时看到的图像）
            drag.setPixmap(pixmap)
            # 设置拖拽热点的位置（鼠标相对于图像的位置）
            drag.setHotSpot(event.pos())
            
            # ========== 执行拖拽操作 ==========
            # 执行拖拽操作，开始拖放流程
            # 该方法会阻塞直到拖放操作完成（用户放下或取消）
            drag.exec()
