#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
控件库管理模块
定义所有可用控件的类型、属性和图标
"""

"""
控件库管理模块
定义所有可用控件的类型、属性和图标
提供控件的元数据信息，包括显示名称、默认属性、默认大小等
"""

# 导入类型提示
from typing import Dict, List, Tuple, Any
# 导入PySide6界面组件（所有支持的控件类型）
from PySide6.QtWidgets import (
    QLabel,  # 标签
    QPushButton,  # 按钮
    QLineEdit,  # 单行文本输入框
    QTextEdit,  # 富文本编辑器
    QCheckBox,  # 复选框
    QRadioButton,  # 单选按钮
    QComboBox,  # 下拉选择框
    QSlider,  # 滑动条
    QProgressBar,  # 进度条
    QFrame,  # 框架
    QGroupBox,  # 分组框
    QTabWidget,  # 选项卡控件
    QStackedWidget,  # 堆叠控件
    QListWidget,  # 列表控件
    QTreeWidget,  # 树形控件
    QTableWidget,  # 表格控件
    QSpinBox,  # 整数旋转框
    QDoubleSpinBox,  # 浮点数旋转框
    QDateEdit,  # 日期编辑框
    QTimeEdit,  # 时间编辑框
    QDateTimeEdit,  # 日期时间编辑框
    QDial,  # 旋钮
    QLCDNumber,  # LCD数字显示
    QScrollArea,  # 滚动区域
    QToolButton,  # 工具按钮
    QCommandLinkButton,  # 命令链接按钮
    QCalendarWidget,  # 日历控件
    QKeySequenceEdit  # 快捷键编辑框
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QIcon,  # 图标
    QPixmap  # 像素图
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QSize  # 大小对象
)


class ControlLibrary:
    """
    控件库类
    管理所有可用控件的类型和属性
    提供控件的元数据信息，包括显示名称、分类、默认属性、默认大小等
    """
    
    # ========== 控件分类定义 ==========
    # 定义控件的分类字典，用于在控件面板中组织控件
    CATEGORIES = {
        "basic": "基础控件",  # 基础控件：Label、Button等
        "container": "容器",  # 容器控件：Frame、GroupBox、TabWidget等
        "input": "输入",  # 输入控件：LineEdit、TextEdit、CheckBox等
        "display": "显示",  # 显示控件：ProgressBar、LCDNumber等
        "data": "数据"  # 数据控件：ListWidget、TableWidget、TreeWidget等
    }
    
    # 控件定义
    CONTROLS = {
        # 基础控件
        "Label": {
            "display_name": "标签",
            "category": "basic",
            "class_name": "QLabel",
            "icon": None,  # 可以设置为图标路径
            "default_properties": {
                "text": "标签",
                "alignment": "center"
            },
            "default_size": (100, 30),
            "resizable": True,
            "description": "用于显示静态文本"
        },
        
        "Button": {
            "display_name": "按钮",
            "category": "basic",
            "class_name": "QPushButton",
            "icon": None,
            "default_properties": {
                "text": "按钮"
            },
            "default_size": (80, 30),
            "resizable": True,
            "description": "点击执行操作"
        },

        "ToolButton": {
            "display_name": "工具按钮",
            "category": "basic",
            "class_name": "QToolButton",
            "icon": None,
            "default_properties": {
                "text": "工具按钮"
            },
            "default_size": (80, 30),
            "resizable": True,
            "description": "工具栏按钮"
        },

        "CommandLinkButton": {
            "display_name": "命令链接按钮",
            "category": "basic",
            "class_name": "QCommandLinkButton",
            "icon": None,
            "default_properties": {
                "text": "命令链接",
                "description": "描述文字"
            },
            "default_size": (180, 60),
            "resizable": True,
            "description": "Vista风格命令按钮"
        },
        
        # 容器控件
        "Frame": {
            "display_name": "框架",
            "category": "container",
            "class_name": "QFrame",
            "icon": None,
            "default_properties": {
                "frame_shape": "Box",
                "frame_shadow": "Raised"
            },
            "default_size": (200, 150),
            "resizable": True,
            "description": "用于组织其他控件的容器",
            "can_contain": True
        },
        
        "GroupBox": {
            "display_name": "组框",
            "category": "container",
            "class_name": "QGroupBox",
            "icon": None,
            "default_properties": {
                "title": "组框"
            },
            "default_size": (200, 150),
            "resizable": True,
            "description": "带标题的容器控件",
            "can_contain": True
        },
        
        "TabWidget": {
            "display_name": "标签页",
            "category": "container",
            "class_name": "QTabWidget",
            "icon": None,
            "default_properties": {
                "tab_position": "North",
                "tabs": ["页面1", "页面2"]
            },
            "default_size": (300, 200),
            "resizable": True,
            "description": "多页容器控件",
            "can_contain": True
        },
        "StackedWidget": {
            "display_name": "堆叠页",
            "category": "container",
            "class_name": "QStackedWidget",
            "icon": None,
            "default_properties": {
                "pages": ["页面1", "页面2"],
                "current_index": 0
            },
            "default_size": (300, 200),
            "resizable": True,
            "description": "多页面切换容器",
            "can_contain": True
        },

        "ScrollArea": {
            "display_name": "滚动区域",
            "category": "container",
            "class_name": "QScrollArea",
            "icon": None,
            "default_properties": {
                "widget_resizable": True
            },
            "default_size": (200, 150),
            "resizable": True,
            "description": "可滚动的容器区域",
            "can_contain": True
        },
        
        # 输入控件
        "LineEdit": {
            "display_name": "输入框",
            "category": "input",
            "class_name": "QLineEdit",
            "icon": None,
            "default_properties": {
                "placeholder_text": "请输入文本",
                "text": ""
            },
            "default_size": (150, 30),
            "resizable": True,
            "description": "单行文本输入"
        },
        
        "TextEdit": {
            "display_name": "文本框",
            "category": "input",
            "class_name": "QTextEdit",
            "icon": None,
            "default_properties": {
                "placeholder_text": "请输入多行文本",
                "plain_text": ""
            },
            "default_size": (200, 100),
            "resizable": True,
            "description": "多行文本输入"
        },
        
        "CheckBox": {
            "display_name": "复选框",
            "category": "input",
            "class_name": "QCheckBox",
            "icon": None,
            "default_properties": {
                "text": "复选框",
                "checked": False
            },
            "default_size": (120, 30),
            "resizable": True,
            "description": "是/否选择"
        },
        
        "RadioButton": {
            "display_name": "单选框",
            "category": "input",
            "class_name": "QRadioButton",
            "icon": None,
            "default_properties": {
                "text": "单选框",
                "checked": False
            },
            "default_size": (120, 30),
            "resizable": True,
            "description": "互斥选择"
        },
        
        "ComboBox": {
            "display_name": "下拉框",
            "category": "input",
            "class_name": "QComboBox",
            "icon": None,
            "default_properties": {
                "items": ["选项1", "选项2", "选项3"],
                "current_index": 0
            },
            "default_size": (150, 30),
            "resizable": True,
            "description": "下拉选项选择"
        },
        
        "SpinBox": {
            "display_name": "整数输入框",
            "category": "input",
            "class_name": "QSpinBox",
            "icon": None,
            "default_properties": {
                "minimum": 0,
                "maximum": 99,
                "value": 0
            },
            "default_size": (100, 30),
            "resizable": True,
            "description": "整数数值输入"
        },
        
        "DoubleSpinBox": {
            "display_name": "小数输入框",
            "category": "input",
            "class_name": "QDoubleSpinBox",
            "icon": None,
            "default_properties": {
                "minimum": 0.0,
                "maximum": 99.99,
                "value": 0.0,
                "decimals": 2
            },
            "default_size": (100, 30),
            "resizable": True,
            "description": "小数数值输入"
        },

        "KeySequenceEdit": {
            "display_name": "快捷键输入",
            "category": "input",
            "class_name": "QKeySequenceEdit",
            "icon": None,
            "default_properties": {
                "key_sequence": ""
            },
            "default_size": (120, 30),
            "resizable": True,
            "description": "快捷键序列输入"
        },
        
        # 显示控件
        "Slider": {
            "display_name": "滑块",
            "category": "display",
            "class_name": "QSlider",
            "icon": None,
            "default_properties": {
                "minimum": 0,
                "maximum": 100,
                "value": 50,
                "orientation": "Horizontal"
            },
            "default_size": (150, 30),
            "resizable": True,
            "description": "数值范围选择"
        },
        
        "ProgressBar": {
            "display_name": "进度条",
            "category": "display",
            "class_name": "QProgressBar",
            "icon": None,
            "default_properties": {
                "minimum": 0,
                "maximum": 100,
                "value": 30
            },
            "default_size": (150, 20),
            "resizable": True,
            "description": "显示进度"
        },

        "Dial": {
            "display_name": "旋钮",
            "category": "display",
            "class_name": "QDial",
            "icon": None,
            "default_properties": {
                "minimum": 0,
                "maximum": 100,
                "value": 0,
                "notches_visible": True
            },
            "default_size": (100, 100),
            "resizable": True,
            "description": "圆形数值调节器"
        },

        "LCDNumber": {
            "display_name": "LCD数字",
            "category": "display",
            "class_name": "QLCDNumber",
            "icon": None,
            "default_properties": {
                "value": 0,
                "digit_count": 5,
                "segment_style": "Flat"
            },
            "default_size": (100, 40),
            "resizable": True,
            "description": "LCD风格数字显示"
        },

        "CalendarWidget": {
            "display_name": "日历",
            "category": "display",
            "class_name": "QCalendarWidget",
            "icon": None,
            "default_properties": {
                "grid_visible": True
            },
            "default_size": (300, 200),
            "resizable": True,
            "description": "日历控件"
        },
        
        "ListWidget": {
            "display_name": "列表框",
            "category": "display",
            "class_name": "QListWidget",
            "icon": None,
            "default_properties": {
                "items": ["项目1", "项目2", "项目3"]
            },
            "default_size": (150, 100),
            "resizable": True,
            "description": "显示项目列表"
        },
        "ImageBox": {
            "display_name": "图片框",
            "category": "display",
            "class_name": "QLabel",
            "icon": None,
            "default_properties": {
                "image_path": "",
                "scaled": True
            },
            "default_size": (160, 120),
            "resizable": True,
            "description": "显示图片"
        },
        # 数据控件
        "TreeWidget": {
            "display_name": "树形视图",
            "category": "data",
            "class_name": "QTreeWidget",
            "icon": None,
            "default_properties": {
                "header_labels": ["名称", "类型"],
                "items": []
            },
            "default_size": (200, 150),
            "resizable": True,
            "description": "层级数据展示"
        },
        "TableWidget": {
            "display_name": "表格",
            "category": "data",
            "class_name": "QTableWidget",
            "icon": None,
            "default_properties": {
                "row_count": 3,
                "column_count": 3,
                "headers": ["列1", "列2", "列3"]
            },
            "default_size": (240, 160),
            "resizable": True,
            "description": "二维表格数据展示"
        },
        "DateEdit": {
            "display_name": "日期选择",
            "category": "input",
            "class_name": "QDateEdit",
            "icon": None,
            "default_properties": {
                "calendar_popup": True,
                "display_format": "yyyy-MM-dd"
            },
            "default_size": (140, 30),
            "resizable": True,
            "description": "选择日期"
        },
        "TimeEdit": {
            "display_name": "时间选择",
            "category": "input",
            "class_name": "QTimeEdit",
            "icon": None,
            "default_properties": {
                "display_format": "HH:mm:ss"
            },
            "default_size": (140, 30),
            "resizable": True,
            "description": "选择时间"
        },
        "DateTimeEdit": {
            "display_name": "日期时间",
            "category": "input",
            "class_name": "QDateTimeEdit",
            "icon": None,
            "default_properties": {
                "display_format": "yyyy-MM-dd HH:mm:ss"
            },
            "default_size": (180, 30),
            "resizable": True,
            "description": "选择日期时间"
        }
    }
    
    # 控件类映射
    CONTROL_CLASSES = {
        "QLabel": QLabel,
        "QPushButton": QPushButton,
        "QLineEdit": QLineEdit,
        "QTextEdit": QTextEdit,
        "QCheckBox": QCheckBox,
        "QRadioButton": QRadioButton,
        "QComboBox": QComboBox,
        "QSlider": QSlider,
        "QProgressBar": QProgressBar,
        "QFrame": QFrame,
        "QGroupBox": QGroupBox,
        "QTabWidget": QTabWidget,
        "QStackedWidget": QStackedWidget,
        "QListWidget": QListWidget,
        "QTableWidget": QTableWidget,
        "QTreeWidget": QTreeWidget,
        "QSpinBox": QSpinBox,
        "QDoubleSpinBox": QDoubleSpinBox,
        "QDateEdit": QDateEdit,
        "QTimeEdit": QTimeEdit,
        "QDateTimeEdit": QDateTimeEdit,
        "QDial": QDial,
        "QLCDNumber": QLCDNumber,
        "QScrollArea": QScrollArea,
        "QToolButton": QToolButton,
        "QCommandLinkButton": QCommandLinkButton,
        "QCalendarWidget": QCalendarWidget,
        "QKeySequenceEdit": QKeySequenceEdit
    }
    
    @classmethod
    def get_control_info(cls, control_type: str) -> Dict[str, Any]:
        """
        获取控件信息
        
        Args:
            control_type: 控件类型
            
        Returns:
            控件信息字典，如果控件不存在则返回空字典
        """
        # 首先从静态定义中查找
        if control_type in cls.CONTROLS:
            return cls.CONTROLS[control_type]
        
        # 如果静态定义中没有，尝试从动态加载的控件注册表中查找
        try:
            from .control_base import control_registry
            dynamic_control = control_registry.get_control(control_type)
            if dynamic_control:
                # 将动态控件转换为字典格式
                return {
                    "display_name": dynamic_control.get_control_name(),
                    "category": dynamic_control.get_category(),
                    "class_name": dynamic_control.get_control_class().__name__ if dynamic_control.get_control_class() else None,
                    "icon": None,
                    "default_properties": dynamic_control.get_default_properties(),
                    "default_size": dynamic_control.get_default_size(),
                    "resizable": True,
                    "description": dynamic_control.get_description()
                }
        except Exception:
            pass
        
        return {}
    
    @classmethod
    def get_control_class(cls, control_type: str):
        """
        获取控件类
        
        Args:
            control_type: 控件类型
            
        Returns:
            控件类，如果控件不存在则返回None
        """
        # ========== 获取控件信息 ==========
        # 调用get_control_info方法获取控件的完整信息字典
        control_info = cls.get_control_info(control_type)
        
        # ========== 提取类名 ==========
        # 从控件信息字典中获取类名（如"QLabel"、"QPushButton"等）
        class_name = control_info.get("class_name")
        
        # ========== 检查类名是否存在 ==========
        # 检查是否成功获取到类名（不为None且不为空）
        if class_name:
            # 如果类名存在，从控件类字典中获取对应的PySide6控件类
            # CONTROL_CLASSES字典存储了类名到类对象的映射
            widget_class = cls.CONTROL_CLASSES.get(class_name)
            if widget_class:
                return widget_class
            
            # 如果静态字典中没有，尝试从动态加载的控件注册表中获取
            try:
                from .control_base import control_registry
                dynamic_control = control_registry.get_control(control_type)
                if dynamic_control:
                    return dynamic_control.get_control_class()
            except Exception:
                pass
        
        # ========== 返回None ==========
        # 如果未找到类名，返回None（控件类型不存在或信息不完整）
        return None
    
    @classmethod
    def get_controls_by_category(cls, category: str) -> List[Tuple[str, Dict[str, Any]]]:
        """
        获取指定分类的所有控件
        
        Args:
            category: 分类名称
            
        Returns:
            控件类型和信息的元组列表
        """
        result = []
        
        # ========== 从静态定义中获取 ==========
        # 遍历所有控件字典中的控件类型和信息
        # 筛选出category属性匹配指定分类的控件
        for control_type, control_info in cls.CONTROLS.items():
            if control_info.get("category") == category:
                result.append((control_type, control_info))
        
        # ========== 从动态加载的控件中获取 ==========
        try:
            from .control_base import control_registry
            dynamic_controls = control_registry.get_controls_by_category(category)
            for dynamic_control in dynamic_controls:
                control_name = dynamic_control.get_control_name()
                # 避免重复添加（如果静态定义中已有）
                if control_name not in [ct[0] for ct in result]:
                    control_info = {
                        "display_name": dynamic_control.get_control_name(),
                        "category": dynamic_control.get_category(),
                        "class_name": dynamic_control.get_control_class().__name__ if dynamic_control.get_control_class() else None,
                        "icon": None,
                        "default_properties": dynamic_control.get_default_properties(),
                        "default_size": dynamic_control.get_default_size(),
                        "resizable": True,
                        "description": dynamic_control.get_description()
                    }
                    result.append((control_name, control_info))
        except Exception:
            pass
        
        return result
    
    @classmethod
    def get_all_categories(cls) -> List[Tuple[str, str]]:
        """
        获取所有分类
        
        Returns:
            分类ID和显示名称的元组列表
        """
        # ========== 使用列表推导式获取所有分类 ==========
        # 遍历分类字典中的所有分类ID和显示名称
        # 返回分类ID和显示名称的元组列表
        return [
            (category_id, display_name)  # 元组：分类ID、显示名称
            for category_id, display_name in cls.CATEGORIES.items()  # 遍历所有分类
        ]
    
    @classmethod
    def is_container(cls, control_type: str) -> bool:
        """
        检查控件是否为容器
        
        Args:
            control_type: 控件类型
            
        Returns:
            是否为容器控件
        """
        # ========== 获取控件信息 ==========
        # 调用get_control_info方法获取控件的完整信息字典
        control_info = cls.get_control_info(control_type)
        
        # ========== 检查容器属性 ==========
        # 从控件信息中获取can_contain属性，如果不存在则默认为False
        # can_contain为True表示该控件可以作为容器，包含其他控件
        return control_info.get("can_contain", False)
    
    @classmethod
    def get_default_size(cls, control_type: str) -> Tuple[int, int]:
        """
        获取控件的默认大小
        
        Args:
            control_type: 控件类型
            
        Returns:
            默认宽度和高度的元组
        """
        # ========== 获取控件信息 ==========
        # 调用get_control_info方法获取控件的完整信息字典
        control_info = cls.get_control_info(control_type)
        
        # ========== 获取默认大小 ==========
        # 从控件信息中获取default_size属性，如果不存在则使用默认值(100, 30)
        # default_size是一个元组，包含宽度和高度
        default_size = control_info.get("default_size", (100, 30))
        # 如果是列表，转换为元组
        if isinstance(default_size, list):
            default_size = tuple(default_size)
        return default_size
    
    @classmethod
    def get_default_properties(cls, control_type: str) -> Dict[str, Any]:
        """
        获取控件的默认属性
        
        Args:
            control_type: 控件类型
            
        Returns:
            默认属性字典
        """
        # ========== 获取控件信息 ==========
        # 调用get_control_info方法获取控件的完整信息字典
        control_info = cls.get_control_info(control_type)
        
        # ========== 获取默认属性 ==========
        # 从控件信息中获取default_properties属性，如果不存在则返回空字典
        # default_properties是一个字典，包含控件的默认属性键值对
        return control_info.get("default_properties", {})
    
    @classmethod
    def create_widget(cls, control_type: str, parent=None):
        """
        创建控件实例
        
        Args:
            control_type: 控件类型
            parent: 父控件
            
        Returns:
            控件实例，如果创建失败则返回None
        """
        # ========== 获取控件类 ==========
        # 调用get_control_class方法获取控件的PySide6类对象
        widget_class = cls.get_control_class(control_type)
        
        # ========== 检查控件类是否存在 ==========
        # 检查是否成功获取到控件类（不为None）
        if widget_class:
            # 如果控件类存在，尝试创建控件实例
            try:
                # ========== 创建控件实例 ==========
                # 调用控件类的构造函数，传入父控件参数，创建控件实例
                widget = widget_class(parent)
                
                # ========== 应用默认属性 ==========
                # 获取控件的默认属性字典
                default_properties = cls.get_default_properties(control_type)
                # 遍历所有默认属性
                for prop_name, prop_value in default_properties.items():
                    # 检查控件实例是否有该属性（使用hasattr检查）
                    if hasattr(widget, prop_name):
                        # 如果实例有该属性，使用setattr动态设置属性值
                        setattr(widget, prop_name, prop_value)
                
                # ========== 返回创建的控件实例 ==========
                # 如果创建和属性设置成功，返回控件实例
                return widget
            except Exception as e:
                # ========== 处理创建异常 ==========
                # 如果创建过程中发生任何异常，打印错误信息
                print(f"创建控件失败: {e}")
                # 返回None表示创建失败
                return None
        
        # ========== 返回None ==========
        # 如果控件类不存在，返回None（控件类型不支持）
        return None
