#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
控件基类和接口定义
定义了所有可视化编程控件的基础接口和通用功能
"""

# 导入抽象基类相关模块
from abc import ABC, abstractmethod, ABCMeta  # ABC用于定义抽象基类，abstractmethod用于定义抽象方法
from typing import Dict, Any, List, Type, Optional  # 类型提示
# 导入PySide6相关模块
from PySide6.QtWidgets import QWidget  # Qt基础窗口部件
from PySide6.QtCore import QObject, Signal  # Qt对象基类和信号定义
from PySide6.QtGui import QIcon  # Qt图标类


class ControlInterface(ABC):
    """控件接口，定义所有控件必须实现的方法"""
    
    @abstractmethod
    def get_control_name(self) -> str:
        """获取控件名称"""
        pass
    
    @abstractmethod
    def get_control_class(self) -> Type[QWidget]:
        """获取控件类"""
        pass
    
    @abstractmethod
    def get_default_properties(self) -> Dict[str, Any]:
        """获取控件默认属性"""
        pass
    
    @abstractmethod
    def get_default_size(self) -> tuple:
        """获取控件默认大小 (width, height)"""
        pass
    
    @abstractmethod
    def get_category(self) -> str:
        """获取控件所属分类"""
        pass
    
    @abstractmethod
    def get_description(self) -> str:
        """获取控件描述"""
        pass
    
    @abstractmethod
    def get_icon(self) -> Optional[QIcon]:
        """获取控件图标"""
        pass
    
    @abstractmethod
    def create_instance(self, parent=None) -> QWidget:
        """创建控件实例"""
        pass


class MetaQObjectABC(type(QObject), ABCMeta):
    pass


class BaseControl(QObject, ControlInterface, metaclass=MetaQObjectABC):
    """控件基类，提供通用功能实现"""
    
    # 定义信号
    property_changed = Signal(str, object)  # 属性变化信号 (property_name, new_value)
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self._name = ""
        self._properties = {}
        self._size = (100, 30)  # 默认大小
        self._category = "基础"
        self._description = ""
        self._icon = None
    
    def get_control_name(self) -> str:
        """获取控件名称"""
        return self._name
    
    def get_control_class(self) -> Type[QWidget]:
        """获取控件类"""
        return QWidget
    
    def get_default_properties(self) -> Dict[str, Any]:
        """获取控件默认属性"""
        return self._properties.copy()
    
    def get_default_size(self) -> tuple:
        """获取控件默认大小"""
        return self._size
    
    def get_category(self) -> str:
        """获取控件所属分类"""
        return self._category
    
    def get_description(self) -> str:
        """获取控件描述"""
        return self._description
    
    def get_icon(self) -> Optional[QIcon]:
        """获取控件图标"""
        return self._icon
    
    def set_property(self, name: str, value: Any) -> None:
        """
        设置控件属性
        如果属性已存在或允许动态属性，则设置属性值并发出属性变化信号
        
        Args:
            name: 属性名称
            value: 属性值
        """
        # ========== 检查是否可以设置属性 ==========
        # 检查属性是否已存在于属性字典中，或者是否允许动态添加新属性
        if name in self._properties or self._allow_dynamic_property():
            # 如果可以设置，执行以下操作
            # 将属性值保存到属性字典中（如果属性不存在则添加，存在则更新）
            self._properties[name] = value
            # 发出属性变化信号，通知其他组件属性已改变
            # 参数：属性名称、新属性值
            self.property_changed.emit(name, value)
    
    def get_property(self, name: str, default=None) -> Any:
        """获取属性"""
        return self._properties.get(name, default)
    
    def _allow_dynamic_property(self) -> bool:
        """是否允许动态属性"""
        return True
    
    def create_instance(self, parent=None) -> QWidget:
        """创建控件实例"""
        control_class = self.get_control_class()
        instance = control_class(parent)
        
        # ========== 应用默认属性 ==========
        # 获取控件的默认属性字典
        properties = self.get_default_properties()
        # 遍历所有默认属性
        for prop_name, prop_value in properties.items():
            # 检查控件实例是否有该属性（使用hasattr检查）
            if hasattr(instance, prop_name):
                # 如果实例有该属性，使用setattr动态设置属性值
                setattr(instance, prop_name, prop_value)
        
        # ========== 设置默认大小 ==========
        # 获取控件的默认大小（宽度和高度）
        width, height = self.get_default_size()
        # 检查宽度和高度是否都大于0（有效尺寸）
        if width > 0 and height > 0:
            # 如果尺寸有效，调用resize方法设置控件大小
            instance.resize(width, height)
        
        return instance


class ControlCategory:
    """控件分类类"""
    
    def __init__(self, name: str, display_name: str, icon: Optional[QIcon] = None):
        self.name = name
        self.display_name = display_name
        self.icon = icon
        self.controls: List[ControlInterface] = []
    
    def add_control(self, control: ControlInterface) -> None:
        """添加控件到分类"""
        if control not in self.controls:
            self.controls.append(control)
    
    def remove_control(self, control: ControlInterface) -> None:
        """从分类中移除控件"""
        if control in self.controls:
            self.controls.remove(control)
    
    def get_controls(self) -> List[ControlInterface]:
        """获取分类中的所有控件"""
        return self.controls.copy()
    
    def get_control_by_name(self, name: str) -> Optional[ControlInterface]:
        """根据名称获取控件"""
        for control in self.controls:
            if control.get_control_name() == name:
                return control
        return None


class ControlRegistry:
    """控件注册表，管理所有控件和分类"""
    
    def __init__(self):
        self._categories: Dict[str, ControlCategory] = {}
        self._controls: Dict[str, ControlInterface] = {}
    
    def register_category(self, category: ControlCategory) -> None:
        """注册控件分类"""
        if category.name not in self._categories:
            self._categories[category.name] = category
    
    def register_control(self, control: ControlInterface) -> None:
        """注册控件"""
        category_name = control.get_category()
        
        # 如果分类不存在，创建新分类
        if category_name not in self._categories:
            self.register_category(ControlCategory(category_name, category_name))
        
        # 将控件添加到分类
        self._categories[category_name].add_control(control)
        
        # 添加到控件字典
        self._controls[control.get_control_name()] = control
    
    def get_categories(self) -> List[ControlCategory]:
        """获取所有分类"""
        return list(self._categories.values())
    
    def get_category(self, name: str) -> Optional[ControlCategory]:
        """获取指定分类"""
        return self._categories.get(name)
    
    def get_all_controls(self) -> List[ControlInterface]:
        """获取所有控件"""
        return list(self._controls.values())
    
    def get_control(self, name: str) -> Optional[ControlInterface]:
        """根据名称获取控件"""
        return self._controls.get(name)
    
    def get_controls_by_category(self, category_name: str) -> List[ControlInterface]:
        """获取指定分类的所有控件"""
        category = self.get_category(category_name)
        if category:
            return category.get_controls()
        return []


# 全局控件注册表实例
control_registry = ControlRegistry()