#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
代码解析器模块
用于解析PySide6代码并提取控件信息，实现"代码->画布"的反向工程
"""

# 导入Python抽象语法树（AST）模块，用于解析Python代码
import ast
# 导入正则表达式模块，用于文本模式匹配
import re
# 导入类型提示
from typing import Dict, List, Any, Optional, Tuple
# 导入控件库，用于获取控件类型信息
from .control_library_pyside import ControlLibrary

class CodeParser:
    """
    代码解析器类
    用于解析PySide6代码，提取窗口属性、控件信息、信号连接等信息
    实现从代码到画布的反向工程
    """
    
    def __init__(self):
        """
        初始化代码解析器
        """
        # 存储解析出的控件列表
        self.controls = []
        # 存储窗口属性字典
        self.window_properties = {}
        # 存储控件属性字典
        self.control_properties = {}
        # 存储信号连接列表
        self.signal_connections = []
        
    def parse_code(self, code: str) -> Dict[str, Any]:
        """
        解析Python代码字符串，提取窗口和控件信息
        
        Args:
            code: Python代码字符串
            
        Returns:
            Dict[str, Any]: 包含解析结果的字典，格式为：
                {
                    "window": {...},  # 窗口属性
                    "controls": [...],  # 控件列表
                    "connections": [...]  # 信号连接列表
                }
                如果解析失败（语法错误），返回None
        """
        # 重置所有解析结果容器
        self.controls = []
        self.window_properties = {}
        self.control_properties = {}
        self.signal_connections = []
        
        try:
            # 将代码字符串解析为AST（抽象语法树）
            tree = ast.parse(code)
        except SyntaxError:
            # 如果代码有语法错误，返回None
            return None
            
        # 遍历AST查找类定义
        for node in ast.walk(tree):
            # 如果找到类定义节点
            if isinstance(node, ast.ClassDef):
                # 解析该类定义
                self._parse_class(node)
                break  # 只解析第一个类（通常是主窗口类）
                
        # 返回解析结果字典
        return {
            "window": self.window_properties,  # 窗口属性
            "controls": self.controls,  # 控件列表
            "connections": self.signal_connections  # 信号连接列表
        }
    
    def _parse_class(self, class_node: ast.ClassDef):
        """
        解析类定义节点
        提取类名和类中的所有方法
        
        Args:
            class_node: AST类定义节点
        """
        # 获取类名并保存到窗口属性中
        self.window_properties["class_name"] = class_node.name
        
        # ========== 遍历类体中的所有节点 ==========
        # 遍历类定义节点体中的所有子节点（方法、属性等）
        for node in class_node.body:
            # ========== 检查节点类型 ==========
            # 检查当前节点是否为函数定义节点（类方法）
            if isinstance(node, ast.FunctionDef):
                # 如果是方法定义，调用解析方法处理该方法
                self._parse_method(node)
                
    def _parse_method(self, method_node: ast.FunctionDef):
        """
        解析方法定义节点
        提取方法体中的控件创建、属性设置、信号连接等操作
        
        Args:
            method_node: AST函数定义节点（方法）
        """
        # ========== 遍历方法体中的所有节点 ==========
        # 遍历方法定义节点体中的所有子节点（语句）
        for node in method_node.body:
            # ========== 解析赋值语句 ==========
            # 检查当前节点是否为赋值语句节点
            if isinstance(node, ast.Assign):
                # 如果是赋值语句，通常是控件创建（如 self.button = QPushButton()）
                self._parse_assignment(node)
            
            # ========== 解析表达式语句 ==========
            # 检查当前节点是否为表达式语句节点
            elif isinstance(node, ast.Expr):
                # 如果是表达式语句，通常是方法调用（如属性设置、信号连接等）
                self._parse_expression(node)

    def _parse_init(self, init_node: ast.FunctionDef):
        """
        解析__init__方法（保留此方法以兼容旧代码调用）
        
        Args:
            init_node: __init__方法的AST节点
        """
        # 实际逻辑已移至_parse_method，直接调用
        self._parse_method(init_node)
                
    def _parse_assignment(self, node: ast.Assign):
        """
        解析赋值语句（控件创建）
        识别类似 self.button = QPushButton() 的语句
        
        Args:
            node: AST赋值节点
        """
        # ========== 检查赋值语句格式 ==========
        # 检查赋值语句是否符合控件创建格式：self.xxx = ClassName(...)
        # 首先检查赋值目标数量是否为1（只有一个赋值目标）
        if len(node.targets) == 1 and isinstance(node.targets[0], ast.Attribute):
            # 如果赋值目标是属性访问，获取目标节点
            target = node.targets[0]
            # 检查属性访问的对象是否为"self"（实例属性）
            if isinstance(target.value, ast.Name) and target.value.id == "self":
                # 如果是self.xxx格式，提取控件ID（属性名，如"button1"）
                control_id = target.attr
                
                # ========== 检查赋值值是否为函数调用 ==========
                # 检查赋值的值是否为函数调用节点（控件构造函数调用）
                if isinstance(node.value, ast.Call):
                    # ========== 处理直接类名调用 ==========
                    # 检查函数调用对象是否为简单名称（如 QPushButton()）
                    if isinstance(node.value.func, ast.Name):
                        # 如果是直接类名，获取类名（如"QPushButton"）
                        class_name = node.value.func.id
                        # 调用添加控件方法，传入控件ID、类名和调用节点
                        self._add_control(control_id, class_name, node.value)
                    # ========== 处理模块.类名形式 ==========
                    # 检查函数调用对象是否为属性访问（如 PySide6.QtWidgets.QPushButton()）
                    elif isinstance(node.value.func, ast.Attribute):
                        # 如果是模块.类名形式，提取类名（属性名部分）
                        class_name = node.value.func.attr
                        # 调用添加控件方法，传入控件ID、类名和调用节点
                        self._add_control(control_id, class_name, node.value)

    def _add_control(self, control_id: str, class_name: str, call_node: ast.Call):
        """添加控件到列表"""
        # ========== 查找对应的控件类型 ==========
        # 初始化控件类型为None
        control_type = None
        # ========== 从控件类字典中查找 ==========
        # 遍历控件库中的控件类字典，查找匹配的控件类型
        for c_type, c_class in ControlLibrary.CONTROL_CLASSES.items():
            # 检查控件类的名称是否与代码中的类名匹配
            if c_class.__name__ == class_name:
                # 如果匹配，保存控件类型并退出循环
                control_type = c_type
                break
        
        # ========== 如果没找到，尝试从控件信息字典中查找 ==========
        # 检查是否在控件类字典中找到了控件类型
        if not control_type:
            # 如果没有找到，尝试从控件信息字典中查找（处理别名情况）
            # 遍历控件库中的控件信息字典
            for c_type, info in ControlLibrary.CONTROLS.items():
                # 检查控件信息中的类名是否与代码中的类名匹配
                if info["class_name"] == class_name:
                    # 如果匹配，保存控件类型并退出循环
                    control_type = c_type
                    break
                    
        if control_type:
            # 解析父控件和文本属性
            parent_id = None
            text_property = None
            
            args = call_node.args
            if args:
                # 检查第一个参数
                arg0 = args[0]
                arg0_val = self._get_arg_value(arg0)
                
                if isinstance(arg0_val, str):
                    # 第一个参数是字符串，认为是文本
                    text_property = arg0_val
                    # 如果有第二个参数，认为是父控件
                    if len(args) > 1:
                        arg1 = args[1]
                        if isinstance(arg1, ast.Attribute) and isinstance(arg1.value, ast.Name) and arg1.value.id == "self":
                            parent_id = arg1.attr
                        elif isinstance(arg1, ast.Name):
                            parent_id = arg1.id
                else:
                    # 第一个参数不是字符串，认为是父控件
                    if isinstance(arg0, ast.Attribute) and isinstance(arg0.value, ast.Name) and arg0.value.id == "self":
                        parent_id = arg0.attr
                    elif isinstance(arg0, ast.Name):
                        parent_id = arg0.id
            
            control_data = {
                "id": control_id,
                "type": control_type,
                "parent": parent_id,
                "properties": {}
            }
            
            if text_property:
                control_data["properties"]["text"] = text_property
                
            self.controls.append(control_data)
            
    def _parse_expression(self, node: ast.Expr):
        """解析表达式"""
        if isinstance(node.value, ast.Call):
            call = node.value
            if isinstance(call.func, ast.Attribute):
                # self.control.method(...)
                if isinstance(call.func.value, ast.Attribute) and isinstance(call.func.value.value, ast.Name) and call.func.value.value.id == "self":
                    control_id = call.func.value.attr
                    method_name = call.func.attr
                    self._parse_property_call(control_id, method_name, call.args)
                
                # self.control.signal.connect(...)
                elif isinstance(call.func.value, ast.Attribute) and call.func.attr == "connect":
                    # 可能是 signal.connect
                    signal_attr = call.func.value
                    if isinstance(signal_attr.value, ast.Attribute) and isinstance(signal_attr.value.value, ast.Name) and signal_attr.value.value.id == "self":
                        control_id = signal_attr.value.attr
                        signal_name = signal_attr.attr
                        self._parse_signal_connection(control_id, signal_name, call.args)

                # self.method(...) (窗口属性)
                elif isinstance(call.func.value, ast.Name) and call.func.value.id == "self":
                    method_name = call.func.attr
                    self._parse_window_property(method_name, call.args)

    def _parse_property_call(self, control_id: str, method_name: str, args: List[ast.AST]):
        """解析控件属性设置"""
        if not args:
            return
            
        val = self._get_arg_value(args[0])
        
        # 找到对应的控件
        control = next((c for c in self.controls if c["id"] == control_id), None)
        if not control:
            return
            
        props = control["properties"]
        
        if method_name == "setGeometry" and len(args) == 4:
            control["x"] = self._get_arg_value(args[0])
            control["y"] = self._get_arg_value(args[1])
            control["width"] = self._get_arg_value(args[2])
            control["height"] = self._get_arg_value(args[3])
        elif method_name == "setText":
            props["text"] = val
        elif method_name == "setTitle":
            props["title"] = val
        elif method_name == "setChecked":
            props["checked"] = val
        elif method_name == "setValue":
            props["value"] = val
        elif method_name == "setMinimum":
            props["minimum"] = val
        elif method_name == "setMaximum":
            props["maximum"] = val
        elif method_name == "setEnabled":
            props["enabled"] = val
        elif method_name == "setVisible":
            props["visible"] = val
        elif method_name == "setObjectName":
            # 忽略 objectName，因为我们用变量名作为ID
            pass
            
    def _parse_window_property(self, method_name: str, args: List[ast.AST]):
        """解析窗口属性"""
        if not args:
            return
        val = self._get_arg_value(args[0])
        
        if method_name == "setWindowTitle":
            self.window_properties["title"] = val
        elif method_name == "resize" and len(args) == 2:
            self.window_properties["width"] = self._get_arg_value(args[0])
            self.window_properties["height"] = self._get_arg_value(args[1])
            
    def _parse_signal_connection(self, control_id: str, signal_name: str, args: List[ast.AST]):
        """解析信号连接"""
        if not args:
            return
        # 获取槽函数名称
        slot = args[0]
        slot_name = ""
        if isinstance(slot, ast.Attribute) and isinstance(slot.value, ast.Name) and slot.value.id == "self":
            slot_name = slot.attr
            
        self.signal_connections.append({
            "control_id": control_id,
            "signal": signal_name,
            "slot": slot_name
        })

    def _get_arg_value(self, node: ast.AST):
        """获取参数的字面值"""
        if isinstance(node, ast.Constant):  # Python 3.8+
            return node.value
        elif isinstance(node, ast.Str):     # Python < 3.8
            return node.s
        elif isinstance(node, ast.Num):     # Python < 3.8
            return node.n
        elif isinstance(node, ast.NameConstant): # True/False/None
            return node.value
        elif isinstance(node, ast.Attribute): # e.g. Qt.AlignCenter
            return f"{self._get_full_attr_name(node)}"
        return None

    def _get_full_attr_name(self, node: ast.Attribute):
        if isinstance(node.value, ast.Name):
            return f"{node.value.id}.{node.attr}"
        elif isinstance(node.value, ast.Attribute):
            return f"{self._get_full_attr_name(node.value)}.{node.attr}"
        return node.attr
