#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
代码生成器模块
实现从可视化设计到PySide6代码的转换
"""

"""
代码生成器模块
实现从可视化设计到PySide6代码的转换
将画布上的控件设计转换为可运行的Python代码
支持PySide6和tkinter两种框架
"""

# 导入标准库模块
import os  # 操作系统接口，用于文件路径处理
# 导入类型提示
from typing import Dict, List, Optional, Tuple
# 导入自定义模块
from .control_pyside import Control, ControlManager  # 控件类和控件管理器
from .control_library_pyside import ControlLibrary  # 控件库，用于获取控件定义信息
from .event_manager_pyside import EventManager  # 事件管理器


class CodeGenerator:
    """
    代码生成器类
    用于将可视化设计转换为PySide6或tkinter代码
    通过遍历控件管理器中的控件，生成对应的Python代码
    """
    
    def __init__(self, control_manager: ControlManager):
        """
        初始化代码生成器
        
        Args:
            control_manager: 控件管理器实例，用于获取所有控件信息
        """
        # 保存控件管理器引用
        self.control_manager = control_manager
        # 窗口标题，默认"我的窗口"
        self.window_title = "我的窗口"
        # 窗口大小，格式为(宽度, 高度)元组，默认800x600
        self.window_size = (800, 600)
        # 窗口类名，用于生成Python类的名称，默认"GeneratedWindow"
        self.window_class_name = "GeneratedWindow"
        # 窗口图标路径，如果为空字符串则不设置图标
        self.window_icon_path = ""
        # 页面列表（来自菜单编辑器），用于多页面应用
        self.pages = []
        # 下拉菜单列表（来自菜单编辑器），用于菜单栏
        self.dropdown_menus = []
        # 窗口名称，用于生成事件函数文件名
        self.window_name = "GeneratedWindow"
    
    def set_window_title(self, title: str):
        """设置窗口标题"""
        self.window_title = title
    
    def set_window_size(self, width: int, height: int):
        """设置窗口大小"""
        self.window_size = (width, height)
    
    def set_window_class_name(self, name: str):
        """设置窗口类名"""
        self.window_class_name = name
    
    def set_window_icon(self, icon_path: str):
        """设置窗口图标路径"""
        self.window_icon_path = icon_path or ""

    def set_pages(self, pages: List[str]):
        """设置页面列表(来自菜单)"""
        self.pages = pages or []

    def set_dropdown_menus(self, menus: List[str]):
        """设置下拉菜单列表"""
        self.dropdown_menus = menus or []
    
    def set_window_name(self, name: str):
        """设置窗口名称（用于生成事件函数文件名）"""
        self.window_name = name or "GeneratedWindow"
    
    def generate_code(self, canvas_width: int = 800, canvas_height: int = 520, framework: str = "PySide") -> str:
        """
        生成完整的PySide6代码
        
        Args:
            canvas_width: 画布宽度
            canvas_height: 画布高度
            
        Returns:
            生成的Python代码字符串
        """
        # 获取所有控件，按Z轴顺序排序
        controls = sorted(
            self.control_manager.controls.values(),
            key=lambda c: c.z_order
        )
        if isinstance(framework, str) and framework.lower().startswith("tk"):
            imports = self._tk_generate_imports(controls)
            window_class = self._tk_generate_window_class(controls, canvas_width, canvas_height)
            main_function = self._tk_generate_main_function()
            code = f"{imports}\n\n{window_class}\n\n{main_function}"
            return code
        imports = self._generate_imports(controls)
        window_class = self._generate_window_class(controls, canvas_width, canvas_height)
        main_function = self._generate_main_function()
        code = f"{imports}\n\n{window_class}\n\n{main_function}"
        return code
    
    def _generate_imports(self, controls: List[Control]) -> str:
        """生成导入语句"""
        # 基础导入
        imports = [
            "import sys",
            "from PySide6.QtWidgets import (",
            "    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,",
            "    QLabel, QPushButton, QLineEdit, QTextEdit, QFrame, QGroupBox,",
            "    QTabWidget, QCheckBox, QRadioButton, QComboBox, QSlider,",
            "    QProgressBar, QListWidget, QTreeWidget, QSpinBox, QDoubleSpinBox,",
            "    QDateEdit, QTimeEdit, QDateTimeEdit, QMenuBar, QMenu, QStackedWidget",
            ")",
            "from PySide6.QtCore import Qt, QSize",
            "from PySide6.QtGui import QFont, QPixmap, QIcon, QAction"
        ]
        
        # ========== 添加函数库导入 ==========
        try:
            from pathlib import Path
            import sys
            sys.path.insert(0, str(Path(__file__).parent.parent))
            from lib.fnpy_library_manager import fnpyLibraryManager
            
            library_manager = fnpyLibraryManager()
            enabled_libraries = library_manager.get_enabled_libraries()
            
            for library in enabled_libraries:
                # 导入函数库（去掉.fnpy扩展名）
                library_name = library.name.replace('.fnpy', '')
                imports.append(f"from lib.fnpy import {library_name}")
        except Exception as e:
            print(f"添加函数库导入失败: {e}")
        
        # ========== 检查是否有事件绑定，如果有则导入事件函数模块 ==========
        has_events = False
        for control in controls:
            if hasattr(control, 'events') and control.events:
                has_events = True
                break
        
        if has_events:
            # 生成事件函数文件名（格式：{窗口名称}_function）
            function_file_name = f"{self.window_name}_function"
            imports.append(f"import {function_file_name}")
        
        # ========== 根据使用的控件添加特定导入 ==========
        # 使用集合推导式收集所有使用的控件类型（set用于去重）
        # 遍历所有控件，提取每个控件的control_type属性
        used_controls = set(control.control_type for control in controls)
        
        # ========== 检查是否需要额外的导入 ==========
        # 检查控件类型集合中是否包含"Image"类型
        if "Image" in used_controls:
            # 如果包含Image控件，添加QPixmap的导入语句（用于图片显示）
            imports.append("from PySide6.QtGui import QPixmap")
        
        return "\n".join(imports)
    
    def _generate_window_class(self, controls: List[Control], canvas_width: int, canvas_height: int) -> str:
        """生成窗口类代码"""
        # 类定义和初始化方法
        class_code = [
            f"class {self.window_class_name}(QMainWindow):",
            "    \"\"\"自动生成的窗口类\"\"\"",
            "",
            "    def __init__(self):",
            "        super().__init__()",
            f"        self.setWindowTitle({repr(self.window_title)})",
            f"        self.resize({self.window_size[0]}, {self.window_size[1]})",
            (f"        from PySide6.QtGui import QIcon\n        self.setWindowIcon(QIcon({repr(self.window_icon_path)}))" if self.window_icon_path else ""),
            "",
            ("        # 创建选项卡作为中央部件\n        self.tabs = QTabWidget()\n        self.setCentralWidget(self.tabs)" if self.pages else "        # 创建中央部件\n        self.central_widget = QWidget()\n        self.setCentralWidget(self.central_widget)"),
            "",
            ("        # 创建菜单栏并添加下拉菜单\n        menubar = self.menuBar()\n" + "\n".join([self._generate_menu_block(i, m) for i, m in enumerate(self.dropdown_menus)]) if self.dropdown_menus else ""),
            ("        # 创建页面容器\n" + "\n".join([f"        page_{i} = QWidget()\n        self.tabs.addTab(page_{i}, {repr(name)})" for i, name in enumerate(self.pages)]) if self.pages else ""),
        ]
        
        # 生成控件创建代码（只要有控件就生成）
        controls_creation = self._generate_controls_creation(controls)
        if controls_creation:
            # 检查是否真的有控件（不是"暂无控件"的注释）
            if controls_creation.strip() != "# 暂无控件" and len(controls) > 0:
                class_code.append("        # 创建控件")
                class_code.append(controls_creation)
        
        # 生成控件属性设置代码
        controls_properties = self._generate_controls_properties(controls)
        if controls_properties:
            # 检查是否真的有属性设置（不是"暂无属性设置"的注释）
            if controls_properties.strip() != "# 暂无属性设置" and len(controls) > 0:
                class_code.append("")
                class_code.append("        # 设置控件属性")
                class_code.append(controls_properties)
        
        # 生成信号连接代码（只要有绑定的事件就生成）
        signal_connections = self._generate_signal_connections(controls)
        if signal_connections:
            # 检查是否真的有信号连接（不是"暂无信号连接"的注释）
            if signal_connections.strip() != "# 暂无信号连接":
                class_code.append("")
                class_code.append("        # 连接信号和槽")
                class_code.append(signal_connections)
        
        # 生成需要重写的事件处理函数
        event_handlers = self._generate_event_handlers(controls)
        if event_handlers:
            class_code.append("")
            class_code.append("    # 事件处理函数（重写）")
            class_code.append(event_handlers)
        
        return "\n".join(class_code)
    
    def _generate_controls_creation(self, controls: List[Control]) -> str:
        """
        生成控件创建代码
        为所有控件生成对应的Python代码，用于创建控件实例
        
        Args:
            controls: 控件列表
            
        Returns:
            str: 控件创建代码字符串（多行，每行一个控件的创建语句）
        """
        # ========== 初始化代码行列表 ==========
        # 创建空列表，用于存储控件创建的代码行
        creation_lines = []
        
        # ========== 遍历所有控件并生成创建代码 ==========
        # 遍历控件列表中的每个控件对象
        for control in controls:
            # 获取控件的类型（如"Button"、"Label"等）
            control_type = control.control_type
            # 将控件ID转换为合法的Python变量名（移除特殊字符、空格等）
            control_id = self._to_valid_variable_name(control.id)
            
            # ========== 获取控件类名 ==========
            # 根据控件类型获取对应的PySide6类名（如"QPushButton"、"QLabel"等）
            class_name = self._get_control_class_name(control_type)
            
            # ========== 生成控件创建代码 ==========
            # 检查项目是否包含多个页面（多页应用）
            if self.pages:
                # 如果包含多个页面，执行以下操作
                # 获取控件所属的页面名称，如果属性中没有指定则使用第一个页面
                page_name = control.properties.get('page', self.pages[0] if self.pages else '')
                # 尝试在页面列表中找到页面名称的索引
                # 如果页面名称存在于页面列表中，使用对应的索引；否则使用0（第一个页面）
                try_index = self.pages.index(page_name) if page_name in self.pages else 0
                # 生成控件创建代码：在指定页面上创建控件
                creation_lines.append(f"        self.{control_id} = {class_name}(page_{try_index})")
            else:
                # 如果不包含多个页面（单页应用），在中央部件上创建控件
                creation_lines.append(f"        self.{control_id} = {class_name}(self.central_widget)")
        
        # ========== 返回生成的代码 ==========
        # 将代码行列表用换行符连接，返回完整的代码字符串
        # 如果没有控件，返回注释说明
        if not creation_lines:
            return "        # 暂无控件"
        return "\n".join(creation_lines)

    def _generate_menu_block(self, idx: int, menu: Dict) -> str:
        """
        生成菜单块代码
        为指定的菜单生成对应的Python代码，包括菜单项、快捷键、状态等
        
        Args:
            idx: 菜单索引（用于生成唯一的变量名）
            menu: 菜单字典，包含标题和菜单项列表
            
        Returns:
            str: 菜单块代码字符串（多行）
        """
        # ========== 获取菜单标题 ==========
        # 从菜单字典中获取标题，如果不存在则使用默认值"菜单"
        # repr()用于将字符串转换为Python代码中的字符串字面量（带引号）
        title = repr(menu.get('title', '菜单'))
        # ========== 初始化代码行列表 ==========
        # 创建代码行列表，首行是菜单创建代码
        lines = [f"        menu_{idx} = menubar.addMenu({title})"]
        # ========== 获取菜单项列表 ==========
        # 从菜单字典中获取菜单项列表，如果不存在则使用空列表
        items = menu.get('items', [])
        
        # ========== 遍历菜单项并生成代码 ==========
        # 遍历菜单项列表，enumerate返回索引和菜单项字典
        for j, it in enumerate(items):
            # 获取菜单项标签文本，如果不存在则使用空字符串
            label = it.get('label', '')
            # 获取菜单项层级（用于嵌套菜单），如果不存在则默认为0
            lvl = int(it.get('level', 0))
            
            # ========== 检查是否为分隔符 ==========
            # 检查标签文本去除空白后是否为分隔符标记（"-"或"—"）
            if label.strip() == '-' or label.strip() == '—':
                # 如果是分隔符，生成添加分隔符的代码
                lines.append(f"        menu_{idx}.addSeparator()")
                # 跳过后续处理，继续下一个菜单项
                continue
            
            # ========== 生成菜单项动作代码 ==========
            # 生成菜单项动作的变量名（格式：action_菜单索引_菜单项索引）
            act_name = f"action_{idx}_{j}"
            # 生成创建菜单项动作的代码（QAction对象）
            lines.append(f"        {act_name} = QAction({repr(label)}, self)")
            
            # ========== 处理快捷键 ==========
            # 获取菜单项的快捷键，如果不存在则使用默认值"(无)"
            sc = it.get('shortcut', '(无)')
            # 检查快捷键是否存在且不为默认值
            if sc and sc != '(无)':
                # 如果存在有效的快捷键，生成设置快捷键的代码
                lines.append(f"        {act_name}.setShortcut({repr(sc)})")
            
            # ========== 处理选中状态 ==========
            # 检查菜单项是否应该被选中（勾选状态）
            if it.get('checked', False):
                # 如果需要选中，生成设置可选中和已选中的代码
                lines.append(f"        {act_name}.setCheckable(True)")
                lines.append(f"        {act_name}.setChecked(True)")
            
            # ========== 处理启用状态 ==========
            # 检查菜单项是否应该被禁用（allow属性为False时禁用）
            if not it.get('allow', True):
                # 如果需要禁用，生成设置禁用的代码
                lines.append(f"        {act_name}.setEnabled(False)")
            
            # ========== 处理可见状态 ==========
            # 检查菜单项是否应该隐藏（visible属性为False时隐藏）
            if not it.get('visible', True):
                # 如果需要隐藏，生成设置隐藏的代码
                lines.append(f"        {act_name}.setVisible(False)")
            
            # ========== 将动作添加到菜单 ==========
            # 生成将菜单项动作添加到菜单的代码
            lines.append(f"        menu_{idx}.addAction({act_name})")
        
        # ========== 返回生成的代码 ==========
        # 将代码行列表用换行符连接，返回完整的代码字符串
        return "\n".join(lines)
    
    def _generate_controls_properties(self, controls: List[Control]) -> str:
        """
        生成控件属性设置代码
        为所有控件生成设置属性的Python代码
        
        Args:
            controls: 控件列表
            
        Returns:
            str: 属性设置代码字符串（多行）
        """
        # ========== 初始化代码行列表 ==========
        # 创建空列表，用于存储属性设置的代码行
        properties_lines = []
        
        # ========== 遍历所有控件并生成属性设置代码 ==========
        # 遍历控件列表中的每个控件对象
        for control in controls:
            # 将控件ID转换为合法的Python变量名
            control_id = self._to_valid_variable_name(control.id)
            # 获取控件的类型
            control_type = control.control_type
            
            # ========== 设置位置和大小 ==========
            # 生成设置控件几何形状的代码（位置和大小）
            properties_lines.append(f"        self.{control_id}.setGeometry({control.x}, {control.y}, {control.width}, {control.height})")
            
            # ========== 设置其他属性 ==========
            # 遍历控件的所有属性（键值对）
            for prop_name, prop_value in control.properties.items():
                if prop_name == "text":
                    properties_lines.append(f"        self.{control_id}.setText({repr(prop_value)})")
                elif prop_name == "title" and control_type == "GroupBox":
                    properties_lines.append(f"        self.{control_id}.setTitle({repr(prop_value)})")
                elif prop_name == "checked" and control_type in ["CheckBox", "RadioButton"]:
                    checked_str = "True" if prop_value else "False"
                    properties_lines.append(f"        self.{control_id}.setChecked({checked_str})")
                elif prop_name == "value" and control_type in ["Slider", "ProgressBar", "SpinBox", "DoubleSpinBox"]:
                    properties_lines.append(f"        self.{control_id}.setValue({prop_value})")
                elif prop_name == "minimum" and control_type in ["Slider", "ProgressBar", "SpinBox", "DoubleSpinBox"]:
                    properties_lines.append(f"        self.{control_id}.setMinimum({prop_value})")
                elif prop_name == "maximum" and control_type in ["Slider", "ProgressBar", "SpinBox", "DoubleSpinBox"]:
                    properties_lines.append(f"        self.{control_id}.setMaximum({prop_value})")
                elif prop_name == "enabled":
                    enabled_str = "True" if prop_value else "False"
                    properties_lines.append(f"        self.{control_id}.setEnabled({enabled_str})")
                elif prop_name == "visible":
                    visible_str = "True" if prop_value else "False"
                    properties_lines.append(f"        self.{control_id}.setVisible({visible_str})")
                elif prop_name == "font_size":
                    properties_lines.append(f"        font = self.{control_id}.font()")
                    properties_lines.append(f"        font.setPointSize({prop_value})")
                    properties_lines.append(f"        self.{control_id}.setFont(font)")
                elif prop_name == "font_bold":
                    bold_str = "True" if prop_value else "False"
                    properties_lines.append(f"        font = self.{control_id}.font()")
                    properties_lines.append(f"        font.setBold({bold_str})")
                    properties_lines.append(f"        self.{control_id}.setFont(font)")
                elif prop_name == "alignment" and control_type == "Label":
                    if prop_value == "center":
                        properties_lines.append(f"        self.{control_id}.setAlignment(Qt.AlignCenter)")
                    elif prop_value == "left":
                        properties_lines.append(f"        self.{control_id}.setAlignment(Qt.AlignLeft)")
                    elif prop_value == "right":
                        properties_lines.append(f"        self.{control_id}.setAlignment(Qt.AlignRight)")
                # TabWidget tabs
                elif control_type == "TabWidget" and prop_name == "tabs":
                    properties_lines.append(f"        self.{control_id}.clear()")
                    for tab in control.properties.get("tabs", []):
                        properties_lines.append(f"        self.{control_id}.addTab(QWidget(), \"{tab}\")")
                # StackedWidget pages
                elif control_type == "StackedWidget" and prop_name == "pages":
                    properties_lines.append(f"        self.{control_id}.clear()")
                    for page in control.properties.get("pages", []):
                        properties_lines.append(f"        self.{control_id}.addWidget(QWidget())  # {page}")
                elif control_type == "StackedWidget" and prop_name == "current_index":
                    properties_lines.append(f"        self.{control_id}.setCurrentIndex({prop_value})")
                
                elif prop_name == "items" and control_type == "ComboBox":
                    # ========== 处理ComboBox的选项列表 ==========
                    # 如果是ComboBox控件的items属性，执行以下操作
                    # 生成清空ComboBox的代码
                    properties_lines.append(f"        self.{control_id}.clear()")
                    # 遍历选项列表中的每个选项
                    for item in control.properties.get("items", []):
                        # 为每个选项生成添加项目的代码
                        properties_lines.append(f"        self.{control_id}.addItem({repr(item)})")
                
                elif prop_name == "items" and control_type == "ListWidget":
                    # ========== 处理ListWidget的列表项 ==========
                    # 如果是ListWidget控件的items属性，执行以下操作
                    # 生成清空ListWidget的代码
                    properties_lines.append(f"        self.{control_id}.clear()")
                    # 遍历列表项中的每个项目
                    for item in control.properties.get("items", []):
                        # 为每个项目生成添加项目的代码
                        properties_lines.append(f"        self.{control_id}.addItem({repr(item)})")
                elif control_type == "DateEdit" and prop_name == "display_format":
                    properties_lines.append(f"        self.{control_id}.setDisplayFormat({repr(prop_value)})")
                elif control_type == "DateEdit" and prop_name == "calendar_popup":
                    popup_str = "True" if prop_value else "False"
                    properties_lines.append(f"        self.{control_id}.setCalendarPopup({popup_str})")
                elif control_type == "TimeEdit" and prop_name == "display_format":
                    properties_lines.append(f"        self.{control_id}.setDisplayFormat({repr(prop_value)})")
                elif control_type == "DateTimeEdit" and prop_name == "display_format":
                    properties_lines.append(f"        self.{control_id}.setDisplayFormat({repr(prop_value)})")
                elif control_type == "TreeWidget" and prop_name == "header_labels":
                    properties_lines.append(f"        self.{control_id}.setHeaderLabels({repr(control.properties.get('header_labels', []))})")
                elif control_type == "TableWidget" and prop_name == "row_count":
                    properties_lines.append(f"        self.{control_id}.setRowCount({prop_value})")
                elif control_type == "TableWidget" and prop_name == "column_count":
                    properties_lines.append(f"        self.{control_id}.setColumnCount({prop_value})")
                elif control_type == "TableWidget" and prop_name == "headers":
                    properties_lines.append(f"        self.{control_id}.setHorizontalHeaderLabels({repr(control.properties.get('headers', []))})")
                elif control_type == "ImageBox" and prop_name == "image_path":
                    properties_lines.append(f"        self.{control_id}.setPixmap(QPixmap({repr(prop_value)}))")
                elif control_type == "ImageBox" and prop_name == "scaled":
                    scaled_str = "True" if prop_value else "False"
                    properties_lines.append(f"        self.{control_id}.setScaledContents({scaled_str})")
        
        # 如果没有属性设置代码，返回注释说明
        if not properties_lines:
            return "        # 暂无属性设置"
        return "\n".join(properties_lines)
    
    def _generate_layout_code(self, controls: List[Control]) -> str:
        """布局代码（当前使用绝对定位，返回空）"""
        return ""
    
    def _generate_signal_connections(self, controls: List[Control]) -> str:
        """
        生成信号连接代码（画布代码绑定函数）
        为所有绑定事件的控件生成PySide6信号连接代码，将控件的信号连接到对应的事件处理函数
        
        此函数遍历所有控件，检查哪些事件已被绑定，然后为每个绑定的事件生成相应的信号连接代码。
        生成的代码会将控件信号（如clicked、textChanged等）连接到事件函数文件中的处理函数。
        
        Args:
            controls: 控件列表，包含所有需要生成连接代码的控件
            
        Returns:
            str: 生成的信号连接代码字符串（多行，每行一个连接语句），如果没有绑定的事件则返回空字符串
            
        生成的代码示例：
            self.button1.clicked.connect(lambda: GeneratedWindow_function._on_button1_clicked(self, self.button1))
            self.lineEdit1.textChanged.connect(lambda text: GeneratedWindow_function._on_lineEdit1_text_changed(self, self.lineEdit1, text))
        """
        # ========== 初始化代码行列表 ==========
        # 创建空列表，用于存储所有信号连接的代码行
        connections = []
        
        # ========== 遍历所有控件 ==========
        # 遍历控件列表中的每个控件对象
        for control in controls:
            # 将控件ID转换为合法的Python变量名（移除特殊字符、空格等）
            control_id = self._to_valid_variable_name(control.id)
            # 获取控件的类型（如"Button"、"Label"等）
            control_type = control.control_type
            
            # ========== 获取控件绑定的事件 ==========
            # 从控件对象中获取已绑定的事件字典
            # 事件字典的格式：{事件ID: True/False}，True表示已绑定，False表示未绑定
            bound_events = getattr(control, 'events', {})
            # 确保bound_events是字典类型，如果不是则初始化为空字典
            if not isinstance(bound_events, dict):
                bound_events = {}
            
            # ========== 为绑定的事件生成连接代码 ==========
            # 遍历控件绑定的事件字典，为每个绑定的事件生成信号连接代码
            for event_id, is_bound in bound_events.items():
                # 如果事件未绑定（is_bound为False），跳过此事件
                if not is_bound:
                    continue
                
                # ========== 获取事件信息 ==========
                # 从事件管理器中获取事件的详细信息（如信号名称、描述等）
                event_info = None
                # 首先检查是否为通用事件（所有控件都支持的事件）
                if event_id in EventManager.COMMON_EVENTS:
                    event_info = EventManager.COMMON_EVENTS[event_id]
                # 如果不是通用事件，检查是否为控件特定事件
                elif control_type in EventManager.CONTROL_SPECIFIC_EVENTS:
                    # 在控件特定事件字典中查找该事件
                    if event_id in EventManager.CONTROL_SPECIFIC_EVENTS[control_type]:
                        event_info = EventManager.CONTROL_SPECIFIC_EVENTS[control_type][event_id]
                
                # ========== 生成信号连接代码 ==========
                # 如果找到事件信息，生成对应的信号连接代码
                if event_info:
                    # 获取事件的信号名称（如果有信号，则使用connect连接；如果没有信号，则需要重写事件处理函数）
                    signal = event_info.get("signal")
                    # 根据控件ID和事件ID生成事件处理函数名称（格式：_on_{控件ID}_{事件名}）
                    function_name = EventManager.get_event_function_name(control.id, event_id)
                    # 生成事件函数文件名（格式：{窗口名称}_function）
                    function_file_name = f"{self.window_name}_function"
                    
                    # ========== 检查是否有信号 ==========
                    # 如果事件有对应的信号（signal不为None），使用connect方法连接信号
                    if signal:
                        # ========== 无参数信号 ==========
                        # 处理无参数信号（如clicked、pressed、released、toggled）
                        # 这些信号触发时不传递任何参数，只需要调用事件处理函数
                        if signal in ["clicked", "pressed", "released", "toggled"]:
                            # 生成无参数的lambda表达式，调用事件处理函数并传递窗口实例和控件实例
                            connections.append(f"        self.{control_id}.{signal}.connect(lambda: {function_file_name}.{function_name}(self, self.{control_id}))")
                        # ========== 文本信号 ==========
                        # 处理文本信号（如textChanged、textEdited、currentTextChanged）
                        # 这些信号触发时会传递文本内容作为参数
                        elif signal in ["textChanged", "textEdited", "currentTextChanged"]:
                            # 生成带文本参数的lambda表达式，将文本内容传递给事件处理函数
                            connections.append(f"        self.{control_id}.{signal}.connect(lambda text: {function_file_name}.{function_name}(self, self.{control_id}, text))")
                        # ========== 值变化信号 ==========
                        # 处理值变化信号（如valueChanged、stateChanged）
                        # 这些信号触发时会传递新的值作为参数
                        elif signal in ["valueChanged", "stateChanged"]:
                            # 生成带值参数的lambda表达式，将值传递给事件处理函数
                            connections.append(f"        self.{control_id}.{signal}.connect(lambda value: {function_file_name}.{function_name}(self, self.{control_id}, value))")
                        # ========== 索引变化信号 ==========
                        # 处理索引变化信号（如currentIndexChanged）
                        # 这些信号触发时会传递新的索引值作为参数
                        elif signal in ["currentIndexChanged"]:
                            # 生成带索引参数的lambda表达式，将索引传递给事件处理函数
                            connections.append(f"        self.{control_id}.{signal}.connect(lambda index: {function_file_name}.{function_name}(self, self.{control_id}, index))")
                        # ========== 项点击信号 ==========
                        # 处理项点击信号（如itemClicked、itemDoubleClicked、cellClicked、cellDoubleClicked）
                        # 这些信号触发时会传递被点击的项对象作为参数
                        elif signal in ["itemClicked", "itemDoubleClicked", "cellClicked", "cellDoubleClicked"]:
                            # 生成带项参数的lambda表达式，将项对象传递给事件处理函数
                            connections.append(f"        self.{control_id}.{signal}.connect(lambda item: {function_file_name}.{function_name}(self, self.{control_id}, item))")
                        # ========== 日期时间信号 ==========
                        # 处理日期时间信号（如dateChanged、timeChanged、dateTimeChanged）
                        # 这些信号触发时会传递日期时间对象作为参数
                        elif signal in ["dateChanged", "timeChanged", "dateTimeChanged"]:
                            # 生成带日期参数的lambda表达式，将日期时间对象传递给事件处理函数
                            connections.append(f"        self.{control_id}.{signal}.connect(lambda date: {function_file_name}.{function_name}(self, self.{control_id}, date))")
                        # ========== 默认情况 ==========
                        # 处理其他未知类型的信号，使用*args传递所有参数
                        else:
                            # 生成带可变参数的lambda表达式，将所有参数传递给事件处理函数
                            connections.append(f"        self.{control_id}.{signal}.connect(lambda *args: {function_file_name}.{function_name}(self, self.{control_id}, *args))")
                    # 注意：如果没有信号（signal为None），表示需要重写事件处理函数，这种情况在_generate_event_handlers中处理
        
        # ========== 返回生成的代码 ==========
        # 如果存在连接代码，用换行符连接所有代码行并返回；否则返回注释说明
        if not connections:
            return "        # 暂无信号连接"
        return "\n".join(connections)
    
    def _generate_event_handlers(self, controls: List[Control]) -> str:
        """
        生成需要重写的事件处理函数代码（使用事件过滤器）
        对于没有对应信号的事件（如mousePressEvent、keyPressEvent等），需要重写事件处理函数
        由于无法直接重写控件的事件处理函数，使用事件过滤器机制来实现事件处理
        
        此函数会：
        1. 收集所有需要事件过滤器的事件（没有信号的事件）
        2. 为每个需要事件过滤器的控件生成installEventFilter调用
        3. 生成统一的eventFilter方法，在其中根据事件类型调用对应的事件处理函数
        
        Args:
            controls: 控件列表，包含所有需要生成事件处理代码的控件
            
        Returns:
            str: 生成的事件处理器代码字符串（包括事件过滤器安装和eventFilter方法），如果没有需要处理的事件则返回空字符串
        """
        # ========== 初始化代码列表 ==========
        # 创建空列表，用于存储生成的事件处理器代码行
        handlers = []
        # 创建字典，用于存储每个控件需要处理的事件列表（格式：{控件ID: [(事件ID, 事件名称), ...]})
        # 这样可以为每个控件生成一个installEventFilter调用，并在注释中列出所有事件
        control_events_map = {}
        # 创建集合，用于记录需要处理的事件类型（用于去重）
        event_types_needed = set()  # 需要处理的事件类型
        
        # ========== 收集所有需要事件过滤器的事件 ==========
        # 遍历所有控件，收集需要事件过滤器的事件
        for control in controls:
            control_id = self._to_valid_variable_name(control.id)
            control_type = control.control_type
            
            # 获取控件绑定的事件
            bound_events = getattr(control, 'events', {})
            if not isinstance(bound_events, dict):
                bound_events = {}
            
            # 为绑定的事件设置事件过滤器
            for event_id, is_bound in bound_events.items():
                if not is_bound:
                    continue
                
                # 获取事件信息
                event_info = None
                if event_id in EventManager.COMMON_EVENTS:
                    event_info = EventManager.COMMON_EVENTS[event_id]
                elif control_type in EventManager.CONTROL_SPECIFIC_EVENTS:
                    if event_id in EventManager.CONTROL_SPECIFIC_EVENTS[control_type]:
                        event_info = EventManager.CONTROL_SPECIFIC_EVENTS[control_type][event_id]
                
                # ========== 检查是否需要事件过滤器 ==========
                # 如果事件没有对应的信号（signal为None），需要使用事件过滤器来处理
                if event_info and not event_info.get("signal"):
                    # 获取事件的显示名称（用于注释）
                    event_display_name = EventManager.get_event_display_name(event_id, control_type)
                    
                    # ========== 将事件添加到控件的事件列表 ==========
                    # 如果控件还没有事件列表，创建一个空列表
                    if control_id not in control_events_map:
                        control_events_map[control_id] = []
                    # 将事件ID和事件显示名称添加到控件的事件列表中
                    control_events_map[control_id].append((event_id, event_display_name))
                    # 将事件类型添加到需要处理的事件类型集合中（用于去重）
                    event_types_needed.add(event_id)
        
        # ========== 生成事件过滤器安装代码 ==========
        # 为每个需要事件过滤器的控件生成installEventFilter调用，并添加注释说明
        event_filter_setup = []
        for control_id, events_list in control_events_map.items():
            # 获取控件对象，用于获取控件类型
            control = next((c for c in controls if self._to_valid_variable_name(c.id) == control_id), None)
            control_type = control.control_type if control else "Unknown"
            
            # ========== 生成注释 ==========
            # 构建注释字符串，格式：组件类型 + 组件ID + 事件列表
            # 如果控件只有一个事件，直接显示；如果有多个事件，用逗号分隔
            if len(events_list) == 1:
                event_name = events_list[0][1]
                comment = f"  # {control_type} + {control_id} + {event_name}"
            else:
                event_names = "、".join([event[1] for event in events_list])
                comment = f"  # {control_type} + {control_id} + {event_names}"
            
            # ========== 生成安装事件过滤器的代码行 ==========
            # 生成installEventFilter调用，并在行末添加注释
            event_filter_setup.append(f"        self.{control_id}.installEventFilter(self){comment}")
        
        # 如果有需要事件过滤器的事件，生成事件过滤器代码
        if event_filter_setup:
            handlers.append("        # 安装事件过滤器（用于处理需要重写的事件）")
            handlers.extend(event_filter_setup)
            handlers.append("")
            handlers.append("    def eventFilter(self, obj, event):")
            handlers.append("        \"\"\"事件过滤器，处理控件的事件\"\"\"")
            handlers.append("        from PySide6.QtCore import QEvent")
            function_file_name = f"{self.window_name}_function"
            handlers.append(f"        import {function_file_name}")
            handlers.append("        ")
            
            # 为每个控件生成事件检查代码
            for control in controls:
                control_id = self._to_valid_variable_name(control.id)
                control_type = control.control_type
                
                bound_events = getattr(control, 'events', {})
                if not isinstance(bound_events, dict):
                    bound_events = {}
                
                control_events = []
                for event_id, is_bound in bound_events.items():
                    if not is_bound:
                        continue
                    
                    event_info = None
                    if event_id in EventManager.COMMON_EVENTS:
                        event_info = EventManager.COMMON_EVENTS[event_id]
                    elif control_type in EventManager.CONTROL_SPECIFIC_EVENTS:
                        if event_id in EventManager.CONTROL_SPECIFIC_EVENTS[control_type]:
                            event_info = EventManager.CONTROL_SPECIFIC_EVENTS[control_type][event_id]
                    
                    if event_info and not event_info.get("signal"):
                        function_name = EventManager.get_event_function_name(control.id, event_id)
                        
                        # 映射事件ID到QEvent类型
                        event_type_map = {
                            "mousePressEvent": "QEvent.Type.MouseButtonPress",
                            "mouseReleaseEvent": "QEvent.Type.MouseButtonRelease",
                            "mouseMoveEvent": "QEvent.Type.MouseMove",
                            "mouseDoubleClickEvent": "QEvent.Type.MouseButtonDblClick",
                            "wheelEvent": "QEvent.Type.Wheel",
                            "keyPressEvent": "QEvent.Type.KeyPress",
                            "keyReleaseEvent": "QEvent.Type.KeyRelease",
                            "enterEvent": "QEvent.Type.Enter",
                            "leaveEvent": "QEvent.Type.Leave",
                            "focusInEvent": "QEvent.Type.FocusIn",
                            "focusOutEvent": "QEvent.Type.FocusOut",
                            "resizeEvent": "QEvent.Type.Resize",
                            "moveEvent": "QEvent.Type.Move",
                            "paintEvent": "QEvent.Type.Paint"
                        }
                        
                        qevent_type = event_type_map.get(event_id, None)
                        if qevent_type:
                            control_events.append((event_id, function_name, qevent_type))
                
                if control_events:
                    handlers.append(f"        if obj == self.{control_id}:")
                    for event_id, function_name, qevent_type in control_events:
                        handlers.append(f"            if event.type() == {qevent_type}:")
                        handlers.append(f"                {function_file_name}.{function_name}(self, self.{control_id}, event)")
                        handlers.append(f"                return True")
                    handlers.append("        ")
            
            handlers.append("        return super().eventFilter(obj, event)")
        
        return "\n".join(handlers) if handlers else ""
    
    def _generate_main_function(self) -> str:
        """生成主函数代码"""
        main_code = [
            "def main():",
            "    \"\"\"主函数\"\"\"",
            "    app = QApplication(sys.argv)",
            "    app.setStyle('Fusion')",
            "    app.setStyleSheet(\"\"\"\n        QMainWindow { background-color: white; color: black; }\n        QWidget { background-color: white; color: black; }\n        QMenuBar { background-color: white; color: black; border-bottom: 1px solid #cccccc; }\n        QMenuBar::item { background-color: white; color: black; }\n    \"\"\")",
            "",
            f"    window = {self.window_class_name}()",
            "    window.show()",
            "",
            "    sys.exit(app.exec())",
            "",
            "if __name__ == \"__main__\":",
            "    main()"
        ]
        
        return "\n".join(main_code)

    def _tk_generate_imports(self, controls: List[Control]) -> str:
        imports = [
            "import sys",
            "import tkinter as tk",
            "from tkinter import ttk"
        ]
        used_controls = set(c.control_type for c in controls)
        if "Image" in used_controls:
            imports.append("from tkinter import PhotoImage")
        return "\n".join(imports)

    def _tk_generate_window_class(self, controls: List[Control], canvas_width: int, canvas_height: int) -> str:
        lines = [
            f"class {self.window_class_name}(tk.Tk):",
            "    def __init__(self):",
            "        super().__init__()",
            f"        self.title({repr(self.window_title)})",
            f"        self.geometry(f\"{self.window_size[0]}x{self.window_size[1]}\")",
        ]
        if self.dropdown_menus:
            lines.extend([
                "        menubar = tk.Menu(self)",
            ])
            for i, m in enumerate(self.dropdown_menus):
                title = m.get('title', '菜单')
                lines.append(f"        menu_{i} = tk.Menu(menubar, tearoff=0)")
                lines.append(f"        menubar.add_cascade(label={repr(title)}, menu=menu_{i})")
                items = m.get('items', [])
                for it in items:
                    label = it.get('label', '')
                    if label.strip() in ['-', '—']:
                        lines.append(f"        menu_{i}.add_separator()")
                        continue
                    state = 'normal' if it.get('allow', True) else 'disabled'
                    visible = it.get('visible', True)
                    if visible:
                        lines.append(f"        menu_{i}.add_command(label={repr(label)}, state='{state}')")
            lines.append("        self.config(menu=menubar)")
        if self.pages:
            lines.append("        self.notebook = ttk.Notebook(self)")
            for i, name in enumerate(self.pages):
                lines.append(f"        page_{i} = tk.Frame(self.notebook)")
                lines.append(f"        self.notebook.add(page_{i}, text={repr(name)})")
            lines.append("        self.notebook.pack(fill='both', expand=True)")
        else:
            lines.append("        self.content_frame = tk.Frame(self)")
            lines.append(f"        self.content_frame.place(x=0, y=0, width={canvas_width}, height={canvas_height})")
        lines.append("        # 创建控件")
        lines.append(self._tk_generate_controls_creation(controls))
        lines.append("        # 设置控件属性")
        lines.append(self._tk_generate_controls_properties(controls))
        return "\n".join(lines)

    def _tk_class_name(self, control_type: str) -> str:
        class_map = {
            "Label": "tk.Label",
            "Button": "tk.Button",
            "LineEdit": "tk.Entry",
            "TextEdit": "tk.Text",
            "Frame": "tk.Frame",
            "GroupBox": "tk.LabelFrame",
            "TabWidget": "ttk.Notebook",
            "CheckBox": "tk.Checkbutton",
            "RadioButton": "tk.Radiobutton",
            "ComboBox": "ttk.Combobox",
            "Slider": "tk.Scale",
            "ProgressBar": "ttk.Progressbar",
            "ListWidget": "tk.Listbox",
            "SpinBox": "tk.Spinbox",
            "DoubleSpinBox": "tk.Spinbox",
            "TreeWidget": "ttk.Treeview",
            "DateEdit": "tk.Entry",
            "TimeEdit": "tk.Entry",
            "DateTimeEdit": "tk.Entry",
            "Image": "tk.Label"
        }
        return class_map.get(control_type, "tk.Frame")

    def _tk_generate_controls_creation(self, controls: List[Control]) -> str:
        creation = []
        for control in controls:
            control_type = control.control_type
            control_id = self._to_valid_variable_name(control.id)
            class_name = self._tk_class_name(control_type)
            parent = None
            if self.pages:
                page_name = control.properties.get('page', self.pages[0] if self.pages else '')
                try_index = self.pages.index(page_name) if page_name in self.pages else 0
                parent = f"page_{try_index}"
            else:
                parent = "self.content_frame"
            args = []
            if control_type in ["Label", "Button", "CheckBox", "RadioButton", "GroupBox"]:
                text = control.properties.get("text", "")
                args.append(f"text={repr(text)}")
            creation.append(f"        self.{control_id} = {class_name}({parent}{', ' if args else ''}{', '.join(args)})")
        return "\n".join(creation)

    def _tk_generate_controls_properties(self, controls: List[Control]) -> str:
        props = []
        for control in controls:
            cid = self._to_valid_variable_name(control.id)
            ctype = control.control_type
            props.append(f"        self.{cid}.place(x={control.x}, y={control.y}, width={control.width}, height={control.height})")
            for prop_name, prop_value in control.properties.items():
                if prop_name == "text":
                    if ctype in ["Label", "Button", "CheckBox", "RadioButton", "GroupBox"]:
                        props.append(f"        self.{cid}['text'] = {repr(prop_value)}")
                    elif ctype == "LineEdit":
                        props.append(f"        self.{cid}.delete(0, 'end')")
                        props.append(f"        self.{cid}.insert(0, {repr(prop_value)})")
                    elif ctype == "TextEdit":
                        props.append(f"        self.{cid}.delete('1.0', 'end')")
                        props.append(f"        self.{cid}.insert('1.0', {repr(prop_value)})")
                elif prop_name == "checked" and ctype == "CheckBox":
                    varname = f"var_{cid}"
                    props.append(f"        self.{varname} = tk.BooleanVar(value={bool(prop_value)})")
                    props.append(f"        self.{cid}['variable'] = self.{varname}")
                elif prop_name == "items" and ctype == "ComboBox":
                    props.append(f"        self.{cid}['values'] = {repr(control.properties.get('items', []))}")
                elif prop_name == "items" and ctype == "ListWidget":
                    props.append(f"        self.{cid}.delete(0, 'end')")
                    for item in control.properties.get('items', []):
                        props.append(f"        self.{cid}.insert('end', {repr(item)})")
                elif prop_name == "value" and ctype == "ProgressBar":
                    props.append(f"        self.{cid}['value'] = {prop_value}")
                elif prop_name == "maximum" and ctype == "ProgressBar":
                    props.append(f"        self.{cid}['maximum'] = {prop_value}")
                elif prop_name == "enabled":
                    state = 'normal' if prop_value else 'disabled'
                    props.append(f"        self.{cid}['state'] = '{state}'")
        return "\n".join(props)

    def _tk_generate_main_function(self) -> str:
        lines = [
            "def main():",
            f"    window = {self.window_class_name}()",
            "    window.mainloop()",
            "",
            "if __name__ == '__main__':",
            "    main()"
        ]
        return "\n".join(lines)
    
    def _to_valid_variable_name(self, control_id: str) -> str:
        """
        将控件ID转换为有效的Python变量名
        
        Args:
            control_id: 原始控件ID
            
        Returns:
            有效的Python变量名
        """
        import re
        
        # 替换非法字符为下划线
        var_name = re.sub(r'[^a-zA-Z0-9_]', '_', control_id)
        
        # 确保不以数字开头
        if var_name and var_name[0].isdigit():
            var_name = f"control_{var_name}"
        
        # 确保不为空
        if not var_name:
            var_name = "control"
        
        return var_name
    
    def _get_control_class_name(self, control_type: str) -> str:
        """获取控件类名"""
        class_map = {
            "Label": "QLabel",
            "Button": "QPushButton",
            "LineEdit": "QLineEdit",
            "TextEdit": "QTextEdit",
            "Frame": "QFrame",
            "GroupBox": "QGroupBox",
            "TabWidget": "QTabWidget",
            "CheckBox": "QCheckBox",
            "RadioButton": "QRadioButton",
            "ComboBox": "QComboBox",
            "Slider": "QSlider",
            "ProgressBar": "QProgressBar",
            "ListWidget": "QListWidget",
            "SpinBox": "QSpinBox",
            "DoubleSpinBox": "QDoubleSpinBox",
            "TreeWidget": "QTreeWidget",
            "DateEdit": "QDateEdit",
            "TimeEdit": "QTimeEdit",
            "DateTimeEdit": "QDateTimeEdit",
            "Image": "QLabel"  
        }
        
        return class_map.get(control_type, "QWidget")
    
    def save_code_to_file(self, code: str, file_path: str) -> bool:
        """
        将生成的代码保存到文件
        
        Args:
            code: 生成的代码
            file_path: 文件路径
            
        Returns:
            是否成功保存
        """
        try:
            # 确保目录存在
            os.makedirs(os.path.dirname(file_path), exist_ok=True)
            
            # 写入文件
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(code)
            
            return True
        except Exception as e:
            print(f"保存代码文件失败: {e}")
            return False
