"""
带行号的代码编辑器模块
提供带行号显示的代码编辑器，支持代码高亮和自动补全功能
"""

# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QPlainTextEdit,  # 纯文本编辑器（基类）
    QWidget,  # 基础窗口部件
    QCompleter,  # 自动补全组件
    QTextEdit  # 富文本编辑器
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QSize,  # 大小对象
    QStringListModel  # 字符串列表模型，用于自动补全
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QPainter,  # 绘图器
    QColor,  # 颜色类
    QTextCursor  # 文本光标
)

# 导入控件库，用于代码补全
from .control_library_pyside import ControlLibrary


class LineNumberArea(QWidget):
    """
    行号区域组件
    显示在代码编辑器左侧，用于显示行号
    """
    def __init__(self, editor):
        """
        初始化行号区域
        
        Args:
            editor: 所属的代码编辑器实例
        """
        # 调用父类构造函数，将editor作为父窗口
        super().__init__(editor)
        # 保存编辑器引用，用于访问编辑器方法
        self.editor = editor
        
    def sizeHint(self):
        """
        返回行号区域的建议大小
        
        Returns:
            QSize: 行号区域的建议大小
        """
        # 调用编辑器的行号区域大小计算方法
        return self.editor.lineNumberAreaSize()
        
    def paintEvent(self, event):
        """
        绘制行号区域
        当需要重绘时，委托给编辑器进行绘制
        
        Args:
            event: 绘制事件对象
        """
        # 调用编辑器的行号区域绘制方法
        self.editor.lineNumberAreaPaintEvent(event)


class LineNumberedCodeEdit(QPlainTextEdit):
    def __init__(self, parent=None):
        super().__init__(parent)
        self._lineNumberArea = LineNumberArea(self)
        self.blockCountChanged.connect(self._updateLineNumberAreaWidth)
        self.updateRequest.connect(self._updateLineNumberArea)
        self.cursorPositionChanged.connect(self._highlightCurrentLine)
        self._updateLineNumberAreaWidth(0)
        self._externalHighlights = []
        self._highlightCurrentLine()
        self._init_completer()
    def lineNumberAreaSize(self):
        return QSize(self._lineNumberAreaWidth(), 0)
    def _lineNumberAreaWidth(self):
        """
        计算行号区域的宽度
        根据代码行数动态计算所需的宽度
        
        Returns:
            int: 行号区域的宽度（像素）
        """
        # ========== 计算行号位数 ==========
        # 获取代码块数量（行数），至少为1（避免除零错误）
        # max(1, self.blockCount())：确保至少有1行
        # str(...)：将行数转换为字符串，以便计算位数
        # len(...)：计算字符串长度，即行号的位数（如100行需要3位数字）
        digits = len(str(max(1, self.blockCount())))
        
        # ========== 计算所需宽度 ==========
        # 计算行号区域的总宽度
        # 3：左右边距（像素）
        # self.fontMetrics().horizontalAdvance('9')：单个数字字符的宽度（像素）
        # digits：行号位数
        # 总宽度 = 边距 + 单个数字宽度 × 位数
        space = 3 + self.fontMetrics().horizontalAdvance('9') * digits
        
        # 返回计算出的宽度
        return space
    def _updateLineNumberAreaWidth(self, _):
        self.setViewportMargins(self._lineNumberAreaWidth(), 0, 0, 0)
    def _updateLineNumberArea(self, rect, dy):
        """
        更新行号区域的显示
        当编辑器内容滚动或更新时调用，同步更新行号区域的显示
        
        Args:
            rect: 需要更新的矩形区域
            dy: 垂直滚动距离（如果为0表示没有滚动，只是内容更新）
        """
        # ========== 处理滚动情况 ==========
        # 检查是否有垂直滚动（dy不为0）
        if dy:
            # 如果有滚动，同步滚动行号区域（保持与编辑器内容同步）
            # 参数：水平滚动距离（0），垂直滚动距离（dy）
            self._lineNumberArea.scroll(0, dy)
        else:
            # ========== 处理内容更新情况 ==========
            # 如果没有滚动，只是内容更新，更新指定区域
            # 参数：X坐标（0，从左侧开始）、Y坐标（rect.top()，矩形顶部）、
            #       宽度（行号区域宽度）、高度（矩形高度）
            self._lineNumberArea.update(0, rect.top(), self._lineNumberArea.width(), rect.height())
        
        # ========== 检查是否需要重新计算宽度 ==========
        # 检查更新区域是否包含整个视口（编辑器可见区域）
        if rect.contains(self.viewport().rect()):
            # 如果包含整个视口，可能需要重新计算行号区域宽度（行数可能变化）
            # 参数0表示不需要立即更新，只是触发宽度计算
            self._updateLineNumberAreaWidth(0)
    def resizeEvent(self, event):
        super().resizeEvent(event)
        cr = self.contentsRect()
        self._lineNumberArea.setGeometry(cr.left(), cr.top(), self._lineNumberAreaWidth(), cr.height())
    def lineNumberAreaPaintEvent(self, event):
        painter = QPainter(self._lineNumberArea)
        painter.fillRect(event.rect(), QColor(245, 245, 245))
        block = self.firstVisibleBlock()
        blockNumber = block.blockNumber()
        top = self.blockBoundingGeometry(block).translated(self.contentOffset()).top()
        bottom = top + self.blockBoundingRect(block).height()
        # ========== 遍历所有可见的代码块（行） ==========
        # 从第一个可见块开始，遍历所有块，直到块无效或超出绘制区域底部
        while block.isValid() and top <= event.rect().bottom():
            # ========== 检查块是否在可见区域内 ==========
            # 检查块是否可见，并且块的底部是否在绘制区域顶部之上（块在可见区域内）
            if block.isVisible() and bottom >= event.rect().top():
                # 如果块在可见区域内，绘制行号
                # 将块号转换为行号（块号从0开始，行号从1开始）
                number = str(blockNumber + 1)
                # 设置画笔颜色为灰色（行号颜色）
                painter.setPen(Qt.gray)
                # 计算文本右对齐位置：行号区域宽度减去4像素边距
                right = self._lineNumberArea.width() - 4
                # 绘制行号文本
                # 参数：X坐标（0）、Y坐标（top，块顶部）、宽度（right）、高度（字体高度）、
                #       对齐方式（右对齐）、文本内容（行号）
                painter.drawText(0, int(top), right, int(self.fontMetrics().height()), Qt.AlignRight, number)
            
            # ========== 移动到下一个块 ==========
            # 获取下一个代码块
            block = block.next()
            # 更新顶部位置为当前块的底部
            top = bottom
            # 计算下一个块的底部位置：当前顶部 + 下一个块的高度
            bottom = top + self.blockBoundingRect(block).height()
            # 增加块号（行号）
            blockNumber += 1
    def _highlightCurrentLine(self):
        """
        高亮显示当前行
        为当前光标所在行添加背景色高亮，并合并外部高亮选择
        """
        # ========== 初始化高亮选择列表 ==========
        # 创建空列表，用于存储所有需要高亮的文本选择区域
        extraSelections = []
        
        # ========== 高亮当前行 ==========
        # 检查编辑器是否不是只读模式（只读模式下不显示当前行高亮）
        if not self.isReadOnly():
            # 如果不是只读模式，创建当前行高亮选择
            selection = QTextEdit.ExtraSelection()
            # 设置高亮背景颜色为浅蓝色（RGB: 235, 235, 255）
            selection.format.setBackground(QColor(235, 235, 255))
            # 获取当前文本光标
            selection.cursor = self.textCursor()
            # 清除光标的选择（只高亮整行，不高亮选中的文本）
            selection.cursor.clearSelection()
            # 将当前行高亮选择添加到列表中
            extraSelections.append(selection)
        
        # ========== 合并外部高亮 ==========
        # 检查是否有外部高亮选择（如代码搜索高亮、选中控件对应代码高亮等）
        if self._externalHighlights:
            # 如果有外部高亮，将它们添加到高亮列表中
            # extend方法将外部高亮列表中的所有元素添加到当前列表
            extraSelections.extend(self._externalHighlights)
        
        # ========== 应用所有高亮 ==========
        # 将所有高亮选择应用到编辑器（同时显示当前行高亮和外部高亮）
        self.setExtraSelections(extraSelections)

    def set_external_highlights(self, selections):
        self._externalHighlights = selections or []
        self._highlightCurrentLine()

    def _init_completer(self):
        keywords = [
            'False','None','True','and','as','assert','async','await','break','class','continue','def','del','elif','else','except','finally','for','from','global','if','import','in','is','lambda','nonlocal','not','or','pass','raise','return','try','while','with','yield'
        ]
        builtins = [
            'print','len','range','dict','list','set','tuple','int','float','str','bool','open','sum','min','max','abs','sorted','enumerate','zip'
        ]
        # ========== 收集控件类型名称 ==========
        # 初始化控件类型列表为空列表
        control_types = []
        try:
            # 尝试从控件库获取所有分类
            cats = ControlLibrary.get_all_categories()
            # 遍历所有分类
            for cat_id, _ in cats:
                # 获取每个分类下的所有控件
                for ct, _info in ControlLibrary.get_controls_by_category(cat_id):
                    # 检查控件类型名称是否存在（不为空）
                    if ct:
                        # 如果控件类型存在，添加到列表中
                        control_types.append(ct)
        except Exception:
            # 如果获取控件类型失败（控件库未初始化等），忽略异常
            pass
        
        # ========== 处理控件类型列表 ==========
        # 对控件类型列表进行去重和排序
        # set(control_types)：去除重复的控件类型
        # sorted(...)：对去重后的列表进行排序（按字母顺序）
        self._control_types = sorted(set(control_types))
        
        # ========== 创建补全模型 ==========
        # 合并所有补全关键词：Python关键字 + 内置函数 + 控件类型
        # set(...)：去除重复项
        # sorted(...)：排序
        # QStringListModel：创建字符串列表模型，用于自动补全组件
        self._completion_model = QStringListModel(sorted(set(keywords + builtins + self._control_types)))
        self._completer = QCompleter(self)
        self._completer.setModel(self._completion_model)
        self._completer.setCaseSensitivity(Qt.CaseInsensitive)
        self._completer.setWidget(self)
        self._completer.activated.connect(self._insert_completion)
        self.textChanged.connect(self._update_completion_words)

    def _text_under_cursor(self):
        cursor = self.textCursor()
        cursor.select(QTextCursor.WordUnderCursor)
        return cursor.selectedText()

    def _insert_completion(self, completion):
        if completion in getattr(self, '_control_types', []):
            self._add_control_to_canvas(completion)
            return
        tc = self.textCursor()
        extra = completion
        prefix = self._text_under_cursor()
        if prefix:
            extra = completion[len(prefix):]
        tc.insertText(extra)
        self.setTextCursor(tc)

    def _update_completion_words(self):
        text = self.toPlainText()
        tokens = set()
        for part in text.split():
            if part.isidentifier():
                tokens.add(part)
        existing = set(self._completion_model.stringList())
        merged = sorted(existing.union(tokens).union(getattr(self, '_control_types', set())))
        self._completion_model.setStringList(merged)

    def _add_control_to_canvas(self, control_type: str):
        main = self.parent()
        while main and not hasattr(main, 'design_canvas'):
            main = main.parent()
        if not main:
            return
        canvas = getattr(main, 'design_canvas', None)
        if not canvas or not hasattr(canvas, 'add_control'):
            return
        try:
            control = canvas.add_control(control_type)
        except Exception:
            control = None
        if not control:
            return
        var_name = control.id
        try:
            cg = getattr(main, 'code_generator', None)
            if cg and hasattr(cg, '_to_valid_variable_name'):
                var_name = cg._to_valid_variable_name(control.id)
        except Exception:
            pass
        class_name = None
        try:
            cls = ControlLibrary.get_control_class(control_type)
            class_name = getattr(cls, '__name__', None)
        except Exception:
            class_name = None
        if not class_name:
            class_name = 'QWidget'
        tc = self.textCursor()
        prefix = self._text_under_cursor()
        if prefix:
            tc.deletePreviousChar()
        snippet = f"self.{var_name} = {class_name}(self.central_widget)\n"
        tc.insertText(snippet)
        self.setTextCursor(tc)

    def keyPressEvent(self, event):
        super().keyPressEvent(event)
        if event.modifiers() == Qt.ControlModifier and event.key() == Qt.Key_Space:
            prefix = self._text_under_cursor()
            if not prefix:
                return
            rect = self.cursorRect()
            self._completer.setCompletionPrefix(prefix)
            self._completer.complete(rect)
