#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
代码对话框模块
用于显示生成的代码并提供保存功能
"""

"""
代码对话框模块
用于显示生成的代码并提供保存、复制等功能
支持切换不同的框架（PySide、TKinter）查看对应代码
"""

# 导入标准库模块
import os  # 操作系统接口，用于文件路径操作
# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QDialog,  # 对话框基类
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QTextEdit,  # 富文本编辑器，用于显示代码
    QPushButton,  # 按钮
    QLabel,  # 标签
    QFileDialog,  # 文件对话框，用于选择保存路径
    QMessageBox,  # 消息框，用于显示提示信息
    QComboBox  # 下拉选择框，用于选择框架
)
# 导入PySide6核心功能
from PySide6.QtCore import Qt  # Qt命名空间和常量
# 导入PySide6图形界面相关类
from PySide6.QtGui import QFont  # 字体类


class CodeDialog(QDialog):
    def __init__(self, parent=None, code="", code_provider=None, initial_framework="PySide"):
        super().__init__(parent)
        self.code = code
        self.code_provider = code_provider
        self.current_framework = initial_framework
        self.setWindowTitle("生成的代码")
        self.resize(900, 650)
        self._setup_ui()
        self._init_framework_selection()
        self._load_code()
    
    def _setup_ui(self):
        # 使用FramelessWindow的内容区域
        content_widget = self.contentWidget()
        # 清除可能存在的布局
        if content_widget.layout():
            from PySide6.QtWidgets import QWidget
            QWidget().setLayout(content_widget.layout())
        
        main_layout = QVBoxLayout(content_widget)
        header_layout = QHBoxLayout()
        header_label = QLabel("框架：")
        self.framework_combo = QComboBox()
        self.framework_combo.addItems(["PySide", "TKinter"])
        header_layout.addWidget(header_label)
        header_layout.addWidget(self.framework_combo)
        header_layout.addStretch()
        self.label = QLabel("生成的代码：")
        header_layout.addWidget(self.label)
        main_layout.addLayout(header_layout)
        self.code_editor = QTextEdit()
        self.code_editor.setReadOnly(True)
        self.code_editor.setFont(QFont("Consolas", 10))
        main_layout.addWidget(self.code_editor)
        button_layout = QHBoxLayout()
        self.save_button = QPushButton("保存代码")
        self.save_button.clicked.connect(self._save_code)
        button_layout.addWidget(self.save_button)
        self.copy_button = QPushButton("复制代码")
        self.copy_button.clicked.connect(self._copy_code)
        button_layout.addWidget(self.copy_button)
        self.close_button = QPushButton("关闭")
        self.close_button.clicked.connect(self.close)
        button_layout.addWidget(self.close_button)
        main_layout.addLayout(button_layout)

    def _init_framework_selection(self):
        """
        初始化框架选择下拉框
        设置当前选中的框架，并连接框架改变信号
        """
        # ========== 确定框架索引 ==========
        try:
            # 检查当前框架是否在支持的框架列表中
            # 如果在列表中，使用index方法获取索引；否则使用0（默认选择第一个）
            index = ["PySide", "TKinter"].index(self.current_framework) if self.current_framework in ["PySide", "TKinter"] else 0
        except Exception:
            # 如果发生任何异常（如current_framework为None），使用默认索引0
            index = 0
        
        # ========== 设置下拉框选中项 ==========
        # 将下拉框的当前选中索引设置为计算出的索引
        self.framework_combo.setCurrentIndex(index)
        
        # ========== 连接框架改变信号 ==========
        # 连接下拉框的文本改变信号到框架改变处理方法
        # 当用户在下拉框中选择不同的框架时，会自动调用_on_framework_changed方法
        self.framework_combo.currentTextChanged.connect(self._on_framework_changed)
    
    def _load_code(self):
        """
        加载代码到代码编辑器
        如果提供了代码提供者函数，则调用它获取代码；否则使用初始代码
        """
        # ========== 检查是否有代码提供者函数 ==========
        # 检查code_provider是否为可调用对象（函数或方法）
        if callable(self.code_provider):
            # 如果是可调用对象，尝试调用它获取代码
            try:
                # 调用代码提供者函数，传入当前选中的框架名称
                # 代码提供者函数应该返回对应框架的代码字符串
                self.code = self.code_provider(self.framework_combo.currentText())
            except Exception:
                # 如果调用失败，忽略异常（保持self.code不变）
                pass
        
        # ========== 显示代码 ==========
        # 将代码文本设置到代码编辑器中（纯文本模式）
        self.code_editor.setPlainText(self.code)
    
    def _save_code(self):
        """
        保存代码到文件
        显示文件保存对话框，让用户选择保存路径，然后将代码写入文件
        """
        # ========== 显示文件保存对话框 ==========
        # 显示文件保存对话框，获取用户选择的保存路径
        # 参数：父窗口、对话框标题、默认文件名、文件类型过滤器
        file_path, _ = QFileDialog.getSaveFileName(
            self,  # 父窗口（当前对话框）
            "保存代码",  # 对话框标题
            "generated_window.py",  # 默认文件名
            "Python文件 (*.py);;所有文件 (*)"  # 文件类型过滤器（支持.py和所有文件）
        )
        
        # ========== 检查是否选择了文件路径 ==========
        # 检查用户是否选择了文件路径（file_path不为空字符串）
        if file_path:
            # 如果选择了路径，执行保存操作
            try:
                # ========== 确保目录存在 ==========
                # 获取文件所在目录的路径
                # os.path.dirname(file_path)返回目录路径，如果路径为空则返回空字符串
                # exist_ok=True：如果目录已存在则不抛出异常
                os.makedirs(os.path.dirname(file_path), exist_ok=True)
                
                # ========== 写入文件 ==========
                # 以写入模式打开文件，使用UTF-8编码
                with open(file_path, 'w', encoding='utf-8') as f:
                    # 将代码文本写入文件
                    f.write(self.code)
                
                # ========== 显示成功消息 ==========
                # 显示信息提示框，告知用户保存成功
                QMessageBox.information(self, "成功", f"代码已保存到: {file_path}")
            except Exception as e:
                # ========== 显示错误消息 ==========
                # 如果保存过程中发生任何异常，显示错误提示框
                QMessageBox.critical(self, "错误", f"保存代码失败: {str(e)}")
    
    def _copy_code(self):
        """
        复制代码到剪贴板
        选中所有代码文本，然后复制到系统剪贴板
        """
        # ========== 选中所有代码 ==========
        # 选中代码编辑器中的所有文本
        self.code_editor.selectAll()
        
        # ========== 复制到剪贴板 ==========
        # 将选中的文本复制到系统剪贴板
        self.code_editor.copy()
        
        # ========== 显示成功消息 ==========
        # 显示信息提示框，告知用户代码已复制
        QMessageBox.information(self, "成功", "代码已复制到剪贴板")

    def _on_framework_changed(self, framework):
        """
        框架选择改变时的处理函数
        当用户在下拉框中选择不同的框架时，重新加载对应框架的代码
        
        Args:
            framework: 新选择的框架名称（"PySide"或"TKinter"）
        """
        # ========== 更新当前框架 ==========
        # 保存当前选择的框架名称
        self.current_framework = framework
        
        # ========== 重新加载代码 ==========
        # 检查是否有代码提供者函数
        if callable(self.code_provider):
            # 如果有代码提供者，尝试获取新框架的代码
            try:
                # 调用代码提供者函数，传入新选择的框架名称
                # 获取对应框架的代码字符串
                self.code = self.code_provider(framework)
                # 将新代码显示到代码编辑器中
                self.code_editor.setPlainText(self.code)
            except Exception:
                # 如果获取代码失败，忽略异常（保持当前代码不变）
                pass