#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
AI提示词常量模块
集中管理所有AI提示词模板，方便后续修改和维护
"""

# ========== 系统提示词模板 ==========

SYSTEM_PROMPT_PYSIDE = """你是一个专业的Python GUI开发工程师。请严格按照以下要求生成代码：

1. **只输出代码**：不要输出任何解释、说明、注释或其他文字，只输出Python代码
2. **代码格式**：必须使用三个反引号（```）包裹全部代码，格式为：
   ```
   python
   你的代码
   ```
3. **框架要求**：必须使用PySide6框架
4. **代码完整性**：
   - 包含所有必要的导入语句（from PySide6.QtWidgets import ...等）
   - 创建一个主窗口类（继承自QMainWindow或QWidget）
   - 设置合适的窗口标题和大小
   - 创建并布局所有控件
   - 为可交互控件添加信号槽连接
   - 包含main()函数和程序入口点（if __name__ == "__main__"）
5. **代码规范**：
   - 使用中文作为界面文案
   - 控件使用绝对定位（setGeometry）
   - 代码要完整可运行，不需要用户额外修改
6. **禁止事项**：
   - 不要在代码块外添加任何文字说明
   - 不要输出"以下是代码"、"代码如下"等提示文字
   - 不要输出代码块外的注释或解释
"""

SYSTEM_PROMPT_TKINTER = """你是一个专业的Python GUI开发工程师。请严格按照以下要求生成代码：

1. **只输出代码**：不要输出任何解释、说明、注释或其他文字，只输出Python代码
2. **代码格式**：必须使用三个反引号（```）包裹全部代码，格式为：
   ```
   python
   你的代码
   ```
3. **框架要求**：必须使用tkinter/ttk框架
4. **代码完整性**：
   - 包含所有必要的导入语句（import tkinter as tk等）
   - 创建一个主窗口（tk.Tk()）
   - 设置合适的窗口标题和大小
   - 创建并布局所有控件
   - 包含基本的事件处理逻辑
   - 包含mainloop()调用
5. **代码规范**：
   - 使用中文作为界面文案
   - 代码要完整可运行，不需要用户额外修改
6. **禁止事项**：
   - 不要在代码块外添加任何文字说明
   - 不要输出"以下是代码"、"代码如下"等提示文字
   - 不要输出代码块外的注释或解释
"""

# ========== 用户提示词模板 ==========

USER_PROMPT_UI_ONLY_TEMPLATE = """请使用{framework}生成完整可运行的Python脚本：{task}。

要求：
1. 窗口标题与大小需设置
2. 控件使用绝对定位（setGeometry或place）
3. 运行即展示界面
4. 所有代码必须用三个反引号包裹，格式：```python\n代码\n```
5. 只输出代码，不要任何解释文字"""

USER_PROMPT_FUNCTIONAL_TEMPLATE = """请使用{framework}生成完整可运行的Python脚本：{task}。

要求：
1. 包含基础功能逻辑与事件处理（如输入校验、按钮点击行为、弹窗提示）
2. 控件使用绝对定位（setGeometry或place）
3. 所有代码必须用三个反引号包裹，格式：```python\n代码\n```
4. 只输出代码，不要任何解释文字
5. 使用PySide6的QMessageBox或tkinter的messagebox进行信息/警告/错误反馈"""

USER_PROMPT_WITH_WINDOW_INFO_TEMPLATE = """请使用{framework}生成完整可运行的Python脚本：{task}。

当前窗口信息：
- 窗口标题：{window_title}
- 窗口大小：{window_width} x {window_height}
- 窗口类名：{window_class_name}

要求：
1. 严格按照上述窗口信息设置窗口属性
2. 包含基础功能逻辑与事件处理
3. 控件使用绝对定位
4. 所有代码必须用三个反引号包裹，格式：```python\n代码\n```
5. 只输出代码，不要任何解释文字
6. 使用PySide6的QMessageBox或tkinter的messagebox进行信息/警告/错误反馈"""

# ========== 代码提取提示 ==========

CODE_EXTRACTION_PATTERNS = [
    r"```(?:python|py)?\s*([\s\S]*?)```",  # 标准代码块格式
    r"```\s*([\s\S]*?)```",  # 无语言标识的代码块
    r"```([\s\S]*?)```",  # 最简格式
]

# ========== 错误提示信息 ==========

ERROR_NO_CODE_EXTRACTED = "未提取到代码内容，请重试或简化问题。\n\n提示：请确保AI返回的代码使用三个反引号（```）包裹。"
ERROR_INVALID_CODE_FORMAT = "代码格式不正确，请检查AI返回的内容。"
ERROR_CODE_TOO_SHORT = "提取的代码过短，可能不完整，请重试。"

# ========== 辅助函数 ==========

def get_system_prompt(framework: str) -> str:
    """
    根据框架获取系统提示词
    
    Args:
        framework: 框架名称（"PySide"或"TKinter"）
        
    Returns:
        str: 系统提示词
    """
    framework_lower = framework.lower().strip()
    if "pyside" in framework_lower or "qt" in framework_lower:
        return SYSTEM_PROMPT_PYSIDE
    else:
        return SYSTEM_PROMPT_TKINTER


def build_user_prompt(framework: str, task: str, mode: str = "单独GUI页面", 
                     logic: str = "无", window_info: dict = None) -> str:
    """
    构建用户提示词
    
    Args:
        framework: 框架名称
        task: 任务描述
        mode: 生成模式（"单独GUI页面"或"功能GUI页面"）
        logic: 逻辑类型（"交互反馈"等）
        window_info: 窗口信息字典（可选），包含window_title, window_width, window_height, window_class_name
        
    Returns:
        str: 构建好的用户提示词
    """
    if window_info:
        # 使用窗口信息模板
        return USER_PROMPT_WITH_WINDOW_INFO_TEMPLATE.format(
            framework=framework,
            task=task,
            window_title=window_info.get("window_title", "我的窗口"),
            window_width=window_info.get("window_width", 800),
            window_height=window_info.get("window_height", 600),
            window_class_name=window_info.get("window_class_name", "GeneratedWindow")
        )
    elif (mode or '').strip() == "功能GUI页面":
        # 功能GUI页面模板
        return USER_PROMPT_FUNCTIONAL_TEMPLATE.format(
            framework=framework,
            task=task
        )
    else:
        # 单独GUI页面模板
        return USER_PROMPT_UI_ONLY_TEMPLATE.format(
            framework=framework,
            task=task
        )

