#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
AI对话模块
提供与GLM-4.5-Flash API和DeepSeek API交互的功能
包含AI对话界面、API密钥管理、代码生成等功能
"""

# 导入标准库模块
import json  # JSON数据处理
import os  # 操作系统接口，用于文件路径操作
import base64  # Base64编码/解码，用于图片等数据处理
import time  # 时间处理，用于超时检测
import requests  # HTTP请求库，用于调用AI API
from typing import List, Dict, Optional, Callable  # 类型提示

# 从PySide6.QtWidgets导入Qt界面组件
from PySide6.QtWidgets import (
    QWidget,  # 基础窗口部件
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QTextEdit,  # 富文本编辑器
    QLineEdit,  # 单行文本输入框
    QPushButton,  # 按钮
    QLabel,  # 标签
    QScrollArea,  # 滚动区域
    QFrame,  # 框架
    QCheckBox,  # 复选框
    QSplitter,  # 分割器
    QDialog,  # 对话框
    QFormLayout,  # 表单布局
    QMessageBox,  # 消息框
    QComboBox,  # 下拉选择框
    QMenu,  # 菜单
    QGroupBox,  # 分组框
    QToolButton,  # 工具按钮
    QColorDialog  # 颜色选择对话框
)
# 从PySide6.QtCore导入Qt核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QThread,  # 线程类，用于异步API调用
    Signal,  # 信号定义
    QTimer  # 定时器
)
# 从PySide6.QtGui导入Qt图形界面相关类
from PySide6.QtGui import (
    QFont,  # 字体
    QTextCursor,  # 文本光标
    QPixmap,  # 像素图
    QGuiApplication,  # 图形应用程序
    QAction,  # 动作
    QIcon,  # 图标
    QColor  # 颜色
)
import re  # 正则表达式模块，用于文本处理
# 导入提示词模板
from .prompt_templates import prompt_templates
# 导入提示词常量
try:
    from .ai_prompt_constants import (
        get_system_prompt, build_user_prompt,
        CODE_EXTRACTION_PATTERNS,
        ERROR_NO_CODE_EXTRACTED, ERROR_INVALID_CODE_FORMAT, ERROR_CODE_TOO_SHORT
    )
except ImportError:
    # 如果导入失败，定义占位函数
    def get_system_prompt(framework: str) -> str:
        return ""
    def build_user_prompt(framework: str, task: str, mode: str = "", logic: str = "", window_info: dict = None) -> str:
        return ""
    CODE_EXTRACTION_PATTERNS = [r"```[A-Za-z0-9_\-]*\s*([\s\S]*?)```"]
    ERROR_NO_CODE_EXTRACTED = "未提取到代码内容"
    ERROR_INVALID_CODE_FORMAT = "代码格式不正确"
    ERROR_CODE_TOO_SHORT = "代码过短"


class AISettingsDialog(QDialog):
    """
    AI设置对话框
    整合所有AI相关的设置选项，包括API密钥、框架选择、类型选择、代码生成等
    """
    def __init__(self, parent=None):
        """
        初始化AI设置对话框
        
        Args:
            parent: 父窗口对象，默认为None
        """
        super().__init__(parent)
        self.setWindowTitle("AI设置")
        self.setModal(True)
        self.parent_widget = parent
        self._init_ui()
    
    def _init_ui(self):
        """
        初始化用户界面
        创建所有设置选项的UI组件
        """
        layout = QVBoxLayout(self)
        layout.setContentsMargins(15, 15, 15, 15)
        layout.setSpacing(10)
        
        # ========== API密钥设置区域 ==========
        api_group = QGroupBox("API密钥设置")
        api_layout = QFormLayout()
        
        self.api_key_edit = QLineEdit()
        self.api_key_edit.setEchoMode(QLineEdit.Password)
        self.api_key_edit.setPlaceholderText("请输入您的 API 密钥")
        api_layout.addRow("API密钥:", self.api_key_edit)
        
        self.provider_select = QComboBox()
        self.provider_select.addItems(["GLM", "DeepSeek"])
        api_layout.addRow("服务商:", self.provider_select)
        
        self.model_select = QComboBox()
        self.model_select.addItem("质量稍好", userData="GLM-4.5-Flash")
        self.model_select.addItem("速度最快", userData="glm-4-flash-250414")
        api_layout.addRow("选择模型:", self.model_select)
        
        self.model_desc_label = QLabel("")
        api_layout.addRow("模型说明:", self.model_desc_label)
        
        api_group.setLayout(api_layout)
        layout.addWidget(api_group)
        
        # ========== 代码生成设置区域 ==========
        code_group = QGroupBox("代码生成设置")
        code_layout = QFormLayout()
        
        self.framework_select = QComboBox()
        self.framework_select.addItems(["PySide", "TKinter"])
        code_layout.addRow("框架:", self.framework_select)
        
        self.mode_select = QComboBox()
        self.mode_select.addItems(["单独GUI页面", "功能GUI页面"])
        code_layout.addRow("类型:", self.mode_select)
        
        self.template_prompt_edit = QLineEdit()
        self.template_prompt_edit.setPlaceholderText("如: 生成一个登录器")
        code_layout.addRow("提问:", self.template_prompt_edit)
        
        code_group.setLayout(code_layout)
        layout.addWidget(code_group)
        
        # ========== 操作按钮区域 ==========
        button_layout = QHBoxLayout()
        
        self.clear_button = QPushButton("清空对话")
        self.clear_button.clicked.connect(self._on_clear_chat)
        button_layout.addWidget(self.clear_button)
        
        button_layout.addStretch()
        
        self.ok_button = QPushButton("确定")
        self.ok_button.clicked.connect(self.accept)
        button_layout.addWidget(self.ok_button)
        
        self.cancel_button = QPushButton("取消")
        self.cancel_button.clicked.connect(self.reject)
        button_layout.addWidget(self.cancel_button)
        
        layout.addLayout(button_layout)
        
        # 连接信号
        self.provider_select.currentIndexChanged.connect(self._on_provider_changed)
        self.model_select.currentIndexChanged.connect(self._update_model_desc)
        
        # 设置对话框大小
        self.setFixedSize(500, 350)
        
        # 初始化对话框状态
        self._load_current_settings()
    
    def _on_provider_changed(self, idx):
        """当服务商选择改变时的处理函数"""
        p = self.provider_select.currentText()
        self.model_select.clear()
        if p == "DeepSeek":
            self.model_select.addItem("deepseek-chat", userData="deepseek-chat")
            self.model_select.addItem("deepseek-reasoner", userData="deepseek-reasoner")
        else:
            self.model_select.addItem("质量稍好", userData="GLM-4.5-Flash")
            self.model_select.addItem("速度最快", userData="glm-4-flash-250414")
        self._update_model_desc()
    
    def _update_model_desc(self):
        """更新模型说明标签的文本"""
        try:
            p = self.provider_select.currentText()
            sel = self.model_select.currentData()
            if p == "DeepSeek":
                if sel == "deepseek-reasoner":
                    self.model_desc_label.setText("模式: reasoning（思考模式）")
                elif sel == "deepseek-chat":
                    self.model_desc_label.setText("模式: normal（对话模式）")
                else:
                    self.model_desc_label.setText("")
            else:
                self.model_desc_label.setText("")
        except Exception:
            self.model_desc_label.setText("")
    
    def _load_current_settings(self):
        """从父组件加载当前设置"""
        if not self.parent_widget:
            return
        
        try:
            # 加载API密钥设置
            self.provider_select.setCurrentText(self.parent_widget.provider)
            self.model_select.setCurrentIndex(0)  # 默认选择第一个
            for i in range(self.model_select.count()):
                if self.model_select.itemData(i) == self.parent_widget.model_id:
                    self.model_select.setCurrentIndex(i)
                    break
            
            # 加载API密钥
            self._load_api_key()
            
            # 加载代码生成设置
            if hasattr(self.parent_widget, 'framework_select'):
                framework = self.parent_widget.framework_select.currentText()
                self.framework_select.setCurrentText(framework)
            
            if hasattr(self.parent_widget, 'mode_select'):
                mode = self.parent_widget.mode_select.currentText()
                self.mode_select.setCurrentText(mode)
            
            if hasattr(self.parent_widget, 'template_prompt_edit'):
                prompt = self.parent_widget.template_prompt_edit.text()
                self.template_prompt_edit.setText(prompt)
        except Exception:
            pass
        
        # 连接服务商和模型改变信号，自动更新密钥
        self.provider_select.currentIndexChanged.connect(lambda _: self._load_api_key())
        self.model_select.currentIndexChanged.connect(lambda _: self._load_api_key())
    
    def _load_api_key(self):
        """从配置文件加载API密钥"""
        if not self.parent_widget:
            return
        
        try:
            p = self.provider_select.currentText()
            m = self.model_select.currentData() or self.parent_widget.model_id
            
            # 读取配置文件
            config_path = self.parent_widget._config_path()
            if os.path.isfile(config_path):
                with open(config_path, "r", encoding="utf-8") as f:
                    cfg = json.load(f)
                keys = cfg.get("keys", {})
                if isinstance(keys, dict):
                    pv = keys.get(p)
                    if isinstance(pv, dict):
                        mv = pv.get(m)
                        if isinstance(mv, dict) and "salt" in mv and "data" in mv:
                            self.api_key_edit.setText(self.parent_widget._decrypt(mv))
                        elif isinstance(mv, str):
                            self.api_key_edit.setText(mv)
                        else:
                            self.api_key_edit.setText("")
                    elif isinstance(pv, dict) and "salt" in pv and "data" in pv:
                        self.api_key_edit.setText(self.parent_widget._decrypt(pv))
                    else:
                        self.api_key_edit.setText("")
                else:
                    self.api_key_edit.setText("")
            else:
                self.api_key_edit.setText("")
        except Exception:
            self.api_key_edit.setText("")
    
    
    def _on_clear_chat(self):
        """清空对话按钮点击处理"""
        if not self.parent_widget:
            return
        self.parent_widget.clear_chat()
    
    def _save_settings(self):
        """保存设置到父组件"""
        if not self.parent_widget:
            return
        
        try:
            # 保存API密钥设置
            new_api_key = self.api_key_edit.text().strip()
            if new_api_key:
                self.parent_widget.api_key = new_api_key
                self.parent_widget.model_id = self.model_select.currentData() or self.parent_widget.model_id
                self.parent_widget.provider = self.provider_select.currentText()
                self.parent_widget._save_settings()
            
            # 保存代码生成设置（这些设置不需要保存到文件，只保存在内存中）
            if hasattr(self.parent_widget, 'framework_select'):
                self.parent_widget.framework_select.setCurrentText(self.framework_select.currentText())
            if hasattr(self.parent_widget, 'mode_select'):
                self.parent_widget.mode_select.setCurrentText(self.mode_select.currentText())
            if hasattr(self.parent_widget, 'template_prompt_edit'):
                self.parent_widget.template_prompt_edit.setText(self.template_prompt_edit.text())
        except Exception:
            pass
    
    def get_api_key(self):
        """获取API密钥"""
        return self.api_key_edit.text().strip()
    
    def get_selected_model(self):
        """获取选中的模型ID"""
        return self.model_select.currentData()
    
    def get_provider(self):
        """获取选中的服务商"""
        return self.provider_select.currentText()
    
    def accept(self):
        """确定按钮点击处理"""
        self._save_settings()
        super().accept()


class APIKeyDialog(QDialog):
    """
    API密钥设置对话框
    用于用户输入和配置AI服务的API密钥、选择服务商和模型
    """
    def __init__(self, parent=None):
        """
        初始化API密钥对话框
        
        Args:
            parent: 父窗口对象，默认为None
        """
        # 调用父类构造函数
        super().__init__(parent)
        # 设置对话框标题
        self.setWindowTitle("设置API密钥")
        # 设置为模态对话框（阻塞父窗口）
        self.setModal(True)
        # 初始化用户界面
        self._init_ui()
    
    def _init_ui(self):
        """
        初始化用户界面
        创建表单布局，包含API密钥输入、服务商选择、模型选择等控件
        """
        # 创建垂直布局作为主布局
        layout = QVBoxLayout(self)
        
        # ========== 创建表单布局 ==========
        form_layout = QFormLayout()
        
        # API密钥输入框：用于输入AI服务的API密钥
        self.api_key_edit = QLineEdit()
        # 设置为密码模式，输入时显示为圆点或星号，保护密钥安全
        self.api_key_edit.setEchoMode(QLineEdit.Password)
        # 设置占位符文本，提示用户输入内容
        self.api_key_edit.setPlaceholderText("请输入您的 API 密钥")
        # 将API密钥输入框添加到表单布局
        form_layout.addRow("API密钥:", self.api_key_edit)

        # 服务商选择下拉框：选择AI服务提供商（GLM或DeepSeek）
        self.provider_select = QComboBox()
        # 添加可选的服务商列表
        self.provider_select.addItems(["GLM", "DeepSeek"])
        # 将服务商选择框添加到表单布局
        form_layout.addRow("服务商:", self.provider_select)

        # 模型选择下拉框：选择具体的AI模型
        self.model_select = QComboBox()
        # 添加"质量稍好"选项，对应的模型ID为"GLM-4.5-Flash"
        self.model_select.addItem("质量稍好", userData="GLM-4.5-Flash")
        # 添加"速度最快"选项，对应的模型ID为"glm-4-flash-250414"
        self.model_select.addItem("速度最快", userData="glm-4-flash-250414")
        # 将模型选择框添加到表单布局
        form_layout.addRow("选择模型:", self.model_select)

        # 模型说明标签：显示当前选择模型的说明信息
        self.model_desc_label = QLabel("")
        # 将模型说明标签添加到表单布局
        form_layout.addRow("模型说明:", self.model_desc_label)

        # 将表单布局添加到主布局中
        layout.addLayout(form_layout)
        
        # ========== 创建按钮布局 ==========
        button_layout = QHBoxLayout()
        # 创建"确定"按钮：用于保存设置并关闭对话框
        self.ok_button = QPushButton("确定")
        # 创建"取消"按钮：用于取消操作并关闭对话框
        self.cancel_button = QPushButton("取消")
        
        # 将确定按钮的点击信号连接到accept槽（关闭对话框并返回QDialog.Accepted）
        self.ok_button.clicked.connect(self.accept)
        # 将取消按钮的点击信号连接到reject槽（关闭对话框并返回QDialog.Rejected）
        self.cancel_button.clicked.connect(self.reject)
        
        # 添加弹性空间，使按钮靠右对齐
        button_layout.addStretch()
        # 将确定按钮添加到按钮布局
        button_layout.addWidget(self.ok_button)
        # 将取消按钮添加到按钮布局
        button_layout.addWidget(self.cancel_button)
        
        # 将按钮布局添加到主布局
        layout.addLayout(button_layout)
        
        # 设置对话框固定大小：宽度520像素，高度220像素
        self.setFixedSize(520, 220)

        def _on_provider_changed(idx):
            """
            当服务商选择改变时的处理函数
            根据选择的服务商更新可用的模型列表
            
            Args:
                idx: 当前选中的服务商索引（未使用，但信号会传递此参数）
            """
            # 获取当前选中的服务商名称
            p = self.provider_select.currentText()
            # 清空模型选择框中的现有选项
            self.model_select.clear()
            # 根据服务商类型添加对应的模型选项
            if p == "DeepSeek":
                # 如果选择DeepSeek，添加deepseek-chat模型（对话模式）
                self.model_select.addItem("deepseek-chat", userData="deepseek-chat")
                # 添加deepseek-reasoner模型（推理模式）
                self.model_select.addItem("deepseek-reasoner", userData="deepseek-reasoner")
            else:
                # 如果选择GLM或其他，添加GLM模型选项
                self.model_select.addItem("质量稍好", userData="GLM-4.5-Flash")
                self.model_select.addItem("速度最快", userData="glm-4-flash-250414")
            # 更新模型说明标签的文本
            _update_model_desc()
        
        def _update_model_desc():
            """
            更新模型说明标签的文本
            根据当前选择的服务商和模型显示相应的说明信息
            """
            try:
                # 获取当前选中的服务商名称
                p = self.provider_select.currentText()
                # 获取当前选中模型的ID（存储在userData中）
                sel = self.model_select.currentData()
                # 根据服务商类型设置说明文本
                if p == "DeepSeek":
                    if sel == "deepseek-reasoner":
                        # DeepSeek推理模式说明
                        self.model_desc_label.setText("模式: reasoning（思考模式）")
                    elif sel == "deepseek-chat":
                        # DeepSeek对话模式说明
                        self.model_desc_label.setText("模式: normal（对话模式）")
                    else:
                        # 其他情况清空说明
                        self.model_desc_label.setText("")
                else:
                    # GLM服务商暂时不显示说明
                    self.model_desc_label.setText("")
            except Exception:
                # 发生异常时清空说明文本
                self.model_desc_label.setText("")
        
        # 连接模型选择框的索引改变信号到更新说明函数
        self.model_select.currentIndexChanged.connect(lambda _: _update_model_desc())
        # 连接服务商选择框的索引改变信号到服务商改变处理函数
        self.provider_select.currentIndexChanged.connect(_on_provider_changed)
    
    def get_api_key(self):
        """
        获取用户输入的API密钥
        
        Returns:
            str: 去除首尾空白字符后的API密钥字符串
        """
        return self.api_key_edit.text().strip()

    def get_selected_model(self):
        """
        获取当前选中的模型ID
        
        Returns:
            str: 当前选中模型对应的userData值（模型ID）
        """
        return self.model_select.currentData()

    def set_selected_model(self, model_id: str):
        """
        设置选中的模型
        根据模型ID查找并选中对应的模型选项
        
        Args:
            model_id: 要选择的模型ID（如"GLM-4.5-Flash"、"deepseek-chat"等）
        """
        # ========== 遍历模型选择框中的所有项 ==========
        # 遍历模型选择框中的所有选项，查找匹配的模型ID
        # range(self.model_select.count())：生成从0到选项总数-1的索引序列
        for i in range(self.model_select.count()):
            # ========== 检查模型ID是否匹配 ==========
            # 获取当前索引对应选项的userData（存储的模型ID），与目标模型ID比较
            if self.model_select.itemData(i) == model_id:
                # 如果找到匹配的模型ID，将该选项设置为当前选中项
                self.model_select.setCurrentIndex(i)
                # 找到匹配项后，退出循环（不需要继续查找）
                break

    def get_provider(self):
        """
        获取当前选中的服务商名称
        
        Returns:
            str: 当前选中服务商的文本（如"GLM"或"DeepSeek"）
        """
        return self.provider_select.currentText()

    def set_provider(self, provider: str):
        """
        设置选中的服务商
        根据服务商名称查找并选中对应的服务商选项
        
        Args:
            provider: 要选择的服务商名称（如"GLM"或"DeepSeek"）
        """
        # ========== 遍历服务商选择框中的所有项 ==========
        # 遍历服务商选择框中的所有选项，查找匹配的服务商名称
        # range(self.provider_select.count())：生成从0到选项总数-1的索引序列
        for i in range(self.provider_select.count()):
            # ========== 检查服务商名称是否匹配 ==========
            # 获取当前索引对应选项的显示文本（服务商名称），与目标服务商名称比较
            if self.provider_select.itemText(i) == provider:
                # 如果找到匹配的服务商名称，将该选项设置为当前选中项
                self.provider_select.setCurrentIndex(i)
                # 找到匹配项后，退出循环（不需要继续查找）
                break

    def get_api_url(self):
        """
        获取API URL（当前未实现，返回空字符串）
        
        Returns:
            str: API URL，当前始终返回空字符串
        """
        return ""

    def set_api_url(self, url: str):
        """
        设置API URL（当前未实现）
        
        Args:
            url: API URL字符串，当前不做任何处理
        """
        pass


class ChatAPIWorker(QThread):
    """
    API请求工作线程
    在后台线程中执行AI API调用，避免阻塞主界面
    使用Qt的信号机制将结果传回主线程
    支持流式返回（streaming）
    """
    # 定义信号：接收到API响应时发出，参数为响应内容字符串（完整响应）
    response_received = Signal(str)
    # 定义信号：接收到流式数据片段时发出，参数为增量内容字符串
    stream_chunk_received = Signal(str)
    # 定义信号：流式传输完成时发出
    stream_finished = Signal()
    # 定义信号：发生错误时发出，参数为错误信息字符串
    error_occurred = Signal(str)
    
    def __init__(self, api_key: str, messages: List[Dict[str, str]], model: str, provider: str, api_url: str):
        """
        初始化API工作线程
        
        Args:
            api_key: AI服务的API密钥
            messages: 对话消息列表，格式为[{"role": "user", "content": "..."}, ...]
            model: 要使用的AI模型ID
            provider: 服务商名称（"GLM"或"DeepSeek"）
            api_url: API端点URL，如果为None则根据provider自动选择默认URL
        """
        # 调用父类QThread的构造函数
        super().__init__()
        # 保存API密钥
        self.api_key = api_key
        # 保存对话消息列表
        self.messages = messages
        # 保存模型ID
        self.model = model
        # 保存服务商名称，默认为"GLM"
        self.provider = provider or "GLM"
        # 设置API URL：如果未提供，则根据服务商选择默认URL
        self.api_url = api_url or ("https://open.bigmodel.cn/api/paas/v4/chat/completions" if (self.provider == "GLM") else "https://api.deepseek.com/v1/chat/completions")
    
    def run(self):
        """
        执行API请求（在线程中运行）
        这是QThread的工作方法，在线程启动时自动调用
        """
        # 设置HTTP请求头：包含认证信息和内容类型
        headers = {"Authorization": f"Bearer {self.api_key}", "Content-Type": "application/json"}
        
        # ========== 处理模型ID ==========
        # 获取模型ID：如果模型ID为空或None，使用默认值"GLM-4.5-Flash"，然后去除首尾空白字符
        mi = (self.model or "GLM-4.5-Flash").strip()
        # ========== 规范化模型ID（针对GLM服务商） ==========
        # 检查当前服务商是否为"GLM"
        if self.provider == "GLM":
            # 如果是GLM服务商，进一步规范化模型ID
            # 检查模型ID是否为中文描述"质量稍好"或包含说明的格式
            if mi in ["质量稍好", "GLM-4.5-Flash（中速）"]:
                # 如果是质量稍好选项，统一转换为标准模型ID
                mi = "GLM-4.5-Flash"
            # 检查模型ID是否为"速度最快"的描述格式
            elif mi in ["速度最快", "glm-4-flash-250414（高速）"]:
                # 如果是速度最快选项，统一转换为高速模型ID
                mi = "glm-4-flash-250414"
            # 检查模型ID是否不在已知的标准模型ID列表中
            elif mi not in ["GLM-4.5-Flash", "glm-4-flash-250414"]:
                # 如果模型ID不在已知列表中，使用默认模型（质量稍好的模型）
                mi = "GLM-4.5-Flash"
        
        # ========== 设置最大token数 ==========
        # 初始化最大token数为4096（GLM服务的默认值）
        mt = 4096
        # 检查当前服务商是否为"DeepSeek"
        if self.provider == "DeepSeek":
            # 如果是DeepSeek服务商，将最大token数设置为8192（DeepSeek支持更大的token数）
            mt = 8192
        # 构建API请求数据，启用流式返回
        data = {"model": mi, "messages": self.messages, "temperature": 0.7, "max_tokens": mt, "stream": True}
        
        try:
            # ========== 流式请求处理 ==========
            max_retries = 3  # 最大重试次数
            retry_count = 0  # 当前重试计数
            success = False  # 是否成功标志
            
            # 在最大重试次数内重试请求
            while retry_count < max_retries and not success:
                try:
                    # 发送流式POST请求
                    # 增加超时时间：连接超时30秒，读取超时600秒（10分钟），适应长响应
                    response = requests.post(
                        self.api_url,  # API端点URL
                        headers=headers,  # 请求头
                        json=data,  # 请求体（JSON格式）
                        stream=True,  # 启用流式传输
                        timeout=(30, 600)  # 超时设置：(连接超时30秒, 读取超时600秒)
                    )
                    
                    # ========== 处理流式API响应 ==========
                    # 检查HTTP响应状态码是否为200（请求成功）
                    if response.status_code == 200:
                        # 初始化完整内容字符串
                        full_content = ""
                        # 初始化块计数器，用于调试
                        chunk_count = 0
                        # 初始化最后活动时间，用于检测连接是否还活着
                        last_activity_time = time.time()
                        
                        # 逐行读取流式响应
                        # 使用iter_lines的chunk_size参数，确保及时读取数据
                        try:
                            for line in response.iter_lines(decode_unicode=False, chunk_size=8192):
                                # 更新最后活动时间
                                current_time = time.time()
                                
                                # 检查是否超时（如果超过60秒没有新数据，认为连接可能已断开）
                                if current_time - last_activity_time > 60:
                                    print(f"[DEBUG] 流式响应超时：超过60秒未收到新数据")
                                    # 发送错误信号
                                    self.error_occurred.emit("流式响应超时：超过60秒未收到新数据，连接可能已断开")
                                    break
                                
                                if not line:
                                    # 空行可能是心跳包，更新活动时间
                                    last_activity_time = current_time
                                    continue
                                
                                # 处理SSE格式（Server-Sent Events）
                                try:
                                    line_text = line.decode('utf-8', errors='replace')
                                except UnicodeDecodeError:
                                    # 如果解码失败，尝试其他编码或跳过
                                    print(f"[DEBUG] 解码失败，跳过该行")
                                    continue
                                
                                # 更新最后活动时间
                                last_activity_time = current_time
                                
                                # 处理不同的SSE格式
                                if line_text.startswith('data: '):
                                    data_str = line_text[6:]  # 移除 'data: ' 前缀
                                elif line_text.startswith('data:'):
                                    data_str = line_text[5:]  # 移除 'data:' 前缀（无空格）
                                else:
                                    # 如果不是标准SSE格式，尝试直接解析
                                    data_str = line_text.strip()
                                
                                # 检查是否是结束标记
                                if data_str.strip() == '[DONE]':
                                    print(f"[DEBUG] 收到结束标记 [DONE]")
                                    break
                                
                                # 跳过空数据
                                if not data_str.strip():
                                    continue
                                
                                try:
                                    # 解析JSON数据
                                    chunk_data = json.loads(data_str)
                                    # 提取增量内容
                                    choices = chunk_data.get("choices", [])
                                    if choices:
                                        delta = choices[0].get("delta", {})
                                        chunk_content = delta.get("content", "")
                                        
                                        if chunk_content:
                                            # 累积内容
                                            full_content += chunk_content
                                            chunk_count += 1
                                            # 发送增量内容到主线程
                                            self.stream_chunk_received.emit(chunk_content)
                                            
                                            # 每100个块打印一次调试信息
                                            if chunk_count % 100 == 0:
                                                print(f"[DEBUG] 已接收 {chunk_count} 个数据块，总长度: {len(full_content)} 字符")
                                    
                                    # 检查是否完成
                                    finish_reason = choices[0].get("finish_reason") if choices else None
                                    if finish_reason:
                                        print(f"[DEBUG] 流式响应完成，原因: {finish_reason}, 总块数: {chunk_count}, 总长度: {len(full_content)}")
                                        break
                                        
                                except json.JSONDecodeError as e:
                                    # JSON解析错误，记录但继续处理
                                    print(f"[DEBUG] JSON解析错误: {e}, 数据: {data_str[:100]}")
                                    continue
                                except Exception as e:
                                    # 其他错误，记录但继续处理
                                    print(f"[DEBUG] 处理数据块时出错: {e}")
                                    import traceback
                                    traceback.print_exc()
                                    continue
                        except requests.exceptions.Timeout:
                            print(f"[DEBUG] 流式响应读取超时")
                            self.error_occurred.emit("流式响应读取超时，请检查网络连接或重试")
                        except requests.exceptions.ConnectionError as e:
                            print(f"[DEBUG] 流式响应连接错误: {e}")
                            self.error_occurred.emit(f"流式响应连接错误: {str(e)}")
                        except Exception as e:
                            print(f"[DEBUG] 流式响应处理异常: {e}")
                            import traceback
                            traceback.print_exc()
                            self.error_occurred.emit(f"流式响应处理异常: {str(e)}")
                        
                        # 流式传输完成
                        self.stream_finished.emit()
                        
                        # 发送完整内容（用于兼容性）
                        if full_content.strip():
                            self.response_received.emit(full_content)
                            success = True
                        else:
                            self.response_received.emit("未获取到回复内容")
                            success = True
                    else:
                        # ========== 处理API请求失败 ==========
                        # 如果HTTP状态码不是200（请求失败），执行以下操作
                        # 构建错误消息：包含状态码和响应文本
                        error_msg = f"API请求失败: {response.status_code} - {response.text}"
                        # 通过信号发送错误消息到主线程
                        self.error_occurred.emit(error_msg)
                        # 跳出重试循环，不再重试（因为可能是认证错误等不需要重试的错误）
                        break
                except requests.exceptions.Timeout:
                    # ========== 处理请求超时异常 ==========
                    # 捕获请求超时异常（timeout异常）
                    # 增加重试计数，表示此次请求失败，需要重试
                    retry_count += 1
                    # 检查重试次数是否小于最大重试次数
                    if retry_count < max_retries:
                        # 如果还可以重试，执行以下操作
                        # 等待2秒后再次尝试请求（避免频繁重试导致服务器压力）
                        time.sleep(2)
                    else:
                        # 如果已达到最大重试次数，不再重试
                        # 发送超时错误消息到主线程
                        self.error_occurred.emit("请求超时，请检查网络连接后重试")
                        # 注意：不设置success = True，表示请求失败
                
                except requests.exceptions.ConnectionError as e:
                    # ========== 处理连接错误异常 ==========
                    # 捕获连接错误异常（网络连接失败、DNS解析失败等）
                    # 检查错误信息中是否包含DNS解析相关的错误标识
                    if "NameResolutionError" in str(e) or "getaddrinfo failed" in str(e):
                        # 如果是DNS解析失败，执行以下操作
                        # 发送DNS解析失败的错误消息到主线程
                        self.error_occurred.emit("DNS解析失败，无法连接到API服务器。请检查网络连接或尝试使用VPN/代理")
                        # 跳出重试循环（DNS解析失败不需要重试，需要用户修复网络问题）
                        break
                    else:
                        # 如果是其他类型的连接错误，执行以下操作
                        # 增加重试计数
                        retry_count += 1
                        # 检查是否还可以重试
                        if retry_count < max_retries:
                            # 如果可以重试，等待后继续循环
                            # 等待2秒
                            time.sleep(2)
                            # 继续下一次循环，重试请求
                            continue
                        else:
                            # 如果已达到最大重试次数，不再重试
                            # 发送连接错误消息到主线程，包含具体错误信息
                            self.error_occurred.emit(f"连接错误: {str(e)}")
                            # 跳出重试循环
                            break
                
                except Exception as e:
                    # ========== 处理其他未知异常 ==========
                    # 捕获所有其他类型的异常（未预期的错误）
                    # 发送异常错误消息到主线程，包含异常信息
                    self.error_occurred.emit(f"请求异常: {str(e)}")
                    # 跳出重试循环（未知异常通常不需要重试）
                    break
        except Exception as e:
            self.error_occurred.emit(f"请求失败: {str(e)}")


class MessageBubble(QFrame):
    """
    消息气泡组件
    用于在聊天界面中显示用户消息和AI回复消息
    支持消息复制功能
    """
    def __init__(self, message: str, is_user: bool = True, parent=None):
        """
        初始化消息气泡
        
        Args:
            message: 要显示的消息内容
            is_user: 是否为用户消息（True表示用户消息，False表示AI回复）
            parent: 父窗口部件
        """
        # 调用父类构造函数
        super().__init__(parent)
        # 保存消息类型标识
        self.is_user = is_user
        # 保存消息内容
        self.message = message
        # 初始化用户界面
        self._init_ui()
    
    def _init_ui(self):
        """
        初始化用户界面
        创建消息标签并设置样式
        """
        # 设置框架形状为无边框
        self.setFrameShape(QFrame.NoFrame)
        # 创建水平布局管理器
        layout = QHBoxLayout(self)
        # 设置布局边距：左、上、右、下各10、5、10、5像素
        layout.setContentsMargins(10, 5, 10, 5)
        
        # ========== 创建消息标签 ==========
        message_label = QLabel(self.message)
        # 启用文本自动换行
        message_label.setWordWrap(True)
        # 设置文本格式为纯文本（不支持HTML）
        message_label.setTextFormat(Qt.PlainText)
        try:
            # 启用文本选择功能（可通过鼠标和键盘选择）
            message_label.setTextInteractionFlags(Qt.TextSelectableByMouse | Qt.TextSelectableByKeyboard)
            # 设置上下文菜单策略为自定义
            message_label.setContextMenuPolicy(Qt.CustomContextMenu)
            # 连接上下文菜单请求信号到显示菜单函数
            message_label.customContextMenuRequested.connect(self._show_context_menu)
        except Exception:
            pass
        
        # ========== 设置样式 ==========
        if self.is_user:
            # 用户消息样式：绿色背景，右对齐
            message_label.setStyleSheet("""
                QLabel {
                    background-color: #DCF8C6;
                    color: #303030;
                    border-radius: 10px;
                    padding: 8px 12px;
                    font-size: 14px;
                }
            """)
            # 添加弹性空间，使消息靠右显示
            layout.addStretch()
            # 将消息标签添加到布局
            layout.addWidget(message_label)
            # 设置最大宽度为400像素，防止消息过宽
            message_label.setMaximumWidth(400)
        else:
            # AI回复样式：浅蓝色背景，左对齐，更明显的气泡效果
            message_label.setStyleSheet("""
                QLabel {
                    background-color: #E3F2FD;
                    color: #1976D2;
                    border: 1px solid #90CAF9;
                    border-radius: 12px;
                    padding: 10px 14px;
                    font-size: 14px;
                }
            """)
            # 设置最大宽度为父窗口宽度的80%，确保内容能完整显示
            # 不设置固定最大宽度，让气泡自适应内容
            message_label.setMaximumWidth(16777215)  # Qt的最大值，实际由布局控制
            # 将消息标签添加到布局
            layout.addWidget(message_label)
            # 添加弹性空间，使消息靠左显示
            layout.addStretch()

        # 保存消息标签引用，用于后续操作
        self._message_label = message_label
    
    def update_message(self, new_text: str):
        """
        更新消息内容（用于流式更新）
        优化长文本更新性能，避免UI卡顿
        
        Args:
            new_text: 新的消息内容
        """
        if self._message_label:
            try:
                # 直接更新文本
                self._message_label.setText(new_text)
                self.message = new_text
                
                # 对于超长文本（>10000字符），限制更新频率
                # 通过检查文本长度变化来决定是否需要立即更新UI
                if len(new_text) > 10000:
                    # 对于超长文本，每增加1000字符才强制刷新一次
                    # 这样可以避免频繁的UI更新导致的卡顿
                    pass  # QLabel的setText已经足够高效
                
            except Exception as e:
                # 如果更新失败，记录错误但不中断流程
                print(f"[DEBUG] 更新消息气泡失败: {e}")
                import traceback
                traceback.print_exc()

    def _show_context_menu(self, pos):
        """
        显示上下文菜单（右键菜单）
        
        Args:
            pos: 鼠标点击位置（相对于消息标签）
        """
        try:
            # 创建上下文菜单
            menu = QMenu(self)
            # 创建"复制"动作
            copy_all = QAction("复制", self)
            # 定义复制功能函数
            def do_copy():
                try:
                    # 将消息内容复制到剪贴板
                    QGuiApplication.clipboard().setText(self.message)
                except Exception:
                    pass
            # 连接复制动作的触发信号到复制函数
            copy_all.triggered.connect(do_copy)
            # 将复制动作添加到菜单
            menu.addAction(copy_all)
            # 获取消息标签并转换为全局坐标
            label = self._message_label
            # 将局部坐标转换为全局坐标，用于显示菜单
            gp = label.mapToGlobal(pos) if label else self.mapToGlobal(pos)
            # 在全局坐标位置显示菜单
            menu.exec(gp)
        except Exception:
            pass


def _validate_license() -> bool:
    """
    验证许可证
    通过访问远程服务器验证软件许可证是否有效
    
    Returns:
        bool: 许可证验证是否通过
    """
    # 导入requests库，用于发送HTTP请求
    return True  # 许可证验证通过


class AIChatWidget(QWidget):
    """
    AI聊天组件
    提供与AI对话的界面，支持代码生成、代码解析等功能
    """
    # 定义信号：代码生成完成时发出，参数为生成的代码和框架名称
    code_generated = Signal(str, str)

    def __init__(self, parent=None):
        """
        初始化AI聊天组件
        
        Args:
            parent: 父窗口部件
        """
        # 调用父类构造函数
        super().__init__(parent)
        
        # ========== 验证许可证 ==========
        # 调用许可证验证函数，检查软件许可证是否有效
        if not _validate_license():
            # 如果许可证验证失败（返回False），执行以下操作
            # 导入消息框模块，用于显示错误提示
            from PySide6.QtWidgets import QMessageBox
            # 显示严重错误消息框，提示用户联系作者
            QMessageBox.critical(None, "错误", "许可证验证错误")
            # 注意：不退出程序，只禁用AI功能，允许用户继续使用其他功能
        
        # ========== 初始化实例属性 ==========
        # API密钥：用于身份验证的密钥字符串，初始为空，需要用户配置
        self.api_key = ""
        # 消息列表：存储对话历史消息，格式为[{"role": "user/assistant", "content": "..."}, ...]
        self.messages = []
        # 模型ID：当前使用的AI模型标识符，默认为GLM-4.5-Flash（质量较好的模型）
        self.model_id = "GLM-4.5-Flash"
        # 服务商：AI服务提供商名称，默认为"GLM"（智谱AI），可选"DeepSeek"
        self.provider = "GLM"
        # API URL：API接口地址，如果为空则使用默认地址（根据provider自动选择）
        self.api_url = ""
        # 输入框颜色设置（将从主题文件读取）
        self._input_bg_color = QColor(255, 255, 255)  # 默认白色背景
        self._input_text_color = QColor(0, 0, 0)  # 默认黑色文字
        # 初始化用户界面：创建所有UI组件
        self._init_ui()
        # 加载设置：从配置文件或本地存储中读取保存的API密钥等配置
        self._load_settings()
        # 从主题文件加载输入框颜色
        self._load_input_colors_from_theme()
        # 代码模式激活标志：标记是否处于代码生成模式（True表示代码模式，False表示对话模式）
        self._code_mode_active = False
        # 请求激活标志：标记当前是否有API请求正在进行（True表示有请求，False表示无请求）
        self._request_active = False
        # 助手更新标志：标记是否已收到AI助手的回复更新（用于控制状态显示）
        self._has_assistant_update = False
        # 等待状态显示标志：标记是否已显示"等待中"状态提示（避免重复显示）
        self._waiting_status_shown = False
        # 当前流式消息气泡（用于逐步更新内容）
        self._current_streaming_bubble = None
    
    def _init_ui(self):
        """
        初始化用户界面
        创建AI聊天组件的所有UI元素，包括设置按钮、聊天区域、输入区域等
        """
        # ========== 创建主布局 ==========
        layout = QVBoxLayout(self)
        layout.setContentsMargins(5, 5, 5, 5)
        layout.setSpacing(5)
        
        # ========== 创建顶部工具栏（只包含设置按钮）==========
        toolbar_layout = QHBoxLayout()
        toolbar_layout.setContentsMargins(0, 0, 0, 0)
        toolbar_layout.setSpacing(5)
        
        # 设置按钮
        self.settings_button = QToolButton()
        self.settings_button.setText("⚙")
        self.settings_button.setToolTip("AI设置")
        self.settings_button.clicked.connect(self._show_settings_dialog)
        self.settings_button.setFixedSize(30, 30)
        toolbar_layout.addWidget(self.settings_button)
        
        toolbar_layout.addStretch()
        layout.addLayout(toolbar_layout)
        
        # ========== 创建聊天区域（占据剩余空间）==========
        self.chat_scroll = QScrollArea()
        self.chat_scroll.setWidgetResizable(True)
        self.chat_scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        
        self.chat_widget = QWidget()
        self.chat_layout = QVBoxLayout(self.chat_widget)
        self.chat_layout.setAlignment(Qt.AlignTop)
        self.chat_layout.setContentsMargins(10, 10, 10, 10)
        self.chat_layout.setSpacing(8)
        
        self.chat_scroll.setWidget(self.chat_widget)
        layout.addWidget(self.chat_scroll)
        
        # ========== 创建底部输入区域 ==========
        input_layout = QHBoxLayout()
        input_layout.setContentsMargins(0, 0, 0, 0)
        input_layout.setSpacing(5)
        
        # 输入框
        self.input_edit = QLineEdit()
        self.input_edit.setPlaceholderText("请输入您的问题...")
        self.input_edit.returnPressed.connect(self.send_message)
        input_layout.addWidget(self.input_edit)
        
        # 是否构建窗口单选框
        self.build_window_checkbox = QCheckBox("构建窗口")
        self.build_window_checkbox.setToolTip("选中后将根据当前窗口信息构建提示词")
        self.build_window_checkbox.setChecked(False)  # 默认不选中
        input_layout.addWidget(self.build_window_checkbox)
        
        # 加号发送按钮
        self.send_button = QPushButton("+")
        self.send_button.setFixedSize(40, 40)
        self.send_button.setToolTip("发送消息")
        self.send_button.clicked.connect(self.send_message)
        input_layout.addWidget(self.send_button)
        
        layout.addLayout(input_layout)
        
        # ========== 创建隐藏的设置控件（用于代码生成功能）==========
        # 这些控件不在界面显示，但保留用于代码生成功能
        self.framework_select = QComboBox()
        self.framework_select.addItems(["PySide", "TKinter"])
        self.framework_select.hide()
        
        self.mode_select = QComboBox()
        self.mode_select.addItems(["单独GUI页面", "功能GUI页面"])
        self.mode_select.hide()
        
        self.template_prompt_edit = QLineEdit()
        self.template_prompt_edit.hide()
        
        self.generate_code_button = QPushButton()
        self.generate_code_button.hide()
        
        self.clear_button = QPushButton()
        self.clear_button.hide()
        
        # 连接隐藏的代码生成按钮（虽然隐藏，但功能保留）
        self.generate_code_button.clicked.connect(self._send_code_generation)
        
        # 应用颜色设置
        self._apply_input_colors()
        
        # 设置基础样式
        self.setStyleSheet("""
            AIChatWidget {
                background-color: #F5F5F5;
            }
            QLineEdit:focus {
                border: 2px solid #0078D4;
            }
            QPushButton {
                background-color: #0078D4;
                color: white;
                border: none;
                border-radius: 20px;
                font-size: 20px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #106EBE;
            }
            QPushButton:pressed {
                background-color: #005A9E;
            }
            QPushButton:disabled {
                background-color: #CCCCCC;
            }
            QToolButton {
                background-color: transparent;
                border: 1px solid #CCCCCC;
                border-radius: 5px;
                font-size: 16px;
            }
            QToolButton:hover {
                background-color: #E0E0E0;
            }
        """)
        
        self.setMinimumHeight(300)
    
    def _load_input_colors_from_theme(self):
        """从主题文件加载输入框颜色"""
        try:
            from PySide6.QtCore import QSettings
            from pathlib import Path
            
            settings = QSettings("VisualProgramming", "Settings")
            theme_name = settings.value("theme/name", "white", type=str)
            
            # 获取主题文件路径
            current_file = Path(__file__).resolve()
            base_dir = current_file.parent.parent
            theme_path = base_dir / "ui" / "color" / f"{theme_name}.qss"
            
            if theme_path.exists():
                with open(theme_path, "r", encoding="utf-8") as f:
                    theme_content = f.read()
                
                # 解析QLineEdit样式，提取背景色和文字颜色
                import re
                
                # 查找QLineEdit样式块（匹配第一个，不包含:focus等伪类）
                # 使用更精确的正则表达式，匹配基础QLineEdit样式
                # 匹配格式：QLineEdit { ... } （不包含冒号，即不是伪类）
                line_edit_pattern = r'^QLineEdit\s*\{[^}]*\}'
                matches = re.findall(line_edit_pattern, theme_content, re.MULTILINE | re.DOTALL)
                
                # 如果上面的模式没匹配到，尝试更宽松的模式
                if not matches:
                    line_edit_pattern = r'QLineEdit\s*\{[^}]*\}'
                    matches = re.findall(line_edit_pattern, theme_content, re.DOTALL)
                
                bg_color = None
                text_color = None
                
                # 优先查找第一个QLineEdit块（基础样式，不包含伪类）
                for match in matches:
                    # 跳过包含伪类的样式块（如QLineEdit:focus, QLineEdit:disabled）
                    # 检查匹配文本的开头部分是否包含冒号（伪类标识）
                    match_start = match.split('{')[0].strip()
                    if ':' in match_start and match_start != 'QLineEdit':
                        continue
                    
                    # 提取background-color（支持多种格式）
                    bg_patterns = [
                        r'background-color:\s*([^;}\n]+)',  # 标准格式
                        r'background:\s*([^;}\n]+)',  # 简写格式
                    ]
                    for pattern in bg_patterns:
                        bg_match = re.search(pattern, match, re.IGNORECASE)
                        if bg_match:
                            bg_color_str = bg_match.group(1).strip()
                            # 移除可能的注释
                            if '/*' in bg_color_str:
                                bg_color_str = bg_color_str.split('/*')[0].strip()
                            bg_color = self._parse_color(bg_color_str)
                            if bg_color:
                                break
                    
                    # 提取color
                    color_patterns = [
                        r'color:\s*([^;}\n]+)',  # 标准格式
                    ]
                    for pattern in color_patterns:
                        color_match = re.search(pattern, match, re.IGNORECASE)
                        if color_match:
                            text_color_str = color_match.group(1).strip()
                            # 移除可能的注释
                            if '/*' in text_color_str:
                                text_color_str = text_color_str.split('/*')[0].strip()
                            text_color = self._parse_color(text_color_str)
                            if text_color:
                                break
                    
                    # 如果找到了背景色和文字颜色，就停止查找
                    if bg_color and text_color:
                        break
                
                # 如果找到了背景色，更新
                if bg_color:
                    self._input_bg_color = bg_color
                    
                    # 如果找到了文字颜色，更新
                    if text_color:
                        self._input_text_color = text_color
                    else:
                        # 如果没有找到文字颜色，根据背景色自动选择对比色
                        self._input_text_color = self._get_contrast_color(bg_color)
                else:
                    # 如果没有找到背景色，根据主题名称判断
                    if 'black' in theme_name.lower():
                        # 所有黑色主题（black, blackblue, blackpink等）：深色背景，浅色文字
                        self._input_bg_color = QColor(45, 45, 45)  # #1a1a1a
                        self._input_text_color = QColor(224, 224, 224)  # #e0e0e0
                    else:
                        # 浅色主题：浅色背景，深色文字
                        self._input_bg_color = QColor(255, 255, 255)
                        self._input_text_color = QColor(0, 0, 0)
                
                # 最终检查：确保文字颜色与背景色有足够的对比度
                # 这是关键步骤，确保在所有情况下文字都可见
                if self._input_bg_color:
                    # 计算背景色亮度
                    brightness = 0.299 * self._input_bg_color.red() + 0.587 * self._input_bg_color.green() + 0.114 * self._input_bg_color.blue()
                    
                    if brightness < 128:
                        # 深色背景（包括所有黑色主题：black, blackblue, blackpink等）
                        # 强制使用浅色文字，确保可见性
                        self._input_text_color = QColor(224, 224, 224)  # #e0e0e0
                    else:
                        # 浅色背景，确保使用深色文字
                        if self._input_text_color:
                            # 计算文字颜色亮度
                            text_brightness = 0.299 * self._input_text_color.red() + 0.587 * self._input_text_color.green() + 0.114 * self._input_text_color.blue()
                            if text_brightness >= 128:
                                # 如果文字是浅色的，在浅色背景下不可见，改为深色
                                self._input_text_color = QColor(0, 0, 0)
                            # 如果文字是深色的（text_brightness < 128），在浅色背景下没问题，保持不变
                        else:
                            # 如果没有文字颜色，使用深色
                            self._input_text_color = QColor(0, 0, 0)
                
                # 应用颜色
                self._apply_input_colors()
        except Exception as e:
            # 如果加载失败，根据主题名称使用默认颜色
            try:
                from PySide6.QtCore import QSettings
                settings = QSettings("VisualProgramming", "Settings")
                theme_name = settings.value("theme/name", "white", type=str)
                
                if 'black' in theme_name.lower():
                    # 所有黑色主题：深色背景，浅色文字
                    self._input_bg_color = QColor(45, 45, 45)  # #1a1a1a
                    self._input_text_color = QColor(224, 224, 224)  # #e0e0e0
                else:
                    # 浅色主题：浅色背景，深色文字
                    self._input_bg_color = QColor(255, 255, 255)
                    self._input_text_color = QColor(0, 0, 0)
                
                self._apply_input_colors()
            except Exception:
                pass
    
    def _get_contrast_color(self, bg_color: QColor) -> QColor:
        """
        根据背景颜色获取对比色（文字颜色）
        如果背景是深色的，返回浅色文字；如果背景是浅色的，返回深色文字
        
        Args:
            bg_color: 背景颜色
            
        Returns:
            对比色（文字颜色）
        """
        # 计算背景色的亮度（使用相对亮度公式）
        # 公式：0.299*R + 0.587*G + 0.114*B
        brightness = 0.299 * bg_color.red() + 0.587 * bg_color.green() + 0.114 * bg_color.blue()
        
        # 如果亮度小于128（深色背景），返回浅色文字
        if brightness < 128:
            return QColor(224, 224, 224)  # 浅灰色文字
        else:
            # 如果亮度大于等于128（浅色背景），返回深色文字
            return QColor(0, 0, 0)  # 黑色文字
    
    def _parse_color(self, color_str: str) -> QColor:
        """
        解析颜色字符串为QColor对象
        支持格式：#RRGGBB, rgb(r,g,b), 颜色名称等
        """
        try:
            color_str = color_str.strip()
            
            # 处理十六进制颜色
            if color_str.startswith('#'):
                return QColor(color_str)
            
            # 处理rgb格式
            if color_str.startswith('rgb'):
                import re
                rgb_match = re.search(r'rgb\((\d+),\s*(\d+),\s*(\d+)\)', color_str)
                if rgb_match:
                    r = int(rgb_match.group(1))
                    g = int(rgb_match.group(2))
                    b = int(rgb_match.group(3))
                    return QColor(r, g, b)
            
            # 处理颜色名称
            color_map = {
                'white': QColor(255, 255, 255),
                'black': QColor(0, 0, 0),
                'red': QColor(255, 0, 0),
                'green': QColor(0, 255, 0),
                'blue': QColor(0, 0, 255),
            }
            if color_str.lower() in color_map:
                return color_map[color_str.lower()]
            
            # 默认返回白色
            return QColor(255, 255, 255)
        except Exception:
            return QColor(255, 255, 255)
    
    def _apply_input_colors(self):
        """应用输入框颜色设置"""
        bg_color = self._input_bg_color.name()
        text_color = self._input_text_color.name()
        self.input_edit.setStyleSheet(f"""
            QLineEdit {{
                padding: 8px 12px;
                border: 1px solid #CCCCCC;
                border-radius: 20px;
                font-size: 14px;
                background-color: {bg_color};
                color: {text_color};
            }}
            QLineEdit:focus {{
                border: 2px solid #0078D4;
            }}
        """)
    
    def _show_settings_dialog(self):
        """
        显示AI设置对话框
        打开包含所有AI设置选项的对话框
        """
        dialog = AISettingsDialog(self)
        if dialog.exec() == QDialog.Accepted:
            # 设置已保存
            pass
    
    def _show_api_key_dialog(self):
        """
        显示API密钥设置对话框
        打开对话框供用户设置API密钥、选择服务商和模型
        支持从配置文件中读取已保存的密钥（包括加密存储的密钥）
        """
        # ========== 创建API密钥对话框 ==========
        # 创建API密钥设置对话框实例，传入self作为父窗口
        dialog = APIKeyDialog(self)
        
        # ========== 定义内部辅助函数 ==========
        def _read_cfg():
            """
            读取配置文件
            从配置文件中读取所有配置信息（包括API密钥等）
            
            Returns:
                dict: 配置字典，如果读取失败则返回空字典
            """
            try:
                # 获取配置文件路径
                p = self._config_path()
                # 检查配置文件是否存在
                if os.path.isfile(p):
                    # 如果文件存在，打开并读取JSON内容
                    with open(p, "r", encoding="utf-8") as f:
                        # 解析JSON格式的配置文件并返回
                        return json.load(f)
            except Exception:
                # 如果读取过程中发生任何异常（文件不存在、格式错误等），返回空字典
                return {}
            # 如果文件不存在，返回空字典
            return {}
        
        def _get_saved_key(provider: str, model_id: str) -> str:
            """
            获取保存的API密钥
            从配置文件中读取指定服务商和模型的API密钥
            支持加密存储的密钥（通过salt和data字段）
            
            Args:
                provider: 服务商名称（"GLM"或"DeepSeek"）
                model_id: 模型ID（如"GLM-4.5-Flash"）
                
            Returns:
                str: API密钥字符串，如果未找到则返回空字符串
            """
            try:
                # 读取配置文件
                cfg = _read_cfg()
                # 获取keys字典（存储所有API密钥）
                ks = cfg.get("keys", {})
                # 获取指定服务商的配置（可能是字典，也可能直接是密钥）
                pv = ks.get(provider)
                # 检查服务商配置是否为字典类型
                if isinstance(pv, dict):
                    # 如果是字典，尝试获取指定模型的配置
                    mv = pv.get(model_id)
                    # 检查模型配置是否为字典且包含加密字段（salt和data）
                    if isinstance(mv, dict) and "salt" in mv and "data" in mv:
                        # 如果包含加密字段，解密后返回
                        return self._decrypt(mv)
                    # 检查模型配置是否为字符串（未加密的密钥）
                    if isinstance(mv, str):
                        # 如果是字符串，直接返回
                        return mv
                # 如果服务商配置本身就是加密的字典（包含salt和data字段）
                if isinstance(pv, dict) and "salt" in pv and "data" in pv:
                    # 解密服务商配置并返回
                    return self._decrypt(pv)
            except Exception:
                # 如果发生任何异常，忽略并继续
                pass
            # 如果未找到密钥，返回空字符串
            return ""
        
        def _update_key_field():
            """
            更新对话框中的API密钥输入框
            根据当前选择的服务商和模型，从配置文件中读取对应的密钥并填充到输入框
            """
            try:
                # 获取对话框中选择的服务商名称
                p = dialog.get_provider()
                # 获取对话框中选择的模型ID，如果未选择则使用当前模型的ID
                m = dialog.get_selected_model() or self.model_id
                # 获取保存的密钥，如果未找到则使用空字符串
                saved_key = _get_saved_key(p, m) or ""
                # 将密钥设置到对话框的API密钥输入框中
                dialog.api_key_edit.setText(saved_key)
            except Exception:
                # 如果发生任何异常，将输入框设置为空
                dialog.api_key_edit.setText("")
        
        # ========== 初始化对话框状态 ==========
        # 尝试设置对话框的服务商选择为当前服务商
        try:
            dialog.set_provider(self.provider)
        except Exception:
            # 如果设置失败，忽略异常
            pass
        # 尝试设置对话框的模型选择为当前模型
        try:
            dialog.set_selected_model(self.model_id)
        except Exception:
            # 如果设置失败，忽略异常
            pass
        # 尝试更新API密钥输入框
        try:
            # 立即更新一次密钥输入框
            _update_key_field()
            # 连接服务商选择框的索引改变信号到更新密钥函数（当服务商改变时自动更新密钥）
            dialog.provider_select.currentIndexChanged.connect(lambda _: _update_key_field())
            # 连接模型选择框的索引改变信号到更新密钥函数（当模型改变时自动更新密钥）
            dialog.model_select.currentIndexChanged.connect(lambda _: _update_key_field())
        except Exception:
            # 如果连接信号失败，忽略异常
            pass
        
        # ========== 显示对话框并处理结果 ==========
        # 显示对话框并等待用户操作，如果用户点击了"确定"按钮（返回Accepted）
        if dialog.exec() == QDialog.Accepted:
            # 如果用户点击了确定，获取用户输入的API密钥
            new_api_key = dialog.get_api_key()
            # 检查API密钥是否不为空
            if new_api_key:
                # 如果API密钥不为空，执行以下操作
                # 更新实例的API密钥属性
                self.api_key = new_api_key
                # 尝试更新模型ID
                try:
                    # 获取对话框中选择的模型ID，如果未选择则保持当前模型ID
                    self.model_id = dialog.get_selected_model() or self.model_id
                except Exception:
                    # 如果获取模型ID失败，保持当前模型ID不变
                    pass
                # 尝试更新服务商
                try:
                    # 获取对话框中选择的服务商，如果未选择则保持当前服务商
                    self.provider = dialog.get_provider() or self.provider
                except Exception:
                    # 如果获取服务商失败，保持当前服务商不变
                    pass
                # 尝试更新API URL
                try:
                    # 获取对话框中的API URL，如果未设置则保持当前URL
                    self.api_url = dialog.get_api_url() or self.api_url
                except Exception:
                    # 如果获取API URL失败，保持当前URL不变
                    pass
                # 尝试保存设置到配置文件
                try:
                    # 调用保存设置方法，将新的API密钥等信息保存到配置文件
                    self._save_settings()
                except Exception:
                    # 如果保存失败，忽略异常（不阻止用户继续使用）
                    pass
                # 显示成功提示消息
                QMessageBox.information(self, "设置成功", "API密钥已更新")
            else:
                # 如果API密钥为空，显示警告提示
                QMessageBox.warning(self, "设置失败", "API密钥不能为空")
    
    def _set_default_api_key(self):
        """
        设置默认API密钥
        当前不设置默认API密钥，用户需要点击"设置API密钥"按钮来手动设置
        """
        # 将API密钥设置为空字符串（不设置默认值）
        self.api_key = ""
    
    def set_api_key(self, api_key: str):
        """
        设置API密钥
        从外部设置API密钥（通常用于程序化设置）
        
        Args:
            api_key: API密钥字符串
        """
        # 更新实例的API密钥属性
        self.api_key = api_key
    
    def add_message(self, message: str, is_user: bool = True):
        """
        添加消息到聊天界面
        创建消息气泡并添加到聊天区域，同时保存到消息历史中
        
        Args:
            message: 要显示的消息内容
            is_user: 是否为用户消息（True表示用户消息，False表示AI回复）
        """
        # 创建消息气泡组件，传入消息内容和消息类型
        message_bubble = MessageBubble(message, is_user)
        # 将消息气泡添加到聊天布局中
        self.chat_layout.addWidget(message_bubble)
        
        # 滚动到底部：延迟100毫秒后自动滚动到聊天区域底部，确保新消息可见
        QTimer.singleShot(100, self._scroll_to_bottom)
        
        # ========== 保存到消息历史 ==========
        # 检查消息类型是否为用户消息
        if is_user:
            # 如果是用户消息，添加用户角色消息到消息列表
            self.messages.append({"role": "user", "content": message})
        else:
            # 如果是AI回复，添加助手角色消息到消息列表
            self.messages.append({"role": "assistant", "content": message})
    
    def send_message(self):
        """
        发送消息
        处理用户输入的消息，发送到AI API并显示回复
        包括输入验证、UI状态管理、API请求创建等
        """
        # 获取输入框中的文本内容，并去除首尾空白字符
        message = self.input_edit.text().strip()
        # 检查消息是否为空（去除空白后）
        if not message:
            # 如果消息为空，直接返回，不执行后续操作
            return
        
        # ========== 检查API密钥是否已设置 ==========
        # 检查当前API密钥是否为空或未设置
        if not self.api_key:
            # 如果API密钥未设置，显示警告消息框，提示用户先设置API密钥
            QMessageBox.warning(self, "API密钥未设置", "请先点击\"设置模型\"按钮设置您的API密钥")
            # 返回，不继续执行发送操作
            return
        
        # ========== 添加用户消息到界面（显示原始消息）==========
        # 调用add_message方法，将用户消息添加到聊天界面，is_user=True表示这是用户消息
        # 注意：显示的是原始消息，而不是构建后的提示词
        self.add_message(message, is_user=True)
        
        # ========== 构建提示词（如果选中了"构建窗口"）==========
        # 检查是否选中了"构建窗口"单选框
        actual_message = message  # 实际发送给AI的消息（可能是构建后的提示词）
        if hasattr(self, 'build_window_checkbox') and self.build_window_checkbox.isChecked():
            # 如果选中了"构建窗口"，构建提示词
            try:
                # 尝试获取主窗口和当前窗口信息
                main_window = self._get_main_window()
                if main_window and hasattr(main_window, 'project_manager'):
                    project_manager = main_window.project_manager
                    if hasattr(project_manager, 'window_class_manager'):
                        window_class_manager = project_manager.window_class_manager
                        current_window = window_class_manager.get_current_window_class()
                        
                        if current_window:
                            # 获取框架（默认PySide）
                            framework = "PySide"
                            if hasattr(main_window, 'framework_combo_left'):
                                framework = main_window.framework_combo_left.currentText().strip() or "PySide"
                            
                            # 获取模式和逻辑（使用默认值）
                            mode = "功能GUI页面"
                            logic = "交互反馈"
                            
                            # 获取窗口信息
                            window_info = {
                                "window_title": getattr(current_window, 'window_title', '我的窗口'),
                                "window_width": getattr(current_window, 'canvas_width', 800),
                                "window_height": getattr(current_window, 'canvas_height', 600),
                                "window_class_name": current_window.name
                            }
                            
                            # 构建用户提示词（传入窗口信息）
                            actual_message = self._build_user_prompt(framework, message, mode, logic, window_info)
            except Exception as e:
                # 如果构建提示词失败，使用原始消息
                print(f"构建提示词失败: {e}")
                import traceback
                traceback.print_exc()
        
        # ========== 保存实际消息到消息历史 ==========
        # 更新消息列表中的最后一条消息（将原始消息替换为构建后的提示词）
        if self.messages and self.messages[-1]["role"] == "user":
            self.messages[-1]["content"] = actual_message
        
        # ========== 清空输入框 ==========
        # 清空输入框中的文本内容，准备接收下一次输入
        self.input_edit.clear()
        
        # ========== 更新UI状态（禁用输入，显示发送中） ==========
        # 禁用输入框，防止用户在请求过程中继续输入
        self.input_edit.setEnabled(False)
        # 禁用发送按钮，防止用户在请求过程中重复发送
        self.send_button.setEnabled(False)
        # 将发送按钮文本改为"..."，提示用户请求正在进行
        self.send_button.setText("...")
        
        # ========== 创建API请求线程 ==========
        # 创建API工作线程实例，传入必要的参数（使用构建后的消息）
        self.api_worker = ChatAPIWorker(self.api_key, self.messages, self.model_id, self.provider, self.api_url)
        # 连接流式数据接收信号
        self.api_worker.stream_chunk_received.connect(self._on_stream_chunk_received)
        # 连接流式传输完成信号
        self.api_worker.stream_finished.connect(self._on_stream_finished)
        # 连接API工作线程的响应接收信号到响应处理方法（用于兼容性）
        self.api_worker.response_received.connect(self._on_response_received)
        # 连接API工作线程的错误发生信号到错误处理方法
        self.api_worker.error_occurred.connect(self._on_error_occurred)
        # 尝试连接线程完成信号到完成处理方法
        try:
            # 连接线程完成信号（当线程执行完毕时发出）
            self.api_worker.finished.connect(self._on_worker_finished)
        except Exception:
            # 如果连接信号失败，忽略异常
            pass
        
        # ========== 更新请求状态标志 ==========
        # 设置请求激活标志为True，表示当前有API请求正在进行
        self._request_active = True
        # 设置助手更新标志为False，表示尚未收到AI回复
        self._has_assistant_update = False
        # 设置等待状态显示标志为False，表示尚未显示等待提示
        self._waiting_status_shown = False
        # 初始化流式消息内容
        self._streaming_content = ""
        # 创建空的AI回复气泡（用于流式更新）
        self._current_streaming_bubble = MessageBubble("", is_user=False)
        self.chat_layout.addWidget(self._current_streaming_bubble)
        QTimer.singleShot(100, self._scroll_to_bottom)
        
        # ========== 设置等待状态定时器 ==========
        # 尝试设置定时器：15秒后触发等待状态检查（如果请求时间过长，显示等待提示）
        try:
            # 使用单次定时器，15秒后调用等待状态检查方法
            QTimer.singleShot(15000, self._on_waiting_status_tick)
        except Exception:
            # 如果设置定时器失败，忽略异常
            pass
        
        # ========== 启动API请求线程 ==========
        # 启动API工作线程，开始执行API请求
        self.api_worker.start()
    
    def _on_stream_chunk_received(self, chunk: str):
        """
        处理流式数据片段
        逐步更新消息气泡内容
        优化长文本更新性能
        
        Args:
            chunk: 接收到的数据片段
        """
        if self._current_streaming_bubble:
            try:
                # 累积内容
                self._streaming_content += chunk
                
                # 对于超长文本，使用节流更新（每累积一定长度才更新一次）
                # 这样可以避免频繁的UI更新导致的卡顿
                content_length = len(self._streaming_content)
                
                # 如果内容较短（<5000字符），每次都更新
                # 如果内容较长，每增加500字符更新一次
                if content_length < 5000 or content_length % 500 < len(chunk):
                    # 更新气泡内容
                    self._current_streaming_bubble.update_message(self._streaming_content)
                    # 自动滚动到底部（延迟执行，避免频繁滚动）
                    QTimer.singleShot(100, self._scroll_to_bottom)
            except Exception as e:
                print(f"[DEBUG] 处理流式数据片段失败: {e}")
                import traceback
                traceback.print_exc()
    
    def _on_stream_finished(self):
        """
        流式传输完成处理
        """
        # 标记已收到助手更新
        self._has_assistant_update = True
        # 保存完整消息到消息历史
        if self._streaming_content.strip():
            self.messages.append({"role": "assistant", "content": self._streaming_content})
        # 清空流式内容
        self._streaming_content = ""
        # 清空当前流式气泡引用
        self._current_streaming_bubble = None
        # 恢复UI状态
        self.input_edit.setEnabled(True)
        self.send_button.setEnabled(True)
        self.send_button.setText("+")

    def _get_main_window(self):
        """
        获取主窗口对象
        通过父窗口链向上查找主窗口
        
        Returns:
            MainWindow对象，如果找不到则返回None
        """
        widget = self.parent()
        while widget:
            # 检查是否是MainWindow（通过类名或属性判断）
            if hasattr(widget, 'project_manager') and hasattr(widget, 'control_manager'):
                return widget
            widget = widget.parent()
        return None

    def _build_system_prompt(self, framework: str) -> str:
        """
        构建系统提示词
        优先使用常量中的提示词，如果失败则使用模板管理器
        
        Args:
            framework: 框架名称
            
        Returns:
            str: 系统提示词
        """
        try:
            # 优先使用常量中的提示词（更明确、更严格）
            return get_system_prompt(framework)
        except Exception:
            try:
                # 如果常量失败，使用模板管理器
                return prompt_templates.build_system_prompt(framework)
            except Exception:
                # 最后的降级方案
                return (
                    f"你是一个资深Python GUI工程师。只输出Python代码，不输出任何解释或说明。"
                    f"必须使用{framework}生成完整、可运行的脚本，包含必要导入与入口。"
                    f"使用一个三引号代码块包裹全部代码，格式：```python\n代码\n```"
                )

    def _build_user_prompt(self, framework: str, prompt: str, mode: str, logic: str, window_info: dict = None) -> str:
        """
        构建用户提示词
        根据框架、提示、模式、逻辑等参数构建发送给AI的用户提示词
        
        Args:
            framework: 目标框架名称（如"PySide"、"TKinter"）
            prompt: 用户输入的提示文本
            mode: 生成模式（如"功能GUI页面"）
            logic: 逻辑类型（如"交互反馈"）
            window_info: 窗口信息字典（可选），包含window_title, window_width, window_height, window_class_name
            
        Returns:
            str: 构建好的用户提示词
        """
        try:
            # 优先使用常量中的提示词构建函数
            return build_user_prompt(framework, prompt, mode, logic, window_info)
        except Exception:
            try:
                # 如果常量失败，使用模板管理器
                return prompt_templates.build_user_prompt(framework, prompt, mode, logic)
            except Exception:
                # 最后的降级方案
                ui_only = (
                    f"请使用{framework}生成完整可运行的Python脚本：{prompt}。"
                    "窗口标题与大小需设置，控件使用绝对定位，运行即展示界面。"
                    "所有代码必须用三个反引号包裹，格式：```python\n代码\n```"
                )
                functional = (
                    f"请使用{framework}生成完整可运行的Python脚本：{prompt}。"
                    "包含基础功能逻辑与事件处理（如输入校验、按钮点击行为、弹窗提示），控件使用绝对定位。"
                    "所有代码必须用三个反引号包裹，格式：```python\n代码\n```"
                )
                base = functional if (mode or '').strip() == '功能GUI页面' else ui_only
                if (mode or '').strip() == '功能GUI页面' and (logic or '').strip() == '交互反馈':
                    base += " 使用PySide6的QMessageBox或tkinter的messagebox进行信息/警告/错误反馈。"
                return base

    def _send_code_generation(self):
        """
        发送代码生成请求
        构建代码生成提示词，发送到AI API并处理响应
        """
        # ========== 检查API密钥 ==========
        # 检查当前API密钥是否为空或未设置
        if not self.api_key:
            # 如果API密钥未设置，显示警告消息框，提示用户先设置API密钥
            QMessageBox.warning(self, "API密钥未设置", "请先设置API密钥")
            # 返回，不继续执行代码生成操作
            return
        
        # ========== 获取用户输入的参数 ==========
        # 获取框架选择下拉框中的当前选中文本（目标框架名称）
        framework = self.framework_select.currentText().strip()
        # 获取提示词输入框中的文本内容（用户的需求描述）
        prompt = self.template_prompt_edit.text().strip()
        
        # ========== 检查提示词是否为空 ==========
        # 检查提示词是否为空（去除空白后）
        if not prompt:
            # 如果提示词为空，直接返回，不执行代码生成操作
            return
        
        # ========== 添加代码生成请求到聊天界面 ==========
        # 构建显示文本：框架名称 + 提示词
        display_text = f"[{framework}] {prompt}"
        # 将代码生成请求添加到聊天界面（显示为用户消息）
        self.add_message(display_text, is_user=True)
        
        # ========== 激活代码模式 ==========
        # 设置代码模式激活标志为True，表示当前处于代码生成模式
        self._code_mode_active = True
        
        # ========== 构建消息列表 ==========
        # 创建空的消息列表，用于发送给AI API
        messages = []
        # 添加系统消息：定义AI的角色和行为（通过提示词模板构建）
        messages.append({"role": "system", "content": self._build_system_prompt(framework)})
        
        # ========== 获取生成模式和逻辑类型 ==========
        # 获取模式选择下拉框中的当前选中文本（如"功能GUI页面"）
        mode = self.mode_select.currentText().strip()
        # 尝试获取逻辑类型选择下拉框中的当前选中文本
        try:
            logic = self.logic_select.currentText().strip()
        except Exception:
            # 如果逻辑选择框不存在或获取失败，使用空字符串
            logic = ""
        
        # ========== 添加用户消息 ==========
        # 添加用户消息：包含代码生成的具体需求（通过提示词模板构建）
        messages.append({"role": "user", "content": self._build_user_prompt(framework, prompt, mode, logic)})
        
        # ========== 更新UI状态（禁用输入） ==========
        # 禁用输入框，防止用户在代码生成过程中继续输入
        self.input_edit.setEnabled(False)
        # 禁用发送按钮，防止用户在代码生成过程中重复发送
        self.send_button.setEnabled(False)
        # 尝试更新代码生成按钮的状态
        try:
            # 禁用代码生成按钮，防止重复生成
            self.generate_code_button.setEnabled(False)
            # 将按钮文本改为"生成中..."，提示用户代码正在生成
            self.generate_code_button.setText("生成中...")
        except Exception:
            # 如果按钮不存在或操作失败，忽略异常
            pass
        
        # ========== 创建API请求线程 ==========
        # 创建API工作线程实例，传入必要的参数（API密钥、消息列表、模型ID、服务商、API URL）
        self.api_worker = ChatAPIWorker(self.api_key, messages, self.model_id, self.provider, self.api_url)
        # 连接流式数据接收信号（代码生成也支持流式）
        self.api_worker.stream_chunk_received.connect(lambda c: self._on_code_stream_chunk_received(c, framework))
        # 连接流式传输完成信号
        self.api_worker.stream_finished.connect(lambda: self._on_code_stream_finished(framework))
        # 连接API工作线程的响应接收信号到代码响应处理方法（用于兼容性）
        # 使用lambda函数传递framework参数
        self.api_worker.response_received.connect(lambda r: self._on_code_response_received(r, framework))
        # 连接API工作线程的错误发生信号到错误处理方法
        self.api_worker.error_occurred.connect(self._on_error_occurred)
        # 尝试连接线程完成信号到完成处理方法
        try:
            # 连接线程完成信号（当线程执行完毕时发出）
            self.api_worker.finished.connect(self._on_worker_finished)
        except Exception:
            # 如果连接信号失败，忽略异常
            pass
        
        # ========== 更新请求状态标志 ==========
        # 设置请求激活标志为True，表示当前有API请求正在进行
        self._request_active = True
        # 设置助手更新标志为False，表示尚未收到AI回复
        self._has_assistant_update = False
        # 设置等待状态显示标志为False，表示尚未显示等待提示
        self._waiting_status_shown = False
        # 初始化流式代码内容
        self._streaming_code_content = ""
        # 创建空的AI回复气泡（用于流式更新代码）
        self._current_streaming_bubble = MessageBubble("", is_user=False)
        self.chat_layout.addWidget(self._current_streaming_bubble)
        QTimer.singleShot(100, self._scroll_to_bottom)
        
        # ========== 设置等待状态定时器 ==========
        # 尝试设置定时器：15秒后触发等待状态检查
        try:
            # 使用单次定时器，15秒后调用等待状态检查方法
            QTimer.singleShot(15000, self._on_waiting_status_tick)
        except Exception:
            # 如果设置定时器失败，忽略异常
            pass
        
        # ========== 启动API请求线程 ==========
        # 启动API工作线程，开始执行代码生成API请求
        self.api_worker.start()
    
    def _on_code_stream_chunk_received(self, chunk: str, framework: str):
        """
        处理代码生成的流式数据片段
        优化长代码更新性能
        
        Args:
            chunk: 接收到的数据片段
            framework: 框架名称
        """
        if self._current_streaming_bubble:
            try:
                # 累积内容
                self._streaming_code_content += chunk
                
                # 对于超长代码，使用节流更新（每累积一定长度才更新一次）
                content_length = len(self._streaming_code_content)
                
                # 如果内容较短（<5000字符），每次都更新
                # 如果内容较长，每增加500字符更新一次
                if content_length < 5000 or content_length % 500 < len(chunk):
                    # 更新气泡内容
                    self._current_streaming_bubble.update_message(self._streaming_code_content)
                    # 自动滚动到底部（延迟执行，避免频繁滚动）
                    QTimer.singleShot(100, self._scroll_to_bottom)
            except Exception as e:
                print(f"[DEBUG] 处理代码流式数据片段失败: {e}")
                import traceback
                traceback.print_exc()
    
    def _on_code_stream_finished(self, framework: str):
        """
        代码生成流式传输完成处理
        
        Args:
            framework: 框架名称
        """
        # 标记已收到助手更新
        self._has_assistant_update = True
        
        # 提取代码
        code = self._extract_code(self._streaming_code_content)
        code = code.strip() if code else ""
        
        # ========== 检查代码是否有效 ==========
        if code and len(code.strip()) > 50:  # 提高最小长度要求，确保代码完整
            # 如果提取到有效代码
            # 更新气泡显示提取的代码
            if self._current_streaming_bubble:
                self._current_streaming_bubble.update_message(code)
            # 发送代码生成信号
            try:
                self.code_generated.emit(code, framework)
                # 显示成功提示
                try:
                    QMessageBox.information(self, "代码生成成功", "代码已成功生成并添加到代码编辑器")
                except Exception:
                    pass
            except Exception as e:
                print(f"发送代码生成信号失败: {e}")
                import traceback
                traceback.print_exc()
        elif code and len(code.strip()) > 10:
            # 代码太短，可能不完整
            error_msg = f"{ERROR_CODE_TOO_SHORT}\n\n提取的代码片段：\n{code[:200]}..."
            if self._current_streaming_bubble:
                self._current_streaming_bubble.update_message(error_msg)
        else:
            # ========== 提取失败时的处理 ==========
            error_msg = f"{ERROR_NO_CODE_EXTRACTED}\n\nAI返回的原始内容：\n{self._streaming_code_content[:500]}..."
            if self._current_streaming_bubble:
                self._current_streaming_bubble.update_message(error_msg)
        
        # 清空流式内容
        self._streaming_code_content = ""
        # 清空当前流式气泡引用
        self._current_streaming_bubble = None
        # 退出代码模式
        self._code_mode_active = False
        # 恢复UI状态
        self.input_edit.setEnabled(True)
        self.send_button.setEnabled(True)
        self.send_button.setText("+")
        try:
            self.generate_code_button.setEnabled(True)
            self.generate_code_button.setText("生成代码")
        except Exception:
            pass
    
    def _on_response_received(self, response: str):
        """
        处理API响应
        接收AI API返回的响应内容，添加到聊天界面并更新UI状态
        
        Args:
            response: AI返回的响应文本内容
        """
        # ========== 处理响应内容 ==========
        # 获取响应文本，如果为None则使用空字符串，然后去除首尾空白字符
        resp = (response or "").strip()
        # ========== 添加响应到聊天界面 ==========
        # 将响应内容添加到聊天界面
        # 如果响应内容不为空，显示响应内容；否则显示默认提示
        self.add_message(resp if resp else "未获取到回复内容", is_user=False)
        
        # ========== 更新助手更新标志 ==========
        # 设置助手更新标志为True，表示已收到AI助手的回复
        self._has_assistant_update = True
        
        # ========== 恢复输入和发送按钮 ==========
        # 启用输入框，允许用户继续输入
        self.input_edit.setEnabled(True)
        # 启用发送按钮，允许用户继续发送消息
        self.send_button.setEnabled(True)
        # 将发送按钮文本改回"发送"
        self.send_button.setText("发送")

    def _extract_code(self, text: str) -> str:
        """
        从文本中提取代码块
        使用多种正则表达式模式查找并提取代码块（被三个反引号包裹的内容）
        
        Args:
            text: 要提取代码的文本内容
            
        Returns:
            str: 提取出的代码内容，如果未找到代码块则返回空字符串
        """
        if not text or not isinstance(text, str):
            return ""
        
        text = text.strip()
        if not text:
            return ""
        
        # ========== 尝试多种提取模式 ==========
        # 使用常量中定义的所有提取模式，按优先级尝试
        for pattern in CODE_EXTRACTION_PATTERNS:
            try:
                matches = re.findall(pattern, text, re.MULTILINE | re.DOTALL)
                if matches:
                    # 找到匹配，合并所有代码块
                    code = "\n".join([m.strip() for m in matches if m.strip()]).strip()
                    if code and len(code) > 10:  # 确保代码长度合理
                        return code
            except Exception:
                continue
        
        # ========== 如果所有模式都失败，尝试直接查找代码块 ==========
        try:
            all_blocks = re.findall(r"```[\s\S]*?```", text, re.MULTILINE | re.DOTALL)
            if all_blocks:
                codes = []
                for block in all_blocks:
                    # 移除开头的```和可能的语言标识
                    content = re.sub(r"^```[a-zA-Z0-9_\-]*\s*", "", block, flags=re.MULTILINE)
                    # 移除结尾的```
                    content = re.sub(r"```\s*$", "", content, flags=re.MULTILINE)
                    if content.strip():
                        codes.append(content.strip())
                if codes:
                    code = "\n".join(codes).strip()
                    if code and len(code) > 10:
                        return code
        except Exception:
            pass
        
        # ========== 如果仍未找到，检查是否整个文本就是代码 ==========
        # 如果文本看起来像代码（包含Python关键字），可能AI没有使用代码块格式
        python_keywords = ["import ", "def ", "class ", "from ", "if __name__", "QApplication", "QMainWindow", "tkinter"]
        if any(keyword in text for keyword in python_keywords):
            # 可能是代码但没有代码块标记，直接返回
            return text.strip()
        
        # ========== 未找到代码 ==========
        # 返回空字符串，表示未提取到有效代码
        return ""

    def _on_code_response_received(self, response: str, framework: str):
        """
        处理代码生成API响应
        从AI响应中提取代码，添加到聊天界面，并发送代码生成信号
        
        Args:
            response: AI返回的响应文本（可能包含代码块）
            framework: 目标框架名称（如"PySide"、"TKinter"）
        """
        # ========== 提取代码内容 ==========
        # 从响应文本中提取代码块（去除代码块标记）
        # self._extract_code(response)：调用提取代码方法
        # or ""：如果提取失败返回None，则使用空字符串
        # .strip()：去除首尾空白字符
        code = (self._extract_code(response) or "").strip()
        
        # ========== 检查代码是否有效 ==========
        if code and len(code.strip()) > 50:  # 提高最小长度要求，确保代码完整
            # 如果提取到有效代码
            self.add_message(code, is_user=False)
            
            # ========== 发送代码生成信号 ==========
            try:
                self.code_generated.emit(code, framework)
                # 显示成功提示
                try:
                    QMessageBox.information(self, "代码生成成功", "代码已成功生成并添加到代码编辑器")
                except Exception:
                    pass
            except Exception as e:
                print(f"发送代码生成信号失败: {e}")
                import traceback
                traceback.print_exc()
        elif code and len(code.strip()) > 10:
            # 代码太短，可能不完整
            error_msg = f"{ERROR_CODE_TOO_SHORT}\n\n提取的代码片段：\n{code[:200]}..."
            self.add_message(error_msg, is_user=False)
        else:
            # ========== 提取失败时的处理 ==========
            error_msg = f"{ERROR_NO_CODE_EXTRACTED}\n\nAI返回的原始内容：\n{response[:500]}..."
            self.add_message(error_msg, is_user=False)
        
        # ========== 更新状态标志 ==========
        # 设置助手更新标志为True，表示已收到AI助手的回复
        self._has_assistant_update = True
        
        # ========== 恢复输入和发送按钮 ==========
        # 启用输入框，允许用户继续输入
        self.input_edit.setEnabled(True)
        # 启用发送按钮，允许用户继续发送消息
        self.send_button.setEnabled(True)
        
        # ========== 恢复代码生成按钮 ==========
        # 尝试恢复代码生成按钮的状态
        try:
            # 启用代码生成按钮，允许用户继续生成代码
            self.generate_code_button.setEnabled(True)
            # 将按钮文本改回"生成代码"
            self.generate_code_button.setText("生成代码")
        except Exception:
            # 如果按钮不存在或操作失败，忽略异常
            pass
        
        # ========== 退出代码模式 ==========
        # 设置代码模式激活标志为False，退出代码生成模式
        self._code_mode_active = False
    
    def _on_error_occurred(self, error_msg: str):
        """
        处理API错误
        当API请求发生错误时调用，显示错误消息并恢复UI状态
        
        Args:
            error_msg: 错误信息字符串
        """
        # ========== 添加错误消息 ==========
        # 将错误消息添加到聊天界面（显示为AI回复，但内容是错误信息）
        self.add_message(f"错误: {error_msg}", is_user=False)
        
        # ========== 更新助手更新标志 ==========
        # 设置助手更新标志为True，表示已收到AI助手的回复（虽然是错误）
        self._has_assistant_update = True
        
        # ========== 恢复输入和发送按钮 ==========
        # 启用输入框，允许用户继续输入
        self.input_edit.setEnabled(True)
        # 启用发送按钮，允许用户继续发送消息
        self.send_button.setEnabled(True)
        # 将发送按钮文本改回"发送"
        self.send_button.setText("发送")
        
        # ========== 恢复代码生成按钮 ==========
        # 尝试恢复代码生成按钮的状态
        try:
            # 启用代码生成按钮，允许用户继续生成代码
            self.generate_code_button.setEnabled(True)
            # 将按钮文本改回"生成代码"
            self.generate_code_button.setText("生成代码")
        except Exception:
            # 如果按钮不存在或操作失败，忽略异常
            pass

    def clear_chat(self):
        """
        清空对话
        移除聊天界面中的所有消息，清空消息历史，并添加欢迎消息
        """
        # ========== 清空聊天界面 ==========
        # 循环移除聊天布局中的所有子控件
        # 检查布局中是否还有子控件
        while self.chat_layout.count():
            # ========== 取出第一个子项 ==========
            # 从布局中取出第一个子项（布局项对象，包含控件信息）
            child = self.chat_layout.takeAt(0)
            # ========== 检查子项是否有控件 ==========
            # 检查子项是否包含控件（widget）
            if child.widget():
                # 如果子项包含控件，延迟删除该控件
                # deleteLater()会在事件循环的下一次迭代中删除控件，避免立即删除导致的错误
                child.widget().deleteLater()
        
        # ========== 清空消息历史 ==========
        # 清空消息列表，移除所有对话历史记录
        self.messages = []
        
        # ========== 添加欢迎消息 ==========
        # 检查API密钥是否已设置
        if self.api_key:
            # 如果API密钥已设置，添加正常的欢迎消息
            self.add_message("您好！我是AI助手，有什么可以帮助您的吗？", is_user=False)
        else:
            # 如果API密钥未设置，添加提示消息，引导用户设置API密钥
            self.add_message("您好！我是AI助手。\n\n使用前请先点击\"设置模型\"按钮设置您的API密钥。", is_user=False)

    def _on_worker_finished(self):
        """
        处理API工作线程完成
        当API请求线程执行完毕时调用，检查是否收到响应并恢复UI状态
        """
        # ========== 检查是否收到响应 ==========
        # 尝试检查请求状态和响应状态
        try:
            # 检查是否有激活的请求且未收到助手更新（未收到响应）
            if self._request_active and not self._has_assistant_update:
                # 如果请求已结束但未收到内容，添加提示消息
                self.add_message("请求已结束，但未收到内容", is_user=False)
        except Exception:
            # 如果检查过程中发生异常，忽略异常（不中断程序运行）
            pass
        
        # ========== 恢复UI控件状态 ==========
        # 尝试恢复输入和发送按钮的状态
        try:
            # 启用输入框，允许用户继续输入
            self.input_edit.setEnabled(True)
            # 启用发送按钮，允许用户继续发送消息
            self.send_button.setEnabled(True)
            # 将发送按钮文本改回"发送"
            self.send_button.setText("发送")
            # 尝试恢复代码生成按钮的状态
            try:
                # 启用代码生成按钮，允许用户继续生成代码
                self.generate_code_button.setEnabled(True)
                # 将按钮文本改回"生成代码"
                self.generate_code_button.setText("生成代码")
            except Exception:
                # 如果代码生成按钮不存在或操作失败，忽略异常
                pass
        except Exception:
            # 如果恢复UI状态过程中发生异常，忽略异常
            pass
        
        # ========== 重置状态标志 ==========
        # 设置请求激活标志为False，表示当前没有API请求正在进行
        self._request_active = False
        # 设置助手更新标志为False，重置助手更新状态
        self._has_assistant_update = False
        # 设置等待状态显示标志为False，重置等待状态显示状态
        self._waiting_status_shown = False

    def _on_waiting_status_tick(self):
        """
        等待状态检查定时器回调
        当请求时间过长时（15秒后），显示等待提示消息
        """
        # ========== 检查是否显示等待提示 ==========
        # 尝试检查是否需要显示等待提示
        try:
            # 检查以下条件是否全部满足：
            # 1. 有激活的请求（请求正在进行）
            # 2. 尚未收到助手更新（未收到响应）
            # 3. 尚未显示等待状态（避免重复显示）
            if self._request_active and not self._has_assistant_update and not self._waiting_status_shown:
                # ========== 显示等待提示 ==========
                # 如果所有条件满足，添加等待提示消息到聊天界面
                self.add_message("正在等待模型响应...", is_user=False)
                # 设置等待状态显示标志为True，表示已显示等待提示
                self._waiting_status_shown = True
        except Exception:
            # 如果检查过程中发生异常，忽略异常（不中断程序运行）
            pass
    
    def _scroll_to_bottom(self):
        """
        滚动到聊天区域底部
        自动滚动到聊天内容的底部，确保新消息可见
        """
        # ========== 获取垂直滚动条 ==========
        # 从聊天滚动区域获取垂直滚动条对象
        scrollbar = self.chat_scroll.verticalScrollBar()
        # ========== 滚动到底部 ==========
        # 将滚动条的值设置为最大值，即滚动到最底部
        # scrollbar.maximum()：获取滚动条的最大值（滚动范围的最大值）
        # scrollbar.setValue(...)：设置滚动条的当前值（滚动位置）
        scrollbar.setValue(scrollbar.maximum())
    
    def set_messages(self, messages: List[Dict[str, str]]):
        """
        设置消息历史
        用新的消息列表替换当前的消息历史，并重新渲染到聊天界面
        
        Args:
            messages: 消息列表，格式为[{"role": "user/assistant", "content": "..."}, ...]
        """
        # ========== 复制消息列表 ==========
        # 复制传入的消息列表，避免修改原始列表（使用浅拷贝）
        self.messages = messages.copy()
        
        # ========== 清空聊天界面 ==========
        # 循环移除聊天布局中的所有子控件，为重新显示消息做准备
        # 检查布局中是否还有子控件
        while self.chat_layout.count():
            # 从布局中取出第一个子项
            child = self.chat_layout.takeAt(0)
            # 检查子项是否包含控件
            if child.widget():
                # 如果子项包含控件，延迟删除该控件
                child.widget().deleteLater()
        
        # ========== 重新显示所有消息 ==========
        # 遍历消息列表中的所有消息，逐个添加到聊天界面
        for msg in self.messages:
            # ========== 判断消息类型 ==========
            # 检查消息的角色是否为"user"（用户消息）
            # 如果role为"user"则is_user为True，否则为False（AI助手消息）
            is_user = msg["role"] == "user"
            # ========== 添加消息到界面 ==========
            # 调用add_message方法，将消息内容添加到聊天界面
            # msg["content"]：获取消息的内容文本
            # is_user：标识消息类型（用户或AI助手）
            self.add_message(msg["content"], is_user)

    def _config_path(self) -> str:
        import sys
        env_p = os.environ.get("APP_SECRETS_PATH", "")
        if isinstance(env_p, str) and env_p:
            try:
                if os.path.isfile(env_p):
                    return env_p
            except Exception:
                pass
        try:
            exe = sys.executable
            exe_is_app = isinstance(exe, str) and exe.lower().endswith(".exe") and (not exe.lower().endswith("python.exe")) and (not exe.lower().endswith("pythonw.exe"))
            base_exe = os.path.dirname(exe) if exe_is_app else ""
        except Exception:
            base_exe = ""
        try:
            base_argv = os.path.dirname(os.path.abspath(sys.argv[0]))
        except Exception:
            base_argv = ""
        try:
            base_cwd = os.getcwd()
        except Exception:
            base_cwd = ""
        try:
            base_proj = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
        except Exception:
            base_proj = ""
        bases = [b for b in [base_exe, base_argv, base_cwd, base_proj] if isinstance(b, str) and b]
        for b in bases:
            for d in ("Python", "python"):
                p = os.path.join(b, d, "app_secrets.json")
                try:
                    if os.path.isfile(p):
                        return p
                except Exception:
                    pass
        fallback = os.path.join(base_proj or base_cwd or base_argv or base_exe or os.path.dirname(os.path.abspath(__file__)), "Python", "app_secrets.json")
        return fallback

    def _encrypt(self, text: str) -> Dict[str, str]:
        salt = os.urandom(16)
        tb = text.encode("utf-8")
        sb = salt
        out = bytearray()
        for i in range(len(tb)):
            out.append(tb[i] ^ sb[i % len(sb)])
        return {"salt": base64.b64encode(salt).decode("ascii"), "data": base64.b64encode(bytes(out)).decode("ascii")}

    def _decrypt(self, obj: Dict[str, str]) -> str:
        try:
            salt = base64.b64decode(obj.get("salt", ""))
            data = base64.b64decode(obj.get("data", ""))
            out = bytearray()
            for i in range(len(data)):
                out.append(data[i] ^ salt[i % len(salt)])
            return bytes(out).decode("utf-8")
        except Exception:
            return ""

    def _load_settings(self):
        """加载设置：从配置文件自动读取API密钥等配置"""
        p = self._config_path()
        try:
            # 确保配置文件目录存在
            config_dir = os.path.dirname(p)
            if config_dir and not os.path.exists(config_dir):
                os.makedirs(config_dir, exist_ok=True)
            
            if os.path.isfile(p):
                with open(p, "r", encoding="utf-8") as f:
                    cfg = json.load(f)
                
                # 加载服务商设置
                self.provider = cfg.get("provider", self.provider)
                
                # 加载模型设置
                m = cfg.get("model")
                if isinstance(m, str) and m:
                    self.model_id = m
                
                # 加载API密钥（自动读取）
                keys = cfg.get("keys", {})
                if isinstance(keys, dict):
                    pv = keys.get(self.provider)
                    if isinstance(pv, dict):
                        # 检查是否是加密格式（服务商级别）
                        if "salt" in pv and "data" in pv:
                            try:
                                self.api_key = self._decrypt(pv)
                            except Exception:
                                self.api_key = ""
                        else:
                            # 检查模型级别的密钥
                            mv = pv.get(self.model_id)
                            if isinstance(mv, dict):
                                # 加密格式
                                if "salt" in mv and "data" in mv:
                                    try:
                                        self.api_key = self._decrypt(mv)
                                    except Exception:
                                        self.api_key = ""
                            elif isinstance(mv, str):
                                # 明文格式（自动加密保存）
                                if mv:
                                    self.api_key = mv
                                    # 自动加密并保存
                                    self._save_settings()
            else:
                # 如果配置文件不存在，创建一个空的配置文件
                try:
                    with open(p, "w", encoding="utf-8") as f:
                        json.dump({}, f, ensure_ascii=False, indent=2)
                except Exception:
                    pass
        except Exception as e:
            # 如果加载失败，使用默认值
            pass

    def _save_settings(self):
        p = self._config_path()
        cfg = {}
        try:
            if os.path.isfile(p):
                with open(p, "r", encoding="utf-8") as f:
                    cfg = json.load(f)
        except Exception:
            cfg = {}
        cfg["provider"] = self.provider
        cfg["model"] = self.model_id
        ks = cfg.get("keys", {}) if isinstance(cfg.get("keys", {}), dict) else {}
        pv = ks.get(self.provider)
        if not isinstance(pv, dict):
            pv = {}
        # 确保所有密钥都以加密形式存储
        for k, v in list(pv.items()):
            if isinstance(v, str) and v:
                # 如果是明文密钥，加密存储
                pv[k] = self._encrypt(v)
            elif isinstance(v, dict) and "salt" in v and "data" in v:
                # 已经是加密格式，保持不变
                pass
            else:
                # 空值或其他格式，保持原样
                pass
        # 为当前模型设置密钥（加密存储）
        if self.api_key:
            pv[self.model_id] = self._encrypt(self.api_key)
        else:
            pv[self.model_id] = ""
        if self.provider == "DeepSeek":
            for mid in ["deepseek-chat", "deepseek-reasoner"]:
                if mid not in pv:
                    pv[mid] = ""
        if self.provider == "GLM":
            for mid in ["GLM-4.5-Flash", "glm-4-flash-250414"]:
                if mid not in pv:
                    pv[mid] = ""
        ks[self.provider] = pv
        cfg["keys"] = ks
        try:
            with open(p, "w", encoding="utf-8") as f:
                json.dump(cfg, f, ensure_ascii=False, indent=2)
        except Exception:
            pass
