#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
将 controls.json 拆分为单个配置文件
每个控件对应一个 JSON 文件
"""

import json
import os
from pathlib import Path


def migrate_controls_to_individual_files():
    """将 controls.json 拆分为单个配置文件"""
    # 获取当前文件所在目录
    current_dir = os.path.dirname(os.path.abspath(__file__))
    controls_file = os.path.join(current_dir, "controls.json")
    output_dir = os.path.join(current_dir, "control_configs")
    
    # 创建输出目录
    if not os.path.exists(output_dir):
        os.makedirs(output_dir, exist_ok=True)
        print(f"已创建目录: {output_dir}")
    
    # 检查源文件是否存在
    if not os.path.exists(controls_file):
        print(f"源文件不存在: {controls_file}")
        return
    
    # 读取 controls.json
    print(f"正在读取: {controls_file}")
    with open(controls_file, 'r', encoding='utf-8') as f:
        controls_data = json.load(f)
    
    # 拆分并保存
    migrated_count = 0
    skipped_count = 0
    
    for control_name, control_config in controls_data.items():
        output_file = os.path.join(output_dir, f"{control_name}.json")
        
        # 如果文件已存在，跳过
        if os.path.exists(output_file):
            print(f"跳过已存在的文件: {control_name}.json")
            skipped_count += 1
            continue
        
        # 确保配置中有名称
        control_config["name"] = control_name
        
        # 保存到单个文件
        try:
            with open(output_file, 'w', encoding='utf-8') as f:
                json.dump(control_config, f, ensure_ascii=False, indent=2)
            print(f"已创建: {control_name}.json")
            migrated_count += 1
        except Exception as e:
            print(f"创建文件失败 {control_name}.json: {e}")
    
    print(f"\n迁移完成:")
    print(f"  - 成功迁移: {migrated_count} 个控件")
    print(f"  - 跳过已存在: {skipped_count} 个控件")
    print(f"  - 输出目录: {output_dir}")


if __name__ == "__main__":
    migrate_controls_to_individual_files()

