#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
层次化配置加载器
从层次化配置文件加载信息
"""

import json
import os
from typing import Dict, Any, Optional


class HierarchicalConfigLoader:
    """层次化配置加载器"""
    
    def __init__(self, config_file: str = None):
        """
        初始化加载器
        
        Args:
            config_file: 配置文件路径，默认为lib/hierarchical_config.json
        """
        if config_file is None:
            lib_dir = os.path.dirname(os.path.abspath(__file__))
            config_file = os.path.join(lib_dir, "hierarchical_config.json")
        
        self.config_file = config_file
        self.config = None
    
    def load(self) -> bool:
        """
        加载配置文件
        
        Returns:
            bool: 是否加载成功
        """
        if not os.path.exists(self.config_file):
            print(f"配置文件不存在: {self.config_file}")
            return False
        
        try:
            with open(self.config_file, 'r', encoding='utf-8') as f:
                self.config = json.load(f)
            print(f"已加载层次化配置: {self.config_file}")
            return True
        except Exception as e:
            print(f"加载配置文件失败: {e}")
            return False
    
    def get_window_class(self, window_name: str) -> Optional[Dict]:
        """获取窗口类信息"""
        if not self.config:
            return None
        return self.config.get("window_classes", {}).get(window_name)
    
    def get_canvas_code(self, window_name: str) -> Optional[Dict]:
        """获取画布代码信息"""
        window_class = self.get_window_class(window_name)
        if window_class:
            return window_class.get("canvas_code")
        return None
    
    def get_functional_code(self, window_name: str) -> Optional[Dict]:
        """获取功能代码信息"""
        window_class = self.get_window_class(window_name)
        if window_class:
            return window_class.get("functional_code")
        return None
    
    def get_window_components(self, window_name: str) -> Optional[Dict]:
        """获取窗口组件信息"""
        window_class = self.get_window_class(window_name)
        if window_class:
            return window_class.get("window_components")
        return None
    
    def get_component_events(self, window_name: str, component_name: str) -> Optional[Dict]:
        """获取组件的事件信息"""
        components = self.get_window_components(window_name)
        if components and component_name in components:
            return components[component_name].get("events")
        return None
    
    def get_functions_and_classes(self, window_name: str) -> Optional[Dict]:
        """获取函数和类信息"""
        window_class = self.get_window_class(window_name)
        if window_class:
            return window_class.get("functions_and_classes")
        return None
    
    def get_class_info(self, window_name: str, class_name: str) -> Optional[Dict]:
        """获取类信息（包括成员和数据结构）"""
        functions_and_classes = self.get_functions_and_classes(window_name)
        if functions_and_classes:
            return functions_and_classes.get("classes", {}).get(class_name)
        return None
    
    def get_class_members(self, window_name: str, class_name: str) -> Optional[Dict]:
        """获取类的成员"""
        class_info = self.get_class_info(window_name, class_name)
        if class_info:
            return class_info.get("members")
        return None
    
    def get_data_structures(self, window_name: str, class_name: str) -> Optional[Dict]:
        """获取类的数据结构"""
        class_info = self.get_class_info(window_name, class_name)
        if class_info:
            return class_info.get("data_structures")
        return None
    
    def get_data_structure_members(self, window_name: str, class_name: str, data_structure_name: str) -> Optional[Dict]:
        """获取数据结构的成员"""
        data_structures = self.get_data_structures(window_name, class_name)
        if data_structures and data_structure_name in data_structures:
            return data_structures[data_structure_name].get("members")
        return None
    
    def list_all_window_classes(self) -> list:
        """列出所有窗口类"""
        if not self.config:
            return []
        return list(self.config.get("window_classes", {}).keys())
    
    def list_all_components(self, window_name: str) -> list:
        """列出所有组件"""
        components = self.get_window_components(window_name)
        if components:
            return list(components.keys())
        return []
    
    def list_all_functions(self, window_name: str) -> list:
        """列出所有函数"""
        functions_and_classes = self.get_functions_and_classes(window_name)
        if functions_and_classes:
            return list(functions_and_classes.get("functions", {}).keys())
        return []
    
    def list_all_classes(self, window_name: str) -> list:
        """列出所有类"""
        functions_and_classes = self.get_functions_and_classes(window_name)
        if functions_and_classes:
            return list(functions_and_classes.get("classes", {}).keys())
        return []


if __name__ == "__main__":
    # 测试加载
    loader = HierarchicalConfigLoader()
    if loader.load():
        print("\n=== 窗口类列表 ===")
        for window_name in loader.list_all_window_classes():
            print(f"  - {window_name}")
        
        window_name = "GeneratedWindow"
        print(f"\n=== {window_name} 信息 ===")
        
        print(f"\n组件数量: {len(loader.list_all_components(window_name))}")
        print(f"函数数量: {len(loader.list_all_functions(window_name))}")
        print(f"类数量: {len(loader.list_all_classes(window_name))}")
        
        # 显示一个组件的详细信息
        components = loader.list_all_components(window_name)
        if components:
            component_name = components[0]
            print(f"\n=== 组件: {component_name} ===")
            events = loader.get_component_events(window_name, component_name)
            if events:
                print(f"  事件数量: {len(events)}")
                for event_id, event_info in list(events.items())[:3]:
                    print(f"    - {event_id}: {event_info.get('name')}")
        
        # 显示一个类的详细信息
        classes = loader.list_all_classes(window_name)
        if classes:
            class_name = classes[0]
            print(f"\n=== 类: {class_name} ===")
            members = loader.get_class_members(window_name, class_name)
            if members:
                print(f"  成员数量: {len(members)}")
                for member_name, member_info in list(members.items())[:5]:
                    member_type = member_info.get('type', 'unknown')
                    print(f"    - {member_name} ({member_type})")
            
            data_structures = loader.get_data_structures(window_name, class_name)
            if data_structures:
                print(f"  数据结构数量: {len(data_structures)}")
                for ds_name, ds_info in data_structures.items():
                    print(f"    - {ds_name}: {ds_info.get('type')}")

