#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
层次化配置导出工具
按照窗口类->画布代码->功能代码->窗口组件->组件事件->函数和类->类成员->数据结构的层次结构导出
"""

import json
import os
import inspect
import ast
from typing import Dict, Any, List, Optional
from pathlib import Path

# 导入相关模块
import sys
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from module.control_base import control_registry
from module.event_manager_pyside import EventManager
from module.control_library_pyside import ControlLibrary
from module.code_generator_pyside import CodeGenerator
from module.control_pyside import Control, ControlManager
from module.event_function_generator_pyside import EventFunctionGenerator


class HierarchicalConfigExporter:
    """层次化配置导出器"""
    
    def __init__(self, output_dir: str = None):
        if output_dir is None:
            self.output_dir = os.path.dirname(os.path.abspath(__file__))
        else:
            self.output_dir = output_dir
        os.makedirs(self.output_dir, exist_ok=True)
    
    def export_all(self):
        """导出所有层次化配置"""
        # 创建主配置结构
        config = {
            "window_classes": {}
        }
        
        # 导出窗口类信息
        self._export_window_classes(config)
        
        # 保存到文件
        output_file = os.path.join(self.output_dir, "hierarchical_config.json")
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(config, f, ensure_ascii=False, indent=2)
        
        print(f"层次化配置已导出到: {output_file}")
        return config
    
    def _export_window_classes(self, config: Dict):
        """导出窗口类信息"""
        # 分析代码生成器中的窗口类
        from module.code_generator_pyside import CodeGenerator
        from module.control_pyside import ControlManager
        
        # 创建一个临时的控件管理器用于分析
        temp_manager = ControlManager()
        code_gen = CodeGenerator(temp_manager)
        
        window_class_name = code_gen.window_class_name
        
        config["window_classes"][window_class_name] = {
            "name": window_class_name,
            "base_class": "QMainWindow",
            "canvas_code": self._export_canvas_code(code_gen),
            "functional_code": self._export_functional_code(),
            "window_components": self._export_window_components(),
            "functions_and_classes": self._export_functions_and_classes()
        }
    
    def _export_canvas_code(self, code_gen: CodeGenerator) -> Dict:
        """导出画布代码信息"""
        return {
            "canvas_width": 800,
            "canvas_height": 600,
            "canvas_background": "#FFFFFF",
            "grid_enabled": True,
            "grid_size": 10,
            "snap_to_grid": True,
            "code_generation_methods": {
                "_generate_imports": "生成导入语句",
                "_generate_window_class": "生成窗口类代码",
                "_generate_controls_creation": "生成控件创建代码",
                "_generate_controls_properties": "生成控件属性设置代码",
                "_generate_signal_connections": "生成信号连接代码",
                "_generate_event_handlers": "生成事件处理函数"
            }
        }
    
    def _export_functional_code(self) -> Dict:
        """导出功能代码信息"""
        functional_code = {
            "files": {}
        }
        
        # 分析各个功能模块
        modules_to_analyze = [
            ("code_generator", "module.code_generator_pyside"),
            ("event_function_generator", "module.event_function_generator_pyside"),
            ("control_manager", "module.control_pyside"),
            ("event_manager", "module.event_manager_pyside")
        ]
        
        for module_name, module_path in modules_to_analyze:
            try:
                module = __import__(module_path, fromlist=[''])
                functional_code["files"][module_name] = self._analyze_module(module)
            except Exception as e:
                print(f"分析模块 {module_name} 失败: {e}")
        
        return functional_code
    
    def _export_window_components(self) -> Dict:
        """导出窗口组件信息"""
        components = {}
        
        # 获取所有控件
        all_controls = control_registry.get_all_controls()
        
        for control in all_controls:
            control_name = control.get_control_name()
            components[control_name] = {
                "name": control_name,
                "display_name": control_name,
                "category": control.get_category(),
                "class_name": control.get_control_class().__name__ if control.get_control_class() else None,
                "default_properties": self._serialize_properties(control.get_default_properties()),
                "default_size": control.get_default_size(),
                "events": self._export_component_events(control_name)
            }
        
        # 同时从ControlLibrary获取
        for control_type, control_info in ControlLibrary.CONTROLS.items():
            if control_type not in components:
                components[control_type] = {
                    "name": control_type,
                    "display_name": control_info.get("display_name", control_type),
                    "category": control_info.get("category", "基础"),
                    "class_name": control_info.get("class_name"),
                    "default_properties": control_info.get("default_properties", {}),
                    "default_size": control_info.get("default_size", (100, 30)),
                    "events": self._export_component_events(control_type)
                }
        
        return components
    
    def _export_component_events(self, control_type: str) -> Dict:
        """导出组件绑定的事件"""
        events = {}
        
        # 获取控件支持的所有事件
        event_list = EventManager.get_events_for_control(control_type)
        
        for event_id, event_info in event_list:
            events[event_id] = {
                "id": event_id,
                "name": event_info.get("name", event_id),
                "signal": event_info.get("signal"),
                "description": event_info.get("description", ""),
                "function_name_template": event_info.get("function_name_template", ""),
                "bound": False  # 默认未绑定
            }
        
        return events
    
    def _export_functions_and_classes(self) -> Dict:
        """导出所有函数和类"""
        functions_and_classes = {
            "functions": {},
            "classes": {}
        }
        
        # 分析主要模块
        modules = [
            ("code_generator", "module.code_generator_pyside", CodeGenerator),
            ("event_function_generator", "module.event_function_generator_pyside", EventFunctionGenerator),
            ("control", "module.control_pyside", Control),
        ]
        
        for module_name, module_path, main_class in modules:
            try:
                module = __import__(module_path, fromlist=[''])
                
                # 分析模块中的函数（包括模块级函数）
                for name, obj in inspect.getmembers(module):
                    if inspect.isfunction(obj):
                        # 检查是否是模块级函数
                        if hasattr(obj, '__module__') and obj.__module__ == module_path:
                            functions_and_classes["functions"][f"{module_name}.{name}"] = self._analyze_function(obj)
                
                # 分析类
                if main_class:
                    class_info = self._analyze_class(main_class)
                    functions_and_classes["classes"][main_class.__name__] = class_info
                    
                    # 将类的方法也添加到函数列表中
                    for method_name, method_info in class_info.get("members", {}).items():
                        if method_info.get("type") == "method":
                            functions_and_classes["functions"][f"{module_name}.{main_class.__name__}.{method_name}"] = {
                                "name": method_name,
                                "belongs_to_class": main_class.__name__,
                                "signature": method_info.get("signature", ""),
                                "docstring": method_info.get("docstring", "")
                            }
                
                # 分析ControlManager
                if module_name == "control":
                    from module.control_pyside import ControlManager
                    class_info = self._analyze_class(ControlManager)
                    functions_and_classes["classes"]["ControlManager"] = class_info
                    
                    # 将ControlManager的方法也添加到函数列表
                    for method_name, method_info in class_info.get("members", {}).items():
                        if method_info.get("type") == "method":
                            functions_and_classes["functions"][f"control_manager.ControlManager.{method_name}"] = {
                                "name": method_name,
                                "belongs_to_class": "ControlManager",
                                "signature": method_info.get("signature", ""),
                                "docstring": method_info.get("docstring", "")
                            }
                    
            except Exception as e:
                print(f"分析模块 {module_name} 失败: {e}")
                import traceback
                traceback.print_exc()
        
        return functions_and_classes
    
    def _analyze_module(self, module) -> Dict:
        """分析模块"""
        return {
            "name": module.__name__,
            "file": getattr(module, '__file__', ''),
            "docstring": module.__doc__ or "",
            "functions": {},
            "classes": {}
        }
    
    def _analyze_function(self, func) -> Dict:
        """分析函数"""
        sig = inspect.signature(func)
        return {
            "name": func.__name__,
            "docstring": func.__doc__ or "",
            "parameters": {
                name: {
                    "name": name,
                    "type": str(param.annotation) if param.annotation != inspect.Parameter.empty else None,
                    "default": str(param.default) if param.default != inspect.Parameter.empty else None,
                    "kind": str(param.kind)
                }
                for name, param in sig.parameters.items()
            },
            "return_type": str(sig.return_annotation) if sig.return_annotation != inspect.Signature.empty else None
        }
    
    def _analyze_class(self, cls) -> Dict:
        """分析类"""
        members = {
            "name": cls.__name__,
            "base_classes": [base.__name__ for base in cls.__bases__],
            "docstring": cls.__doc__ or "",
            "members": {},
            "data_structures": {}
        }
        
        # 分析类成员（包括私有成员，但标记为私有）
        for name, obj in inspect.getmembers(cls):
            # 跳过特殊方法（但保留重要的）
            if name.startswith('__') and name.endswith('__') and name not in ['__init__', '__str__', '__repr__']:
                continue
            
            try:
                if inspect.ismethod(obj) or inspect.isfunction(obj):
                    try:
                        sig = inspect.signature(obj)
                        members["members"][name] = {
                            "type": "method",
                            "signature": str(sig),
                            "docstring": obj.__doc__ or "",
                            "is_private": name.startswith('_'),
                            "parameters": {
                                param_name: {
                                    "name": param_name,
                                    "type": str(param.annotation) if param.annotation != inspect.Parameter.empty else None,
                                    "default": str(param.default) if param.default != inspect.Parameter.empty else None
                                }
                                for param_name, param in sig.parameters.items()
                            }
                        }
                    except Exception:
                        members["members"][name] = {
                            "type": "method",
                            "signature": f"{name}(...)",
                            "docstring": obj.__doc__ or "",
                            "is_private": name.startswith('_')
                        }
                elif isinstance(obj, property):
                    members["members"][name] = {
                        "type": "property",
                        "docstring": obj.__doc__ or "",
                        "is_private": name.startswith('_')
                    }
                elif not inspect.ismodule(obj) and not inspect.isclass(obj) and not callable(obj):
                    # 类变量或数据成员
                    try:
                        members["members"][name] = {
                            "type": "attribute",
                            "value": str(obj)[:100] if not callable(obj) else "callable",
                            "is_private": name.startswith('_')
                        }
                    except:
                        pass
            except Exception:
                pass
        
        # 分析数据结构（类的属性字典等）
        if hasattr(cls, '__annotations__'):
            for attr_name, attr_type in cls.__annotations__.items():
                members["data_structures"][attr_name] = {
                    "name": attr_name,
                    "type": str(attr_type),
                    "members": self._analyze_data_structure_members(attr_type)
                }
        
        # 分析__init__方法中的实例变量（作为数据结构）
        if hasattr(cls, '__init__'):
            try:
                init_sig = inspect.signature(cls.__init__)
                # 尝试从文档字符串或代码中提取实例变量
                init_doc = cls.__init__.__doc__ or ""
                # 这里可以进一步解析，暂时先记录
            except:
                pass
        
        return members
    
    def _analyze_data_structure_members(self, data_type) -> Dict:
        """分析数据结构成员"""
        members = {}
        
        # 如果是Dict类型，尝试分析
        if "Dict" in str(data_type):
            members["type"] = "dict"
            members["structure"] = "key-value pairs"
        
        # 如果是List类型
        elif "List" in str(data_type):
            members["type"] = "list"
            members["structure"] = "ordered collection"
        
        # 如果是Tuple类型
        elif "Tuple" in str(data_type):
            members["type"] = "tuple"
            members["structure"] = "immutable sequence"
        
        else:
            members["type"] = str(data_type)
            members["structure"] = "unknown"
        
        return members
    
    def _serialize_properties(self, properties: Dict[str, Any]) -> Dict[str, Any]:
        """序列化属性"""
        serialized = {}
        for key, value in properties.items():
            if hasattr(value, '__class__') and 'Qt' in str(value.__class__):
                try:
                    serialized[key] = str(value)
                except:
                    serialized[key] = repr(value)
            elif isinstance(value, tuple):
                serialized[key] = list(value)
            else:
                try:
                    json.dumps(value)
                    serialized[key] = value
                except:
                    serialized[key] = str(value)
        return serialized


if __name__ == "__main__":
    exporter = HierarchicalConfigExporter()
    config = exporter.export_all()
    print(f"\n导出完成！")
    print(f"窗口类数量: {len(config['window_classes'])}")
    
    # 打印统计信息
    for window_name, window_info in config['window_classes'].items():
        print(f"\n窗口类: {window_name}")
        print(f"  组件数量: {len(window_info.get('window_components', {}))}")
        functions = window_info.get('functions_and_classes', {}).get('functions', {})
        classes = window_info.get('functions_and_classes', {}).get('classes', {})
        print(f"  函数数量: {len(functions)}")
        print(f"  类数量: {len(classes)}")

