#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
函数库管理器
管理.fnpy格式的函数库文件
"""

import os
import ast
import json
from pathlib import Path
from typing import Dict, List, Optional, Any
from dataclasses import dataclass, field


@dataclass
class FunctionInfo:
    """函数信息"""
    name: str
    chinese_name: str  # 中文名称
    line: int
    docstring: Optional[str] = None
    parameters: List[str] = field(default_factory=list)


@dataclass
class ClassInfo:
    """类信息"""
    name: str
    chinese_name: str  # 中文名称
    line: int
    docstring: Optional[str] = None
    members: List[Dict[str, Any]] = field(default_factory=list)
    data_structures: List[Dict[str, Any]] = field(default_factory=list)


@dataclass
class DataTypeInfo:
    """数据类型信息"""
    name: str
    chinese_name: str  # 中文名称
    line: int
    docstring: Optional[str] = None
    members: List[Dict[str, Any]] = field(default_factory=list)


@dataclass
class ConstantInfo:
    """常量信息"""
    name: str
    chinese_name: str  # 中文名称
    line: int
    value: Optional[str] = None


@dataclass
class LibraryInfo:
    """函数库信息"""
    name: str
    chinese_name: str  # 中文名称
    file_path: str
    version: str = "1.0.0"
    create_time: str = ""
    author: str = ""
    author_contact: str = ""
    python_version: str = ""
    required_packages: Dict[str, str] = field(default_factory=dict)
    functions: List[FunctionInfo] = field(default_factory=list)
    classes: List[ClassInfo] = field(default_factory=list)
    data_types: List[DataTypeInfo] = field(default_factory=list)
    constants: List[ConstantInfo] = field(default_factory=list)
    code: str = ""  # 实现代码
    enabled: bool = True


class fnpyLibraryManager:
    """函数库管理器"""
    
    def __init__(self, library_dir: Optional[str] = None):
        """
        初始化函数库管理器
        
        Args:
            library_dir: 函数库目录路径，如果为None则使用默认路径
        """
        if library_dir is None:
            # 默认路径：项目根目录下的 lib/fnpy 文件夹
            base_dir = Path(__file__).parent
            self.library_dir = base_dir / "fnpy"
        else:
            self.library_dir = Path(library_dir)
        
        # 确保目录存在
        self.library_dir.mkdir(parents=True, exist_ok=True)
        
        # 存储所有函数库信息
        self.libraries: Dict[str, LibraryInfo] = {}
        
        # 加载设置
        self._load_settings()
    
    def _load_settings(self):
        """从设置中加载启用的函数库列表"""
        try:
            from PySide6.QtCore import QSettings
            settings = QSettings("VisualProgramming", "Settings")
            enabled_libraries = settings.value("fnpy_libraries/enabled", [], type=list)
            self.enabled_library_names = set(enabled_libraries) if enabled_libraries else set()
        except Exception:
            self.enabled_library_names = set()
    
    def _save_settings(self):
        """保存启用的函数库列表到设置"""
        try:
            from PySide6.QtCore import QSettings
            settings = QSettings("VisualProgramming", "Settings")
            settings.setValue("fnpy_libraries/enabled", list(self.enabled_library_names))
        except Exception:
            pass
    
    def scan_libraries(self) -> List[str]:
        """
        扫描函数库目录，返回所有函数库文件名列表
        
        Returns:
            函数库文件名列表（不含路径）
        """
        if not self.library_dir.exists():
            return []
        
        libraries = []
        for file_path in self.library_dir.glob("*.fnpy"):
            libraries.append(file_path.name)
        
        return sorted(libraries)
    
    def load_library(self, library_name: str) -> Optional[LibraryInfo]:
        """
        加载并解析.fnpy函数库文件
        
        Args:
            library_name: 函数库文件名（不含路径）
            
        Returns:
            函数库信息对象，如果加载失败则返回None
        """
        library_path = self.library_dir / library_name
        
        if not library_path.exists():
            return None
        
        try:
            with open(library_path, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # 解析.fnpy文件格式（JSON + 代码）
            # 格式：JSON元数据 + "\n---CODE---\n" + Python代码
            if "---CODE---" in content:
                parts = content.split("---CODE---", 1)
                metadata_json = parts[0].strip()
                code = parts[1].strip() if len(parts) > 1 else ""
            else:
                # 兼容旧格式：尝试解析为纯JSON或纯Python
                try:
                    json.loads(content)
                    metadata_json = content
                    code = ""
                except:
                    metadata_json = "{}"
                    code = content
            
            # 解析JSON元数据
            try:
                metadata = json.loads(metadata_json)
            except:
                metadata = {}
            
            # 创建函数库信息对象
            library_info = LibraryInfo(
                name=library_name.replace('.fnpy', ''),
                chinese_name=metadata.get("chinese_name", library_name.replace('.fnpy', '')),
                file_path=str(library_path),
                version=metadata.get("version", "1.0.0"),
                create_time=metadata.get("create_time", ""),
                author=metadata.get("author", ""),
                author_contact=metadata.get("author_contact", ""),
                python_version=metadata.get("python_version", ""),
                required_packages=metadata.get("required_packages", {}),
                code=code,
                enabled=library_name in self.enabled_library_names
            )
            
            # 解析Python代码
            if code:
                tree = ast.parse(code)
                
                # 获取函数和类的中文名称映射
                function_names = metadata.get("function_names", {})
                class_names = metadata.get("class_names", {})
                data_type_names = metadata.get("data_type_names", {})
                constant_names = metadata.get("constant_names", {})
                
                # 直接遍历模块级别的节点
                for node in tree.body:
                    if isinstance(node, ast.FunctionDef):
                        func_info = FunctionInfo(
                            name=node.name,
                            chinese_name=function_names.get(node.name, node.name),
                            line=node.lineno,
                            docstring=ast.get_docstring(node),
                            parameters=[arg.arg for arg in node.args.args]
                        )
                        library_info.functions.append(func_info)
                    
                    elif isinstance(node, ast.ClassDef):
                        # 判断是普通类还是数据类型
                        is_data_type = len(node.name) <= 15 and node.name[0].isupper()
                        method_count = sum(1 for item in node.body if isinstance(item, ast.FunctionDef))
                        if is_data_type and method_count <= 2:
                            # 作为数据类型处理
                            dt_info = DataTypeInfo(
                                name=node.name,
                                chinese_name=data_type_names.get(node.name, node.name),
                                line=node.lineno,
                                docstring=ast.get_docstring(node)
                            )
                            # 提取数据类型的成员
                            for dt_item in node.body:
                                if isinstance(dt_item, ast.Assign):
                                    for target in dt_item.targets:
                                        if isinstance(target, ast.Name):
                                            dt_info.members.append({
                                                "name": target.id,
                                                "type": "属性",
                                                "line": dt_item.lineno
                                            })
                                elif isinstance(dt_item, ast.FunctionDef):
                                    dt_info.members.append({
                                        "name": dt_item.name,
                                        "type": "方法",
                                        "line": dt_item.lineno
                                    })
                            library_info.data_types.append(dt_info)
                        else:
                            # 作为普通类处理
                            class_info = ClassInfo(
                                name=node.name,
                                chinese_name=class_names.get(node.name, node.name),
                                line=node.lineno,
                                docstring=ast.get_docstring(node)
                            )
                            
                            # 提取类的成员
                            for class_item in node.body:
                                if isinstance(class_item, ast.FunctionDef):
                                    class_info.members.append({
                                        "name": class_item.name,
                                        "type": "方法",
                                        "line": class_item.lineno
                                    })
                                elif isinstance(class_item, ast.Assign):
                                    for target in class_item.targets:
                                        if isinstance(target, ast.Name):
                                            class_info.members.append({
                                                "name": target.id,
                                                "type": "属性",
                                                "line": class_item.lineno
                                            })
                                elif isinstance(class_item, ast.ClassDef):
                                    # 嵌套类（数据结构）
                                    ds_members = []
                                    for ds_item in class_item.body:
                                        if isinstance(ds_item, ast.Assign):
                                            for target in ds_item.targets:
                                                if isinstance(target, ast.Name):
                                                    ds_members.append({
                                                        "name": target.id,
                                                        "type": "属性",
                                                        "line": ds_item.lineno
                                                    })
                                        elif isinstance(ds_item, ast.FunctionDef):
                                            ds_members.append({
                                                "name": ds_item.name,
                                                "type": "方法",
                                                "line": ds_item.lineno
                                            })
                                    class_info.data_structures.append({
                                        "name": class_item.name,
                                        "line": class_item.lineno,
                                        "members": ds_members
                                    })
                            
                            library_info.classes.append(class_info)
                    
                    elif isinstance(node, ast.Assign):
                        # 模块级别的常量
                        for target in node.targets:
                            if isinstance(target, ast.Name):
                                # 检查是否全大写（常量命名约定）
                                if target.id.isupper() or target.id.startswith("_"):
                                    const_info = ConstantInfo(
                                        name=target.id,
                                        chinese_name=constant_names.get(target.id, target.id),
                                        line=node.lineno,
                                        value=ast.unparse(node.value) if hasattr(ast, 'unparse') else str(node.value)
                                    )
                                    library_info.constants.append(const_info)
            
            self.libraries[library_name] = library_info
            return library_info
            
        except Exception as e:
            print(f"加载函数库 '{library_name}' 失败: {e}")
            import traceback
            traceback.print_exc()
            return None
    
    def get_enabled_libraries(self) -> List[LibraryInfo]:
        """
        获取所有启用的函数库
        
        Returns:
            启用的函数库信息列表
        """
        enabled = []
        for library_name in self.scan_libraries():
            if library_name not in self.libraries:
                self.load_library(library_name)
            
            library = self.libraries.get(library_name)
            if library and library.enabled:
                enabled.append(library)
        
        return enabled
    
    def set_library_enabled(self, library_name: str, enabled: bool):
        """
        设置函数库的启用状态
        
        Args:
            library_name: 函数库文件名
            enabled: 是否启用
        """
        if library_name not in self.libraries:
            self.load_library(library_name)
        
        library = self.libraries.get(library_name)
        if library:
            library.enabled = enabled
            if enabled:
                self.enabled_library_names.add(library_name)
            else:
                self.enabled_library_names.discard(library_name)
            self._save_settings()
    
    def get_library_code(self, library_name: str) -> Optional[str]:
        """
        获取函数库的代码内容
        
        Args:
            library_name: 函数库文件名
            
        Returns:
            函数库代码内容，如果不存在则返回None
        """
        library = self.libraries.get(library_name)
        if library:
            return library.code
        
        # 如果还没加载，尝试加载
        library_info = self.load_library(library_name)
        if library_info:
            return library_info.code
        
        return None
    
    def get_all_enabled_library_code(self) -> str:
        """
        获取所有启用的函数库代码（合并）
        
        Returns:
            所有启用函数库的代码（用换行分隔）
        """
        codes = []
        for library in self.get_enabled_libraries():
            code = library.code
            if code:
                codes.append(code)
        return "\n\n".join(codes)
