#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
窗口工具函数库
提供窗口相关的工具函数
"""

from typing import Optional, Dict
from PySide6.QtWidgets import QWidget, QMainWindow

# 全局窗口实例注册表：存储已打开的窗口实例
_window_instances: Dict[str, QWidget] = {}


def open_window(window_id: str, parent_window_id: Optional[str] = None) -> Optional[QWidget]:
    """
    打开窗口
    
    根据窗口ID（窗口类名）打开对应的窗口实例。
    如果窗口已经打开，则显示已存在的窗口；否则创建新实例。
    
    Args:
        window_id: 窗口ID（窗口类名，例如 "GeneratedWindow"）
        parent_window_id: 父窗口ID（可选），如果提供则作为父窗口
        
    Returns:
        打开的窗口对象，如果失败则返回None
        
    Example:
        # 打开主窗口
        window = open_window("GeneratedWindow")
        
        # 打开子窗口（模态对话框）
        dialog = open_window("SettingsDialog", parent_window_id="GeneratedWindow")
    """
    try:
        import sys
        import inspect
        
        # 如果窗口已经打开，直接显示并返回
        if window_id in _window_instances:
            window = _window_instances[window_id]
            if window and window.isVisible():
                window.show()
                window.raise_()
                window.activateWindow()
                return window
        
        # 从全局命名空间中查找窗口类
        # 方法1: 从调用者的全局命名空间查找（通常是主文件）
        window_class = None
        try:
            # 获取调用者的全局命名空间
            frame = sys._getframe(1)
            caller_globals = frame.f_globals
            
            if window_id in caller_globals:
                cls = caller_globals[window_id]
                if inspect.isclass(cls) and issubclass(cls, (QWidget, QMainWindow)):
                    window_class = cls
        except (ValueError, AttributeError):
            pass
        
        # 方法2: 如果没找到，尝试从所有已加载的模块中查找
        if window_class is None:
            for module_name, module in list(sys.modules.items()):
                try:
                    if hasattr(module, window_id):
                        cls = getattr(module, window_id)
                        if inspect.isclass(cls) and issubclass(cls, (QWidget, QMainWindow)):
                            window_class = cls
                            break
                except (AttributeError, TypeError):
                    continue
        
        # 方法3: 尝试从 __main__ 模块查找（通常是运行的主文件）
        if window_class is None:
            try:
                import __main__
                if hasattr(__main__, window_id):
                    cls = getattr(__main__, window_id)
                    if inspect.isclass(cls) and issubclass(cls, (QWidget, QMainWindow)):
                        window_class = cls
            except (AttributeError, TypeError):
                pass
        
        if window_class is None:
            print(f"错误: 找不到窗口类 '{window_id}'。请确保窗口类 '{window_id}' 已在代码中定义。")
            return None
        
        # 创建窗口实例
        window = window_class()
        
        # 设置父窗口（如果提供）
        if parent_window_id:
            parent_window = _window_instances.get(parent_window_id)
            if parent_window:
                window.setParent(parent_window)
                # 如果是QMainWindow，设置为模态窗口
                if isinstance(window, QMainWindow):
                    window.setWindowModality(2)  # Qt.WindowModality.WindowModal
            else:
                # 尝试从全局命名空间查找父窗口类
                if parent_window_id in caller_globals:
                    parent_cls = caller_globals[parent_window_id]
                    if inspect.isclass(parent_cls) and issubclass(parent_cls, (QWidget, QMainWindow)):
                        # 查找已存在的父窗口实例
                        for instance in _window_instances.values():
                            if isinstance(instance, parent_cls):
                                window.setParent(instance)
                                if isinstance(window, QMainWindow):
                                    window.setWindowModality(2)
                                break
        
        # 注册窗口实例
        _window_instances[window_id] = window
        
        # 连接窗口关闭信号，从注册表中移除
        def on_window_destroyed():
            if window_id in _window_instances:
                del _window_instances[window_id]
        
        window.destroyed.connect(on_window_destroyed)
        
        # 显示窗口
        window.show()
        window.raise_()
        window.activateWindow()
        
        return window
        
    except Exception as e:
        print(f"打开窗口失败: {e}")
        import traceback
        traceback.print_exc()
        return None


def get_window_instance(window_id: str) -> Optional[QWidget]:
    """
    获取已打开的窗口实例
    
    Args:
        window_id: 窗口ID（窗口类名）
        
    Returns:
        窗口实例，如果不存在则返回None
    """
    return _window_instances.get(window_id)


def close_window(window_id: str) -> bool:
    """
    关闭窗口
    
    Args:
        window_id: 窗口ID（窗口类名）
        
    Returns:
        是否成功关闭
    """
    if window_id in _window_instances:
        window = _window_instances[window_id]
        if window:
            window.close()
            return True
    return False

