#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
控件配置动态加载器
从lib文件夹加载控件配置文件并动态注册控件
"""

import json
import os
from typing import Dict, Any, List, Optional, Type
from pathlib import Path

# 导入PySide6组件
from PySide6.QtWidgets import QWidget
from PySide6.QtCore import Qt

# 导入控件基类和注册表
import sys
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from module.control_base import BaseControl, control_registry
from module.event_manager_pyside import EventManager


class DynamicControl(BaseControl):
    """动态加载的控件类"""
    
    def __init__(self, config: Dict[str, Any]):
        """
        初始化动态控件
        
        Args:
            config: 控件配置字典
        """
        super().__init__()
        self._config = config
        self._name = config.get("name", "")
        self._category = config.get("category", "基础")
        self._description = config.get("description", "")
        self._size = tuple(config.get("default_size", [100, 30]))
        self._properties = self._deserialize_properties(config.get("default_properties", {}))
        self._class_name = config.get("class_name")
        self._can_contain = config.get("can_contain", False)
        self._icon = None  # 可以后续加载图标
    
    def get_control_class(self) -> Type[QWidget]:
        """获取控件类"""
        if self._class_name:
            # 动态导入PySide6组件
            from PySide6.QtWidgets import (
                QLabel, QPushButton, QLineEdit, QTextEdit, QCheckBox,
                QRadioButton, QComboBox, QSlider, QProgressBar, QFrame,
                QGroupBox, QTabWidget, QStackedWidget, QListWidget,
                QTreeWidget, QTableWidget, QSpinBox, QDoubleSpinBox,
                QDateEdit, QTimeEdit, QDateTimeEdit, QDial, QLCDNumber,
                QScrollArea, QToolButton, QCommandLinkButton,
                QCalendarWidget, QKeySequenceEdit
            )
            
            # 映射类名到类对象
            class_map = {
                "QLabel": QLabel,
                "QPushButton": QPushButton,
                "QLineEdit": QLineEdit,
                "QTextEdit": QTextEdit,
                "QCheckBox": QCheckBox,
                "QRadioButton": QRadioButton,
                "QComboBox": QComboBox,
                "QSlider": QSlider,
                "QProgressBar": QProgressBar,
                "QFrame": QFrame,
                "QGroupBox": QGroupBox,
                "QTabWidget": QTabWidget,
                "QStackedWidget": QStackedWidget,
                "QListWidget": QListWidget,
                "QTreeWidget": QTreeWidget,
                "QTableWidget": QTableWidget,
                "QSpinBox": QSpinBox,
                "QDoubleSpinBox": QDoubleSpinBox,
                "QDateEdit": QDateEdit,
                "QTimeEdit": QTimeEdit,
                "QDateTimeEdit": QDateTimeEdit,
                "QDial": QDial,
                "QLCDNumber": QLCDNumber,
                "QScrollArea": QScrollArea,
                "QToolButton": QToolButton,
                "QCommandLinkButton": QCommandLinkButton,
                "QCalendarWidget": QCalendarWidget,
                "QKeySequenceEdit": QKeySequenceEdit
            }
            
            return class_map.get(self._class_name, QWidget)
        
        return QWidget
    
    def create_instance(self, parent=None) -> QWidget:
        """创建控件实例"""
        widget_class = self.get_control_class()
        instance = widget_class(parent)
        
        # 应用属性
        properties = self.get_default_properties()
        for prop_name, prop_value in properties.items():
            # 尝试使用setter方法
            setter_name = f"set{prop_name[0].upper()}{prop_name[1:]}"
            if hasattr(instance, setter_name):
                try:
                    getattr(instance, setter_name)(prop_value)
                except:
                    pass
            # 如果setter不存在，尝试直接设置属性
            elif hasattr(instance, prop_name):
                try:
                    setattr(instance, prop_name, prop_value)
                except:
                    pass
        
        # 设置大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            instance.resize(width, height)
        
        return instance
    
    def _deserialize_properties(self, properties: Dict[str, Any]) -> Dict[str, Any]:
        """反序列化属性（处理Qt枚举等）"""
        deserialized = {}
        for key, value in properties.items():
            # 处理Qt枚举字符串
            if isinstance(value, str) and "Qt." in value:
                try:
                    # 尝试解析Qt枚举
                    deserialized[key] = self._parse_qt_enum(value)
                except:
                    deserialized[key] = value
            # 处理列表（可能是元组）
            elif isinstance(value, list) and key in ["default_size", "iconSize"]:
                deserialized[key] = tuple(value)
            else:
                deserialized[key] = value
        return deserialized
    
    def _parse_qt_enum(self, enum_str: str):
        """解析Qt枚举字符串"""
        # 简单的枚举解析，可以根据需要扩展
        if "AlignLeft" in enum_str:
            return Qt.AlignmentFlag.AlignLeft
        elif "AlignRight" in enum_str:
            return Qt.AlignmentFlag.AlignRight
        elif "AlignCenter" in enum_str:
            return Qt.AlignmentFlag.AlignCenter
        elif "Horizontal" in enum_str:
            return Qt.Orientation.Horizontal
        elif "Vertical" in enum_str:
            return Qt.Orientation.Vertical
        # 默认返回字符串
        return enum_str


class ControlConfigLoader:
    """控件配置加载器 - 支持一个组件一个配置文件"""
    
    def __init__(self, config_dir: str = None, use_individual_files: bool = True):
        """
        初始化加载器
        
        Args:
            config_dir: 配置文件目录，默认为lib目录
            use_individual_files: 是否使用单个配置文件模式（一个组件一个文件）
        """
        if config_dir is None:
            self.config_dir = os.path.dirname(os.path.abspath(__file__))
        else:
            self.config_dir = config_dir
        
        self.use_individual_files = use_individual_files
        self._loaded_controls = set()  # 记录已加载的控件
        self._config_cache = {}  # 配置缓存
        
        # 单个配置文件目录
        if use_individual_files:
            self.individual_config_dir = os.path.join(self.config_dir, "control_configs")
            # 如果目录不存在，创建它
            if not os.path.exists(self.individual_config_dir):
                os.makedirs(self.individual_config_dir, exist_ok=True)
    
    def load_all(self, lazy: bool = False) -> bool:
        """
        加载所有控件配置
        
        Args:
            lazy: 是否使用懒加载模式（只加载文件列表，不实际加载配置）
        
        Returns:
            bool: 是否加载成功
        """
        try:
            if self.use_individual_files:
                # 使用单个配置文件模式
                if lazy:
                    # 懒加载：只扫描文件，不实际加载
                    self._scan_control_files()
                    print(f"已扫描 {self.individual_config_dir} 目录，共找到 {len(self._get_control_files())} 个控件配置文件")
                else:
                    # 立即加载所有配置
                    self.load_controls_from_individual_files()
            else:
                # 使用旧的单个文件模式（向后兼容）
                self.load_controls()
            
            # 加载事件定义（可选）
            # self.load_events()
            
            # 加载分类信息（可选）
            # self.load_categories()
            
            print(f"已从 {self.config_dir} 加载控件配置")
            return True
        except Exception as e:
            print(f"加载控件配置失败: {e}")
            import traceback
            traceback.print_exc()
            return False
    
    def _get_control_files(self) -> List[str]:
        """获取所有控件配置文件路径"""
        if not self.use_individual_files:
            return []
        
        if not os.path.exists(self.individual_config_dir):
            return []
        
        config_files = []
        for filename in os.listdir(self.individual_config_dir):
            if filename.endswith('.json') and filename != 'README.md':
                config_files.append(os.path.join(self.individual_config_dir, filename))
        return sorted(config_files)
    
    def _scan_control_files(self):
        """扫描控件配置文件（不实际加载）"""
        self._control_files = self._get_control_files()
    
    def load_controls_from_individual_files(self):
        """从单个配置文件加载控件定义"""
        if not self.use_individual_files:
            return
        
        config_files = self._get_control_files()
        
        if not config_files:
            print(f"未找到控件配置文件，目录: {self.individual_config_dir}")
            # 尝试从旧的 controls.json 迁移
            self._migrate_from_legacy_config()
            return
        
        loaded_count = 0
        for config_file in config_files:
            try:
                control_name = self._load_control_from_file(config_file)
                if control_name:
                    loaded_count += 1
            except Exception as e:
                print(f"加载控件配置文件失败 {config_file}: {e}")
        
        print(f"已从单个配置文件加载 {loaded_count} 个控件")
    
    def _load_control_from_file(self, config_file: str) -> Optional[str]:
        """
        从单个配置文件加载控件
        
        Args:
            config_file: 配置文件路径
        
        Returns:
            控件名称，如果加载失败则返回None
        """
        try:
            with open(config_file, 'r', encoding='utf-8') as f:
                control_config = json.load(f)
            
            # 获取控件名称
            control_name = control_config.get("name", "")
            if not control_name:
                # 从文件名获取
                control_name = os.path.splitext(os.path.basename(config_file))[0]
                control_config["name"] = control_name
            
            # 缓存配置
            self._config_cache[control_name] = control_config
            
            # 创建并注册控件
            dynamic_control = DynamicControl(control_config)
            control_registry.register_control(dynamic_control)
            self._loaded_controls.add(control_name)
            
            return control_name
        except Exception as e:
            print(f"加载控件配置文件失败 {config_file}: {e}")
            return None
    
    def load_control(self, control_name: str) -> bool:
        """
        按需加载单个控件配置（懒加载）
        
        Args:
            control_name: 控件名称
        
        Returns:
            bool: 是否加载成功
        """
        if control_name in self._loaded_controls:
            return True  # 已经加载
        
        if not self.use_individual_files:
            return False
        
        # 查找配置文件
        config_file = os.path.join(self.individual_config_dir, f"{control_name}.json")
        if not os.path.exists(config_file):
            return False
        
        return self._load_control_from_file(config_file) is not None
    
    def save_control_config(self, control_name: str, config: Dict[str, Any]) -> bool:
        """
        保存单个控件配置
        
        Args:
            control_name: 控件名称
            config: 控件配置字典
        
        Returns:
            bool: 是否保存成功
        """
        if not self.use_individual_files:
            return False
        
        try:
            # 确保配置中有名称
            config["name"] = control_name
            
            # 保存到文件
            config_file = os.path.join(self.individual_config_dir, f"{control_name}.json")
            with open(config_file, 'w', encoding='utf-8') as f:
                json.dump(config, f, ensure_ascii=False, indent=2)
            
            # 更新缓存
            self._config_cache[control_name] = config
            
            # 如果控件已加载，重新加载
            if control_name in self._loaded_controls:
                self.reload_control(control_name)
            
            return True
        except Exception as e:
            print(f"保存控件配置失败 {control_name}: {e}")
            return False
    
    def get_control_config(self, control_name: str) -> Optional[Dict[str, Any]]:
        """
        获取控件配置（如果未加载则自动加载）
        
        Args:
            control_name: 控件名称
        
        Returns:
            控件配置字典，如果不存在则返回None
        """
        # 先检查缓存
        if control_name in self._config_cache:
            return self._config_cache[control_name]
        
        # 尝试加载
        if self.load_control(control_name):
            return self._config_cache.get(control_name)
        
        return None
    
    def reload_control(self, control_name: str) -> bool:
        """
        重新加载单个控件配置
        
        Args:
            control_name: 控件名称
        
        Returns:
            bool: 是否重新加载成功
        """
        if not self.use_individual_files:
            return False
        
        # 从注册表中移除旧控件
        # 注意：control_registry 可能没有直接的移除方法，这里先重新注册
        config_file = os.path.join(self.individual_config_dir, f"{control_name}.json")
        if os.path.exists(config_file):
            # 清除缓存
            if control_name in self._config_cache:
                del self._config_cache[control_name]
            if control_name in self._loaded_controls:
                self._loaded_controls.remove(control_name)
            
            # 重新加载
            return self._load_control_from_file(config_file) is not None
        
        return False
    
    def reload_all_controls(self) -> bool:
        """重新加载所有控件配置"""
        if not self.use_individual_files:
            return False
        
        # 清除缓存和已加载记录
        self._config_cache.clear()
        self._loaded_controls.clear()
        
        # 重新加载
        self.load_controls_from_individual_files()
        return True
    
    def _migrate_from_legacy_config(self):
        """从旧的 controls.json 迁移到单个配置文件"""
        controls_file = os.path.join(self.config_dir, "controls.json")
        if not os.path.exists(controls_file):
            return
        
        print("检测到旧的 controls.json，开始迁移到单个配置文件...")
        try:
            with open(controls_file, 'r', encoding='utf-8') as f:
                controls_data = json.load(f)
            
            migrated_count = 0
            for control_name, control_config in controls_data.items():
                config_file = os.path.join(self.individual_config_dir, f"{control_name}.json")
                if not os.path.exists(config_file):
                    with open(config_file, 'w', encoding='utf-8') as f:
                        json.dump(control_config, f, ensure_ascii=False, indent=2)
                    migrated_count += 1
            
            print(f"已迁移 {migrated_count} 个控件配置到单个文件")
        except Exception as e:
            print(f"迁移失败: {e}")
    
    def load_controls(self):
        """加载控件定义（旧方法，向后兼容）"""
        controls_file = os.path.join(self.config_dir, "controls.json")
        
        if not os.path.exists(controls_file):
            print(f"控件配置文件不存在: {controls_file}")
            return
        
        with open(controls_file, 'r', encoding='utf-8') as f:
            controls_data = json.load(f)
        
        # 注册每个控件
        for control_name, control_config in controls_data.items():
            try:
                dynamic_control = DynamicControl(control_config)
                control_registry.register_control(dynamic_control)
                print(f"已加载控件: {control_name}")
            except Exception as e:
                print(f"加载控件 {control_name} 失败: {e}")
    
    def load_events(self):
        """加载事件定义（可选，用于覆盖默认事件）"""
        events_file = os.path.join(self.config_dir, "events.json")
        
        if not os.path.exists(events_file):
            return
        
        with open(events_file, 'r', encoding='utf-8') as f:
            events_data = json.load(f)
        
        # 更新EventManager的事件定义
        if "common_events" in events_data:
            EventManager.COMMON_EVENTS.update(events_data["common_events"])
        
        if "control_specific_events" in events_data:
            EventManager.CONTROL_SPECIFIC_EVENTS.update(events_data["control_specific_events"])
    
    def load_categories(self):
        """加载分类信息（可选）"""
        categories_file = os.path.join(self.config_dir, "categories.json")
        
        if not os.path.exists(categories_file):
            return
        
        with open(categories_file, 'r', encoding='utf-8') as f:
            categories_data = json.load(f)
        
        # 分类信息已经在加载控件时自动创建，这里可以用于验证或补充


if __name__ == "__main__":
    # 测试加载
    loader = ControlConfigLoader()
    loader.load_all()
    
    # 验证加载的控件
    all_controls = control_registry.get_all_controls()
    print(f"\n已加载 {len(all_controls)} 个控件:")
    for control in all_controls:
        print(f"  - {control.get_control_name()} ({control.get_category()})")

