#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
控件配置导出工具
将所有控件信息导出为JSON配置文件
"""

import json
import os
from typing import Dict, Any, List
from pathlib import Path

# 导入控件注册表和事件管理器
import sys
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..'))

from module.control_base import control_registry
from module.event_manager_pyside import EventManager
from module.control_library_pyside import ControlLibrary


class ControlConfigExporter:
    """控件配置导出器"""
    
    def __init__(self, output_dir: str = None):
        """
        初始化导出器
        
        Args:
            output_dir: 输出目录，默认为lib目录
        """
        if output_dir is None:
            # 获取当前文件所在目录
            self.output_dir = os.path.dirname(os.path.abspath(__file__))
        else:
            self.output_dir = output_dir
        
        # 确保输出目录存在
        os.makedirs(self.output_dir, exist_ok=True)
    
    def export_all(self):
        """导出所有控件配置"""
        # 导出控件定义
        self.export_controls()
        
        # 导出事件定义
        self.export_events()
        
        # 导出分类信息
        self.export_categories()
        
        # 导出代码生成模板
        self.export_code_templates()
        
        print(f"所有配置已导出到: {self.output_dir}")
    
    def export_controls(self):
        """导出所有控件定义"""
        controls_data = {}
        
        # 从control_registry获取所有控件
        all_controls = control_registry.get_all_controls()
        for control in all_controls:
            control_name = control.get_control_name()
            controls_data[control_name] = {
                "name": control_name,
                "display_name": control_name,  # 可以后续扩展
                "category": control.get_category(),
                "description": control.get_description(),
                "class_name": control.get_control_class().__name__ if control.get_control_class() else None,
                "default_properties": self._serialize_properties(control.get_default_properties()),
                "default_size": control.get_default_size(),
                "icon": None,  # 图标路径，如果有的话
                "resizable": True,
                "can_contain": getattr(control, '_can_contain', False),
                "events": self._get_control_events(control_name)
            }
        
        # 同时从ControlLibrary获取控件信息（兼容旧系统）
        for control_type, control_info in ControlLibrary.CONTROLS.items():
            if control_type not in controls_data:
                controls_data[control_type] = {
                    "name": control_type,
                    "display_name": control_info.get("display_name", control_type),
                    "category": control_info.get("category", "基础"),
                    "description": control_info.get("description", ""),
                    "class_name": control_info.get("class_name"),
                    "default_properties": control_info.get("default_properties", {}),
                    "default_size": control_info.get("default_size", (100, 30)),
                    "icon": control_info.get("icon"),
                    "resizable": control_info.get("resizable", True),
                    "can_contain": control_info.get("can_contain", False),
                    "events": self._get_control_events(control_type)
                }
        
        # 保存到文件
        output_file = os.path.join(self.output_dir, "controls.json")
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(controls_data, f, ensure_ascii=False, indent=2)
        
        print(f"已导出 {len(controls_data)} 个控件定义到 {output_file}")
    
    def export_events(self):
        """导出事件定义"""
        events_data = {
            "common_events": EventManager.COMMON_EVENTS,
            "control_specific_events": EventManager.CONTROL_SPECIFIC_EVENTS
        }
        
        # 序列化事件数据（处理Qt枚举等）
        events_data = self._serialize_events(events_data)
        
        # 保存到文件
        output_file = os.path.join(self.output_dir, "events.json")
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(events_data, f, ensure_ascii=False, indent=2)
        
        print(f"已导出事件定义到 {output_file}")
    
    def export_categories(self):
        """导出分类信息"""
        categories_data = {}
        
        # 从control_registry获取分类
        categories = control_registry.get_categories()
        for category in categories:
            categories_data[category.name] = {
                "name": category.name,
                "display_name": category.display_name,
                "controls": [c.get_control_name() for c in category.get_controls()]
            }
        
        # 同时从ControlLibrary获取分类（兼容旧系统）
        for cat_id, cat_name in ControlLibrary.CATEGORIES.items():
            if cat_id not in categories_data:
                controls = ControlLibrary.get_controls_by_category(cat_id)
                categories_data[cat_id] = {
                    "name": cat_id,
                    "display_name": cat_name,
                    "controls": [control_type for control_type, _ in controls]
                }
        
        # 保存到文件
        output_file = os.path.join(self.output_dir, "categories.json")
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(categories_data, f, ensure_ascii=False, indent=2)
        
        print(f"已导出分类信息到 {output_file}")
    
    def export_code_templates(self):
        """导出代码生成模板"""
        # 这里可以导出代码生成的模板或规则
        # 目前先导出一个基础模板结构
        templates_data = {
            "import_template": {
                "pyside": "from PySide6.QtWidgets import {widgets}\nfrom PySide6.QtCore import {core}\nfrom PySide6.QtGui import {gui}",
                "tkinter": "import tkinter as tk\nfrom tkinter import {widgets}"
            },
            "creation_template": {
                "pyside": "self.{var_name} = {class_name}({parent})",
                "tkinter": "{var_name} = {class_name}({parent})"
            },
            "property_setting_template": {
                "pyside": "self.{var_name}.{method}({value})",
                "tkinter": "{var_name}.{method}({value})"
            }
        }
        
        # 保存到文件
        output_file = os.path.join(self.output_dir, "code_templates.json")
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(templates_data, f, ensure_ascii=False, indent=2)
        
        print(f"已导出代码模板到 {output_file}")
    
    def _get_control_events(self, control_type: str) -> List[str]:
        """获取控件支持的事件列表"""
        events = EventManager.get_events_for_control(control_type)
        return [event_id for event_id, _ in events]
    
    def _serialize_properties(self, properties: Dict[str, Any]) -> Dict[str, Any]:
        """序列化属性（处理Qt枚举等）"""
        serialized = {}
        for key, value in properties.items():
            # 处理Qt枚举
            if hasattr(value, '__class__') and 'Qt' in str(value.__class__):
                # 尝试获取枚举的字符串表示
                try:
                    serialized[key] = str(value)
                except:
                    serialized[key] = repr(value)
            # 处理元组
            elif isinstance(value, tuple):
                serialized[key] = list(value)
            # 处理其他可序列化类型
            else:
                try:
                    json.dumps(value)  # 测试是否可序列化
                    serialized[key] = value
                except:
                    serialized[key] = str(value)
        return serialized
    
    def _serialize_events(self, events_data: Dict[str, Any]) -> Dict[str, Any]:
        """序列化事件数据"""
        serialized = {}
        for key, value in events_data.items():
            if isinstance(value, dict):
                serialized[key] = {}
                for k, v in value.items():
                    if isinstance(v, dict):
                        serialized[key][k] = {}
                        for k2, v2 in v.items():
                            # 跳过signal字段（可能是Qt信号对象）
                            if k2 == "signal" and v2 is not None:
                                serialized[key][k][k2] = str(v2) if not isinstance(v2, str) else v2
                            else:
                                serialized[key][k][k2] = v2
                    else:
                        serialized[key][k] = v
            else:
                serialized[key] = value
        return serialized


if __name__ == "__main__":
    exporter = ControlConfigExporter()
    exporter.export_all()

