#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
统一组件配置加载器
从组件配置文件加载样式和属性，应用到各个组件
"""

import json
from pathlib import Path
from typing import Dict, Optional, List
from PySide6.QtWidgets import QWidget
from PySide6.QtGui import QFont


class ComponentConfigLoader:
    """组件配置加载器"""
    
    def __init__(self, config_dir: Optional[Path] = None):
        """
        初始化配置加载器
        
        Args:
            config_dir: 配置文件目录，如果为None则使用默认目录
        """
        if config_dir is None:
            current_file = Path(__file__).resolve()
            config_dir = current_file.parent / "component_configs"
        
        self.config_dir = Path(config_dir)
        self._configs = {}  # 缓存配置
        self._load_all_configs()
    
    def _load_all_configs(self):
        """加载所有组件配置"""
        if not self.config_dir.exists():
            return
        
        for config_file in self.config_dir.glob("*.json"):
            try:
                with open(config_file, 'r', encoding='utf-8') as f:
                    config = json.load(f)
                    component_name = config.get('name', config_file.stem)
                    self._configs[component_name] = config
            except Exception as e:
                print(f"加载组件配置失败 {config_file}: {e}")
    
    def get_config(self, component_name: str) -> Optional[Dict]:
        """
        获取组件配置
        
        Args:
            component_name: 组件名称
        
        Returns:
            组件配置字典，如果不存在则返回None
        """
        return self._configs.get(component_name)
    
    def apply_config_to_widget(self, widget: QWidget, component_name: str):
        """
        将配置应用到组件
        
        Args:
            widget: 要应用配置的组件
            component_name: 组件名称
        """
        config = self.get_config(component_name)
        if not config:
            return
        
        # 应用样式
        if 'style' in config:
            widget.setStyleSheet(config['style'])
        
        # 应用字体
        if 'font' in config:
            font_config = config['font']
            font_family = font_config.get('family', 'Microsoft YaHei UI')
            font_size = font_config.get('size', 10)
            font = QFont(font_family, font_size)
            widget.setFont(font)
    
    def get_all_component_names(self) -> List[str]:
        """
        获取所有组件名称
        
        Returns:
            组件名称列表
        """
        return list(self._configs.keys())
    
    def reload_configs(self):
        """重新加载所有配置"""
        self._configs.clear()
        self._load_all_configs()

