#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
清除API密钥脚本
用于清除app_secrets.json文件中的所有API密钥
"""

import json
import os
import sys


def _resolve_app_secrets_path() -> str:
    """
    解析app_secrets.json文件的路径
    使用与ai_chat_module.py相同的逻辑
    """
    # 检查环境变量
    env_p = os.environ.get("APP_SECRETS_PATH", "")
    if isinstance(env_p, str) and env_p:
        try:
            if os.path.isfile(env_p):
                return env_p
        except Exception:
            pass
    
    # 获取可能的基路径
    try:
        exe = sys.executable
        exe_is_app = isinstance(exe, str) and exe.lower().endswith(".exe") and (not exe.lower().endswith("python.exe")) and (not exe.lower().endswith("pythonw.exe"))
        base_exe = os.path.dirname(exe) if exe_is_app else ""
    except Exception:
        base_exe = ""
    
    try:
        base_argv = os.path.dirname(os.path.abspath(sys.argv[0]))
    except Exception:
        base_argv = ""
    
    try:
        base_cwd = os.getcwd()
    except Exception:
        base_cwd = ""
    
    try:
        # 获取项目根目录（当前脚本的父目录的父目录）
        base_proj = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    except Exception:
        base_proj = ""
    
    # 尝试查找配置文件
    bases = [b for b in [base_exe, base_argv, base_cwd, base_proj] if isinstance(b, str) and b]
    for b in bases:
        for d in ("Python", "python"):
            p = os.path.join(b, d, "app_secrets.json")
            try:
                if os.path.isfile(p):
                    return p
            except Exception:
                pass
    
    # 默认路径
    fallback = os.path.join(
        base_proj or base_cwd or base_argv or base_exe or os.path.dirname(os.path.abspath(__file__)),
        "Python",
        "app_secrets.json"
    )
    return fallback


def clear_api_keys():
    """
    清除app_secrets.json文件中的所有API密钥
    """
    config_path = _resolve_app_secrets_path()
    
    print(f"正在查找配置文件: {config_path}")
    
    if not os.path.isfile(config_path):
        print(f"错误: 配置文件不存在: {config_path}")
        return False
    
    try:
        # 读取配置文件
        with open(config_path, "r", encoding="utf-8") as f:
            config = json.load(f)
        
        print(f"成功读取配置文件: {config_path}")
        print(f"当前配置:")
        print(f"  - Provider: {config.get('provider', 'N/A')}")
        print(f"  - Model: {config.get('model', 'N/A')}")
        
        # 检查是否有密钥
        keys = config.get("keys", {})
        if keys:
            print(f"\n找到以下服务商的密钥:")
            for provider, models in keys.items():
                if isinstance(models, dict):
                    print(f"  - {provider}: {len(models)} 个模型")
                    for model in models.keys():
                        print(f"    * {model}")
        else:
            print("\n未找到任何密钥")
            return True
        
        # 确认清除
        print("\n" + "="*50)
        response = input("确定要清除所有API密钥吗? (yes/no): ").strip().lower()
        if response not in ("yes", "y", "是"):
            print("操作已取消")
            return False
        
        # 清除所有密钥
        config["keys"] = {}
        
        # 保存配置文件
        with open(config_path, "w", encoding="utf-8") as f:
            json.dump(config, f, ensure_ascii=False, indent=2)
        
        print(f"\n✓ 成功清除所有API密钥")
        print(f"配置文件已更新: {config_path}")
        return True
        
    except json.JSONDecodeError as e:
        print(f"错误: 配置文件格式无效: {e}")
        return False
    except Exception as e:
        print(f"错误: 处理配置文件时发生异常: {e}")
        return False


def main():
    """主函数"""
    print("="*50)
    print("API密钥清除工具")
    print("="*50)
    print()
    
    success = clear_api_keys()
    
    print()
    print("="*50)
    if success:
        print("操作完成")
    else:
        print("操作失败")
    print("="*50)


if __name__ == "__main__":
    main()

