package tech.konata.phosphate.rendering.notification;

import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.WorldRenderer;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import tech.konata.phosphate.interfaces.SharedRenderingConstants;
import tech.konata.phosphate.management.FontManager;
import tech.konata.phosphate.management.ThemeManager;
import tech.konata.phosphate.rendering.Stencil;
import tech.konata.phosphate.rendering.animation.Animation;
import tech.konata.phosphate.rendering.animation.Easing;
import tech.konata.phosphate.rendering.animation.Interpolations;
import tech.konata.phosphate.rendering.entities.impl.Rect;
import tech.konata.phosphate.rendering.font.CFontRenderer;
import tech.konata.phosphate.rendering.rendersystem.RenderSystem;
import tech.konata.phosphate.utils.timing.Timer;

import java.awt.*;
import java.text.DecimalFormat;
import java.time.Duration;

public class Notification implements SharedRenderingConstants {
    public static Minecraft mc = Minecraft.getMinecraft();

    public String message, title;
    public String icon;
    public Type type;
    public long stayTime;
    public boolean forever = false;
    public double renderX = Double.MAX_VALUE;
    public double renderY = -50;
    public double progress = 0;
    public Timer timer;

    public Notification(String title, String message, Type type, long stayTime) {
        this.message = message;
        this.title = title;

        switch (type) {
            case INFO:
                this.icon = "A";
                break;
            case WARNING:
                this.icon = "B";
                break;
            case ERROR:
                this.icon = "C";
                break;
            case SUCCESS:
                this.icon = "D";
                break;
        }
        this.type = type;
        this.stayTime = stayTime;

        this.timer = new Timer();
        timer.reset();
    }

    // no stay time = show forever
    public Notification(String title, String message, Type type) {
        this.message = message;
        this.title = title;

        switch (type) {
            case INFO:
                this.icon = "A";
                break;
            case WARNING:
                this.icon = "B";
                break;
            case ERROR:
                this.icon = "C";
                break;
            case SUCCESS:
                this.icon = "D";
                break;
        }
        this.type = type;
        this.forever = true;

        this.timer = new Timer();
        timer.reset();
    }

    public void show() {
        NotificationManager.getNotifications().add(this);
    }

    DecimalFormat df = new DecimalFormat("##.#");

    double width = 160;
    double height = 40;

    Animation openCloseAnimation = new Animation(Easing.EASE_IN_OUT_CUBIC, Duration.ofMillis(500L));

    public void draw(double offsetX, double offsetY) {

        renderY = Interpolations.interpBezier(renderY, offsetY, 0.2);

        renderX = RenderSystem.getWidth() * 0.5 - width * 0.5;

        boolean closing = (timer.delayed().toMillis() > stayTime && !forever);

        openCloseAnimation.run(closing ? 0 : width * 0.5);

        NORMAL.add(() -> {
            RenderSystem.doScissor(renderX + width * 0.5 - openCloseAnimation.getValue(), renderY, openCloseAnimation.getValue() * 2, height + 4);
            roundedRect(renderX, renderY, width, height, 6, ThemeManager.getAsColor(ThemeManager.ThemeColor.Surface, 255));

            int fontColor;
            switch (this.type) {
                case INFO:
                    fontColor = ThemeManager.get(ThemeManager.ThemeColor.Text);
                    break;
                case WARNING:
                    fontColor = 0xffefbc12;
                    break;
                case ERROR:
                    fontColor = 0xfff04747;
                    break;
                case SUCCESS:
                    fontColor = 0xff23ad5c;
                    break;
                default:
                    throw new IllegalArgumentException();
            }

            FontManager.pf20bold.drawCenteredString(this.title, renderX + width * 0.5, renderY + 4, fontColor);

            if (!forever) {
                double leftTime = (this.stayTime - timer.delayed().toMillis()) / 1000.0;

//                String fmt = df.format(leftTime);
//
//                if (fmt.length() == 1)
//                    fmt += ".0";

//                FontManager.pf20.drawString(" (" + fmt + "s)", renderX + 8 + FontManager.pf20bold.getStringWidth(this.title), renderY + 4.5, fontColor);

                progress = ((double) timer.delayed().toMillis() / this.stayTime);
            }

            Stencil.write();
            Rect.draw(renderX, renderY + height - 3, width * progress, 3, -1, Rect.RectType.EXPAND);
            Stencil.erase();
            roundedRect(renderX, renderY, width, height, 6, new Color(fontColor));
            Stencil.dispose();

            CFontRenderer fontRenderer = FontManager.pf18bold;

            fontRenderer.drawCenteredString(this.message, renderX + width * 0.5, renderY + 10 + fontRenderer.getHeight(), fontColor);
            width = Math.max(fontRenderer.getStringWidth(this.message) + 8, 160);
            RenderSystem.endScissor();
        });

    }

    public boolean shouldDelete() {
        if (forever)
            return false;

        return isFinished() && openCloseAnimation.getValue() <= 0;
    }

    private boolean isFinished() {
        return timer.isDelayed(stayTime);
    }

    public enum Type {
        INFO, WARNING, ERROR, SUCCESS
    }

}
