package processing.sound;

import com.jsyn.util.SampleLoader;
import lombok.Getter;

import java.io.File;
import java.io.IOException;


// calls to amp(), pan() etc affect both the LAST initiated and still running sample, AND all subsequently started ones
/**
 * This is a Soundfile player which allows to play back and manipulate sound
 * files. Supported formats are: WAV, AIF/AIFF, and MP3.
 * 
 * MP3 decoding can be very slow on ARM processors (Android/Raspberry Pi), we generally recommend you use lossless WAV or AIF files.
 * @webref Sampling:SoundFile
 * @webBrief This is a Soundfile Player which allows to play back and manipulate soundfiles.
 **/
public class SoundFile extends AudioSample {

	public SoundFile(String path) {
		super();

		try {
			// load WAV or AIF using JSyn
			this.sample = SampleLoader.loadStreamedFloatSample(new File(path));
		} catch (IOException e) {
			e.printStackTrace();
		}

		this.initiatePlayer();
	}

	// Below are just duplicated methods from the AudioSample superclass which
	// are required for the reference to build the corresponding pages.

	/**
	 * Returns the number of channels of the soundfile as an int (1 for mono, 2 for stereo).
	 * 
	 * @return Returns the number of channels of the soundfile (1 for mono, 2 for
	 *         stereo)
	 * @webref Sampling:SoundFile
	 * @webBrief Returns the number of channels of the soundfile as an int (1 for mono, 2 for stereo).
	 **/
	public int channels() {
		return super.channels();
	}

	/**
	 * Cues the playhead to a fixed position in the soundfile. Note that <b>cue()</b> only 
	 * affects the playhead for future calls to <b>play()</b>, but not to <b>loop()</b>.
	 * 
	 * @param time
	 *            position in the soundfile that the next playback should start
	 *            from, in seconds.
	 * @webref Sampling:SoundFile
	 * @webBrief Cues the playhead to a fixed position in the soundfile.
	 **/
	public void cue(float time) {
		super.cue(time);
	}

	/**
	 * Returns the duration of the soundfile in seconds.
	 * 
	 * @webref Sampling:SoundFile
	 * @webBrief Returns the duration of the soundfile in seconds.
	 * @return The duration of the soundfile in seconds.
	 **/
	public float duration() {
		return super.duration();
	}

	public int getTotalTimeSeconds() {
		return (int) this.duration();
	}

	public int getCurrentTimeSeconds() {
		return (int) (getCurrentTimeMillis() / 1000);
	}

	public int getTotalTimeMillis() {
		return getTotalTimeSeconds() * 1000;
	}

	public float getCurrentTimeMillis() {
		return this.position() * 1000;
	}

	/**
	 * Returns the number of frames of this soundfile.
	 * 
	 * @webref Sampling:SoundFile
	 * @webBrief Returns the number of frames of this soundfile.
	 * @return The number of frames of this soundfile.
	 **/
	public int frames() {
		return super.frames();
	}

	public void play() {
		super.play();
	}

	public void play(float rate) {
		super.play(rate);
	}

	public void play(float rate, float amp) {
		super.play(rate, amp);
	}

	public void play(float rate, float pos, float amp) {
		super.play(rate, pos, amp);
	}

	public void play(float rate, float pos, float amp, float add) {
		super.play(rate, pos, amp, add);
	}

	private boolean muted = false;

	public boolean isMuted() {
		return muted;
	}

	public void setMuted(boolean muted) {
		this.muted = muted;

		if (muted) {
			float volume1 = this.volume;
			this.amp(0.001f);
			this.volume = volume1;
		} else {
			this.amp(volume);
		}
	}

	/**
	 * Starts the playback of the soundfile. Only plays to the end of the
	 * audiosample once. If <b>cue()</b> or <b>pause()</b> were called previously, playback 
	 * will resume from the cued position.
	 * 
	 * @param rate
	 *            relative playback rate to use. 1 is the original speed. 0.5 is
	 *            half speed and one octave down. 2 is double the speed and one
	 *            octave up.
	 * @param amp
	 *            the desired playback amplitude of the audiosample as a value from
	 *            0.0 (complete silence) to 1.0 (full volume)
	 * @param pos
	 *            the panoramic position of this sound unit from -1.0 (left) to 1.0
	 *            (right). Only works for mono soundfiles!
	 * @param cue
	 *            position in the audiosample that playback should start from, in
	 *            seconds.
	 * @param add
	 *            offset the output of the generator by the given value
	 * @webref Sampling:SoundFile
	 * @webBrief Starts the playback of the soundfile.
	 **/
	public void play(float rate, float pos, float amp, float add, float cue) {
		super.play(rate, pos, amp, add, cue);
	}


	/**
	 * Jump to a specific position in the soundfile while continuing to play 
	 * (or starting to play if it wasn't playing already).
	 * 
	 * @webref Sampling:SoundFile
	 * @webBrief Jump to a specific position in the soundfile while continuing to play (or starting to play if it wasn't playing already).
	 * @param time
	 *            position to jump to, in seconds.
	 **/
	public void jump(float time) {
		super.jump(time);
	}

	/**
	 * Stop the playback of the file, but cue it to the current position. The
	 * next call to <b>play()</b> will continue playing where it left off.
	 * 
	 * @see SoundFile#stop()
	 * @webref Sampling:SoundFile
	 * @webBrief Stop the playback of the file, but cue it to the current position.
	 */
	public void pause() {
		super.pause();
	}

	/**
	 * Check whether this soundfile is currently playing.
	 *
	 * @return `true` if the soundfile is currently playing, `false` if it is not.
	 * @webref Sampling:SoundFile
	 * @webBrief Check whether this soundfile is currently playing.
	 */
	public boolean isPlaying() {
		return super.isPlaying();
	}

	public void loop() {
		super.loop();
	}

	public void loop(float rate) {
		super.loop(rate);
	}

	public void loop(float rate, float amp) {
		super.loop(rate, amp);
	}

	public void loop(float rate, float pos, float amp) {
		super.loop(rate, pos, amp);
	}

	/**
	 * Starts playback which will loop at the end of the soundfile.
	 * 
	 * @param rate
	 *            relative playback rate to use. 1 is the original speed. 0.5 is
	 *            half speed and one octave down. 2 is double the speed and one
	 *            octave up.
	 * @param pos
	 *            the panoramic position of this sound unit from -1.0 (left) to 1.0
	 *            (right). Only works for mono soundfiles!
	 * @param amp
	 *            the desired playback amplitude of the audiosample as a value from
	 *            0.0 (complete silence) to 1.0 (full volume)
	 * @param add
	 *            offset the output of the generator by the given value
	 * @webref Sampling:SoundFile
	 * @webBrief Starts playback which will loop at the end of the soundfile.
	 */
	public void loop(float rate, float pos, float amp, float add) {
		super.loop(rate, pos, amp, add);
	}

	/**
	 * Changes the amplitude/volume of the player. Allowed values are between 0.0 and 1.0.
	 * 
	 * @param cue
	 *            position in the audiosample that the next playback or loop should
	 *            start from, in seconds. public void loop(float rate, float pos,
	 *            float amp, float add, float cue) { super.loop(rate, pos, amp, add,
	 *            cue); }
	 */

	/**
	 * Change the amplitude/volume of this audiosample.
	 *
	 * @param amp
	 *            A float value between 0.0 (complete silence) and 1.0 (full volume)
	 *            controlling the amplitude/volume of this sound.
	 * @webref Sampling:SoundFile
	 * @webBrief Changes the amplitude/volume of the player.
	 **/
	public void amp(float amp) {
		super.amp(amp);
		volume = amp;
	}

	@Getter
	public float volume = 1;

	/**
	 * Move the sound in a stereo panorama.-1.0 pans to the left channel and 1.0 to the 
	 * right channel. Note that panning is only supported for mono (1 channel) soundfiles.
	 * 
	 * @param pos
	 *            the panoramic position of this sound unit from -1.0 (left) to 1.0
	 *            (right).
	 * @webref Sampling:SoundFile
	 * @webBrief Move the sound in a stereo panorama.
	 **/
	public void pan(float pos) {
		super.pan(pos);
	}

	/**
	 * Set the playback rate of the soundfile. 1 is the original speed. 0.5 is half speed 
	 * and one octave down. 2 is double the speed and one octave up.
	 * 
	 * @param rate
	 *            Relative playback rate to use. 1 is the original speed. 0.5 is
	 *            half speed and one octave down. 2 is double the speed and one
	 *            octave up.
	 * @webref Sampling:SoundFile
	 * @webBrief Set the playback rate of the soundfile.
	 **/
	public void rate(float rate) {
		super.rate(rate);
	}

}
