#pragma once

#include <cstdint>
#include <Client/BuzzHouse/AST/SQLProtoStr.h>

namespace BuzzHouse
{

class CHAggregate
{
public:
    const bool support_nulls_clause;
    const uint32_t fnum, min_params, max_params, min_args, max_args;

    CHAggregate(const uint32_t f, const uint32_t min_p, const uint32_t max_p, const uint32_t min_a, const uint32_t m_args, const bool snc)
        : support_nulls_clause(snc)
        , fnum(f)
        , min_params(min_p)
        , max_params(max_p)
        , min_args(min_a)
        , max_args(m_args)
    {
    }
};

class CHFunction
{
public:
    const uint32_t fnum, min_lambda_param, max_lambda_param, min_args, max_args;

    CHFunction(const uint32_t f, const uint32_t min_lambda, const uint32_t max_lambda, const uint32_t min_a, const uint32_t m_args)
        : fnum(f)
        , min_lambda_param(min_lambda)
        , max_lambda_param(max_lambda)
        , min_args(min_a)
        , max_args(m_args)
    {
    }
};

const constexpr uint32_t ulimited_params = 10000;

const std::vector<CHAggregate> CHAggrs = {
    /// Standard aggregates
    CHAggregate(SQLFunc::FUNCcount, 0, 0, 0, 1, false),
    CHAggregate(SQLFunc::FUNCmin, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCmax, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCsum, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCavg, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCstddevPop, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCstddevPopStable, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCstddevSamp, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCstddevSampStable, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCvarPop, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCvarSamp, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCcorr, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcorrStable, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcorrMatrix, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCcovarPop, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcovarPopStable, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcovarPopMatrix, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCcovarSamp, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcovarSampStable, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcovarSampMatrix, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCentropy, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCexponentialMovingAverage, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCexponentialTimeDecayedAvg, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCexponentialTimeDecayedCount, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCexponentialTimeDecayedMax, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCexponentialTimeDecayedSum, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCgrouping, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCintervalLengthSum, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmedian, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCmedianDeterministic, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmedianExact, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCmedianExactWeighted, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmedianTiming, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCmedianTimingWeighted, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmedianTDigest, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCmedianTDigestWeighted, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmedianBFloat16, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCmedianDD, 1, 2, 1, 1, false),
    CHAggregate(SQLFunc::FUNCrankCorr, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCsumKahan, 0, 0, 1, 1, false),
    /// Clickhouse aggregates
    CHAggregate(SQLFunc::FUNCaggThrow, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCanalysisOfVariance, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCboundingRatio, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCargMin, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCargMax, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCapprox_top_count, 0, 2, 2, 2, false),
    CHAggregate(SQLFunc::FUNCapprox_top_k, 0, 2, 2, 2, false),
    CHAggregate(SQLFunc::FUNCapprox_top_sum, 0, 2, 2, 2, false),
    CHAggregate(SQLFunc::FUNCavgWeighted, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCtopK, 1, 2, 1, 1, false),
    CHAggregate(SQLFunc::FUNCtopKWeighted, 1, 2, 1, 1, false),
    CHAggregate(SQLFunc::FUNCdeltaSum, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCdeltaSumTimestamp, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCgroupArray, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupArrayArray, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupArrayLast, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupUniqArray, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupArrayInsertAt, 2, 2, 2, 2, false),
    CHAggregate(SQLFunc::FUNCgroupArrayMovingAvg, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupArrayMovingSum, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupArraySorted, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupArrayIntersect, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupBitAnd, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupBitOr, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupBitXor, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupBitmap, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupBitmapAnd, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupBitmapOr, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupBitmapXor, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCsumWithOverflow, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCsumMap, 0, 0, 1, 2, false),
    CHAggregate(SQLFunc::FUNCsumMapWithOverflow, 0, 0, 1, 2, false),
    CHAggregate(SQLFunc::FUNCsumMapFiltered, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCsumMapFilteredWithOverflow, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCminMap, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmaxMap, 0, 0, 1, 2, false),
    CHAggregate(SQLFunc::FUNCskewSamp, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCskewPop, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCkolmogorovSmirnovTest, 0, 2, 2, 2, false),
    CHAggregate(SQLFunc::FUNCkurtSamp, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCkurtPop, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCuniq, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCuniqExact, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCuniqCombined, 0, 1, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCuniqCombined64, 0, 1, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCuniqHLL12, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCuniqTheta, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCquantile, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileBFloat16, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileBFloat16Weighted, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileDD, 1, 2, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileDeterministic, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantileExact, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileExactLow, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileExactHigh, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileExactExclusive, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileExactInclusive, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileExactWeighted, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantileGK, 1, 2, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileExactWeightedInterpolated, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantileInterpolatedWeighted, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantiles, 1, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesBFloat16, 0, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesBFloat16Weighted, 0, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesDD, 1, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesDeterministic, 0, ulimited_params, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantilesExact, 0, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesExactLow, 0, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesExactHigh, 0, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesExactExclusive, 0, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesExactInclusive, 0, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesExactWeighted, 0, ulimited_params, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantilesGK, 1, ulimited_params, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantilesExactWeightedInterpolated, 0, ulimited_params, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantilesInterpolatedWeighted, 0, ulimited_params, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantileTDigest, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileTDigestWeighted, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCquantileTiming, 0, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCquantileTimingWeighted, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCsimpleLinearRegression, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCsingleValueOrNull, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCcontingency, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcramersV, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcramersVBiasCorrected, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCcategoricalInformationValue, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCtheilsU, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCwelchTTest, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCstudentTTest, 0, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmannWhitneyUTest, 0, 2, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmaxIntersections, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmaxIntersectionsPosition, 0, 0, 2, 2, false),
    CHAggregate(SQLFunc::FUNCmeanZTest, 3, 3, 2, 2, false),
    CHAggregate(SQLFunc::FUNCsparkBar, 1, 3, 2, 2, false),
    CHAggregate(SQLFunc::FUNCsumCount, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNClargestTriangleThreeBuckets, 1, 1, 2, 2, false),
    CHAggregate(SQLFunc::FUNCestimateCompressionRatio, 2, 2, 1, 1, false),
    CHAggregate(SQLFunc::FUNCgroupConcat, 0, 2, 1, 1, false),
    /// JSON and Dynamic tyoes
    CHAggregate(SQLFunc::FUNCdistinctDynamicTypes, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCdistinctJSONPaths, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCdistinctJSONPathsAndTypes, 0, 0, 1, 1, false),
    /// Parametric aggregates
    CHAggregate(SQLFunc::FUNChistogram, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCsequenceMatch, 1, 1, 3, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCsequenceCount, 1, 1, 3, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCwindowFunnel, 2, ulimited_params, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCretention, 0, 0, 1, ulimited_params, false),
    CHAggregate(SQLFunc::FUNCuniqUpTo, 1, 1, 1, 1, false),
    CHAggregate(SQLFunc::FUNCsequenceNextNode, 2, 2, 4, ulimited_params, false),
    /// Not deterministic
    CHAggregate(SQLFunc::FUNCany, 0, 0, 1, 1, true),
    CHAggregate(SQLFunc::FUNCanyHeavy, 0, 0, 1, 1, false),
    CHAggregate(SQLFunc::FUNCanyLast, 0, 0, 1, 1, true),
    CHAggregate(SQLFunc::FUNCfirst_value, 0, 0, 1, 1, true),
    CHAggregate(SQLFunc::FUNClast_value, 0, 0, 1, 1, true),
    CHAggregate(SQLFunc::FUNCgroupArraySample, 1, 2, 1, 1, false)};

const CHFunction materialize = CHFunction(SQLFunc::FUNCmaterialize, 0, 0, 1, 1);

const std::vector<CHFunction> CHFuncs = {
    /// Arithmetic Functions
    CHFunction(SQLFunc::FUNCplus, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCminus, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiply, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCdivide, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCintDiv, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCintDivOrZero, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCisFinite, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisInfinite, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCifNotFinite, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCisNaN, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCmodulo, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmoduloOrZero, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCpositiveModulo, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCnegate, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCabs, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCgcd, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNClcm, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmax2, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmin2, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiplyDecimal, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCdivideDecimal, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCbyteSwap, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCicebergBucket, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCicebergTruncate, 0, 0, 2, 2),
    /// String and array functions
    CHFunction(SQLFunc::FUNCempty, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnotEmpty, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClength, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreverse, 0, 0, 1, 1),
    /// Arrays
    CHFunction(SQLFunc::FUNCarray, 0, 0, 0, ulimited_params),
    CHFunction(SQLFunc::FUNCemptyArrayUInt8, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayUInt16, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayUInt32, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayUInt64, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayInt8, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayInt16, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayInt32, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayInt64, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayFloat32, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayFloat64, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayDate, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayDateTime, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayString, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCemptyArrayToSingle, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCarrayWithConstant, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayConcat, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayElement, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChas, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayElementOrNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasAll, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasAny, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasSubstr, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCindexOf, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayCount, 0, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayDotProduct, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCcountEqual, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayEnumerate, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayEnumerateUniq, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayEnumerateUniqRanked, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCarrayPopBack, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayPopFront, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayPushBack, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayPushFront, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayResize, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCarraySlice, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCarrayShingles, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarraySort, 0, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayPartialSort, 0, 1, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayReverseSort, 0, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayPartialReverseSort, 0, 1, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayReverseSort, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayUniq, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayJoin, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayDifference, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayDistinct, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayEnumerateDense, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayEnumerateDenseRanked, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCarrayUnion, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayIntersect, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayJaccardIndex, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayFold, 1, 1, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayReverse, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayFlatten, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayCompact, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayZip, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayZipUnaligned, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayROCAUC, 0, 0, 2, 4),
    CHFunction(SQLFunc::FUNCarrayAUC, 0, 0, 2, 4),
    CHFunction(SQLFunc::FUNCarrayAUCPR, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCarrayPRAUC, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCarrayMap, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayFilter, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayFill, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayReverseFill, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarraySplit, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayReverseSplit, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayExists, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayAll, 0, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayFirst, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayFirstOrNull, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayLast, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayLastOrNull, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayFirstIndex, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayLastIndex, 1, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayMin, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCarrayMax, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCarraySum, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCarrayAvg, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCarrayCumSum, 0, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayCumSumNonNegative, 0, 1, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayProduct, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCarrayRotateLeft, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayRotateRight, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayShiftLeft, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCarrayShiftRight, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCindexOfAssumeSorted, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayNormalizedGini, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarraySymmetricDifference, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayLevenshteinDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCarrayLevenshteinDistanceWeighted, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCarraySimilarity, 0, 0, 4, 4),
    /// Bit functions
    CHFunction(SQLFunc::FUNCbitAnd, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitOr, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitXor, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitNot, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitShiftLeft, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitShiftRight, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitRotateLeft, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitRotateRight, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitSlice, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCbyteSlice, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCbitTest, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitTestAll, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCbitTestAny, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCbitCount, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitHammingDistance, 0, 0, 2, 2),
    /// Bitmap functions
    CHFunction(SQLFunc::FUNCbitmapBuild, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitmapToArray, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitmapSubsetInRange, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCbitmapSubsetLimit, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCsubBitmap, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCbitmapContains, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapHasAny, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapHasAll, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapCardinality, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitmapMin, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitmapMax, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitmapTransform, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCbitmapAnd, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapOr, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapXor, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapAndnot, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapAndCardinality, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapOrCardinality, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapXorCardinality, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCbitmapAndnotCardinality, 0, 0, 2, 2),
    /// Comparison functions
    CHFunction(SQLFunc::FUNCequals, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCnotEquals, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCless, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCgreater, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNClessOrEquals, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCgreaterOrEquals, 0, 0, 2, 2),
    /// Conditional functions
    CHFunction(SQLFunc::FUNCif, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCmultiIf, 0, 0, 3, ulimited_params),
    CHFunction(SQLFunc::FUNCgreatest, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCleast, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCclamp, 0, 0, 3, 3),
    /// Time functions
    CHFunction(SQLFunc::FUNCmakeDate, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCmakeDate32, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCmakeDateTime, 0, 0, 6, 7),
    CHFunction(SQLFunc::FUNCmakeDateTime64, 0, 0, 6, 7),
    CHFunction(SQLFunc::FUNCtoTimeZone, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtimeZoneOf, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtimeZoneOffset, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoYear, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoQuarter, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoMonth, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDayOfYear, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDayOfMonth, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDayOfWeek, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoHour, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoMinute, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoSecond, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoMillisecond, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUnixTimestamp, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoStartOfYear, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfISOYear, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfQuarter, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfMonth, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoLastDayOfMonth, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoMonday, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfWeek, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCtoLastDayOfWeek, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCtoStartOfDay, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfHour, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfMinute, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfSecond, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoStartOfMillisecond, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoStartOfMicrosecond, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoStartOfNanosecond, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoStartOfFiveMinutes, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfTenMinutes, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfFifteenMinutes, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoStartOfInterval, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoTime, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeYearNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeQuarterNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeMonthNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeWeekNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeDayNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeHourNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeMinuteNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoRelativeSecondNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoISOYear, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoISOWeek, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoWeek, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoYearWeek, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCtoDaysSinceYearZero, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCfromDaysSinceYearZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCfromDaysSinceYearZero32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCage, 0, 0, 3, 4),
    CHFunction(SQLFunc::FUNCdate_diff, 0, 0, 3, 4),
    CHFunction(SQLFunc::FUNCDATE_TRUNC, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCdate_add, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCdate_sub, 0, 0, 2, 4),
    CHFunction(SQLFunc::FUNCtimestamp_add, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCtimestamp_sub, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCaddDate, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubDate, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtimeSlot, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCchangeYear, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCchangeMonth, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCchangeDay, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCchangeHour, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCchangeMinute, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCchangeSecond, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddYears, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddQuarters, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddMonths, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddWeeks, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddDays, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddHours, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddMinutes, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddSeconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddMilliseconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddMicroseconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddNanoseconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddInterval, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCaddTupleOfIntervals, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractYears, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractQuarters, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractMonths, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractWeeks, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractDays, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractHours, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractMinutes, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractSeconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractMilliseconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractMicroseconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractNanoseconds, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractInterval, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsubtractTupleOfIntervals, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtimeSlots, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCdateName, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmonthName, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtimeDiff, 0, 0, 2, 2),
    /// Distance
    CHFunction(SQLFunc::FUNCL1Norm, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCL2Norm, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCL2SquaredNorm, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCLinfNorm, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCLpNorm, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCL1Distance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCL2Distance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCL2SquaredDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCLinfDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCLpDistance, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCL1Normalize, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCL2Normalize, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCLinfNormalize, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCLpNormalize, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCcosineDistance, 0, 0, 2, 2),
    /// Encoding
    CHFunction(SQLFunc::FUNCchar, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNChex, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCunhex, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbin, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCunbin, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitmaskToList, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitmaskToArray, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbitPositionsToArray, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCmortonEncode, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCmortonDecode, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChilbertEncode, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNChilbertDecode, 0, 0, 2, 2),
    /// Hash
    CHFunction(SQLFunc::FUNChalfMD5, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCMD4, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCMD5, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsipHash64, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCsipHash64Keyed, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCsipHash128, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCsipHash128Keyed, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCsipHash128Reference, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCsipHash128ReferenceKeyed, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCcityHash64, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCintHash32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCintHash64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCSHA1, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCSHA224, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCSHA256, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCSHA512, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCSHA512_256, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCBLAKE3, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCURLHash, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCfarmFingerprint64, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCfarmHash64, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCjavaHash, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCjavaHashUTF16LE, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNChiveHash, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCmetroHash64, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCjumpConsistentHash, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCkeccak256, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCkostikConsistentHash, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCRIPEMD160, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCmurmurHash2_32, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCmurmurHash2_64, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCgccMurmurHash, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCkafkaMurmurHash, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCmurmurHash3_32, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCmurmurHash3_64, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCmurmurHash3_128, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCxxh3, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCxxHash32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCxxHash64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCngramSimHash, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCngramSimHashCaseInsensitive, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCngramSimHashUTF8, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCngramSimHashCaseInsensitiveUTF8, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCwordShingleSimHash, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCwordShingleSimHashCaseInsensitive, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCwordShingleSimHashUTF8, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCwordShingleSimHashCaseInsensitiveUTF8, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCwyHash64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCngramMinHash, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCngramMinHashCaseInsensitive, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCngramMinHashUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCngramMinHashCaseInsensitiveUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCngramMinHashArg, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCngramMinHashArgCaseInsensitive, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCngramMinHashArgUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCngramMinHashArgCaseInsensitiveUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHash, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHashCaseInsensitive, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHashUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHashCaseInsensitiveUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHashArg, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHashArgCaseInsensitive, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHashArgUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCwordShingleMinHashArgCaseInsensitiveUTF8, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCsqidEncode, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCsqidDecode, 0, 0, 1, 1),
    /// JSON
    CHFunction(SQLFunc::FUNCsimpleJSONHas, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsimpleJSONExtractUInt, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsimpleJSONExtractInt, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsimpleJSONExtractFloat, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsimpleJSONExtractBool, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsimpleJSONExtractRaw, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsimpleJSONExtractString, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCisValidJSON, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONHas, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONLength, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONType, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractUInt, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractInt, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractFloat, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractBool, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractString, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtract, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractKeysAndValues, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractKeys, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractRaw, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractArrayRaw, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONExtractKeysAndValuesRaw, 0, 0, 1, 5),
    CHFunction(SQLFunc::FUNCJSON_EXISTS, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCJSON_QUERY, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCJSON_VALUE, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoJSONString, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONArrayLength, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONMergePatch, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCJSONAllPaths, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONAllPathsWithTypes, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONDynamicPaths, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONDynamicPathsWithTypes, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONSharedDataPaths, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCJSONSharedDataPathsWithTypes, 0, 0, 1, 1),
    /// Dynamic
    CHFunction(SQLFunc::FUNCdynamicType, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdynamicElement, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCisDynamicElementInSharedData, 0, 0, 1, 1),
    /// Logical
    CHFunction(SQLFunc::FUNCand, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCor, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCnot, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCxor, 0, 0, 2, ulimited_params),
    /// Map
    CHFunction(SQLFunc::FUNCmap, 0, 0, 0, ulimited_params),
    CHFunction(SQLFunc::FUNCmapFromArrays, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCextractKeyValuePairs, 0, 0, 1, 4),
    CHFunction(SQLFunc::FUNCextractKeyValuePairsWithEscaping, 0, 0, 1, 4),
    CHFunction(SQLFunc::FUNCmapAdd, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCmapSubtract, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCmapPopulateSeries, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCmapKeys, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCmapContains, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmapContainsKeyLike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmapExtractKeyLike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmapValues, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCmapContainsValue, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmapContainsKeyLike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmapExtractKeyLike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmapApply, 1, 1, 1, 1),
    CHFunction(SQLFunc::FUNCmapFilter, 1, 1, 1, 1),
    CHFunction(SQLFunc::FUNCmapUpdate, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmapConcat, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCmapExists, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCmapAll, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCmapSort, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCmapPartialSort, 0, 1, 2, 2),
    CHFunction(SQLFunc::FUNCmapReverseSort, 0, 1, 1, 1),
    CHFunction(SQLFunc::FUNCmapPartialReverseSort, 0, 1, 2, 2),
    /// Math
    CHFunction(SQLFunc::FUNCe, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCpi, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCexp, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClog, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCexp2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCintExp2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClog2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCexp10, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCintExp10, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClog10, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsqrt, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCcbrt, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCerf, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCerfc, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClgamma, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtgamma, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsin, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCcos, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtan, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCasin, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCacos, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCatan, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCpow, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCcosh, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCacosh, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsinh, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCasinh, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtanh, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCatanh, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCatan2, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChypot, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNClog1p, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsign, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsigmoid, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdegrees, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCradians, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCfactorial, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCwidth_bucket, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCproportionsZTest, 0, 0, 6, 6),
    /// NLP
    CHFunction(SQLFunc::FUNCdetectCharset, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdetectLanguage, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdetectLanguageMixed, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdetectProgrammingLanguage, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdetectLanguageUnknown, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdetectTonality, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClemmatize, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCstem, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCsynonyms, 0, 0, 2, 2),
    /// Nullable
    CHFunction(SQLFunc::FUNCisNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisNullable, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisNotNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisNotDistinctFrom, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCisZeroOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCcoalesce, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCifNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCnullIf, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCassumeNotNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoNullable, 0, 0, 1, 1),
    /// String replace
    CHFunction(SQLFunc::FUNCoverlay, 0, 0, 3, 4),
    CHFunction(SQLFunc::FUNCoverlayUTF8, 0, 0, 3, 4),
    CHFunction(SQLFunc::FUNCreplaceOne, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCreplaceAll, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCreplaceRegexpOne, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCreplaceRegexpAll, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCregexpQuoteMeta, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCformat, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCtranslate, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCtranslateUTF8, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCprintf, 0, 0, 1, ulimited_params),
    /// Rounding
    CHFunction(SQLFunc::FUNCfloor, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCceiling, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtruncate, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCround, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCroundBankers, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCroundToExp2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCroundDuration, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCroundAge, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCroundDown, 0, 0, 2, 2),
    /// String search
    CHFunction(SQLFunc::FUNCposition, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNClocate, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCpositionCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCpositionUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCpositionCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAllPositions, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAllPositionsCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAllPositionsUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAllPositionsCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstPosition, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstPositionCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstPositionUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstPositionCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstIndex, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstIndexCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstIndexUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchFirstIndexCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAny, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAnyCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAnyUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiSearchAnyCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmatch, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiMatchAny, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiMatchAnyIndex, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiMatchAllIndices, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCmultiFuzzyMatchAny, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCmultiFuzzyMatchAnyIndex, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCmultiFuzzyMatchAllIndices, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCextract, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCextractAll, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCextractAllGroupsHorizontal, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCextractAllGroupsVertical, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNClike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCnotLike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCilike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCnotILike, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramDistanceCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramDistanceUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramDistanceCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramSearch, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramSearchCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramSearchUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngramSearchCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCcountSubstrings, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCcountSubstringsCaseInsensitive, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCcountSubstringsCaseInsensitiveUTF8, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCcountMatches, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCcountMatchesCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCregexpExtract, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNChasSubsequence, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasSubsequenceCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasSubsequenceUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasSubsequenceCaseInsensitiveUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasToken, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasTokenOrNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasTokenCaseInsensitive, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNChasTokenCaseInsensitiveOrNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCcompareSubstrings, 0, 0, 5, 5),
    /// Split strings
    CHFunction(SQLFunc::FUNCsplitByChar, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCsplitByString, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCsplitByRegexp, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCsplitByWhitespace, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCsplitByNonAlpha, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCarrayStringConcat, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCalphaTokens, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCextractAllGroups, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCngrams, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtokens, 0, 0, 1, 1),
    /// Strings
    CHFunction(SQLFunc::FUNClengthUTF8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCleft, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCleftUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCleftPad, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCleftPadUTF8, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCright, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrightUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrightPad, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCrightPadUTF8, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNClower, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCupper, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClowerUTF8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCupperUTF8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisValidUTF8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoValidUTF8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCspace, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreverseUTF8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCconcat, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCconcatAssumeInjective, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCconcatWithSeparator, 0, 0, 3, ulimited_params),
    CHFunction(SQLFunc::FUNCconcatWithSeparatorAssumeInjective, 0, 0, 3, ulimited_params),
    CHFunction(SQLFunc::FUNCsubstring, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCsubstringUTF8, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCsubstringIndex, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCsubstringIndexUTF8, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCappendTrailingCharIfAbsent, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCconvertCharset, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCbase58Encode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbase58Decode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtryBase58Decode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbase64Encode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbase64URLEncode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbase64Decode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbase64URLDecode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtryBase64Decode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtryBase64URLDecode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCendsWith, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCendsWithUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCstartsWith, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCstartsWithUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtrimLeft, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtrimRight, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtrimBoth, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCCRC32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCCRC32IEEE, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCCRC64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizeQuery, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizeQueryKeepNames, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizedQueryHash, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizedQueryHashKeepNames, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizeUTF8NFC, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizeUTF8NFD, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizeUTF8NFKC, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCnormalizeUTF8NFKD, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCencodeXMLComponent, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdecodeXMLComponent, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdecodeHTMLComponent, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCextractTextFromHTML, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCascii, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsoundex, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtryPunycodeDecode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCidnaEncode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtryIdnaEncode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCidnaDecode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbyteHammingDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCstringJaccardIndex, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCstringJaccardIndexUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCeditDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCeditDistanceUTF8, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCdamerauLevenshteinDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCjaroSimilarity, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCjaroWinklerSimilarity, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCinitcap, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCinitcapUTF8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCfirstLine, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCstringCompare, 0, 0, 2, 5),
    CHFunction(SQLFunc::FUNCsparseGrams, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCsparseGramsUTF8, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCsparseGramsHashes, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCsparseGramsHashesUTF8, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCstringBytesUniq, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCstringBytesEntropy, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbase32Encode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbase32Decode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtryBase32Decode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbech32Encode, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCbech32Decode, 0, 0, 1, 1),
    /// Timeseries
    CHFunction(SQLFunc::FUNCseriesOutliersDetectTukey, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCseriesPeriodDetectFFT, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCseriesDecomposeSTL, 0, 0, 2, 2),
    /// Time window
    CHFunction(SQLFunc::FUNCtumble, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCtumbleStart, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCtumbleEnd, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNChop, 0, 0, 3, 4),
    CHFunction(SQLFunc::FUNChopStart, 0, 0, 3, 4),
    CHFunction(SQLFunc::FUNChopEnd, 0, 0, 3, 4),
    /// Tuples
    CHFunction(SQLFunc::FUNCtuple, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCtupleElement, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCuntuple, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtupleHammingDistance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleToNameValuePairs, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtupleNames, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtuplePlus, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleMinus, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleMultiply, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleDivide, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleNegate, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtupleMultiplyByNumber, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleDivideByNumber, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleConcat, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCtupleIntDiv, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleIntDivOrZero, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleIntDivByNumber, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleIntDivOrZeroByNumber, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleModulo, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtupleModuloByNumber, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCflattenTuple, 0, 0, 1, 1),
    /// Type conversion
    CHFunction(SQLFunc::FUNCtoInt8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt8OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt8OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt8OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoInt16, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt16OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt16OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt16OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoInt32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt32OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt32OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt32OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoInt64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt64OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt64OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt64OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoInt128, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt128OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt128OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt128OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoInt256, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt256OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt256OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInt256OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoUInt8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt8OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt8OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt8OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoUInt16, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt16OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt16OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt16OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoUInt32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt32OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt32OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt32OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoUInt64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt64OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt64OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt64OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoUInt128, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt128OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt128OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt128OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoUInt256, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt256OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt256OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUInt256OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoFloat32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoFloat32OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoFloat32OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoFloat32OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoFloat64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoFloat64OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoFloat64OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoFloat64OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoBFloat16, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoBFloat16OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoBFloat16OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDate, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateOrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateOrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoDateTime, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateTimeOrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateTimeOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateTimeOrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoDate32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDate32OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDate32OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDate32OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoDateTime64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateTime64OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateTime64OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDateTime64OrDefault, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtoDecimal32, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal32OrZero, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal32OrNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal32OrDefault, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCtoDecimal64, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal64OrZero, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal64OrNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal64OrDefault, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCtoDecimal128, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal128OrZero, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal128OrNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal128OrDefault, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCtoDecimal256, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal256OrZero, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal256OrNull, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoDecimal256OrDefault, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCtoString, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoFixedString, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoStringCutToZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoDecimalString, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCreinterpretAsUInt8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsUInt16, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsUInt32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsUInt64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsUInt128, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsUInt256, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsInt8, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsInt16, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsInt32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsInt64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsInt128, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsInt256, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsFloat32, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsFloat64, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsDate, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsDateTime, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsString, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsFixedString, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreinterpretAsUUID, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoInterval, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoIntervalYear, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalQuarter, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalMonth, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalWeek, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalDay, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalHour, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalMinute, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalSecond, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalMillisecond, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalMicrosecond, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIntervalNanosecond, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCparseDateTime, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTimeOrZero, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTimeOrNull, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTimeInJodaSyntax, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTimeInJodaSyntaxOrZero, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTimeInJodaSyntaxOrNull, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTimeBestEffort, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTime32BestEffort, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTimeBestEffortUS, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTimeBestEffortOrNull, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTime32BestEffortOrNull, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTimeBestEffortOrZero, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTime32BestEffortOrZero, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTimeBestEffortUSOrNull, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTimeBestEffortUSOrZero, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCparseDateTime64BestEffort, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTime64BestEffortUS, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTime64BestEffortOrNull, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTime64BestEffortOrZero, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTime64BestEffortUSOrNull, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCparseDateTime64BestEffortUSOrZero, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCtoLowCardinality, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUnixTimestamp64Milli, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUnixTimestamp64Micro, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUnixTimestamp64Nano, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUnixTimestamp64Second, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCfromUnixTimestamp64Milli, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCfromUnixTimestamp64Micro, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCfromUnixTimestamp64Nano, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCfromUnixTimestamp64Second, 0, 0, 1, 2),
    /// ULID
    CHFunction(SQLFunc::FUNCULIDStringToDateTime, 0, 0, 1, 2),
    /// UUID
    CHFunction(SQLFunc::FUNCtoUUID, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUUIDOrDefault, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoUUIDOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUUIDOrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCUUIDStringToNum, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCUUIDNumToString, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCUUIDToNum, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCUUIDv7ToDateTime, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCserverUUID, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCsnowflakeToDateTime, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCsnowflakeToDateTime64, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCdateTimeToSnowflake, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdateTime64ToSnowflake, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCsnowflakeIDToDateTime, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCsnowflakeIDToDateTime64, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCdateTimeToSnowflakeID, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCdateTime64ToSnowflakeID, 0, 0, 1, 2),
    /// UniqTheta
    CHFunction(SQLFunc::FUNCuniqThetaUnion, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCuniqThetaIntersect, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCuniqThetaNot, 0, 0, 2, 2),
    /// IPv4 and IPv6
    CHFunction(SQLFunc::FUNCIPv4NumToString, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv4StringToNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv4StringToNumOrDefault, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv4StringToNumOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv4NumToStringClassC, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv6NumToString, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv6StringToNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv6StringToNumOrDefault, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv6StringToNumOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv4ToIPv6, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCcutIPv6, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCIPv4CIDRToRange, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCIPv6CIDRToRange, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoIPv4, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIPv4OrDefault, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIPv4OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIPv4OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIPv6, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIPv6OrDefault, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIPv6OrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoIPv6OrZero, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv6StringToNumOrDefault, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCIPv6StringToNumOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisIPv4String, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisIPv6String, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisIPAddressInRange, 0, 0, 2, 2),
    /// Geo
    CHFunction(SQLFunc::FUNCgreatCircleDistance, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCgeoDistance, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCgreatCircleAngle, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCpointInEllipses, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCpointInPolygon, 0, 0, 2, ulimited_params),
    CHFunction(SQLFunc::FUNCgeohashEncode, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCgeohashDecode, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCgeohashesInBox, 0, 0, 5, 5),
    CHFunction(SQLFunc::FUNCh3IsValid, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetResolution, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3EdgeAngle, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3EdgeLengthM, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3EdgeLengthKm, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCgeoToH3, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCh3ToGeo, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3ToGeoBoundary, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3kRing, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3GetBaseCell, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3HexAreaM2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3HexAreaKm2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3IndexesAreNeighbors, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3ToChildren, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3ToParent, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3ToString, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCstringToH3, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetResolution, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3IsResClassIII, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3IsPentagon, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetFaces, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3CellAreaM2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3CellAreaRads2, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3ToCenterChild, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3ExactEdgeLengthM, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3ExactEdgeLengthKm, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3ExactEdgeLengthRads, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3NumHexagons, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3PointDistM, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCh3PointDistKm, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCh3PointDistRads, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCh3GetRes0Indexes, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCh3GetPentagonIndexes, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3Line, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3Distance, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3HexRing, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3GetUnidirectionalEdge, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCh3UnidirectionalEdgeIsValid, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetOriginIndexFromUnidirectionalEdge, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetDestinationIndexFromUnidirectionalEdge, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetIndexesFromUnidirectionalEdge, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetUnidirectionalEdgesFromHexagon, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCh3GetUnidirectionalEdgeBoundary, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCgeoToS2, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCs2ToGeo, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCs2GetNeighbors, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCs2CellsIntersect, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCs2CapContains, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCs2CapUnion, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCs2RectAdd, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCs2RectContains, 0, 0, 3, 3),
    CHFunction(SQLFunc::FUNCs2RectUnion, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCs2RectIntersection, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCsvg, 0, 0, 1, 2),
    /// Other
    CHFunction(SQLFunc::FUNChostname, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCfqdn, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCbasename, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCvisibleWidth, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoTypeName, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbyteSize, 0, 0, 1, ulimited_params),
    materialize,
    CHFunction(SQLFunc::FUNCignore, 0, 0, 0, ulimited_params),
    CHFunction(SQLFunc::FUNCisConstant, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCbar, 0, 0, 4, 4),
    CHFunction(SQLFunc::FUNCtransform, 0, 0, 3, 4),
    CHFunction(SQLFunc::FUNCformatReadableDecimalSize, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCformatReadableSize, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCformatReadableQuantity, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCformatReadableTimeDelta, 0, 0, 1, 3),
    CHFunction(SQLFunc::FUNCneighbor, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCMACNumToString, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCMACStringToNum, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCMACStringToOUI, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCgetSizeOfEnumType, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCblockSerializedSize, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCtoColumnTypeName, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdumpColumnStructure, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdefaultValueOfArgumentType, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCdefaultValueOfTypeName, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCidentity, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCisDecimalOverflow, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCcountDigits, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCerrorCodeToName, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCgetOSKernelVersion, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCvariantElement, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCvariantType, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClowCardinalityIndices, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNClowCardinalityKeys, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCgetSubcolumn, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCgetTypeSerializationStreams, 0, 0, 1, 1),
    /// Not Deterministic
    /// Array
    CHFunction(SQLFunc::FUNCarrayShuffle, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCarrayPartialShuffle, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCarrayRandomSample, 0, 0, 2, 2),
    /// UUID
    CHFunction(SQLFunc::FUNCgenerateUUIDv4, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCgenerateUUIDv7, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCgenerateSnowflakeID, 0, 0, 0, 2),
    /// ULID
    CHFunction(SQLFunc::FUNCgenerateULID, 0, 0, 1, 2),
    /// Time
    CHFunction(SQLFunc::FUNCtimestamp, 0, 0, 1, 2),
    CHFunction(SQLFunc::FUNCtimeZone, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCserverTimeZone, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCnow, 0, 0, 0, 1),
    CHFunction(SQLFunc::FUNCnow64, 0, 0, 0, 1),
    CHFunction(SQLFunc::FUNCnowInBlock, 0, 0, 0, 1),
    CHFunction(SQLFunc::FUNCtoday, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCyesterday, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCformatDateTime, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCformatDateTimeInJodaSyntax, 0, 0, 2, 3),
    CHFunction(SQLFunc::FUNCUTCTimestamp, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCfromUnixTimestamp, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCfromUnixTimestampInJodaSyntax, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCtoModifiedJulianDay, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoModifiedJulianDayOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCfromModifiedJulianDay, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCfromModifiedJulianDayOrNull, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCtoUTCTimestamp, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCfromUTCTimestamp, 0, 0, 2, 2),
    /// Rand
    CHFunction(SQLFunc::FUNCrand, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCrand64, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCrandCanonical, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCrandConstant, 0, 0, 0, 1),
    CHFunction(SQLFunc::FUNCrandUniform, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrandNormal, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrandLogNormal, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrandBinomial, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrandNegativeBinomial, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrandPoisson, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCrandBernoulli, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCrandExponential, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCrandChiSquared, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCrandStudentT, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCrandFisherF, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCfuzzBits, 0, 0, 2, 2),
    /// Other
    CHFunction(SQLFunc::FUNCshowCertificate, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCdisplayName, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCcurrentDatabase, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCcurrentUser, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCcurrentSchemas, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCindexHint, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCreplicate, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCrevision, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCversion, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCbuildId, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCtcpPort, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCcurrentProfiles, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCenabledProfiles, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCdefaultProfiles, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCcurrentRoles, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCenabledRoles, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCdefaultRoles, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCblockSize, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNChasThreadFuzzer, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCuptime, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCblockNumber, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCrowNumberInBlock, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCrowNumberInAllBlocks, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCrunningDifference, 0, 0, 1, 1),
    CHFunction(SQLFunc::FUNCrunningConcurrency, 0, 0, 2, 2),
    CHFunction(SQLFunc::FUNCfilesystemAvailable, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCfilesystemUnreserved, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCfilesystemCapacity, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCqueryID, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCinitialQueryID, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCpartitionID, 0, 0, 1, ulimited_params),
    CHFunction(SQLFunc::FUNCshardNum, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCshardCount, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCzookeeperSessionUptime, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCgenerateRandomStructure, 0, 0, 0, 2),
    CHFunction(SQLFunc::FUNCconnectionId, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCtransactionID, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCtransactionLatestSnapshot, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCtransactionOldestSnapshot, 0, 0, 0, 0),
    CHFunction(SQLFunc::FUNCinitialQueryStartTime, 0, 0, 0, 0)};

const std::vector<CHFunction> CHTableFuncs = {
    /// Table Functions
    CHFunction(SQLTableFunc::TFazureBlobStorage, 0, 0, 3, 8),
    CHFunction(SQLTableFunc::TFazureBlobStorageCluster, 0, 0, 4, 9),
    CHFunction(SQLTableFunc::TFcluster, 0, 0, 2, 4),
    CHFunction(SQLTableFunc::TFclusterAllReplicas, 0, 0, 2, 4),
    CHFunction(SQLTableFunc::TFdictionary, 0, 0, 1, 1),
    CHFunction(SQLTableFunc::TFfile, 0, 0, 1, 5),
    CHFunction(SQLTableFunc::TFfileCluster, 0, 0, 2, 5),
    CHFunction(SQLTableFunc::TFformat, 0, 0, 2, 3),
    CHFunction(SQLTableFunc::TFgcs, 0, 0, 1, 8),
    CHFunction(SQLTableFunc::TFgenerateSeries, 0, 0, 2, 3),
    CHFunction(SQLTableFunc::TFgenerateRandom, 0, 0, 1, 4),
    CHFunction(SQLTableFunc::TFmerge, 0, 0, 1, 2),
    CHFunction(SQLTableFunc::TFmergeTreeIndex, 0, 0, 2, 4),
    CHFunction(SQLTableFunc::TFmergeTreeProjection, 0, 0, 3, 3),
    CHFunction(SQLTableFunc::TFmysql, 0, 0, 5, 7),
    CHFunction(SQLTableFunc::TFpostgresql, 0, 0, 5, 7),
    CHFunction(SQLTableFunc::TFremote, 0, 0, 1, 6),
    CHFunction(SQLTableFunc::TFremoteSecure, 0, 0, 1, 6),
    CHFunction(SQLTableFunc::TFS3, 0, 0, 1, 8),
    CHFunction(SQLTableFunc::TFS3Cluster, 0, 0, 2, 9),
    CHFunction(SQLTableFunc::TFsqlite, 0, 0, 2, 2),
    CHFunction(SQLTableFunc::TFurl, 0, 0, 1, 4),
    CHFunction(SQLTableFunc::TFurlCluster, 0, 0, 4, 4),
    CHFunction(SQLTableFunc::TFvalues, 0, 0, 1, ulimited_params)};

}
