---
description: 'Documentation for Functions for Generating Random Numbers'
sidebar_label: 'Random number'
slug: /sql-reference/functions/random-functions
title: 'Functions for Generating Random Numbers'
---

# Functions for Generating Random Numbers

All functions in this section accept zero or one arguments. The only use of the argument (if provided) is to prevent [common subexpression
elimination](/sql-reference/functions/overview#common-subexpression-elimination) such that two different executions within a row of the same random
function return different random values.

Related content

- Blog: [Generating random data in ClickHouse](https://clickhouse.com/blog/generating-random-test-distribution-data-for-clickhouse)

:::note
The random numbers are generated by non-cryptographic algorithms.
:::

## rand {#rand}

Returns a random UInt32 number with uniform distribution.

Uses a linear congruential generator with an initial state obtained from the system, which means that while it appears random, it's not truly random and can be predictable if the initial state is known. For scenarios where true randomness is crucial, consider using alternative methods like system-level calls or integrating with external libraries.

**Syntax**

```sql
rand()
```

Alias: `rand32`

**Arguments**

None.

**Returned value**

Returns a number of type UInt32.

**Example**

```sql
SELECT rand();
```

```response
1569354847 -- Note: The actual output will be a random number, not the specific number shown in the example
```

## rand64 {#rand64}

Returns a random UInt64 integer (UInt64) number

**Syntax**

```sql
rand64()
```

**Arguments**

None.

**Arguments**

Returns a number UInt64 number with uniform distribution.

Uses a linear congruential generator with an initial state obtained from the system, which means that while it appears random, it's not truly random and can be predictable if the initial state is known. For scenarios where true randomness is crucial, consider using alternative methods like system-level calls or integrating with external libraries.

**Example**

```sql
SELECT rand64();
```

```response
15030268859237645412 -- Note: The actual output will be a random number, not the specific number shown in the example.
```

## randCanonical {#randcanonical}

Returns a random Float64 number.

**Syntax**

```sql
randCanonical()
```

**Arguments**

None.

**Arguments**

Returns a Float64 value between 0 (inclusive) and 1 (exclusive).

**Example**

```sql
SELECT randCanonical();
```

```response
0.3452178901234567 - Note: The actual output will be a random Float64 number between 0 and 1, not the specific number shown in the example.
```

## randConstant {#randconstant}

Generates a single constant column filled with a random value. Unlike `rand`, this function ensures the same random value appears in every row of the generated column, making it useful for scenarios requiring a consistent random seed across rows in a single query.

**Syntax**

```sql
randConstant([x]);
```

**Arguments**

- **[x] (Optional):** An optional expression that influences the generated random value. Even if provided, the resulting value will still be constant within the same query execution. Different queries using the same expression will likely generate different constant values.

**Arguments**

Returns a column of type UInt32 containing the same random value in each row.

**Implementation details**

The actual output will be different for each query execution, even with the same optional expression. The optional parameter may not significantly change the generated value compared to using `randConstant` alone.

**Example**

```sql
SELECT randConstant() AS random_value;
```

```response
| random_value |
|--------------|
| 1234567890   |
```

```sql
SELECT randConstant(10) AS random_value;
```

```response
| random_value |
|--------------|
| 9876543210   |
```

## randUniform {#randuniform}

Returns a random Float64 drawn uniformly from interval [`min`, `max`].

**Syntax**

```sql
randUniform(min, max)
```

**Arguments**

- `min` - `Float64` - left boundary of the range,
- `max` - `Float64` - right boundary of the range.

**Returned value**

A random number of type [Float64](../data-types/float.md).

**Example**

```sql
SELECT randUniform(5.5, 10) FROM numbers(5)
```

```response
┌─randUniform(5.5, 10)─┐
│    8.094978491443102 │
│   7.3181248914450885 │
│    7.177741903868262 │
│    6.483347380953762 │
│    6.122286382885112 │
└──────────────────────┘
```

## randNormal {#randnormal}

Returns a random Float64 drawn from a [normal distribution](https://en.wikipedia.org/wiki/Normal_distribution).

**Syntax**

```sql
randNormal(mean, stddev)
```

**Arguments**

- `mean` - `Float64` - mean value of distribution,
- `stddev` - `Float64` - [standard deviation](https://en.wikipedia.org/wiki/Standard_deviation) of the distribution.

**Returned value**

- Random number. [Float64](../data-types/float.md).

**Example**

```sql
SELECT randNormal(10, 2) FROM numbers(5)
```

Result:

```result
┌──randNormal(10, 2)─┐
│ 13.389228911709653 │
│  8.622949707401295 │
│ 10.801887062682981 │
│ 4.5220192605895315 │
│ 10.901239123982567 │
└────────────────────┘
```

## randLogNormal {#randlognormal}

Returns a random Float64 drawn from a [log-normal distribution](https://en.wikipedia.org/wiki/Log-normal_distribution).

**Syntax**

```sql
randLogNormal(mean, stddev)
```

**Arguments**

- `mean` - `Float64` - mean value of distribution,
- `stddev` - `Float64` - [standard deviation](https://en.wikipedia.org/wiki/Standard_deviation) of the distribution.

**Returned value**

- Random number. [Float64](../data-types/float.md).

**Example**

```sql
SELECT randLogNormal(100, 5) FROM numbers(5)
```

Result:

```result
┌─randLogNormal(100, 5)─┐
│  1.295699673937363e48 │
│  9.719869109186684e39 │
│  6.110868203189557e42 │
│  9.912675872925529e39 │
│ 2.3564708490552458e42 │
└───────────────────────┘
```

## randBinomial {#randbinomial}

Returns a random UInt64 drawn from a [binomial distribution](https://en.wikipedia.org/wiki/Binomial_distribution).

**Syntax**

```sql
randBinomial(experiments, probability)
```

**Arguments**

- `experiments` - `UInt64` - number of experiments,
- `probability` - `Float64` - probability of success in each experiment, a value between 0 and 1.

**Returned value**

- Random number. [UInt64](../data-types/int-uint.md).

**Example**

```sql
SELECT randBinomial(100, .75) FROM numbers(5)
```

Result:

```result
┌─randBinomial(100, 0.75)─┐
│                      74 │
│                      78 │
│                      76 │
│                      77 │
│                      80 │
└─────────────────────────┘
```

## randNegativeBinomial {#randnegativebinomial}

Returns a random UInt64 drawn from a [negative binomial distribution](https://en.wikipedia.org/wiki/Negative_binomial_distribution).

**Syntax**

```sql
randNegativeBinomial(experiments, probability)
```

**Arguments**

- `experiments` - `UInt64` - number of experiments,
- `probability` - `Float64` - probability of failure in each experiment, a value between 0 and 1.

**Returned value**

- Random number. [UInt64](../data-types/int-uint.md).

**Example**

```sql
SELECT randNegativeBinomial(100, .75) FROM numbers(5)
```

Result:

```result
┌─randNegativeBinomial(100, 0.75)─┐
│                              33 │
│                              32 │
│                              39 │
│                              40 │
│                              50 │
└─────────────────────────────────┘
```

## randPoisson {#randpoisson}

Returns a random UInt64 drawn from a [Poisson distribution](https://en.wikipedia.org/wiki/Poisson_distribution).

**Syntax**

```sql
randPoisson(n)
```

**Arguments**

- `n` - `UInt64` - mean number of occurrences.

**Returned value**

- Random number. [UInt64](../data-types/int-uint.md).

**Example**

```sql
SELECT randPoisson(10) FROM numbers(5)
```

Result:

```result
┌─randPoisson(10)─┐
│               8 │
│               8 │
│               7 │
│              10 │
│               6 │
└─────────────────┘
```

## randBernoulli {#randbernoulli}

Returns a random UInt64 drawn from a [Bernoulli distribution](https://en.wikipedia.org/wiki/Bernoulli_distribution).

**Syntax**

```sql
randBernoulli(probability)
```

**Arguments**

- `probability` - `Float64` - probability of success, a value between 0 and 1.

**Returned value**

- Random number. [UInt64](../data-types/int-uint.md).

**Example**

```sql
SELECT randBernoulli(.75) FROM numbers(5)
```

Result:

```result
┌─randBernoulli(0.75)─┐
│                   1 │
│                   1 │
│                   0 │
│                   1 │
│                   1 │
└─────────────────────┘
```

## randExponential {#randexponential}

Returns a random Float64 drawn from a [exponential distribution](https://en.wikipedia.org/wiki/Exponential_distribution).

**Syntax**

```sql
randExponential(lambda)
```

**Arguments**

- `lambda` - `Float64` - lambda value.

**Returned value**

- Random number. [Float64](../data-types/float.md).

**Example**

```sql
SELECT randExponential(1/10) FROM numbers(5)
```

Result:

```result
┌─randExponential(divide(1, 10))─┐
│              44.71628934340778 │
│              4.211013337903262 │
│             10.809402553207766 │
│              15.63959406553284 │
│             1.8148392319860158 │
└────────────────────────────────┘
```

## randChiSquared {#randchisquared}

Returns a random Float64 drawn from a [Chi-square distribution](https://en.wikipedia.org/wiki/Chi-squared_distribution) - a distribution of a sum of the squares of k independent standard normal random variables.

**Syntax**

```sql
randChiSquared(degree_of_freedom)
```

**Arguments**

- `degree_of_freedom` - `Float64` - degree of freedom.

**Returned value**

- Random number. [Float64](../data-types/float.md).

**Example**

```sql
SELECT randChiSquared(10) FROM numbers(5)
```

Result:

```result
┌─randChiSquared(10)─┐
│ 10.015463656521543 │
│  9.621799919882768 │
│   2.71785015634699 │
│ 11.128188665931908 │
│  4.902063104425469 │
└────────────────────┘
```

## randStudentT {#randstudentt}

Returns a random Float64 drawn from a [Student's t-distribution](https://en.wikipedia.org/wiki/Student%27s_t-distribution).

**Syntax**

```sql
randStudentT(degree_of_freedom)
```

**Arguments**

- `degree_of_freedom` - `Float64` - degree of freedom.

**Returned value**

- Random number. [Float64](../data-types/float.md).

**Example**

```sql
SELECT randStudentT(10) FROM numbers(5)
```

Result:

```result
┌─────randStudentT(10)─┐
│   1.2217309938538725 │
│   1.7941971681200541 │
│ -0.28192176076784664 │
│   0.2508897721303792 │
│  -2.7858432909761186 │
└──────────────────────┘
```

## randFisherF {#randfisherf}

Returns a random Float64 drawn from a [F-distribution](https://en.wikipedia.org/wiki/F-distribution).

**Syntax**

```sql
randFisherF(d1, d2)
```

**Arguments**

- `d1` - `Float64` - d1 degree of freedom in `X = (S1 / d1) / (S2 / d2)`,
- `d2` - `Float64` - d2 degree of freedom in `X = (S1 / d1) / (S2 / d2)`,

**Returned value**

- Random number. [Float64](../data-types/float.md).

**Example**

```sql
SELECT randFisherF(10, 3) FROM numbers(5)
```

Result:

```result
┌──randFisherF(10, 3)─┐
│   7.286287504216609 │
│ 0.26590779413050386 │
│ 0.22207610901168987 │
│  0.7953362728449572 │
│ 0.19278885985221572 │
└─────────────────────┘
```

## randomString {#randomString}

Generates a string of the specified length filled with random bytes (including zero bytes). Not all characters may be printable.

**Syntax**

```sql
randomString(length)
```

**Arguments**

- `length` — String length in bytes. Positive integer.

**Returned value**

- String filled with random bytes. [String](../data-types/string.md).

**Example**

Query:

```sql
SELECT randomString(30) AS str, length(str) AS len FROM numbers(2) FORMAT Vertical;
```

Result:

```text
Row 1:
──────
str: 3 G  :   pT ?w тi  k aV f6
len: 30

Row 2:
──────
str: 9 ,]    ^   )  ]??  8
len: 30
```

## randomFixedString {#randomfixedstring}

Generates a binary string of the specified length filled with random bytes (including zero bytes). Not all characters may be printable.

**Syntax**

```sql
randomFixedString(length);
```

**Arguments**

- `length` — String length in bytes. [UInt64](../data-types/int-uint.md).

**Returned value(s)**

- String filled with random bytes. [FixedString](../data-types/fixedstring.md).

**Example**

Query:

```sql
SELECT randomFixedString(13) AS rnd, toTypeName(rnd)
```

Result:

```text
┌─rnd──────┬─toTypeName(randomFixedString(13))─┐
│ j▒h㋖HɨZ'▒ │ FixedString(13)                 │
└──────────┴───────────────────────────────────┘
```

## randomPrintableASCII {#randomprintableascii}

Generates a string with a random set of [ASCII](https://en.wikipedia.org/wiki/ASCII#Printable_characters) characters. All characters are printable.
If you pass `length < 0`, the behavior of the function is undefined.

**Syntax**

```sql
randomPrintableASCII(length)
```

**Arguments**

- `length` — String length in bytes. Positive integer.

**Returned value**

- String with a random set of [ASCII](https://en.wikipedia.org/wiki/ASCII#Printable_characters) printable characters. [String](../data-types/string.md)

**Example**

```sql
SELECT number, randomPrintableASCII(30) AS str, length(str) FROM system.numbers LIMIT 3
```

```text
┌─number─┬─str────────────────────────────┬─length(randomPrintableASCII(30))─┐
│      0 │ SuiCOSTvC0csfABSw=UcSzp2.`rv8x │                               30 │
│      1 │ 1Ag NlJ &RCN:*>HVPG;PE-nO"SUFD │                               30 │
│      2 │ /"+<"wUTh:=LjJ Vm!c&hI*m#XTfzz │                               30 │
└────────┴────────────────────────────────┴──────────────────────────────────┘
```

## randomStringUTF8 {#randomstringutf8}

Generates a random string of a specified length. Result string contains valid UTF-8 code points. The value of code points may be outside of the range of assigned Unicode.

**Syntax**

```sql
randomStringUTF8(length);
```

**Arguments**

- `length` — Length of the string in code points. [UInt64](../data-types/int-uint.md).

**Returned value(s)**

- UTF-8 random string. [String](../data-types/string.md).

**Example**

Query:

```sql
SELECT randomStringUTF8(13)
```

Result:

```text
┌─randomStringUTF8(13)─┐
│ 𘤗𙉝д兠庇󡅴󱱎󦐪􂕌𔊹𓰛   │
└──────────────────────┘
```

## fuzzBits {#fuzzBits}

**Syntax**

Flips the bits of String or FixedString `s`, each with probability `prob`.

**Syntax**

```sql
fuzzBits(s, prob)
```

**Arguments**

- `s` - `String` or `FixedString`,
- `prob` - constant `Float32/64` between 0.0 and 1.0.

**Returned value**

Fuzzed string with same type as `s`.

**Example**

```sql
SELECT fuzzBits(materialize('abacaba'), 0.1)
FROM numbers(3)
```

Result:

```result
┌─fuzzBits(materialize('abacaba'), 0.1)─┐
│ abaaaja                               │
│ a*cjab+                               │
│ aeca2A                                │
└───────────────────────────────────────┘
```

<!-- 
The inner content of the tags below are replaced at doc framework build time with 
docs generated from system.functions. Please do not modify or remove the tags.
See: https://github.com/ClickHouse/clickhouse-docs/blob/main/contribute/autogenerated-documentation-from-source.md
-->

<!--AUTOGENERATED_START-->
<!--AUTOGENERATED_END-->
