use serde_json::Value;
/* 
这里提一下微信的机制: 
微信你手机的"微信"页面里面所有的内容都有wxid,这是微信识别这些内容的唯一标识

>>>>好友的wxid不是微信号,wxid是你账号刚注册时的"微信号"一栏里面显示的内容,你之后改过的微信号在微信后台是另一个字段,wxid在账号注册时随机生成且不可修改,这个才是你账号的唯一凭证,微信号本质就是一个不可重复且可以用来登录和加好友的昵称

微信中好友的wxid以"wxid_"开头、公众号以"gh_"开头 、群聊以"@chatroom"结尾
还有, wxid为weixin是微信团队, filehelper是文件传输助手 ,qqmail是qq邮箱提醒的聊天, fmessage是朋友推荐消息, medianote是语音记事本, floatbottle是漂流瓶(这个功能不是早没了吗,为什么还留着)
而微信支付,微信运动是由公众号提供的内容所以wxid格式和普通公众号相同
*/


/// 用来判断wxid是否是好友, 微信中 好友的wxid以"wxid_"开头、公众号以"gh_"开头 、群聊以"@chatroom"结尾
/// 过滤包含"wxid"字段的JSON数组，保留以"wxid_"开头的元素
/// 
/// # 参数
/// * `items` - 包含JSON对象的数组
/// 
/// # 返回
/// 过滤后的数组
pub fn filter_wxid_items(items: Vec<Value>) -> Vec<Value> {
    items.into_iter()
        .filter(|item| {
            item.get("wxid")
                .and_then(Value::as_str)
                .map_or(false, |s| s.starts_with("wxid_"))
        })
        .collect()
}

/// 从官网api获取数据,用于检查更新和获取公告(暂时没有)与changelog功能
/// 这不会上传用户的隐私信息
pub async fn get_api_data() -> Result<serde_json::Value, reqwest::Error> {
    println!("正在访问api");
    const APP_API: &str = "https://we.freespace.host/wf/v1/app.json";
    let client = reqwest::Client::builder()
        .timeout(std::time::Duration::from_secs(5))
        .user_agent("WeFriends-app")
        .build()?;
    
    let resp = client.get(APP_API).send().await?;
    resp.json::<serde_json::Value>().await
}

/// 自动选择检测好友用的群,通过随机数识别是因为用户可能有成百上千的群聊,手动选择不现实
pub fn select_chatroom_helper(chatroom_rand: u32, contact_list: &Value) -> Option<String> {
    let rand_str = chatroom_rand.to_string();

    if let Some(items) = contact_list.as_array() {
        for item in items {
            // 安全获取 nickname 字段
            if let Some(nickname) = item.get("wxNickName").and_then(|v| v.as_str()) {
                if nickname.contains(&rand_str) && nickname.contains("@chatroom") {
                    // 成功找到，返回 wxid 
                    return item.get("wxid")
                        .and_then(|v| v.as_str())
                        .map(|s| s.to_string()); 
                }
            }
        }
    }
    None // 没找到
}

/// 从json数组批量取出wxid放到Vec里面
pub fn wxid_json2vec(data_chunk: &[Value]) -> Vec<String> {
    let mut id_batch = Vec::with_capacity(data_chunk.len());
    
    for item in data_chunk {
        if let Some(id_value) = item.get("wxid") {
            id_batch.push(id_value.to_string().to_owned());
        } else {
            eprint!("数据格式不正确")
        }
    }
    id_batch
}
/*
/ 分批处理，每批最多30个
    for chunk in data.chunks(35) {
        // 现在 id_batch 是这一批的用户 ID 列表
        println!("本批 {} 个 ID: {:?}", id_batch.len(), id_batch);
        
        // 在这里处理这批 ID，比如：
        // send_to_api(&id_batch);
    }
 */
