'use strict';
const electron = require('electron');
const { writeFile, readFileSync } = require('fs');
const path = require('path');

if (typeof electron === 'string') {
  throw new TypeError('Not running in an Electron environment!');
}

module.exports.init = (customPath = '', ignoreXY = false) => {
  const { app, BrowserWindow } = electron;
  const confPath =
    customPath ||
    path.join(app.getPath('userData'), `${app.getName()}-window-bounds.json`);
  let confObj = {};
  try {
    confObj = JSON.parse(readFileSync(confPath, 'utf8'));
  } catch (e) { }

  const bounds = new Proxy(confObj, {
    set: (target, key, value) => {
      target[key] = value;
      writeFile(confPath, JSON.stringify(target, null, 2), () => { });
      return true;
    },
  });

  const isOutOfScreens = ({ x, y, width }) => {
    const { screen } = electron;
    return (
      screen.getAllDisplays().find(({ bounds }) => {
        const dotInBounds = ({ x, y }) => {
          const { x: dx, y: dy, width, height } = bounds;
          return x >= dx && x < dx + width && y >= dy && y < dy + height;
        };
        return dotInBounds({ x, y }) || dotInBounds({ x: x + width, y });
      }) === undefined
    );
  };

  const register = (win) => {
    const wid = win.id;
    const { [wid]: cached } = bounds;
    const update = (e) => {
      bounds[wid] = { ...win.getNormalBounds(), isMaximized: win.isMaximized(), isFullScreen: win.isFullScreen() };
    };
    if (cached) {
      const { isFullScreen = false, isMaximized = false } = cached
      // check if author change window bounds
      const currentBounds = win.getBounds();
      const w = Math.max(currentBounds.width, cached.width);
      const h = Math.max(currentBounds.height, cached.height);
      const ignorePosition = ignoreXY || isOutOfScreens(cached);
      if (win.resizable) {
        if (isFullScreen) {
          win.setBounds({ x: cached.x, y: cached.y })
          win.setFullScreen(true);
        } else if (isMaximized) {
          win.setBounds({ x: cached.x, y: cached.y })
          win.maximize();
        } else {
          if (ignorePosition) {
            win.setBounds({ width: w, height: h });
          } else {
            win.setBounds({ ...cached, width: w, height: h });
          }
        }
      } else {
        if (!ignorePosition) {
          win.setBounds({ x: cached.x, y: cached.y });
        }
      }
    } else {
      update();
    }
    win.on('resize', update);
    win.on('moved', update);
    win.on('maximize', update)
  };

  const wins = BrowserWindow.getAllWindows();

  wins.forEach(register);

  app.on('browser-window-created', (_, win) => {
    register(win);
  });
};
