import os
import sys
import threading
import subprocess
import tkinter as tk
from tkinter import ttk, filedialog, messagebox, scrolledtext
import re
import json
from datetime import datetime

# 多语言支持
class I18n:
    def __init__(self):
        self.current_lang = "zh"  # 默认中文
        self.translations = {}
        # 获取脚本所在目录
        self.script_dir = os.path.dirname(os.path.abspath(__file__))
        self.load_translations()
        self.load_language_preference()
    
    def load_translations(self):
        """加载语言文件"""
        try:
            i18n_path = os.path.join(self.script_dir, "i18n.json")
            with open(i18n_path, "r", encoding="utf-8") as f:
                self.translations = json.load(f)
        except Exception as e:
            print(f"加载语言文件失败: {e}")
            # 使用默认中文
            self.translations = {"zh": {}}
    
    def load_language_preference(self):
        """加载用户的语言偏好"""
        try:
            config_path = os.path.join(self.script_dir, "config.json")
            if os.path.exists(config_path):
                with open(config_path, "r", encoding="utf-8") as f:
                    config = json.load(f)
                    self.current_lang = config.get("language", "zh")
        except:
            pass
    
    def save_language_preference(self):
        """保存用户的语言偏好"""
        try:
            config_path = os.path.join(self.script_dir, "config.json")
            config = {"language": self.current_lang}
            with open(config_path, "w", encoding="utf-8") as f:
                json.dump(config, f, ensure_ascii=False, indent=2)
        except Exception as e:
            print(f"保存语言偏好失败: {e}")
    
    def set_language(self, lang):
        """切换语言"""
        if lang in self.translations:
            self.current_lang = lang
            self.save_language_preference()
            return True
        return False
    
    def t(self, key, default=None):
        """翻译文本"""
        if default is None:
            default = key
        return self.translations.get(self.current_lang, {}).get(key, default)
    
    def get_available_languages(self):
        """获取可用的语言列表"""
        lang_names = {
            "zh": "中文",
            "en": "English",
            "ja": "日本語",
            "es": "Español"
        }
        return [(code, lang_names.get(code, code)) for code in self.translations.keys()]

# 全局i18n实例
i18n = I18n()

# 尝试检测可用的显卡
def detect_gpus():
    """检测系统中可用的GPU"""
    try:
        import torch
        if torch.cuda.is_available():
            gpu_count = torch.cuda.device_count()
            gpus = []
            for i in range(gpu_count):
                gpu_name = torch.cuda.get_device_name(i)
                gpus.append((f"cuda:{i}", f"GPU {i}: {gpu_name}"))
            return gpus
        else:
            return []
    except Exception as e:
        print(f"检测GPU失败: {e}")
        return []

APP_TITLE = "IndexTTS2 批量 & 单句情绪控制 GUI"

EMO_LEVELS = ["low", "mid", "high"]

# 和 QwenEmotion / infer_v2 里的顺序保持一致：
# ["高兴", "愤怒", "悲伤", "恐惧", "反感", "低落", "惊讶", "自然"] -> 英文 key
EMO_VECTOR_KEYS = [
    "happy",
    "angry",
    "sad",
    "afraid",
    "disgusted",
    "melancholic",
    "surprised",
    "calm",
]


def find_venv_python():
    base = os.path.dirname(os.path.abspath(__file__))
    cand = os.path.join(base, "venv", "Scripts", "python.exe")
    if os.path.isfile(cand):
        return cand
    return "python"


class TTSBatchGUI:
    def __init__(self, root):
        self.root = root
        self.root.title(i18n.t("app_title", APP_TITLE))
        self.python_exe = find_venv_python()
        
        # 获取脚本所在目录（用于资源文件路径）
        self.script_dir = os.path.dirname(os.path.abspath(__file__))

        # 检测可用GPU
        self.available_gpus = detect_gpus()

        # 全局状态
        self.current_emo_vector = [0.0] * len(EMO_VECTOR_KEYS)
        self.use_vector_for_batch = tk.BooleanVar(value=False)
        self.preserve_voice = tk.BooleanVar(value=False)
        
        # 显卡设置
        self.gpu_mode = tk.StringVar(value="auto")  # auto, single, multi, cpu
        self.selected_gpu = tk.StringVar(value="cuda:0" if self.available_gpus else "auto")

        # 正在运行的子进程
        self.current_process = None
        self.current_processes = []  # 多进程列表（多GPU模式）
        self.process_lock = threading.Lock()
        self.is_running = False  # 标记是否有任务正在运行
        self.is_stopping = False  # 标记是否正在停止
        
        # 日志记录
        self.log_buffer = []  # 保存所有日志
        self.max_log_lines = 10000  # 最多保留10000行日志
        
        # 进度跟踪
        self.current_progress = 0  # 当前音频的生成进度 (0-100)
        self.total_progress = 0  # 总体任务进度 (0-100)
        self.current_audio_index = 0  # 当前正在生成的音频序号
        self.total_audio_count = 0  # 总共需要生成的音频数量
        
        # 多GPU进度跟踪
        self.multi_gpu_mode = False  # 是否处于多GPU模式
        self.gpu_progress = {}  # 每个GPU的完成数量 {gpu_id: completed_count}

        self._build_ui()

    # ---------- UI 构建 ----------

    def _build_ui(self):
        # 顶部：语言选择和全局设置
        lang_frame = ttk.Frame(self.root)
        lang_frame.pack(fill="x", padx=8, pady=(6, 0))
        
        self.lang_label = ttk.Label(lang_frame, text=i18n.t("language"))
        self.lang_label.pack(side="left", padx=4)
        self.lang_var = tk.StringVar(value=i18n.current_lang)
        lang_combo = ttk.Combobox(
            lang_frame,
            textvariable=self.lang_var,
            values=[f"{code} - {name}" for code, name in i18n.get_available_languages()],
            state="readonly",
            width=15
        )
        lang_combo.pack(side="left", padx=4)
        lang_combo.bind("<<ComboboxSelected>>", self._on_language_change)
        
        # 捐赠按钮（放在顶部右侧）
        self.donate_btn = ttk.Button(
            lang_frame,
            text=i18n.t("donate"),
            command=self._show_donate_window
        )
        self.donate_btn.pack(side="right", padx=4)
        
        # 顶部：全局设置
        self.global_settings_frame = ttk.LabelFrame(self.root, text=i18n.t("global_settings"))
        self.global_settings_frame.pack(fill="x", padx=8, pady=6)
        top = self.global_settings_frame  # 保持原变量名以兼容后续代码

        # 声纹音频
        self.label_voice = ttk.Label(top, text=i18n.t("voice_audio"))
        self.label_voice.grid(row=0, column=0, sticky="w", padx=4, pady=2)
        self.entry_voice = ttk.Entry(top, width=45)
        self.entry_voice.grid(row=0, column=1, sticky="we", padx=4, pady=2)
        self.btn_voice = ttk.Button(top, text=i18n.t("browse"), command=self._choose_voice)
        self.btn_voice.grid(row=0, column=2, padx=4, pady=2)

        # 情绪参考音频（可空）
        self.label_emo_ref = ttk.Label(top, text=i18n.t("emotion_ref_audio"))
        self.label_emo_ref.grid(row=1, column=0, sticky="w", padx=4, pady=2)
        self.entry_emo_ref = ttk.Entry(top, width=45)
        self.entry_emo_ref.grid(row=1, column=1, sticky="we", padx=4, pady=2)
        self.btn_emo_ref = ttk.Button(top, text=i18n.t("browse"), command=self._choose_emo_ref)
        self.btn_emo_ref.grid(row=1, column=2, padx=4, pady=2)

        # 声纹保护 + 继承向量
        self.chk_preserve = ttk.Checkbutton(top, text=i18n.t("enable_voice_protection"), variable=self.preserve_voice)
        self.chk_preserve.grid(row=0, column=3, padx=6, pady=2, sticky="w")
        ttk.Button(top, text="?", width=2, command=self._show_preserve_help).grid(row=0, column=4, padx=2)

        self.chk_use_vector = ttk.Checkbutton(
            top,
            text=i18n.t("batch_use_emotion_vector"),
            variable=self.use_vector_for_batch,
            command=self._update_vector_status_indicator
        )
        self.chk_use_vector.grid(row=1, column=3, padx=6, pady=2, sticky="w")

        # 显卡设置
        self.label_gpu = ttk.Label(top, text=i18n.t("gpu_settings"))
        self.label_gpu.grid(row=2, column=0, sticky="w", padx=4, pady=2)
        
        gpu_frame = ttk.Frame(top)
        gpu_frame.grid(row=2, column=1, columnspan=2, sticky="we", padx=4, pady=2)
        
        # 显卡模式选择
        self.radio_auto = ttk.Radiobutton(
            gpu_frame, text=i18n.t("auto_select"), variable=self.gpu_mode, value="auto",
            command=self._on_gpu_mode_change
        )
        self.radio_auto.pack(side="left", padx=4)
        
        self.radio_single = ttk.Radiobutton(
            gpu_frame, text=i18n.t("single_gpu"), variable=self.gpu_mode, value="single",
            command=self._on_gpu_mode_change
        )
        self.radio_single.pack(side="left", padx=4)
        
        # 多显卡选项（只有在检测到2张及以上GPU时才显示）
        if len(self.available_gpus) >= 2:
            self.radio_multi = ttk.Radiobutton(
                gpu_frame, text=i18n.t("multi_gpu"), variable=self.gpu_mode, value="multi",
                command=self._on_gpu_mode_change
            )
            self.radio_multi.pack(side="left", padx=4)
        
        # CPU模式选项
        self.radio_cpu = ttk.Radiobutton(
            gpu_frame, text=i18n.t("cpu_mode"), variable=self.gpu_mode, value="cpu",
            command=self._on_gpu_mode_change
        )
        self.radio_cpu.pack(side="left", padx=4)
        
        # 单显卡选择下拉框
        self.gpu_combo = ttk.Combobox(
            gpu_frame,
            textvariable=self.selected_gpu,
            state="readonly",
            width=30
        )
        if self.available_gpus:
            self.gpu_combo['values'] = [f"{gpu[0]} - {gpu[1]}" for gpu in self.available_gpus]
            self.gpu_combo.set(f"{self.available_gpus[0][0]} - {self.available_gpus[0][1]}")
        else:
            self.gpu_combo['values'] = ["未检测到GPU"]
            self.gpu_combo.set("未检测到GPU")
        self.gpu_combo.pack(side="left", padx=4)
        self.gpu_combo.config(state="disabled")  # 默认禁用
        
        ttk.Button(top, text="?", width=2, command=self._show_gpu_help).grid(row=2, column=4, padx=2)

        top.columnconfigure(1, weight=1)

        # 主 Notebook：批量 / 单句
        self.main_nb = ttk.Notebook(self.root)
        self.main_nb.pack(fill="both", expand=True, padx=8, pady=4)

        # 批量模式 Tab
        self.batch_tab = ttk.Frame(self.main_nb)
        self.main_nb.add(self.batch_tab, text=i18n.t("batch_mode"))

        # 单句模式 Tab
        self.single_tab = ttk.Frame(self.main_nb)
        self.main_nb.add(self.single_tab, text=i18n.t("single_mode"))

        self._build_batch_tab()
        self._build_single_tab()

        # 底部：控制台
        bottom = ttk.Frame(self.root)
        bottom.pack(fill="both", expand=True, padx=8, pady=(0, 6))

        # 控制台控制栏
        console_control = ttk.Frame(bottom)
        console_control.pack(fill="x", pady=(0, 4))

        self.console_visible = tk.BooleanVar(value=True)
        self.chk_console = ttk.Checkbutton(
            console_control, text=i18n.t("show_console"), variable=self.console_visible,
            command=self._toggle_console
        )
        self.chk_console.pack(side="left")

        # 停止按钮
        self.stop_btn = ttk.Button(
            console_control, 
            text=i18n.t("stop_task"), 
            command=self._stop_current_task,
            state="disabled"
        )
        self.stop_btn.pack(side="right", padx=4)
        
        # 导出日志按钮
        self.export_log_btn = ttk.Button(
            console_control,
            text=i18n.t("export_log"),
            command=self._export_log
        )
        self.export_log_btn.pack(side="right", padx=4)

        # 运行状态标签
        status_text = i18n.t("status") + i18n.t("status_idle")
        self.status_label = ttk.Label(console_control, text=status_text, foreground="gray")
        self.status_label.pack(side="right", padx=8)

        self.console = scrolledtext.ScrolledText(bottom, height=12, state="disabled")
        self.console.pack(fill="both", expand=True)

    def _build_batch_tab(self):
        # 内部 Notebook：多个任务
        upper = ttk.Frame(self.batch_tab)
        upper.pack(fill="both", expand=True)

        self.task_nb = ttk.Notebook(upper)
        self.task_nb.pack(fill="both", expand=True, padx=4, pady=4)

        btn_frame = ttk.Frame(self.batch_tab)
        btn_frame.pack(fill="x", padx=4, pady=(0, 4))

        self.btn_add_task = ttk.Button(btn_frame, text=i18n.t("add_task"), command=self._add_task)
        self.btn_add_task.pack(side="left")
        self.btn_remove_task = ttk.Button(btn_frame, text=i18n.t("remove_task"), command=self._remove_current_task)
        self.btn_remove_task.pack(side="left", padx=4)
        self.btn_start_all_tasks = ttk.Button(btn_frame, text=i18n.t("start_all_tasks"), command=self._run_all_tasks)
        self.btn_start_all_tasks.pack(side="right")
        self.btn_start_task = ttk.Button(btn_frame, text=i18n.t("start_task"), command=self._run_current_task)
        self.btn_start_task.pack(side="right", padx=4)
        
        # 情绪向量状态指示器
        self.vector_status_label = ttk.Label(
            btn_frame, 
            text="", 
            foreground="gray",
            font=("", 9)
        )
        self.vector_status_label.pack(side="right", padx=10)
        self._update_vector_status_indicator()
        
        # 进度条区域
        progress_frame = ttk.Frame(self.batch_tab)
        progress_frame.pack(fill="x", padx=4, pady=(0, 4))
        
        # 当前音频进度
        current_prog_frame = ttk.Frame(progress_frame)
        current_prog_frame.pack(fill="x", pady=2)
        
        self.current_progress_label = ttk.Label(
            current_prog_frame, 
            text=i18n.t("current_audio_progress"),
            font=("", 9)
        )
        self.current_progress_label.pack(side="left", padx=(0, 5))
        
        self.current_progress_bar = ttk.Progressbar(
            current_prog_frame,
            length=400,
            mode='determinate',
            style='Blue.Horizontal.TProgressbar'
        )
        self.current_progress_bar.pack(side="left", fill="x", expand=True)
        
        self.current_progress_text = ttk.Label(
            current_prog_frame,
            text="0%",
            font=("", 9),
            width=10
        )
        self.current_progress_text.pack(side="left", padx=(5, 0))
        
        # 整体任务进度
        total_prog_frame = ttk.Frame(progress_frame)
        total_prog_frame.pack(fill="x", pady=2)
        
        self.total_progress_label = ttk.Label(
            total_prog_frame,
            text=i18n.t("total_task_progress"),
            font=("", 9)
        )
        self.total_progress_label.pack(side="left", padx=(0, 5))
        
        self.total_progress_bar = ttk.Progressbar(
            total_prog_frame,
            length=400,
            mode='determinate',
            style='Blue.Horizontal.TProgressbar'
        )
        self.total_progress_bar.pack(side="left", fill="x", expand=True)
        
        self.total_progress_text = ttk.Label(
            total_prog_frame,
            text="0/0",
            font=("", 9),
            width=10
        )
        self.total_progress_text.pack(side="left", padx=(5, 0))
        
        # 配置进度条样式为蓝色
        style = ttk.Style()
        style.configure('Blue.Horizontal.TProgressbar', 
                       troughcolor='white',
                       background='#4A90E2',
                       darkcolor='#4A90E2',
                       lightcolor='#4A90E2',
                       bordercolor='gray')

        self.tasks = []
        self._add_task()  # 默认一个任务

    def _build_single_tab(self):
        frame = ttk.Frame(self.single_tab)
        frame.pack(fill="both", expand=True, padx=6, pady=6)

        # 单句文本
        self.label_single_text = ttk.Label(frame, text=i18n.t("single_text"))
        self.label_single_text.grid(row=0, column=0, sticky="nw")
        self.single_text = tk.Text(frame, height=4, width=50)
        self.single_text.grid(row=0, column=1, columnspan=3, sticky="nwe", padx=4, pady=2)

        # 情绪档位
        self.label_single_emo = ttk.Label(frame, text=i18n.t("emotion_level"))
        self.label_single_emo.grid(row=1, column=0, sticky="w", padx=2, pady=4)
        self.single_emo_level = tk.StringVar(value="mid")
        emo_combo = ttk.Combobox(
            frame,
            textvariable=self.single_emo_level,
            values=EMO_LEVELS,
            width=8,
            state="readonly",
        )
        emo_combo.grid(row=1, column=1, sticky="w", padx=4, pady=4)
        ttk.Button(frame, text="?", width=2, command=self._show_emo_level_help).grid(row=1, column=2, padx=2, pady=4, sticky="w")

        # 情绪向量 slider 区域（单列整齐布局）
        self.vec_frame = ttk.LabelFrame(frame, text=i18n.t("emotion_vector_title"))
        self.vec_frame.grid(row=2, column=0, columnspan=4, sticky="nwe", padx=2, pady=4)
        self.vec_frame.columnconfigure(1, weight=1)

        self.emo_scales = {}
        self.emo_labels = {}  # 保存情绪标签引用以便更新

        for row_index, key in enumerate(EMO_VECTOR_KEYS):
            # 使用翻译获取情绪标签
            nice_label = i18n.t(key)

            label = ttk.Label(self.vec_frame, text=nice_label, width=18)
            label.grid(row=row_index, column=0, sticky="w", padx=4, pady=2)
            self.emo_labels[key] = label  # 保存引用

            # 使用 tk.Scale 而不是 ttk.Scale，因为 ttk.Scale 有精度问题
            scale = tk.Scale(
                self.vec_frame,
                from_=0.0,
                to=1.0,
                resolution=0.01,  # 设置为0.01的精度，可以平滑滑动
                orient="horizontal",
                command=lambda val, k=key: self._on_emo_scale_change(k, val),
                showvalue=True,  # 显示当前值
                length=300,  # 设置长度
                bg="#f0f0f0",  # 背景色
                troughcolor="#e0e0e0",  # 滑槽颜色
                highlightthickness=0,  # 去除高亮边框
            )
            scale.set(0.0)
            scale.grid(row=row_index, column=1, sticky="we", padx=4, pady=2)
            self.emo_scales[key] = scale

            help_btn = ttk.Button(
                self.vec_frame,
                text="?",
                width=2,
                command=lambda k=key: self._show_emo_help(k),
            )
            help_btn.grid(row=row_index, column=2, padx=2, pady=2, sticky="e")

        # 单句按钮
        btn_frame = ttk.Frame(frame)
        btn_frame.grid(row=3, column=0, columnspan=4, sticky="we", pady=8)

        self.btn_generate_single = ttk.Button(btn_frame, text=i18n.t("generate_single"), command=self._run_single)
        self.btn_generate_single.pack(side="left", padx=4)
        self.btn_apply_vector = ttk.Button(
            btn_frame,
            text=i18n.t("apply_vector"),
            command=self._apply_vector_to_batch,
        )
        self.btn_apply_vector.pack(side="left", padx=4)
        self.btn_clear_vector = ttk.Button(
            btn_frame,
            text=i18n.t("clear_vector"),
            command=self._clear_vector_from_batch,
        )
        self.btn_clear_vector.pack(side="left", padx=4)

        frame.columnconfigure(1, weight=1)

    # ---------- 任务管理 ----------

    def _add_task(self):
        idx = len(self.tasks) + 1
        frame = ttk.Frame(self.task_nb)
        task_title = f"{i18n.t('task')} {idx}"
        self.task_nb.add(frame, text=task_title)
        self.task_nb.select(frame)

        # 文本文件
        label_txt = ttk.Label(frame, text=i18n.t("script_txt"))
        label_txt.grid(row=0, column=0, sticky="w", padx=4, pady=4)
        entry_txt = ttk.Entry(frame, width=40)
        entry_txt.grid(row=0, column=1, sticky="we", padx=4, pady=4)
        btn_txt = ttk.Button(frame, text=i18n.t("browse"), command=lambda e=entry_txt: self._choose_file(e))
        btn_txt.grid(row=0, column=2, padx=4, pady=4)

        # 输出目录
        label_out = ttk.Label(frame, text=i18n.t("output_dir"))
        label_out.grid(row=1, column=0, sticky="w", padx=4, pady=4)
        entry_out = ttk.Entry(frame, width=40)
        entry_out.grid(row=1, column=1, sticky="we", padx=4, pady=4)
        btn_out = ttk.Button(frame, text=i18n.t("select"), command=lambda e=entry_out: self._choose_dir(e))
        btn_out.grid(row=1, column=2, padx=4, pady=4)

        # 情绪档位
        label_emo = ttk.Label(frame, text=i18n.t("emotion_level"))
        label_emo.grid(row=2, column=0, sticky="w", padx=4, pady=4)
        emo_var = tk.StringVar(value="mid")
        emo_combo = ttk.Combobox(
            frame,
            textvariable=emo_var,
            values=EMO_LEVELS,
            state="readonly",
            width=8,
        )
        emo_combo.grid(row=2, column=1, sticky="w", padx=4, pady=4)
        ttk.Button(frame, text="?", width=2, command=self._show_emo_level_help).grid(row=2, column=2, padx=2, pady=4)

        frame.columnconfigure(1, weight=1)

        task = {
            "frame": frame,
            "entry_txt": entry_txt,
            "entry_out": entry_out,
            "emo_var": emo_var,
            "label_txt": label_txt,
            "label_out": label_out,
            "label_emo": label_emo,
            "btn_txt": btn_txt,
            "btn_out": btn_out,
        }
        self.tasks.append(task)

    def _remove_current_task(self):
        if not self.tasks:
            return
        cur = self.task_nb.select()
        if not cur:
            return
        index = None
        for i, t in enumerate(self.tasks):
            if str(t["frame"]) == cur:
                index = i
                break
        if index is None:
            return
        self.task_nb.forget(self.tasks[index]["frame"])
        del self.tasks[index]

    # ---------- 事件处理 ----------

    def _choose_voice(self):
        path = filedialog.askopenfilename(
            title="选择声纹参考音频",
            filetypes=[("Audio files", "*.wav *.mp3 *.flac *.m4a"), ("All files", "*.*")],
        )
        if path:
            self.entry_voice.delete(0, tk.END)
            self.entry_voice.insert(0, path)

    def _choose_emo_ref(self):
        path = filedialog.askopenfilename(
            title="选择情绪参考音频（可选）",
            filetypes=[("Audio files", "*.wav *.mp3 *.flac *.m4a"), ("All files", "*.*")],
        )
        if path:
            self.entry_emo_ref.delete(0, tk.END)
            self.entry_emo_ref.insert(0, path)

    def _choose_file(self, entry):
        path = filedialog.askopenfilename(
            title="选择台词 TXT 文件",
            filetypes=[("Text files", "*.txt"), ("All files", "*.*")],
        )
        if path:
            entry.delete(0, tk.END)
            entry.insert(0, path)

    def _choose_dir(self, entry):
        path = filedialog.askdirectory(title="选择输出目录")
        if path:
            entry.delete(0, tk.END)
            entry.insert(0, path)

    def _show_preserve_help(self):
        messagebox.showinfo(i18n.t("preserve_help_title"), i18n.t("preserve_help_text"))

    def _show_gpu_help(self):
        gpu_info = i18n.t("gpu_help_detected") + "\n"
        if self.available_gpus:
            for gpu_id, gpu_name in self.available_gpus:
                gpu_info += f"  {gpu_id}: {gpu_name}\n"
        else:
            gpu_info += "  " + i18n.t("gpu_help_no_gpu") + "\n"
        
        multi_gpu_info = ""
        if len(self.available_gpus) >= 2:
            speedup = len(self.available_gpus) * 0.8
            multi_gpu_info = f"\n{i18n.t('gpu_help_multi_gpu')}\n"
            multi_gpu_info += f"  {i18n.t('gpu_help_multi_desc')}\n"
            multi_gpu_info += f"  {len(self.available_gpus)} GPUs\n"
            multi_gpu_info += f"  {i18n.t('gpu_help_multi_speedup')} {speedup:.1f}x\n"
            multi_gpu_info += f"  {i18n.t('gpu_help_multi_note')}\n"
        
        msg = f"{gpu_info}\n{i18n.t('gpu_help_modes')}"
        msg += multi_gpu_info
        messagebox.showinfo(i18n.t("gpu_help_title"), msg)

    def _on_gpu_mode_change(self):
        """显卡模式改变时的回调"""
        mode = self.gpu_mode.get()
        if mode == "single":
            self.gpu_combo.config(state="readonly")
        else:
            self.gpu_combo.config(state="disabled")
        
        # CPU模式时记录日志
        if mode == "cpu":
            self._log(i18n.t("cpu_mode_switched"))

    def _show_emo_level_help(self):
        messagebox.showinfo(i18n.t("emo_level_help_title"), i18n.t("emo_level_help_text"))

    def _show_emo_help(self, key):
        help_key = f"emo_help_{key}"
        desc = i18n.t(help_key)
        messagebox.showinfo(i18n.t("emo_help_title"), desc)

    def _on_emo_scale_change(self, key, val):
        try:
            v = float(val)
        except ValueError:
            v = 0.0
        idx = EMO_VECTOR_KEYS.index(key)
        self.current_emo_vector[idx] = max(0.0, min(1.0, v))
        # 如果已启用批量使用向量，实时更新指示器
        if self.use_vector_for_batch.get():
            self._update_vector_status_indicator()

    def _toggle_console(self):
        if self.console_visible.get():
            self.console.pack(fill="both", expand=True, pady=(4, 0))
        else:
            self.console.pack_forget()

    # ---------- 日志 ----------

    def _log(self, text: str):
        # 解析进度信息并更新进度条
        self._parse_progress_info(text)
        
        # 翻译子进程输出中的常见中文信息
        translations_map = {
            ">> 全部任务完成！": i18n.t("log_all_tasks_done"),
            ">> 使用 GUI 情绪向量": ">> Using GUI emotion vector" if i18n.current_lang == "en" else 
                                    ">> GUIの感情ベクトルを使用" if i18n.current_lang == "ja" else
                                    ">> Usando vector emocional de GUI" if i18n.current_lang == "es" else 
                                    ">> 使用 GUI 情绪向量",
            ">> 使用指定的GPU设备": ">> Using specified GPU device" if i18n.current_lang == "en" else
                                    ">> 指定されたGPUデバイスを使用" if i18n.current_lang == "ja" else
                                    ">> Usando dispositivo GPU especificado" if i18n.current_lang == "es" else
                                    ">> 使用指定的GPU设备",
            ">> 情绪档位": ">> Emotion level" if i18n.current_lang == "en" else
                          ">> 感情レベル" if i18n.current_lang == "ja" else
                          ">> Nivel emocional" if i18n.current_lang == "es" else
                          ">> 情绪档位",
            "基础强度": "base intensity" if i18n.current_lang == "en" else
                       "基本強度" if i18n.current_lang == "ja" else
                       "intensidad base" if i18n.current_lang == "es" else
                       "基础强度",
            "最终情绪强度": "final emotion intensity" if i18n.current_lang == "en" else
                          "最終感情強度" if i18n.current_lang == "ja" else
                          "intensidad emocional final" if i18n.current_lang == "es" else
                          "最终情绪强度",
            ">> 生成中": ">> Generating" if i18n.current_lang == "en" else
                        ">> 生成中" if i18n.current_lang == "ja" else
                        ">> Generando" if i18n.current_lang == "es" else
                        ">> 生成中",
            "台词": "Text" if i18n.current_lang == "en" else
                   "台詞" if i18n.current_lang == "ja" else
                   "Texto" if i18n.current_lang == "es" else
                   "台词",
            "情绪向量": "Emotion vector" if i18n.current_lang == "en" else
                       "感情ベクトル" if i18n.current_lang == "ja" else
                       "Vector emocional" if i18n.current_lang == "es" else
                       "情绪向量",
        }
        
        # 应用翻译
        for cn_text, translated in translations_map.items():
            if cn_text in text:
                text = text.replace(cn_text, translated)
        
        # 过滤无法正确显示的Unicode字符（进度条等特殊字符）
        # 保留常用字符：中英文、日文、西班牙文、数字、标点、空格等
        filtered_text = ""
        for char in text:
            code = ord(char)
            # 允许的字符范围：
            # - ASCII可打印字符 (32-126)
            # - 拉丁扩展 (0x00A0-0x00FF) - 西班牙语等
            # - 中文 (0x4E00-0x9FFF)
            # - 日文平假名片假名 (0x3040-0x30FF)
            # - 中日韩符号和标点 (0x3000-0x303F)
            # - 换行符等
            if (32 <= code <= 126) or \
               (0x00A0 <= code <= 0x00FF) or \
               (0x3000 <= code <= 0x30FF) or \
               (0x4E00 <= code <= 0x9FFF) or \
               char in ['\n', '\r', '\t']:
                filtered_text += char
            elif code > 127:  # 其他特殊Unicode字符（如进度条字符）过滤掉
                pass
        
        text = filtered_text.strip()
        
        # 添加时间戳
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        log_line = f"[{timestamp}] {text}"
        
        # 保存到日志缓冲区
        self.log_buffer.append(log_line)
        if len(self.log_buffer) > self.max_log_lines:
            self.log_buffer.pop(0)  # 移除最早的日志
        
        # 显示到控制台
        self.console.configure(state="normal")
        self.console.insert(tk.END, text + "\n")
        self.console.see(tk.END)
        self.console.configure(state="disabled")
        self.root.update_idletasks()

    def _update_progress_bars(self, current_pct=None, total_current=None, total_count=None):
        """更新进度条显示
        Args:
            current_pct: 当前音频生成进度百分比 (0-100)
            total_current: 当前完成的音频数量
            total_count: 总共需要生成的音频数量
        """
        if current_pct is not None:
            self.current_progress = current_pct
            self.current_progress_bar['value'] = current_pct
            self.current_progress_text.config(text=f"{int(current_pct)}%")
        
        if total_current is not None:
            self.current_audio_index = total_current
        
        if total_count is not None:
            self.total_audio_count = total_count
        
        # 更新总体进度
        if self.total_audio_count > 0:
            total_pct = (self.current_audio_index / self.total_audio_count) * 100
            self.total_progress = total_pct
            self.total_progress_bar['value'] = total_pct
            self.total_progress_text.config(text=f"{self.current_audio_index}/{self.total_audio_count}")
        
        self.root.update_idletasks()
    
    def _reset_progress_bars(self):
        """重置进度条"""
        self.current_progress = 0
        self.total_progress = 0
        self.current_audio_index = 0
        self.total_audio_count = 0
        self.multi_gpu_mode = False
        self.gpu_progress = {}
        self.current_progress_bar['value'] = 0
        self.total_progress_bar['value'] = 0
        self.current_progress_text.config(text="0%")
        self.total_progress_text.config(text="0/0")
        self.root.update_idletasks()
    
    def _parse_progress_info(self, text: str):
        """从日志文本中解析进度信息并更新进度条"""
        import re
        
        # 检查是否是多GPU日志（带有[cuda:X]前缀）
        gpu_prefix_match = re.match(r'\[(cuda:\d+)\]\s+(.+)', text)
        if gpu_prefix_match:
            gpu_id = gpu_prefix_match.group(1)
            content = gpu_prefix_match.group(2)
            
            # 解析多GPU模式下的进度: [1/82] 生成中
            match = re.search(r'\[(\d+)/(\d+)\]', content)
            if match and self.multi_gpu_mode:
                current = int(match.group(1))
                # 更新该GPU的进度
                self.gpu_progress[gpu_id] = current
                
                # 计算所有GPU的总进度
                total_completed = sum(self.gpu_progress.values())
                self._update_progress_bars(total_current=total_completed, total_count=self.total_audio_count)
                
                # 当开始新的音频时，重置当前进度
                if "生成中" in content or "Generating" in content:
                    self._update_progress_bars(current_pct=0)
                return
        
        # 单GPU模式：解析总体进度 >> [1/10] 生成中: 或 >> [5/10]
        match = re.search(r'\[(\d+)/(\d+)\]', text)
        if match and not self.multi_gpu_mode:
            current = int(match.group(1))
            total = int(match.group(2))
            self._update_progress_bars(total_current=current, total_count=total)
            # 当开始新的音频时，重置当前进度
            if "生成中" in text or "Generating" in text:
                self._update_progress_bars(current_pct=0)
        
        # 解析当前音频进度: 查找百分比
        # 匹配 50% 或 50.0% 格式
        match = re.search(r'(\d+(?:\.\d+)?)\s*%', text)
        if match:
            pct = float(match.group(1))
            # 只更新当前音频进度，保留总体进度
            self._update_progress_bars(current_pct=pct)
        
        # 任务完成时的进度
        if "全部任务完成" in text or "All tasks completed" in text or "任务成功完成" in text or "所有GPU任务完成" in text:
            # 设置为100%
            if self.total_audio_count > 0:
                self._update_progress_bars(
                    current_pct=100,
                    total_current=self.total_audio_count,
                    total_count=self.total_audio_count
                )

    def _update_status(self, status_text, color="gray"):
        """更新状态标签"""
        # 状态文本映射
        status_map = {
            "空闲": "status_idle",
            "运行中": "status_running",
            "正在停止...": "status_stopping",
            "完成": "status_completed",
            "已停止": "status_stopped",
            "失败": "status_failed"
        }
        
        # 尝试翻译状态文本
        status_key = status_map.get(status_text, None)
        if status_key:
            translated_status = i18n.t(status_key)
        else:
            translated_status = status_text
        
        full_text = i18n.t("status") + translated_status
        self.status_label.config(text=full_text, foreground=color)

    def _stop_current_task(self):
        """停止当前正在运行的任务"""
        with self.process_lock:
            # 检查单进程模式
            if self.current_process is None and not self.current_processes:
                self._log(">> 当前没有运行中的任务。")
                return
            
            if self.is_stopping:
                self._log(">> 已经在停止中，请稍候...")
                return
            
            self.is_stopping = True
            proc = self.current_process
            procs = list(self.current_processes)  # 复制列表
        
        self._log(">> 正在停止任务...")
        self._update_status("正在停止...", "orange")
        
        def terminate_processes():
            try:
                # 单进程模式
                if proc:
                    proc.terminate()
                    try:
                        proc.wait(timeout=3)
                        self._log(">> 任务已成功停止。")
                    except subprocess.TimeoutExpired:
                        self._log(">> 进程未响应，强制终止...")
                        proc.kill()
                        proc.wait()
                        self._log(">> 任务已强制终止。")
                
                # 多进程模式
                if procs:
                    self._log(f">> 正在停止 {len(procs)} 个并行任务...")
                    for i, p in enumerate(procs):
                        try:
                            p.terminate()
                        except Exception as e:
                            self._log(f">> GPU{i} 进程终止失败: {e}")
                    
                    # 等待所有进程
                    for i, p in enumerate(procs):
                        try:
                            p.wait(timeout=3)
                        except subprocess.TimeoutExpired:
                            p.kill()
                            p.wait()
                    
                    self._log(f">> 所有 {len(procs)} 个任务已停止。")
                
            except Exception as e:
                self._log(f">> 停止任务时出错: {e}")
            finally:
                with self.process_lock:
                    self.current_process = None
                    self.current_processes.clear()
                    self.is_stopping = False
                self._update_status("空闲", "gray")
                self.stop_btn.config(state="disabled")
        
        # 在新线程中执行停止操作，避免阻塞 GUI
        t = threading.Thread(target=terminate_processes, daemon=True)
        t.start()

    # ---------- 运行子进程 ----------

    def _run_subprocess(self, args, cwd=None):
        with self.process_lock:
            if self.current_process is not None:
                self._log(i18n.t("log_task_running"))
                return

            self._log(f">> 启动进程: {' '.join(args)}")
            self._update_status("运行中", "green")
            
            # 重置进度条
            self._reset_progress_bars()

            creationflags = 0
            if os.name == "nt":
                creationflags = subprocess.CREATE_NO_WINDOW  # 隐藏控制台窗口

            try:
                proc = subprocess.Popen(
                    args,
                    cwd=cwd,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.STDOUT,
                    text=True,
                    encoding="utf-8",
                    errors="replace",
                    creationflags=creationflags,
                )
            except Exception as e:
                self._log(f"启动进程失败: {e}")
                self._update_status("失败", "red")
                return

            self.current_process = proc
            # 启用停止按钮
            self.stop_btn.config(state="normal")

        def reader():
            try:
                assert proc.stdout is not None
                for line in proc.stdout:
                    line = line.rstrip("\n")
                    self._log(line)
                    # 检查是否被停止
                    if proc.poll() is not None:
                        break
            except Exception as e:
                self._log(i18n.t("log_read_error").format(error=e))
            finally:
                with self.process_lock:
                    if not self.is_stopping:
                        self.current_process = None
                
                # 检查进程是否正常结束
                if proc.returncode is not None:
                    # 如果是用户主动停止，不管退出码是什么都显示"已停止"
                    if self.is_stopping:
                        self._log(i18n.t("log_task_terminated").format(signal=proc.returncode))
                        self._update_status("已停止", "orange")
                    elif proc.returncode == 0:
                        self._log(i18n.t("log_task_completed"))
                        self._update_status("完成", "blue")
                    elif proc.returncode < 0:
                        self._log(i18n.t("log_task_terminated").format(signal=-proc.returncode))
                        self._update_status("已停止", "orange")
                    else:
                        self._log(i18n.t("log_task_failed").format(code=proc.returncode))
                        self._update_status("失败", "red")
                else:
                    self._log(i18n.t("log_task_ended"))
                    self._update_status("空闲", "gray")
                
                # 禁用停止按钮（如果不是在停止过程中）
                if not self.is_stopping:
                    self.stop_btn.config(state="disabled")

        t = threading.Thread(target=reader, daemon=True)
        t.start()

    # ---------- 批量模式运行（当前任务） ----------

    def _run_current_task(self):
        """执行当前选中的任务"""
        voice = self.entry_voice.get().strip()
        if not voice:
            messagebox.showwarning("缺少声纹", "请先选择声纹参考音频。")
            return

        if not self.tasks:
            messagebox.showwarning("没有任务", "请先新增至少一个任务。")
            return

        cur_id = self.task_nb.select()
        current_task = None
        for t in self.tasks:
            if str(t["frame"]) == cur_id:
                current_task = t
                break

        if current_task is None:
            messagebox.showwarning("未选择任务", "请先选择一个任务标签页。")
            return

        txt = current_task["entry_txt"].get().strip()
        outdir = current_task["entry_out"].get().strip()
        emo = current_task["emo_var"].get().strip() or "mid"

        if not txt or not outdir:
            messagebox.showwarning("任务不完整", "请为当前任务选择台词 TXT 和输出目录。")
            return

        gpu_mode = self.gpu_mode.get()
        
        # 多显卡并行模式
        if gpu_mode == "multi" and len(self.available_gpus) >= 2:
            self._run_multi_gpu_tasks(voice, txt, outdir, emo)
            return

        # 单显卡或自动模式
        args = [self.python_exe, "batch_generate.py",
                "--voice", voice,
                "--text", txt,
                "--outdir", outdir,
                "--emo", emo,
                "--preserve", "true" if self.preserve_voice.get() else "false"]

        emo_ref = self.entry_emo_ref.get().strip()
        if emo_ref:
            args += ["--emo-ref", emo_ref]

        if self.use_vector_for_batch.get():
            vec_str = ",".join(f"{v:.3f}" for v in self.current_emo_vector)
            args += ["--use-vector", "true", "--emo-vec", vec_str]
        else:
            args += ["--use-vector", "false"]

        # 添加显卡设置
        if gpu_mode == "cpu":
            # CPU模式
            args += ["--device", "cpu"]
        elif gpu_mode == "single" and self.available_gpus:
            # 从下拉框中提取实际的device（例如从 "cuda:0 - NVIDIA ..." 中提取 "cuda:0"）
            selected = self.gpu_combo.get()
            device = selected.split(" - ")[0] if " - " in selected else "cuda:0"
            args += ["--device", device]
        # auto模式不传device参数，让程序自动选择

        self._run_subprocess(args)

    def _run_all_tasks(self):
        """按顺序执行所有任务"""
        voice = self.entry_voice.get().strip()
        if not voice:
            messagebox.showwarning("缺少声纹", "请先选择声纹参考音频。")
            return

        if not self.tasks:
            messagebox.showwarning("没有任务", "请先新增至少一个任务。")
            return

        # 检查所有任务是否都配置完整
        incomplete_tasks = []
        for idx, task in enumerate(self.tasks):
            txt = task["entry_txt"].get().strip()
            outdir = task["entry_out"].get().strip()
            if not txt or not outdir:
                incomplete_tasks.append(idx + 1)
        
        if incomplete_tasks:
            msg = f"任务 {', '.join(map(str, incomplete_tasks))} 配置不完整，请为这些任务选择台词 TXT 和输出目录。"
            messagebox.showwarning("任务不完整", msg)
            return

        # 确认执行
        msg = f"即将依次执行 {len(self.tasks)} 个任务，是否继续？"
        if not messagebox.askyesno("确认执行", msg):
            return

        # 开始执行所有任务
        self._log(f">> 开始执行所有任务，共 {len(self.tasks)} 个")
        # 重置任务序列标志
        self._task_sequence_started = False
        self._execute_tasks_sequentially(0)

    def _execute_tasks_sequentially(self, task_index):
        """递归执行任务序列"""
        try:
            if task_index >= len(self.tasks):
                # 所有任务完成
                self._log(f">> 🎉 所有 {len(self.tasks)} 个任务已完成！")
                self._update_status("status_completed")
                messagebox.showinfo("完成", f"所有 {len(self.tasks)} 个任务已成功完成！")
                return

            # 获取当前任务
            current_task = self.tasks[task_index]
            txt = current_task["entry_txt"].get().strip()
            outdir = current_task["entry_out"].get().strip()
            emo = current_task["emo_var"].get().strip() or "mid"
            
            self._log(f">> ▶ 开始执行任务 {task_index + 1}/{len(self.tasks)}")
            self._log(f"   - 台词文件: {txt}")
            self._log(f"   - 输出目录: {outdir}")
            self._log(f"   - 情绪档位: {emo}")

            # 准备参数
            voice = self.entry_voice.get().strip()
            emo_ref = self.entry_emo_ref.get().strip()
            gpu_mode = self.gpu_mode.get()

            # 多显卡并行模式
            if gpu_mode == "multi" and len(self.available_gpus) >= 2:
                self._run_multi_gpu_tasks_for_sequence(voice, txt, outdir, emo, task_index)
                return

            # 单显卡或自动模式
            args = [self.python_exe, "batch_generate.py",
                    "--voice", voice,
                    "--text", txt,
                    "--outdir", outdir,
                    "--emo", emo,
                    "--preserve", "true" if self.preserve_voice.get() else "false"]

            if emo_ref:
                args += ["--emo-ref", emo_ref]

            if self.use_vector_for_batch.get():
                vec_str = ",".join(f"{v:.3f}" for v in self.current_emo_vector)
                args += ["--use-vector", "true", "--emo-vec", vec_str]
            else:
                args += ["--use-vector", "false"]

            # 添加显卡设置
            if gpu_mode == "cpu":
                args += ["--device", "cpu"]
            elif gpu_mode == "single" and self.available_gpus:
                selected = self.gpu_combo.get()
                device = selected.split(" - ")[0] if " - " in selected else "cuda:0"
                args += ["--device", device]

            # 执行子进程，完成后调用回调
            self._run_subprocess_with_callback(args, lambda: self._execute_tasks_sequentially(task_index + 1))
            
        except Exception as e:
            self._log(f">> 错误：执行任务 {task_index + 1} 时出错: {e}")
            import traceback
            self._log(f">> 详细错误: {traceback.format_exc()}")
            self.is_running = False
            self._update_status("status_failed")
            messagebox.showerror("任务执行失败", f"执行任务 {task_index + 1} 时出错：\n{e}")

    def _run_subprocess_with_callback(self, args, callback):
        """运行子进程并在完成后调用回调函数"""
        if self.is_running:
            messagebox.showwarning("正在运行", "当前有任务正在运行，请等待完成。")
            return
        
        # 重置进度条（仅在第一个任务开始时）
        if not hasattr(self, '_task_sequence_started'):
            self._task_sequence_started = True
            self._reset_progress_bars()

        self.is_running = True
        self.is_stopping = False
        self._update_status("status_running")

        try:
            process = subprocess.Popen(
                args,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                text=True,
                encoding="utf-8",
                errors="replace",
                creationflags=subprocess.CREATE_NO_WINDOW if os.name == 'nt' else 0
            )
            self.current_process = process
            self.current_processes = [process]

            def read_output():
                try:
                    for line in iter(process.stdout.readline, ""):
                        if line:
                            self._log(line.rstrip())
                    process.wait()
                    
                    # 任务完成
                    self.is_running = False
                    self.current_process = None
                    self.current_processes = []
                    
                    if self.is_stopping:
                        self._log(">> 任务被用户停止")
                        self._update_status("status_stopped")
                        self.is_stopping = False
                    else:
                        self._log(">> ✓ 任务完成")
                        # 调用回调函数执行下一个任务
                        if callback:
                            self.root.after(100, callback)
                            
                except Exception as e:
                    self._log(f">> 读取输出时出错: {e}")
                    self.is_running = False
                    self.current_process = None
                    self.current_processes = []

            thread = threading.Thread(target=read_output, daemon=True)
            thread.start()

        except Exception as e:
            self.is_running = False
            self.current_processes = []
            self._update_status("status_failed")
            self._log(f">> 启动失败: {e}")
            messagebox.showerror("错误", f"启动任务失败：{e}")

    def _run_multi_gpu_tasks_for_sequence(self, voice, txt_path, outdir, emo, task_index):
        """为序列任务执行多GPU并行（完成后继续下一个任务）"""
        # 读取所有文本行
        try:
            with open(txt_path, "r", encoding="utf-8") as f:
                lines = [line.strip() for line in f if line.strip()]
        except Exception as e:
            messagebox.showerror("读取文件失败", f"无法读取文本文件：{e}")
            self.root.after(100, lambda: self._execute_tasks_sequentially(task_index + 1))  # 继续下一个任务
            return
        
        if not lines:
            messagebox.showwarning("文件为空", "文本文件中没有内容。")
            self.root.after(100, lambda: self._execute_tasks_sequentially(task_index + 1))  # 继续下一个任务
            return
        
        num_gpus = len(self.available_gpus)
        self._log(f">> 多GPU并行模式：使用 {num_gpus} 张显卡")
        self._log(f">> 总共 {len(lines)} 行文本，将分配到各显卡")
        
        # 将文本行分配到各个GPU
        lines_per_gpu = [[] for _ in range(num_gpus)]
        for i, line in enumerate(lines):
            gpu_idx = i % num_gpus
            lines_per_gpu[gpu_idx].append(line)
        
        # 创建临时文本文件（记录GPU索引）
        temp_files = []  # 格式：[(gpu_idx, temp_file_path), ...]
        for i in range(num_gpus):
            if not lines_per_gpu[i]:
                continue
            temp_file = f"temp_gpu{i}_{os.getpid()}.txt"
            with open(temp_file, "w", encoding="utf-8") as f:
                f.write("\n".join(lines_per_gpu[i]))
            temp_files.append((i, temp_file))  # 保存GPU索引和文件路径的元组
        
        # 启动多个进程
        self.is_running = True
        self.is_stopping = False
        self._update_status("status_running")
        self.current_processes = []
        
        # 设置多GPU模式和进度跟踪
        self.multi_gpu_mode = True
        self.total_audio_count = len(lines)
        self.gpu_progress = {}  # 重置GPU进度跟踪
        for gpu_idx, _ in temp_files:
            gpu_id = self.available_gpus[gpu_idx][0]
            self.gpu_progress[gpu_id] = 0
        
        # 添加一个标志来确保回调只被触发一次
        callback_triggered = [False]  # 使用列表以便在闭包中修改
        
        emo_ref = self.entry_emo_ref.get().strip()
        
        for gpu_idx, temp_file in temp_files:
            device = self.available_gpus[gpu_idx][0]  # 提取设备ID（如"cuda:0"）
            args = [self.python_exe, "batch_generate.py",
                    "--voice", voice,
                    "--text", temp_file,
                    "--outdir", outdir,
                    "--emo", emo,
                    "--preserve", "true" if self.preserve_voice.get() else "false",
                    "--device", device]
            
            if emo_ref:
                args += ["--emo-ref", emo_ref]
            
            if self.use_vector_for_batch.get():
                vec_str = ",".join(f"{v:.3f}" for v in self.current_emo_vector)
                args += ["--use-vector", "true", "--emo-vec", vec_str]
            else:
                args += ["--use-vector", "false"]
            
            try:
                # 打印完整命令用于调试
                cmd_str = ' '.join(args)
                self._log(f">> [DEBUG] 启动命令: {cmd_str}")
                
                process = subprocess.Popen(
                    args,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.STDOUT,
                    text=True,
                    encoding="utf-8",
                    errors="replace",
                    creationflags=subprocess.CREATE_NO_WINDOW if os.name == 'nt' else 0
                )
                self.current_processes.append(process)
                self._log(f">> 启动 GPU {gpu_idx} ({device}) 进程 (PID: {process.pid})，处理 {len(lines_per_gpu[gpu_idx])} 行")
                
                def read_process_output_for_sequence(proc, gpu_id, temp_file):
                    try:
                        for line in iter(proc.stdout.readline, ""):
                            if line:
                                self._log(f"[{gpu_id}] {line.rstrip()}")
                        proc.wait()
                        self._log(f"[{gpu_id}] >> GPU任务完成")
                    except Exception as e:
                        self._log(f"[{gpu_id}] >> 读取输出时出错: {e}")
                    finally:
                        try:
                            if os.path.exists(temp_file):
                                os.remove(temp_file)
                        except:
                            pass
                        
                        # 检查是否所有进程都完成（只触发一次回调）
                        all_done = all(p.poll() is not None for p in self.current_processes)
                        if all_done and not callback_triggered[0]:
                            callback_triggered[0] = True
                            self.is_running = False
                            self.current_processes = []
                            # 重置多GPU模式标志
                            self.multi_gpu_mode = False
                            self._log(">> 所有GPU任务完成！")
                            # 调用回调执行下一个任务
                            self.root.after(100, lambda: self._execute_tasks_sequentially(task_index + 1))
                
                t = threading.Thread(
                    target=read_process_output_for_sequence,
                    args=(process, device, temp_file),
                    daemon=True
                )
                t.start()
                
            except Exception as e:
                self._log(f">> 启动 GPU {gpu_idx} 进程失败: {e}")
                if os.path.exists(temp_file):
                    os.remove(temp_file)

    def _run_multi_gpu_tasks(self, voice, txt_path, outdir, emo):
        """多显卡并行运行批量任务"""
        # 读取所有文本行
        try:
            with open(txt_path, "r", encoding="utf-8") as f:
                lines = [line.strip() for line in f if line.strip()]
        except Exception as e:
            messagebox.showerror("读取文件失败", f"无法读取文本文件：{e}")
            return
        
        if not lines:
            messagebox.showwarning("文件为空", "文本文件中没有内容。")
            return
        
        num_gpus = len(self.available_gpus)
        self._log(f">> 多GPU并行模式：使用 {num_gpus} 张显卡")
        self._log(f">> 总共 {len(lines)} 行文本，将分配到各显卡")
        
        # 将文本行分配到各个GPU
        lines_per_gpu = [[] for _ in range(num_gpus)]
        for i, line in enumerate(lines):
            gpu_idx = i % num_gpus
            lines_per_gpu[gpu_idx].append(line)
        
        # 创建临时文本文件（记录GPU索引）
        temp_files = []  # 格式：[(gpu_idx, temp_file_path), ...]
        for i in range(num_gpus):
            if not lines_per_gpu[i]:
                continue
            temp_file = f"temp_gpu{i}_{os.getpid()}.txt"
            with open(temp_file, "w", encoding="utf-8") as f:
                f.write("\n".join(lines_per_gpu[i]))
            temp_files.append((i, temp_file))  # 保存元组：(GPU索引, 文件路径)
            self._log(f"   GPU{i} ({self.available_gpus[i][0]}): {len(lines_per_gpu[i])} 行")
        
        # 检查是否有任务在运行
        with self.process_lock:
            if self.current_process is not None or self.current_processes:
                self._log(i18n.t("log_task_running"))
                # 清理临时文件
                for _, temp_file in temp_files:
                    try:
                        os.remove(temp_file)
                    except:
                        pass
                return
        
        self._log(">> 启动多GPU并行任务...")
        self._update_status(f"运行中 ({num_gpus}GPU)", "green")
        
        # 设置多GPU模式和进度跟踪
        self.multi_gpu_mode = True
        self.total_audio_count = len(lines)
        self.gpu_progress = {}  # 重置GPU进度跟踪
        for gpu_idx, _ in temp_files:
            gpu_id = self.available_gpus[gpu_idx][0]
            self.gpu_progress[gpu_id] = 0
        
        # 为每个GPU创建进程
        processes = []
        creationflags = subprocess.CREATE_NO_WINDOW if os.name == "nt" else 0
        
        for gpu_idx, temp_file in temp_files:
            gpu_id = self.available_gpus[gpu_idx][0]
            
            args = [self.python_exe, "batch_generate.py",
                    "--voice", voice,
                    "--text", temp_file,
                    "--outdir", outdir,
                    "--emo", emo,
                    "--device", gpu_id,
                    "--preserve", "true" if self.preserve_voice.get() else "false"]
            
            emo_ref = self.entry_emo_ref.get().strip()
            if emo_ref:
                args += ["--emo-ref", emo_ref]
            
            if self.use_vector_for_batch.get():
                vec_str = ",".join(f"{v:.3f}" for v in self.current_emo_vector)
                args += ["--use-vector", "true", "--emo-vec", vec_str]
            else:
                args += ["--use-vector", "false"]
            
            try:
                # 打印完整命令用于调试
                cmd_str = ' '.join(args)
                self._log(f"   [DEBUG] 启动命令: {cmd_str}")
                
                proc = subprocess.Popen(
                    args,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.STDOUT,
                    text=True,
                    encoding="utf-8",
                    errors="replace",
                    creationflags=creationflags,
                )
                processes.append((gpu_idx, gpu_id, proc, temp_file))
                self._log(f"   启动 {gpu_id} 进程（PID: {proc.pid}），处理 {len(lines_per_gpu[gpu_idx])} 行")
            except Exception as e:
                self._log(f"   启动 {gpu_id} 进程失败: {e}")
        
        if not processes:
            self._log(">> 没有成功启动任何进程")
            self._update_status("失败", "red")
            return
        
        # 保存进程列表
        with self.process_lock:
            self.current_processes = [p[2] for p in processes]
            self.stop_btn.config(state="normal")
        
        # 为每个进程创建独立的输出读取线程
        def read_process_output(idx, gpu_id, proc, temp_file):
            try:
                for line in proc.stdout:
                    self._log(f"[{gpu_id}] {line.rstrip()}")
            except Exception as e:
                self._log(i18n.t("log_gpu_read_error").format(gpu=gpu_id, error=e))
            finally:
                proc.wait()
                self._log(i18n.t("log_gpu_completed").format(gpu=gpu_id, code=proc.returncode))
                # 删除临时文件
                try:
                    os.remove(temp_file)
                except:
                    pass
        
        # 启动所有输出读取线程
        threads = []
        for idx, gpu_id, proc, temp_file in processes:
            t = threading.Thread(
                target=read_process_output,
                args=(idx, gpu_id, proc, temp_file),
                daemon=True
            )
            t.start()
            threads.append(t)
        
        # 监控所有线程是否完成
        def wait_all_complete():
            for t in threads:
                t.join()
            
            # 所有进程完成
            with self.process_lock:
                self.current_processes.clear()
                if not self.is_stopping:
                    self.stop_btn.config(state="disabled")
            
            # 重置多GPU模式标志
            self.multi_gpu_mode = False
            
            self._log(">> 所有GPU任务完成！")
            self._update_status("完成", "blue")
        
        # 启动监控线程
        monitor_thread = threading.Thread(target=wait_all_complete, daemon=True)
        monitor_thread.start()

    # ---------- 单句模式运行 ----------

    def _run_single(self):
        voice = self.entry_voice.get().strip()
        if not voice:
            messagebox.showwarning("缺少声纹", "请先选择声纹参考音频。")
            return

        line = self.single_text.get("1.0", tk.END).strip()
        if not line:
            messagebox.showwarning("缺少文本", "请输入单句文本。")
            return

        # 生成建议的文件名
        safe = line.replace("\n", " ")
        base = safe[:20] if safe else "single"
        base = re.sub(r'[\\/:*?"<>|]', "", base)
        default_filename = f"{base}.wav"
        
        # 让用户选择保存位置
        out_path = filedialog.asksaveasfilename(
            title="保存单句音频 / Save Single Audio",
            defaultextension=".wav",
            initialfile=default_filename,
            initialdir=os.path.join(os.getcwd(), "single_outputs"),
            filetypes=[
                ("WAV Audio", "*.wav"),
                ("All files", "*.*")
            ]
        )
        
        if not out_path:
            # 用户取消了保存
            return
        
        # 确保输出目录存在
        outdir = os.path.dirname(out_path)
        if outdir:
            os.makedirs(outdir, exist_ok=True)

        emo = self.single_emo_level.get().strip() or "mid"

        args = [self.python_exe, "batch_generate.py",
                "--voice", voice,
                "--outdir", outdir,
                "--emo", emo,
                "--preserve", "true" if self.preserve_voice.get() else "false",
                "--single",
                "--line", line]

        emo_ref = self.entry_emo_ref.get().strip()
        if emo_ref:
            args += ["--emo-ref", emo_ref]

        # 单句模式总是使用当前向量
        vec_str = ",".join(f"{v:.3f}" for v in self.current_emo_vector)
        args += ["--use-vector", "true", "--emo-vec", vec_str]

        # 指定单句输出文件
        args += ["--single-out", out_path]

        # 添加显卡设置
        gpu_mode = self.gpu_mode.get()
        if gpu_mode == "cpu":
            # CPU模式
            args += ["--device", "cpu"]
        elif gpu_mode == "single" and self.available_gpus:
            selected = self.gpu_combo.get()
            device = selected.split(" - ")[0] if " - " in selected else "cuda:0"
            args += ["--device", device]

        self._run_subprocess(args)
        self._log(f"单句将输出到: {out_path}")

    def _apply_vector_to_batch(self):
        self.use_vector_for_batch.set(True)
        self._update_vector_status_indicator()
        self._log(i18n.t("vector_applied_log"))
        messagebox.showinfo(
            i18n.t("vector_applied_title"),
            i18n.t("vector_applied_message")
        )
    
    def _clear_vector_from_batch(self):
        """清除批量模式的情绪向量设置"""
        self.use_vector_for_batch.set(False)
        self._update_vector_status_indicator()
        self._log(i18n.t("vector_cleared_log"))
        messagebox.showinfo(
            i18n.t("vector_cleared_title"),
            i18n.t("vector_cleared_message")
        )
    
    def _update_vector_status_indicator(self):
        """更新批量模式的情绪向量状态指示器"""
        if not hasattr(self, 'vector_status_label'):
            return
        
        if self.use_vector_for_batch.get():
            # 显示当前向量的摘要
            active_emotions = []
            for i, key in enumerate(EMO_VECTOR_KEYS):
                if self.current_emo_vector[i] > 0.01:  # 大于0.01才显示
                    # 使用翻译获取情绪名称
                    name = i18n.t(key)
                    active_emotions.append(f"{name}:{self.current_emo_vector[i]:.2f}")
            
            if active_emotions:
                emotion_text = ", ".join(active_emotions[:3])  # 最多显示3个
                if len(active_emotions) > 3:
                    emotion_text += "..."
                status_text = f"{i18n.t('vector_status_using')} {emotion_text}"
                color = "green"
            else:
                status_text = i18n.t("vector_status_all_zero")
                color = "orange"
            
            self.vector_status_label.config(text=status_text, foreground=color)
        else:
            self.vector_status_label.config(text=i18n.t("vector_status_not_using"), foreground="gray")
    
    # ---------- 语言切换 ----------
    
    def _on_language_change(self, event=None):
        """语言切换回调"""
        selected = self.lang_var.get()
        lang_code = selected.split(" - ")[0] if " - " in selected else selected
        
        if i18n.set_language(lang_code):
            # 更新所有UI文本
            self._update_ui_texts()
            
            messagebox.showinfo(
                "Language / 语言",
                i18n.t("language_changed")
            )
            self._log(f"Language changed to: {selected}")
    
    def _update_ui_texts(self):
        """更新所有UI元素的文本"""
        try:
            # 更新窗口标题
            self.root.title(i18n.t("app_title"))
            
            # 更新语言标签
            if hasattr(self, 'lang_label'):
                self.lang_label.config(text=i18n.t("language"))
            
            # 更新全局设置框
            if hasattr(self, 'global_settings_frame'):
                self.global_settings_frame.config(text=i18n.t("global_settings"))
            
            # 更新主标签
            if hasattr(self, 'label_voice'):
                self.label_voice.config(text=i18n.t("voice_audio"))
            if hasattr(self, 'label_emo_ref'):
                self.label_emo_ref.config(text=i18n.t("emotion_ref_audio"))
            if hasattr(self, 'btn_voice'):
                self.btn_voice.config(text=i18n.t("browse"))
            if hasattr(self, 'btn_emo_ref'):
                self.btn_emo_ref.config(text=i18n.t("browse"))
            
            # 更新复选框
            if hasattr(self, 'chk_preserve'):
                self.chk_preserve.config(text=i18n.t("enable_voice_protection"))
            if hasattr(self, 'chk_use_vector'):
                self.chk_use_vector.config(text=i18n.t("batch_use_emotion_vector"))
            if hasattr(self, 'chk_console'):
                self.chk_console.config(text=i18n.t("show_console"))
            
            # 更新GPU设置
            if hasattr(self, 'label_gpu'):
                self.label_gpu.config(text=i18n.t("gpu_settings"))
            if hasattr(self, 'radio_auto'):
                self.radio_auto.config(text=i18n.t("auto_select"))
            if hasattr(self, 'radio_single'):
                self.radio_single.config(text=i18n.t("single_gpu"))
            if hasattr(self, 'radio_multi'):
                self.radio_multi.config(text=i18n.t("multi_gpu"))
            if hasattr(self, 'radio_cpu'):
                self.radio_cpu.config(text=i18n.t("cpu_mode"))
            
            # 更新Notebook标签
            if hasattr(self, 'main_nb'):
                self.main_nb.tab(0, text=i18n.t("batch_mode"))
                self.main_nb.tab(1, text=i18n.t("single_mode"))
            
            # 更新按钮
            if hasattr(self, 'stop_btn'):
                self.stop_btn.config(text=i18n.t("stop_task"))
            if hasattr(self, 'export_log_btn'):
                self.export_log_btn.config(text=i18n.t("export_log"))
            if hasattr(self, 'donate_btn'):
                self.donate_btn.config(text=i18n.t("donate"))
            
            # 更新批量模式按钮
            if hasattr(self, 'btn_add_task'):
                self.btn_add_task.config(text=i18n.t("add_task"))
            if hasattr(self, 'btn_remove_task'):
                self.btn_remove_task.config(text=i18n.t("remove_task"))
            if hasattr(self, 'btn_start_task'):
                self.btn_start_task.config(text=i18n.t("start_task"))
            if hasattr(self, 'btn_start_all_tasks'):
                self.btn_start_all_tasks.config(text=i18n.t("start_all_tasks"))
            
            # 更新单句模式标签和按钮
            if hasattr(self, 'label_single_text'):
                self.label_single_text.config(text=i18n.t("single_text"))
            if hasattr(self, 'label_single_emo'):
                self.label_single_emo.config(text=i18n.t("emotion_level"))
            if hasattr(self, 'vec_frame'):
                self.vec_frame.config(text=i18n.t("emotion_vector_title"))
            if hasattr(self, 'btn_generate_single'):
                self.btn_generate_single.config(text=i18n.t("generate_single"))
            if hasattr(self, 'btn_apply_vector'):
                self.btn_apply_vector.config(text=i18n.t("apply_vector"))
            if hasattr(self, 'btn_clear_vector'):
                self.btn_clear_vector.config(text=i18n.t("clear_vector"))
            
            # 更新情绪标签
            if hasattr(self, 'emo_labels'):
                for key, label in self.emo_labels.items():
                    label.config(text=i18n.t(key))
            
            # 更新情绪向量状态指示器
            self._update_vector_status_indicator()
            
            # 更新所有任务标签页的标签和按钮
            for idx, task in enumerate(self.tasks):
                # 更新标签页标题
                task_title = f"{i18n.t('task')} {idx + 1}"
                self.task_nb.tab(idx, text=task_title)
                
                # 更新任务内的标签
                if "label_txt" in task:
                    task["label_txt"].config(text=i18n.t("script_txt"))
                if "label_out" in task:
                    task["label_out"].config(text=i18n.t("output_dir"))
                if "label_emo" in task:
                    task["label_emo"].config(text=i18n.t("emotion_level"))
                if "btn_txt" in task:
                    task["btn_txt"].config(text=i18n.t("browse"))
                if "btn_out" in task:
                    task["btn_out"].config(text=i18n.t("select"))
            
            # 更新状态标签
            if hasattr(self, 'status_label'):
                current_color = self.status_label.cget("foreground")
                # 尝试从当前文本判断状态
                current_text = self.status_label.cget("text")
                if "空闲" in current_text or "Idle" in current_text:
                    status_key = "status_idle"
                elif "运行" in current_text or "Running" in current_text:
                    status_key = "status_running"
                elif "完成" in current_text or "Completed" in current_text:
                    status_key = "status_completed"
                elif "停止" in current_text or "Stopped" in current_text or "Stopping" in current_text:
                    status_key = "status_stopped"
                elif "失败" in current_text or "Failed" in current_text:
                    status_key = "status_failed"
                else:
                    status_key = "status_idle"
                
                new_text = i18n.t("status") + i18n.t(status_key)
                self.status_label.config(text=new_text)
            
            self._log(">> UI language updated / 界面语言已更新")
            
        except Exception as e:
            self._log(f">> Error updating UI texts: {e}")
    
    # ---------- 日志导出 ----------
    
    def _export_log(self):
        """导出日志到文件"""
        if not self.log_buffer:
            messagebox.showwarning(
                "No Logs / 无日志",
                "No logs to export. / 没有可导出的日志。"
            )
            return
        
        # 生成默认文件名
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        default_filename = f"tts_log_{timestamp}.txt"
        
        # 选择保存位置
        filepath = filedialog.asksaveasfilename(
            title="Export Log / 导出日志",
            defaultextension=".txt",
            initialfile=default_filename,
            filetypes=[
                ("Text files", "*.txt"),
                ("All files", "*.*")
            ]
        )
        
        if not filepath:
            return
        
        try:
            with open(filepath, "w", encoding="utf-8") as f:
                f.write("=" * 80 + "\n")
                f.write("IndexTTS2 Batch Generation Log / IndexTTS2 批量生成日志\n")
                f.write(f"Export Time / 导出时间: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
                f.write("=" * 80 + "\n\n")
                
                for log_line in self.log_buffer:
                    f.write(log_line + "\n")
                
                f.write("\n" + "=" * 80 + "\n")
                f.write(f"Total Lines / 总行数: {len(self.log_buffer)}\n")
                f.write("=" * 80 + "\n")
            
            self._log(f">> Log exported to: {filepath}")
            messagebox.showinfo(
                "Success / 成功",
                f"Log exported successfully! / 日志导出成功！\n\n{filepath}"
            )
        except Exception as e:
            self._log(f">> Export log failed: {e}")
            messagebox.showerror(
                "Error / 错误",
                f"Failed to export log: / 导出日志失败：\n{e}"
            )
    
    # ---------- 捐赠窗口 ----------
    
    def _show_donate_window(self):
        """显示捐赠窗口"""
        try:
            # 创建顶层窗口
            donate_window = tk.Toplevel(self.root)
            donate_window.title(i18n.t("donate_title"))
            donate_window.geometry("700x750")
            donate_window.resizable(True, True)
            
            # 设置窗口图标（如果有的话）
            try:
                donate_window.iconbitmap(self.root.iconbitmap())
            except:
                pass
            
            # 主容器
            main_frame = ttk.Frame(donate_window, padding="20")
            main_frame.pack(fill="both", expand=True)
            
            # 标题
            title_label = ttk.Label(
                main_frame,
                text=i18n.t("donate_title"),
                font=("", 16, "bold")
            )
            title_label.pack(pady=(0, 10))
            
            # 描述
            desc_label = ttk.Label(
                main_frame,
                text=i18n.t("donate_description"),
                font=("", 10),
                wraplength=650  # 增加换行宽度
            )
            desc_label.pack(pady=(0, 20))
            
            # 创建滚动区域
            canvas = tk.Canvas(main_frame, height=600)
            scrollbar = ttk.Scrollbar(main_frame, orient="vertical", command=canvas.yview)
            scrollable_frame = ttk.Frame(canvas)
        
            scrollable_frame.bind(
                "<Configure>",
                lambda e: canvas.configure(scrollregion=canvas.bbox("all"))
            )
            
            canvas.create_window((0, 0), window=scrollable_frame, anchor="nw")
            canvas.configure(yscrollcommand=scrollbar.set)
            
            # 1. 支付宝
            alipay_frame = ttk.LabelFrame(scrollable_frame, text=i18n.t("donate_alipay"), padding="10")
            alipay_frame.pack(fill="x", pady=(0, 15))
            
            # 尝试加载支付宝二维码
            alipay_img_path = os.path.join(self.script_dir, "assets", "alipay_qr.png")
            img_loaded = False
            
            try:
                from PIL import Image, ImageTk
                
                if os.path.exists(alipay_img_path):
                    try:
                        img = Image.open(alipay_img_path)
                        
                        # 保持原始宽高比，缩放到合适大小
                        target_size = 280
                        try:
                            img.thumbnail((target_size, target_size), Image.Resampling.LANCZOS)
                        except AttributeError:
                            img.thumbnail((target_size, target_size), Image.LANCZOS)
                        
                        photo = ImageTk.PhotoImage(img)
                        
                        # 创建容器居中显示
                        img_container = tk.Frame(alipay_frame)
                        img_container.pack(pady=10)
                        
                        img_label = tk.Label(img_container, image=photo)
                        img_label.image = photo  # 保持引用
                        img_label.pack()
                        
                        img_loaded = True
                    except Exception as img_error:
                        pass
                
                if not img_loaded:
                    if not os.path.exists(alipay_img_path):
                        msg = "⚠️ 支付宝二维码图片未找到\n\n请将图片命名为 alipay_qr.png\n放置到: assets 文件夹"
                        if i18n.current_lang != "zh":
                            msg = "⚠️ Alipay QR code not found\n\nPlease name the image: alipay_qr.png\nPlace in: assets folder"
                    else:
                        msg = f"⚠️ 图片路径: {alipay_img_path}\n但加载失败"
                        if i18n.current_lang != "zh":
                            msg = f"⚠️ Image path: {alipay_img_path}\nBut failed to load"
                    
                    tk.Label(
                        alipay_frame,
                        text=msg,
                        foreground="orange",
                        justify="left"
                    ).pack(pady=10)
                    
            except ImportError:
                msg = "📱 未安装PIL/Pillow库\n\n运行以下命令安装:\nvenv\\Scripts\\pip install Pillow\n\n或手动扫描支付宝收款码"
                if i18n.current_lang != "zh":
                    msg = "📱 PIL/Pillow not installed\n\nRun:\nvenv\\Scripts\\pip install Pillow\n\nOr scan Alipay QR manually"
                
                tk.Label(
                    alipay_frame,
                    text=msg,
                    justify="left"
                ).pack(pady=10)
            
            # 2. PayPal
            paypal_frame = ttk.LabelFrame(scrollable_frame, text=i18n.t("donate_paypal"), padding="10")
            paypal_frame.pack(fill="x", pady=(0, 15))
            
            paypal_account = "1250857665@qq.com"
            
            paypal_entry_frame = ttk.Frame(paypal_frame)
            paypal_entry_frame.pack(fill="x", pady=5)
            
            paypal_entry = ttk.Entry(paypal_entry_frame, width=50, font=("", 10))
            paypal_entry.insert(0, paypal_account)
            paypal_entry.config(state="readonly")
            paypal_entry.pack(side="left", padx=(0, 5), fill="x", expand=True)
            
            paypal_copy_btn = ttk.Button(
                paypal_entry_frame,
                text=i18n.t("copy"),
                width=8,
                command=lambda: self._copy_to_clipboard(paypal_account, donate_window)
            )
            paypal_copy_btn.pack(side="left")
            
            # 3. 虚拟货币
            crypto_frame = ttk.LabelFrame(scrollable_frame, text=i18n.t("donate_crypto"), padding="10")
            crypto_frame.pack(fill="x", pady=(0, 15))
            
            # ETH/BNB/Polygon地址
            eth_label = ttk.Label(crypto_frame, text=i18n.t("donate_eth"), font=("", 9, "bold"))
            eth_label.pack(anchor="w", pady=(5, 2))
            
            eth_address = "0xfedecd3fa0c0f8fa1a893bb761a1253aaf2636f6"
            
            eth_entry_frame = ttk.Frame(crypto_frame)
            eth_entry_frame.pack(fill="x", pady=5)
            
            eth_entry = ttk.Entry(eth_entry_frame, width=55, font=("Courier", 9))
            eth_entry.insert(0, eth_address)
            eth_entry.config(state="readonly")
            eth_entry.pack(side="left", padx=(0, 5), fill="x", expand=True)
            
            eth_copy_btn = ttk.Button(
                eth_entry_frame,
                text=i18n.t("copy"),
                width=8,
                command=lambda: self._copy_to_clipboard(eth_address, donate_window)
            )
            eth_copy_btn.pack(side="left")
            
            # BTC地址
            btc_label = ttk.Label(crypto_frame, text=i18n.t("donate_btc"), font=("", 9, "bold"))
            btc_label.pack(anchor="w", pady=(15, 2))
            
            btc_address = "0xfedecd3fa0c0f8fa1a893bb761a1253aaf2636f6"
            
            btc_entry_frame = ttk.Frame(crypto_frame)
            btc_entry_frame.pack(fill="x", pady=5)
            
            btc_entry = ttk.Entry(btc_entry_frame, width=55, font=("Courier", 9))
            btc_entry.insert(0, btc_address)
            btc_entry.config(state="readonly")
            btc_entry.pack(side="left", padx=(0, 5), fill="x", expand=True)
            
            btc_copy_btn = ttk.Button(
                btc_entry_frame,
                text=i18n.t("copy"),
                width=8,
                command=lambda: self._copy_to_clipboard(btc_address, donate_window)
            )
            btc_copy_btn.pack(side="left")
            
            # 底部感谢信息（放在滚动区域内）
            thank_label = ttk.Label(
                scrollable_frame,
                text=i18n.t("thank_you"),
                font=("", 11, "bold"),
                foreground="green"
            )
            thank_label.pack(pady=(20, 10))
            
            # 打包canvas和scrollbar
            canvas.pack(side="left", fill="both", expand=True)
            scrollbar.pack(side="right", fill="y")
            
        except Exception as e:
            # 捕获所有错误，避免GUI闪退
            error_msg = f"打开捐赠窗口时出错：\n{str(e)}"
            self._log(f">> 错误：{error_msg}")
            import traceback
            self._log(f">> 详细错误：\n{traceback.format_exc()}")
            messagebox.showerror(
                "错误 / Error",
                error_msg
            )
    
    def _copy_to_clipboard(self, text, parent_window):
        """复制文本到剪贴板"""
        try:
            self.root.clipboard_clear()
            self.root.clipboard_append(text)
            self.root.update()  # 确保剪贴板更新
            
            # 显示提示
            messagebox.showinfo(
                i18n.t("copied"),
                i18n.t("copy_success"),
                parent=parent_window
            )
        except Exception as e:
            messagebox.showerror(
                "Error / 错误",
                f"Failed to copy: / 复制失败：\n{e}",
                parent=parent_window
            )


def main():
    root = tk.Tk()
    app = TTSBatchGUI(root)
    root.mainloop()


if __name__ == "__main__":
    main()
