import os
import re
import argparse
import json
import traceback
import sys

from indextts.infer_v2 import IndexTTS2


# ========== 辅助函数：自动检测编码读取 ==========
def read_file_auto_encoding(path):
    """尝试使用不同编码读取文件"""
    encodings = ["utf-8", "gbk", "utf-16"]
    for enc in encodings:
        try:
            with open(path, "r", encoding=enc) as f:
                return [x.strip() for x in f if x.strip()]
        except UnicodeDecodeError:
            continue
        except Exception as e:
            print(f"读取文件出错 {path}: {e}")
            return []
    
    print(f"错误：无法识别文件编码 {path}，请尝试保存为 UTF-8 格式。")
    return []


# ========== 辅助函数：获取唯一文件名 ==========
def get_unique_filename(directory, base_name, extension=".wav"):
    """确保文件名不冲突，如果存在则自动添加序号 (1), (2)..."""
    # 移除非法字符
    base_name = re.sub(r'[\\/:*?"<>|]', "", base_name)
    if not base_name:
        base_name = "line"
        
    # 动态计算最大允许长度，以避免 Windows 路径超限
    # Windows MAX_PATH 通常为 260 字符
    # 我们需要预留: 目录路径长度 + 分隔符(1) + 扩展名长度 + 序号占位(例如 " (99)" 占 5 字符)
    try:
        abs_dir = os.path.abspath(directory)
        dir_len = len(abs_dir)
    except:
        dir_len = 100 # 兜底预估
        
    ext_len = len(extension)
    reserved_buffer = 5 # 为 " (99)" 预留空间
    
    # 1. 基于全路径的限制 (MAX_PATH - dir - / - ext - buffer)
    max_path_limit = 259 - dir_len - 1 - ext_len - reserved_buffer
    
    # 2. 基于 NTFS 单文件名的限制 (255 - ext - buffer)
    ntfs_limit = 255 - ext_len - reserved_buffer
    
    # 取两者较小值作为最终限制
    final_limit = min(max_path_limit, ntfs_limit)
    
    # 兜底：至少保留 20 个字符，如果路径实在太深，让它报错也比截断成空字符串好
    final_limit = max(20, final_limit)
    
    if len(base_name) > final_limit:
        base_name = base_name[:final_limit]
        
    filename = f"{base_name}{extension}"
    full_path = os.path.join(directory, filename)
    
    if not os.path.exists(full_path):
        return full_path
        
    # 如果文件存在，尝试添加序号
    counter = 1
    while True:
        new_filename = f"{base_name} ({counter}){extension}"
        new_full_path = os.path.join(directory, new_filename)
        if not os.path.exists(new_full_path):
            return new_full_path
        counter += 1


# ========== 情绪提取（支持所有括号，兼容不规范写法） ==========
def split_emo_and_text(line: str):
    line = line.strip()
    if not line:
        return None, ""

    # 1. 尝试标准格式：(xxx) text / [xxx] text / 【xxx】text
    #    支持：(开心)你好 / [生气] 滚！ / 【难过】... / (小声) ...
    pattern_std = r"^[\(\（\[\【](.*?)[\)\）\]\】]\s*(.*)$"
    m = re.match(pattern_std, line)
    if m:
        emo_text = m.group(1).strip()
        real_text = m.group(2).strip()
        # 只有当括号内不为空时才认为是情绪标签
        if emo_text:
            return emo_text, real_text

    # 2. 如果没有匹配到情绪标签，则整句作为文本
    #    这会覆盖 "..." / "......" 等特殊开头的情况，确保它们被视为普通文本处理
    return None, line


# ========== 主流程 ==========
def main():
    parser = argparse.ArgumentParser()

    parser.add_argument("--voice", required=True, type=str)
    parser.add_argument("--text", required=False, type=str)  # 单句模式下不需要
    parser.add_argument("--outdir", required=True, type=str)

    parser.add_argument("--emo", type=str, default="mid")
    parser.add_argument("--preserve", type=str, default="false")
    parser.add_argument("--use-vector", type=str, default="false")
    parser.add_argument("--vecfile", type=str, default=None)
    parser.add_argument("--emo-vec", type=str, default=None)  # 逗号分隔的向量字符串
    parser.add_argument("--emo-ref", type=str, default=None)  # 情绪参考音频
    
    # 单句模式参数
    parser.add_argument("--single", action="store_true")  # 单句模式标志
    parser.add_argument("--line", type=str, default=None)  # 单句文本
    parser.add_argument("--single-out", type=str, default=None)  # 单句输出路径
    
    # 显卡设置
    parser.add_argument("--device", type=str, default=None)  # 指定GPU设备（例如：cuda:0, cuda:1）

    # 高级参数
    parser.add_argument("--interval", type=int, default=200, help="句间停顿(ms)")
    parser.add_argument("--max-tokens", type=int, default=120, help="最大分段token数")

    args = parser.parse_args()

    # 转换布尔
    use_vector = args.use_vector.lower() == "true"
    preserve_voice = args.preserve.lower() == "true"

    # ========== 单句模式 vs 批量模式 ==========
    lines = []
    if args.single:
        # 单句模式
        if not args.line:
            print("错误: 单句模式需要 --line 参数")
            return
        lines = [args.line]
    else:
        # 批量模式
        if not args.text:
            print("错误: 批量模式需要 --text 参数")
            return
        if not os.path.isfile(args.text):
            print(f"错误: 找不到文本文件: {args.text}")
            return
        
        # 使用自动编码检测读取
        lines = read_file_auto_encoding(args.text)
        if not lines:
            print("警告: 文件为空或无法读取")
            return

    # ========== 加载情绪向量 ==========
    emotion_vector = None
    if use_vector:
        # 优先使用 --emo-vec（GUI 传递的字符串）
        if args.emo_vec:
            try:
                emotion_vector = [float(x.strip()) for x in args.emo_vec.split(",")]
                print(">> 使用 GUI 情绪向量:", emotion_vector)
            except Exception as e:
                print(f"警告: 解析情绪向量失败: {e}")
                emotion_vector = None
        # 其次使用 --vecfile（JSON 文件）
        elif args.vecfile and os.path.isfile(args.vecfile):
            try:
                with open(args.vecfile, "r", encoding="utf-8") as f:
                    emotion_vector = json.load(f)
                print(">> 使用文件情绪向量:", emotion_vector)
            except Exception as e:
                print(f"警告: 读取向量文件失败: {e}")

    # ========== 输出目录 ==========
    try:
        os.makedirs(args.outdir, exist_ok=True)
    except Exception as e:
        print(f"错误: 无法创建输出目录 {args.outdir}: {e}")
        return

    # ========== 初始化模型 ==========
    print(">> 正在加载模型 IndexTTS2...")
    
    # 获取脚本所在目录，确保能找到 checkpoints
    script_dir = os.path.dirname(os.path.abspath(__file__))
    cfg_path = os.path.join(script_dir, "checkpoints", "config.yaml")
    model_dir = os.path.join(script_dir, "checkpoints")
    
    if not os.path.exists(cfg_path):
        print(f"错误: 找不到配置文件 {cfg_path}")
        return

    try:
        if args.device:
            print(f">> 使用指定的GPU设备: {args.device}")
            tts = IndexTTS2(
                cfg_path=cfg_path,
                model_dir=model_dir,
                use_fp16=True,
                use_cuda_kernel=True,
                device=args.device,
            )
        else:
            print(">> 自动选择GPU设备")
            tts = IndexTTS2(
                cfg_path=cfg_path,
                model_dir=model_dir,
                use_fp16=True,
                use_cuda_kernel=True,
            )
        print(f">> [确认] 模型已加载到设备: {tts.device}")
    except Exception as e:
        print(f"错误: 模型加载失败: {e}")
        traceback.print_exc()
        return

    # ========== 检查声纹 ==========
    if not os.path.isfile(args.voice):
        print(f"错误: 找不到参考音频: {args.voice}")
        return

    print(f">> 使用参考音频（声纹）: {args.voice}")
    
    # ========== 情绪参考音频 ==========
    emo_audio_prompt = None
    if args.emo_ref and os.path.isfile(args.emo_ref):
        emo_audio_prompt = args.emo_ref
        print(f">> 使用情绪参考音频: {args.emo_ref}")
    
    # ========== 根据情绪档位和声纹保护设置 emo_alpha ==========
    # 第一步：根据情绪档位设置基础强度
    emo_level_map = {
        "low": 0.5,      # 低档：温和的情绪
        "mid": 0.8,      # 中档：适中的情绪（默认）
        "high": 1.0,     # 高档：强烈的情绪
    }
    base_alpha = emo_level_map.get(args.emo.lower(), 0.8)
    print(f">> 情绪档位: {args.emo} -> 基础强度 {base_alpha}")
    
    # 第二步：如果启用声纹保护，进一步降低强度
    if preserve_voice:
        emo_alpha = base_alpha * 0.7  # 降低到70%
        print(f">> 启用声纹保护模式，最终 emo_alpha={emo_alpha:.2f}")
    else:
        emo_alpha = base_alpha
        print(f">> 最终情绪强度 emo_alpha={emo_alpha:.2f}")

    # ========== 批量生成 ==========
    success_count = 0
    fail_count = 0

    for idx, raw in enumerate(lines, 1):
        try:
            emo_text, real_text = split_emo_and_text(raw)
            if not real_text:
                continue

            # 启用情绪文本只有当未使用向量模式
            use_emo_text = (not use_vector)

            # 单句模式：使用指定的输出路径
            if args.single and args.single_out:
                out_path = args.single_out
                # 确保父目录存在
                os.makedirs(os.path.dirname(out_path), exist_ok=True)
            else:
                # 批量模式：生成唯一文件名
                out_path = get_unique_filename(args.outdir, real_text)

            print(f"\n>> [{idx}/{len(lines)}] 生成中:")
            print("   台词:", real_text)
            if use_emo_text and emo_text:
                print("   情绪标签:", emo_text)
            if emotion_vector:
                print("   情绪向量:", [f"{v:.2f}" for v in emotion_vector])

            # ========== 推理 ==========
            tts.infer(
                spk_audio_prompt=args.voice,
                text=real_text,
                output_path=out_path,
                emo_audio_prompt=emo_audio_prompt,
                emo_alpha=emo_alpha,
                emo_vector=emotion_vector,
                use_emo_text=use_emo_text,
                emo_text=emo_text,
                use_random=False,
                interval_silence=args.interval,
                verbose=False,
                max_text_tokens_per_segment=args.max_tokens,
            )

            print("   输出:", out_path)
            success_count += 1

        except Exception as e:
            print(f"   [错误] 第 {idx} 行生成失败: {e}")
            traceback.print_exc()
            fail_count += 1
            continue

    print(f"\n>> 全部任务完成！成功: {success_count}, 失败: {fail_count}")


if __name__ == "__main__":
    main()
