/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.util.resource;

import java.io.Closeable;
import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.ProviderNotFoundException;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.URIUtil;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.annotation.ManagedOperation;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.eclipse.jetty.util.resource.MountedPathResource;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.thread.AutoLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject(value="Pool of FileSystems used to mount Resources")
public class FileSystemPool
implements Dumpable {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemPool.class);
    public static final FileSystemPool INSTANCE = new FileSystemPool();
    private static final Map<String, String> ENV_MULTIRELEASE_RUNTIME;
    private final Map<URI, Bucket> pool = new HashMap<URI, Bucket>();
    private final AutoLock poolLock = new AutoLock();
    private Listener listener;

    private FileSystemPool() {
    }

    Mount mount(URI uri) throws IOException {
        if (!uri.isAbsolute()) {
            throw new IllegalArgumentException("not an absolute uri: " + String.valueOf(uri));
        }
        if (!uri.getScheme().equalsIgnoreCase("jar")) {
            throw new IllegalArgumentException("not an supported scheme: " + String.valueOf(uri));
        }
        FileSystem fileSystem = null;
        URI jarURIRoot = this.toJarURIRoot(uri);
        AutoLock ignore = this.poolLock.lock();
        try {
            try {
                fileSystem = FileSystems.newFileSystem(jarURIRoot, ENV_MULTIRELEASE_RUNTIME);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Mounted new FS {}", (Object)jarURIRoot);
                }
            }
            catch (FileSystemAlreadyExistsException fsaee) {
                fileSystem = Paths.get(jarURIRoot).getFileSystem();
                if (!fileSystem.isOpen()) {
                    LOG.warn("FileSystem {} of URI {} already exists but is not open (bug JDK-8291712)", (Object)fileSystem, (Object)uri);
                } else if (LOG.isDebugEnabled()) {
                    LOG.debug("Using existing FS {}", (Object)jarURIRoot);
                }
            }
            catch (ProviderNotFoundException pnfe) {
                throw new IllegalArgumentException("Unable to mount FileSystem from unsupported URI: " + String.valueOf(jarURIRoot), pnfe);
            }
            URI rootURI = fileSystem.getPath("/", new String[0]).toUri();
            Mount mount = new Mount(rootURI, new MountedPathResource(jarURIRoot));
            this.retain(rootURI, fileSystem, mount);
            Mount mount2 = mount;
            if (ignore != null) {
                ignore.close();
            }
            return mount2;
        }
        catch (Throwable throwable) {
            try {
                if (ignore != null) {
                    try {
                        ignore.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (Exception e) {
                IO.close(fileSystem);
                throw e;
            }
        }
    }

    private URI toJarURIRoot(URI uri) {
        String rawURI = uri.toASCIIString();
        int idx = rawURI.indexOf("!/");
        if (idx > 0) {
            return URI.create(rawURI.substring(0, idx + 2));
        }
        return uri;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void unmount(URI fsUri) {
        block22: {
            try (AutoLock ignore = this.poolLock.lock();){
                Bucket bucket = this.pool.get(fsUri);
                if (bucket == null) {
                    LOG.warn("Unable to release Mount (not in pool): {}", (Object)fsUri);
                    return;
                }
                int count = bucket.counter.decrementAndGet();
                if (count == 0) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Ref counter reached 0, closing pooled FS {}", (Object)bucket);
                    }
                    try {
                        Path rootOfCreatedPath = null;
                        if (!Files.exists(bucket.path, new LinkOption[0])) {
                            rootOfCreatedPath = this.createEmptyFileWithParents(bucket.path);
                        }
                        try {
                            bucket.fileSystem.close();
                        }
                        finally {
                            IO.delete(rootOfCreatedPath);
                        }
                        this.pool.remove(fsUri);
                        if (this.listener != null) {
                            this.listener.onClose(fsUri);
                        }
                        break block22;
                    }
                    catch (IOException e) {
                        LOG.warn("Unable to close FileSystem {} of URI {} (bug JDK-8291712)", bucket.fileSystem, fsUri, e);
                    }
                    break block22;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Decremented ref counter to {} for FS {}", (Object)count, (Object)bucket);
                }
                if (this.listener != null) {
                    this.listener.onDecrement(fsUri);
                }
            }
            catch (FileSystemNotFoundException fileSystemNotFoundException) {
                // empty catch block
            }
        }
    }

    private Path createEmptyFileWithParents(Path path) throws IOException {
        Path createdRootPath = null;
        if (!Files.exists(path.getParent(), new LinkOption[0])) {
            createdRootPath = this.createDirWithAllParents(path.getParent());
        }
        Files.createFile(path, new FileAttribute[0]);
        if (createdRootPath == null) {
            createdRootPath = path;
        }
        return createdRootPath;
    }

    private Path createDirWithAllParents(Path path) throws IOException {
        Path parentPath = path.getParent();
        if (!Files.exists(parentPath, new LinkOption[0])) {
            Path createdRootPath = this.createDirWithAllParents(parentPath);
            Files.createDirectory(path, new FileAttribute[0]);
            return createdRootPath;
        }
        Files.createDirectory(path, new FileAttribute[0]);
        return path;
    }

    @ManagedAttribute(value="The mounted FileSystems")
    public Collection<Mount> mounts() {
        try (AutoLock ignore = this.poolLock.lock();){
            List<Mount> list = this.pool.values().stream().map(m4 -> m4.mount).toList();
            return list;
        }
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        Collection<Bucket> values;
        try (AutoLock ignore = this.poolLock.lock();){
            values = this.pool.values();
        }
        Dumpable.dumpObjects(out, indent, this, new DumpableCollection("buckets", values));
    }

    @ManagedOperation(value="Sweep the pool for deleted mount points", impact="ACTION")
    public void sweep() {
        Set<Map.Entry<URI, Bucket>> entries;
        try (AutoLock ignore = this.poolLock.lock();){
            entries = this.pool.entrySet();
        }
        for (Map.Entry<URI, Bucket> entry : entries) {
            URI fsUri = entry.getKey();
            Bucket bucket = entry.getValue();
            FileSystem fileSystem = bucket.fileSystem;
            try {
                AutoLock ignore = this.poolLock.lock();
                try {
                    if ((!fileSystem.isOpen() || Files.isReadable(bucket.path)) && Files.getLastModifiedTime(bucket.path, new LinkOption[0]).equals(bucket.lastModifiedTime) && Files.size(bucket.path) == bucket.size) continue;
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("File {} backing filesystem {} has been removed or changed, closing it", (Object)bucket.path, (Object)fileSystem);
                    }
                    IO.close(fileSystem);
                    this.pool.remove(fsUri);
                }
                finally {
                    if (ignore == null) continue;
                    ignore.close();
                }
            }
            catch (IOException e) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Cannot read last access time or size of file {} backing filesystem {}", (Object)bucket.path, (Object)fileSystem);
            }
        }
    }

    private void retain(URI fsUri, FileSystem fileSystem, Mount mount) {
        assert (this.poolLock.isHeldByCurrentThread());
        Bucket bucket = this.pool.get(fsUri);
        if (bucket == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Pooling new FS {}", (Object)fileSystem);
            }
            bucket = new Bucket(fsUri, fileSystem, mount);
            this.pool.put(fsUri, bucket);
            if (this.listener != null) {
                this.listener.onRetain(fsUri);
            }
        } else {
            int count = bucket.counter.incrementAndGet();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Incremented ref counter to {} for FS {}", (Object)count, (Object)fileSystem);
            }
            if (this.listener != null) {
                this.listener.onIncrement(fsUri);
            }
        }
    }

    int getReferenceCount(URI fsUri) {
        Bucket bucket = this.pool.get(fsUri);
        if (bucket == null) {
            return 0;
        }
        return bucket.counter.get();
    }

    public void setListener(Listener listener) {
        try (AutoLock ignore = this.poolLock.lock();){
            this.listener = listener;
        }
    }

    public String toString() {
        return "%s@%x{%d}".formatted(TypeUtil.toShortName(this.getClass()), this.hashCode(), this.pool.size());
    }

    static {
        HashMap<String, String> env = new HashMap<String, String>();
        env.put("releaseVersion", "runtime");
        ENV_MULTIRELEASE_RUNTIME = env;
    }

    public static class Mount
    implements Closeable {
        private final URI fsUri;
        private final Resource root;

        private Mount(URI fsUri, Resource resource) {
            this.fsUri = fsUri;
            this.root = resource;
        }

        public Resource root() {
            return this.root;
        }

        @Override
        public void close() {
            INSTANCE.unmount(this.fsUri);
        }

        public String toString() {
            return String.format("%s[uri=%s,root=%s]", TypeUtil.toShortName(this.getClass()), this.fsUri, this.root);
        }
    }

    private static class Bucket {
        private final AtomicInteger counter;
        private final FileSystem fileSystem;
        private final FileTime lastModifiedTime;
        private final long size;
        private final Path path;
        private final Mount mount;

        private Bucket(URI fsUri, FileSystem fileSystem, Mount mount) {
            FileTime lastModifiedTime;
            long size;
            Path path;
            block2: {
                URI containerUri = URIUtil.unwrapContainer(fsUri);
                path = Paths.get(containerUri);
                size = -1L;
                lastModifiedTime = null;
                try {
                    size = Files.size(path);
                    lastModifiedTime = Files.getLastModifiedTime(path, new LinkOption[0]);
                }
                catch (IOException ioe) {
                    if (!LOG.isDebugEnabled()) break block2;
                    LOG.debug("Cannot read size or last modified time from {} backing filesystem at {}", (Object)path, (Object)fsUri);
                }
            }
            this.counter = new AtomicInteger(1);
            this.fileSystem = fileSystem;
            this.path = path;
            this.size = size;
            this.lastModifiedTime = lastModifiedTime;
            this.mount = mount;
        }

        public String toString() {
            return this.fileSystem.toString() + "#" + String.valueOf(this.counter);
        }
    }

    public static interface Listener {
        public void onRetain(URI var1);

        public void onIncrement(URI var1);

        public void onDecrement(URI var1);

        public void onClose(URI var1);
    }

    public static class StackLoggingListener
    implements Listener {
        private static final Logger LOG = LoggerFactory.getLogger(StackLoggingListener.class);

        @Override
        public void onRetain(URI uri) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Retain {}", (Object)uri, (Object)new Throwable("Retain"));
            }
        }

        @Override
        public void onIncrement(URI uri) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Increment {}", (Object)uri, (Object)new Throwable("Increment"));
            }
        }

        @Override
        public void onDecrement(URI uri) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Decrement {}", (Object)uri, (Object)new Throwable("Decrement"));
            }
        }

        @Override
        public void onClose(URI uri) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Close {}", (Object)uri, (Object)new Throwable("Close"));
            }
        }
    }
}

