from django.db import models


# Create your models here.

class Store(models.Model):
    # 本实验此由用户提供
    storeName = models.CharField('店铺名', max_length=20, unique=True)

    owner = models.ForeignKey(to='user.User', on_delete=models.CASCADE)
    address = models.TextField('店铺地址', default='')
    info = models.TextField('店铺介绍')
    sold = models.IntegerField('售出书本数量', default=0)
    createDate = models.DateField('创建时间', auto_now_add=True)

    class Meta:
        db_table = 'store'

    def info_json(self):
        return {
            'store_id': self.storeName,
            'address': self.address,
            'info': self.info,
            'sold': self.sold,
            'create_date': str(self.createDate),
        }


class Book(models.Model):
    # 本实验此由用户提供
    bookID = models.CharField('书本ID', max_length=50, unique=True)

    # 事务相关
    store = models.ForeignKey(Store, on_delete=models.CASCADE)
    price = models.DecimalField('价格', max_digits=10, decimal_places=2)
    stock = models.IntegerField('库存', default=0)
    sold = models.IntegerField('售出书本数量', default=0)
    off = models.BooleanField('是否下架', default=False)
    createDate = models.DateTimeField('上传时间', auto_now=True)

    # 书籍信息
    isbn = models.CharField('ISBN', max_length=50)
    cover = models.ImageField('封面', default='cover/default.png')
    title = models.CharField('书名', max_length=200)
    author = models.CharField('作者', max_length=50, null=True)
    publisher = models.CharField('出版社', max_length=50, null=True)
    originalTitle = models.CharField('原名', max_length=200, null=True)
    translator = models.CharField('翻译', max_length=50, null=True)
    pubYear = models.CharField('出版时间', max_length=50, null=True)
    pages = models.IntegerField('页数', null=True)
    currencyUnit = models.CharField('货币单位', max_length=50, null=True)
    binding = models.CharField('封面', max_length=50, null=True)
    authorIntro = models.TextField('作者简介', null=True)
    bookIntro = models.TextField('书本简介', null=True)
    content = models.TextField('目录', null=True)

    class Meta:
        db_table = 'book'

    def info_simple_json(self):
        return {
            'store_id': self.store.storeName,
            'book_id': self.bookID,
            'stock_level': self.stock,
            'sold': self.sold,
            'price': self.price,
            'off': self.off,
            'title': self.title,
            'author': self.author,
            'pub_year': self.pubYear,
        }

    def info_json(self):
        return {
            'store_id': self.store.storeName,
            'id': self.bookID,
            'stock_level': self.stock,
            'sold': self.sold,
            'price': self.price,
            'off': self.off,
            'isbn': self.isbn,
            'title': self.title,
            'author': self.author,
            'publisher': self.publisher,
            'original_title': self.originalTitle,
            'translator': self.translator,
            'pub_year': self.pubYear,
            'pages': self.pages,
            'binding': self.binding,
            'author_intro': self.authorIntro,
            'book_intro': self.bookIntro,
            'content': self.content,
            "tags": [tag.tag for tag in Tag.objects.filter(book=self)],
            "pictures": [picture.picture.url for picture in Picture.objects.filter(book=self)],
        }


class Tag(models.Model):
    book = models.ForeignKey(Book, on_delete=models.CASCADE)
    tag = models.CharField('标签', max_length=50)

    class Meta:
        db_table = 'tag'


class Picture(models.Model):
    book = models.ForeignKey(Book, on_delete=models.CASCADE)
    picture = models.ImageField('书本相关图片', upload_to='picture/')

    class Meta:
        db_table = 'picture'