from django.test import TestCase

from django.test import TestCase, Client
from django.utils.timezone import localtime

from BookStore import error, function
# Create your tests here.
from cart.models import Cart
from order import state
from order.models import Order, OrderBook
from store.models import Store, Book
from user.models import User, Address


class HomeRecommendTest(TestCase):
    def setUp(self) -> None:
        data = {
            'user_id': 'yzm',
            'password': 'aaaaaa'
        }
        buyer = User.objects.create(username=data['user_id'], password=data['password'], balance=10000000)
        seller = User.objects.create(username='stm', password='aaaaaa', balance=0.01)
        store = Store.objects.create(storeName='648全球连锁店', owner=seller, address='648')
        order = Order.objects.create(store=store, user=buyer, sumPrice=10, state=state.order_received(), paid=True)
        self.book = Book.objects.create(bookID='book_id_1', stock=10, store=store, isbn='1', title='book',
                                        price='100.00')
        OrderBook.objects.create(order=order, book=self.book, count=10, booksPrice=100)
        self.client = Client(HTTP_TOKEN=function.generate_token({'user_id': 'yzm', 'terminal': '127.0.0.1'}))

    def test_home_recommend_success(self):
        response = self.client.get('/recommend/home/', content_type='application/json')

        code, message = 200, 'ok'
        self.assertEqual(response.status_code, code)
        expect = {'message': 'ok',
                  'new_books': [{'author': None,
                                 'create_date': localtime(self.book.createDate).strftime("%Y-%m-%d %H:%M:%S"),
                                 'pub_year': None,
                                 'title': 'book'}],
                  'weekly_books': [{'author': None,
                                    'pub_year': None,
                                    'sold': 10,
                                    'title': 'book'}],
                  'weekly_stores': [{'address': '648',
                                     'info': '',
                                     'sold': 0,
                                     'store_id': '648全球连锁店'}]}

        self.assertJSONEqual(response.content, expected_data=expect)


class StoreRecommendGoodBookTest(TestCase):
    def setUp(self) -> None:
        data = {
            'user_id': 'yzm',
            'password': 'aaaaaa'
        }
        buyer = User.objects.create(username=data['user_id'], password=data['password'], balance=10000000)
        seller = User.objects.create(username='stm', password='aaaaaa', balance=0.01)
        store = Store.objects.create(storeName='648全球连锁店', owner=seller, address='648')
        order = Order.objects.create(store=store, user=buyer, sumPrice=10, state=state.order_received(), paid=True)
        self.book = Book.objects.create(bookID='book_id_1', stock=10, store=store, isbn='1', title='book',
                                        price='100.00')
        OrderBook.objects.create(order=order, book=self.book, count=10, booksPrice=100)
        self.client = Client(HTTP_TOKEN=function.generate_token({'user_id': 'yzm', 'terminal': '127.0.0.1'}))

    def test_store_recommend_store_id_not_exist(self):
        data = {
            'store_id': '648环球港'
        }
        response = self.client.get('/recommend/stores/good_books/', data=data, content_type='application/json')

        code, message = error.store_id_doesnt_exist(data["store_id"])
        self.assertEqual(response.status_code, code)
        expect = {'message': "store id 648环球港 doesn't exist (店铺不存在)"}

        self.assertJSONEqual(response.content, expected_data=expect)

    def test_store_recommend_success(self):
        data = {
            'store_id': '648全球连锁店'
        }
        response = self.client.get('/recommend/stores/good_books/', data=data, content_type='application/json')

        code, message = 200, 'ok'
        self.assertEqual(response.status_code, code)
        expect = {'books': [{'author': None,
                             'off': False,
                             'pub_year': None,
                             'sold': 10,
                             'title': 'book'}],
                  'message': 'ok'}



        self.assertJSONEqual(response.content, expected_data=expect)


class StoreRecommendNewBookTest(TestCase):
    def setUp(self) -> None:
        data = {
            'user_id': 'yzm',
            'password': 'aaaaaa'
        }
        buyer = User.objects.create(username=data['user_id'], password=data['password'], balance=10000000)
        seller = User.objects.create(username='stm', password='aaaaaa', balance=0.01)
        store = Store.objects.create(storeName='648全球连锁店', owner=seller, address='648')
        order = Order.objects.create(store=store, user=buyer, sumPrice=10, state=state.order_received(), paid=True)
        self.book = Book.objects.create(bookID='book_id_1', stock=10, store=store, isbn='1', title='book',
                                        price='100.00')
        OrderBook.objects.create(order=order, book=self.book, count=10, booksPrice=100)
        self.client = Client(HTTP_TOKEN=function.generate_token({'user_id': 'yzm', 'terminal': '127.0.0.1'}))

    def test_store_recommend_store_id_not_exist(self):
        data = {
            'store_id': '648环球港'
        }
        response = self.client.get('/recommend/stores/new_books/', data=data, content_type='application/json')

        code, message = error.store_id_doesnt_exist(data["store_id"])
        self.assertEqual(response.status_code, code)
        expect = {'message': "store id 648环球港 doesn't exist (店铺不存在)"}

        self.assertJSONEqual(response.content, expected_data=expect)

    def test_store_recommend_success(self):
        data = {
            'store_id': '648全球连锁店'
        }
        response = self.client.get('/recommend/stores/new_books/', data=data, content_type='application/json')

        code, message = 200, 'ok'
        self.assertEqual(response.status_code, code)
        expect = {'books': [{'author': None,
                             'create_date': localtime(self.book.createDate).strftime("%Y-%m-%d %H:%M:%S"),
                             'price': '100.00',
                             'pub_year': None,
                             'title': 'book'}],
                  'message': 'ok'}


        self.assertJSONEqual(response.content, expected_data=expect)
