package com.fr.value;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * 一次计算永久持有的非{@code null}懒加载值.
 *
 * <p>
 * 线程安全版本 {@link AtomicNullableLazyValue}.
 * </p>
 *
 * @author vito
 * created on 2019-09-17
 * @IncludeIntoJavadoc
 */
public abstract class NullableLazyValue<T> {

    private boolean myComputed;
    @Nullable
    private T myValue;

    /**
     * 计算值.
     *
     * @return 值
     */
    @Nullable
    protected abstract T compute();

    /**
     * 获取值.
     *
     * @return 值
     */
    @Nullable
    public T getValue() {
        T value = myValue;
        if (!myComputed) {
            value = compute();
            myValue = value;
            myComputed = true;
        }
        return value;
    }

    /**
     * 创建可计算的懒加载值.
     *
     * @param value 可计算的值
     * @param <T> 类型
     * @return 懒加载值
     */
    @NotNull
    public static <T> NullableLazyValue<T> createValue(@NotNull final Computable<? extends T> value) {
        return new NullableLazyValue<T>() {

            @Nullable
            @Override
            protected T compute() {
                return value.compute();
            }
        };
    }
}

