package com.fr.value;

import org.jetbrains.annotations.NotNull;

/**
 * 一次计算永久持有的非{@code null}懒加载值.
 *
 * <p>
 * 线程安全版本 {@link AtomicNotNullLazyValue}.
 * </p>
 *
 * @author vito
 * created on 2019-09-17
 * @IncludeIntoJavadoc
 */
public abstract class NotNullLazyValue<T> {

    private T myValue;

    /**
     * 计算值.
     *
     * @return 值
     */
    @NotNull
    protected abstract T compute();

    /**
     * 获取值.
     *
     * @return 值
     */
    @NotNull
    public T getValue() {
        T result = myValue;
        if (result == null) {
            result = compute();
            myValue = result;
        }
        return result;
    }

    /**
     * 是否已经计算过.
     *
     * @return 是否已经计算过
     */
    public boolean isComputed() {
        return myValue != null;
    }

    /**
     * 创建常量懒加载值.
     *
     * @param value 常量
     * @param <T> 类型
     * @return 懒加载值
     */
    @NotNull
    public static <T> NotNullLazyValue<T> createConstantValue(@NotNull final T value) {
        return new NotNullLazyValue<T>() {
            @NotNull
            @Override
            protected T compute() {
                return value;
            }
        };
    }

    /**
     * 创建可计算的懒加载值.
     *
     * @param value 可计算的值
     * @param <T> 类型
     * @return 懒加载值
     */
    @NotNull
    public static <T> NotNullLazyValue<T> createValue(@NotNull final Computable<? extends T> value) {
        return new NotNullLazyValue<T>() {
            @NotNull
            @Override
            protected T compute() {
                return value.compute();
            }
        };
    }
}
