package com.fr.value;


import org.jetbrains.annotations.NotNull;

/**
 * 一次计算永久持有的懒加载值，可重复计算.
 *
 * <p>
 * 线程安全版本 {@link AtomicClearableLazyValue}.
 * </p>
 *
 * @author vito
 * created on 2019-09-17
 * @IncludeIntoJavadoc
 */
public abstract class ClearableLazyValue<T> {

    private T myValue;

    /**
     * 计算值.
     *
     * @return 值
     */
    @NotNull
    protected abstract T compute();

    /**
     * 获取值.
     *
     * @return 值
     */
    @NotNull
    public T getValue() {
        T result = myValue;
        if (result == null) {
            result = compute();
            myValue = result;
        }
        return result;
    }

    /**
     * 清除内部保存的值.
     */
    public void drop() {
        myValue = null;
    }

    /**
     * 创建可计算的懒加载值.
     *
     * @param value 可计算的值
     * @param <T> 类型
     * @return 懒加载值
     */
    @NotNull
    public static <T> ClearableLazyValue<T> create(final Computable<? extends T> value) {
        return new ClearableLazyValue<T>() {
            @NotNull
            @Override
            protected T compute() {
                return value.compute();
            }
        };
    }
}
