package com.fr.value;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * 一次计算永久持有的非{@code null}懒加载值.
 *
 * <p>
 * 线程安全.
 * 非线程安全版本 {@link NullableLazyValue}.
 * </p>
 *
 *
 * @author vito
 * created on 2019-09-17
 * @IncludeIntoJavadoc
 */
public abstract class AtomicNullableLazyValue<T> extends NullableLazyValue<T> {

    private volatile T myValue;
    private volatile boolean myComputed;

    /**
     * 获取值.
     *
     * @return 值
     */
    @Override
    public final T getValue() {
        boolean computed = myComputed;
        T value = myValue;
        if (computed) {
            return value;
        }
        //noinspection SynchronizeOnThis
        synchronized (this) {
            computed = myComputed;
            value = myValue;
            if (!computed) {
                value = compute();
                myValue = value;
                myComputed = true;
            }
        }
        return value;
    }

    /**
     * 创建可计算的懒加载值.
     *
     * @param value 可计算的值
     * @param <T> 类型
     * @return 懒加载值
     */
    @SuppressWarnings("MethodOverridesStaticMethodOfSuperclass")
    @NotNull
    public static <T> AtomicNullableLazyValue<T> createValue(@NotNull final Computable<? extends T> value) {
        return new AtomicNullableLazyValue<T>() {
            @Nullable
            @Override
            protected T compute() {
                return value.compute();
            }
        };
    }
}

