package com.fr.value;

import org.jetbrains.annotations.NotNull;

/**
 * 一次计算永久持有的非{@code null}懒加载值.
 *
 * <p>
 * 线程安全.
 * 非线程安全版本 {@link NotNullLazyValue}.
 * </p>
 *
 * @author vito
 * created on 2019-09-17
 * @IncludeIntoJavadoc
 */
public abstract class AtomicNotNullLazyValue<T> extends NotNullLazyValue<T> {
    private volatile T myValue;

    /**
     * 获取值.
     *
     * @return 值
     */
    @Override
    @NotNull
    public final T getValue() {
        T value = myValue;
        if (value == null) {
            //no inspection SynchronizeOnThis
            synchronized (this) {
                value = myValue;
                if (value == null) {
                    value = compute();
                    myValue = value;
                }
            }
        }
        return value;
    }

    /**
     * 是否已经计算过.
     *
     * @return 是否已经计算过
     */
    @Override
    public boolean isComputed() {
        return myValue != null;
    }

    /**
     * 创建可计算的懒加载值.
     *
     * @param value 可计算的值
     * @param <T> 类型
     * @return 懒加载值
     */
    @SuppressWarnings("MethodOverridesStaticMethodOfSuperclass")
    @NotNull
    public static <T> AtomicNotNullLazyValue<T> createValue(@NotNull final Computable<? extends T> value) {
        return new AtomicNotNullLazyValue<T>() {
            @NotNull
            @Override
            protected T compute() {
                return value.compute();
            }
        };
    }
}

