package com.fr.stable.collections.combination;

import com.fr.common.annotations.Open;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * 三元组.
 *
 * @param <L> 第一个对象的类型
 * @param <M> 第二个对象的类型
 * @param <R> 第三个对象的类型
 * @author juhaoyu
 * created on 2018-06-13
 */
@Open
public class Triple<L, M, R> {
    
    private L left;

    /**
     * 获取第一个对象.
     *
     * @return 第一个对象
     */
    @Nullable
    public L getLeft() { return left; }
    
    
    private M middle;

    /**
     * 获取第二个对象.
     *
     * @return 第二个对象
     */
    @Nullable
    public M getMiddle() { return middle; }
    
    
    private R right;

    /**
     * 获取第三个对象.
     *
     * @return 第三个对象
     */
    @Nullable
    public R getRight() {
        
        return right;
    }

    /**
     * 生成一个{@code Triple}.
     *
     * @param l 第一个对象
     * @param m 第二个对象
     * @param r 第三个对象
     * @param <L> 第一个对象的类型
     * @param <M> 第二个对象的类型
     * @param <R> 第三个对象的类型
     * @return {@code Triple}
     */
    @NotNull
    public static <L, M, R> Triple<L, M, R> wrap(@Nullable L l, @Nullable M m, @Nullable R r) {
        
        return new Triple<L, M, R>(l, m, r);
    }
    
    private Triple(L left, M middle, R right) {
        
        this.left = left;
        this.middle = middle;
        this.right = right;
    }

    /**
     * 生成{@code Triple}的哈希值.
     *
     * @return 哈希值
     */
    @Override
    public int hashCode() {
        
        return left.hashCode() * 13 + (middle == null ? 0 : middle.hashCode());
    }

    /**
     * 判断{@code Triple}与另一个对象是否相等.
     *
     * <p>
     * 如果另一个对象是{@code null}或者不是{@code Triple}，返回{@code false}.<br>
     * 当且仅当两个{@code Triple}的第一、第二对象均相等时才返回{@code true}.
     * </p>
     *
     * @param o 要比较的另一个对象
     * @return 是否相等
     */
    @Override
    public boolean equals(Object o) {
        
        if (this == o) return true;
        if (o instanceof Triple) {
            Triple triple = (Triple) o;
            if (left != null ? !left.equals(triple.left) : triple.left != null) return false;
            if (middle != null ? !middle.equals(triple.middle) : triple.middle != null) return false;
            if (right != null ? !right.equals(triple.right) : triple.right != null) return false;
            return true;
        }
        return false;
    }
}
