package com.fr.stable.collections.combination;

import com.fr.common.annotations.Open;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.Serializable;

/**
 * 元组.
 *
 * @param <U> 第一个对象的类型
 * @param <V> 第二个对象的类型
 * @author juhaoyu
 * created on 2018-06-13
 */
@Open
public class Pair<U, V> implements Serializable {
    
    /**
     * Key of this <code>Pair</code>.
     */
    private U first;
    
    /**
     * 获取第一个对象.
     *
     * @return 第一个对象
     */
    @Nullable
    public U getFirst() { return first; }
    
    /**
     * Value of this this <code>Pair</code>.
     */
    private V second;

    /**
     * 获取第二个对象.
     *
     * @return 第二个对象
     */
    @Nullable
    public V getSecond() { return second; }

    /**
     * 创建一个空{@code Pair}.
     */
    public Pair() {
    }

    /**
     * 创建一个{@code Pair}.
     *
     * @param first 第一个对象
     * @param second 第二个对象
     */
    @NotNull
    public Pair(@Nullable U first, @Nullable V second) {
        
        this.first = first;
        this.second = second;
    }

    /**
     * 设置第一个对象.
     *
     * @param first 第一个对象
     */
    public void setFirst(@Nullable U first) {
        this.first = first;
    }

    /**
     * 设置第二个对象.
     *
     * @param second 第二个对象
     */
    public void setSecond(@Nullable V second) {
        this.second = second;
    }

    /**
     * 获取{@code Pair}的字符串表示.
     *
     * <p>
     * 默认为"第一个对象=第二个对象".
     * </p>
     *
     * @return 字符串表示
     */
    @Override
    public String toString() {
        
        return first + "=" + second;
    }

    /**
     * 生成{@code Pair}的哈希值.
     *
     * @return 哈希值
     */
    @Override
    public int hashCode() {
        // name's hashCode is multiplied by an arbitrary prime number (13)
        // in order to make sure there is a difference in the hashCode between
        // these two parameters:
        //  name: a  second: aa
        //  name: aa second: a
        return first.hashCode() * 13 + (second == null ? 0 : second.hashCode());
    }

    /**
     * 判断{@code Pair}与另一个对象是否相等.
     *
     * <p>
     * 如果另一个对象是{@code null}或者不是{@code Pair}，返回{@code false}.<br>
     * 当且仅当两个{@code Pair}的第一、第二对象均相等时才返回{@code true}.
     * </p>
     *
     * @param o 要比较的另一个对象
     * @return 是否相等
     */
    @Override
    public boolean equals(Object o) {
        
        if (this == o) return true;
        if (o instanceof Pair) {
            Pair pair = (Pair) o;
            if (first != null ? !first.equals(pair.first) : pair.first != null) return false;
            if (second != null ? !second.equals(pair.second) : pair.second != null) return false;
            return true;
        }
        return false;
    }
}

