package com.fr.stable;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * 运算工具类.
 *
 * @author fanruan
 * created on 2020-07-28
 * @IncludeIntoJavadoc
 */
public class OperationUtils {

	/** 正无穷大 */
	public static final Double POSITIVE_INFINITY = new Double(Double.POSITIVE_INFINITY);
	/** 负无穷大 */
	public static final Double NEGATIVE_INFINITY = new Double(Double.NEGATIVE_INFINITY);
	/** {@code Integer}零 */
	public static final Integer ZERO = new Integer(0);
	/** {@code BigDecimal}零 */
	public static final BigDecimal BZERO = new BigDecimal(0);

	/** 非 */
	private static final String BOOL = "!";
	/** 加 */
	private static final String PLUS = "+";
	/** 减 */
	private static final String MINUS = "-";

	private OperationUtils() {
	}

	/**
	 * 判断对象是否为false.
	 *
	 * <p>
	 * 如果对象为{@code null}，或为{@link Primitive#NULL}，或与{@code false}相等，或为{@code 0}，或为{@link StringUtils#EMPTY}，则返回true
	 * </p>
	 *
	 * @param obj 要判断的对象
	 * @return 是否为false
	 */
	public static boolean isFalse(@Nullable Object obj) {
		return obj == null
				|| obj == Primitive.NULL
				|| Boolean.FALSE.equals(obj)
				|| new Integer(0).equals(obj)
				|| StringUtils.EMPTY.equals(obj);
	}

	/**
	 * 对布尔值使用一元操作符.
	 *
	 * <pre>
	 * OperationUtils.intUnaryOperation(false, "!")	= true
	 * </pre>
	 *
	 * @param B 布尔值
	 * @param kind 一元操作符，仅支持"!"
	 * @return 结果
	 * @throws UtilEvalError 操作符错误
	 */
	public static boolean booleanUnaryOperation(@NotNull Boolean B, @NotNull String kind) throws UtilEvalError {

		boolean operand = B.booleanValue();

		if (BOOL.equals(kind)) {
			return !operand;
		} else {
			throw new UtilEvalError("Operator inappropriate for boolean:\n" + kind + B);
		}
	}

	/**
	 * 对{@code Integer}值使用一元操作符.
	 *
	 * <pre>
	 * OperationUtils.intUnaryOperation(2, "+")	= 2
	 * OperationUtils.intUnaryOperation(2, "-")	= -2
	 * </pre>
	 *
	 * @param I {@code Integer}值
	 * @param kind 一元操作符，支持"+"和"-"
	 * @return 结果
	 * @throws InterpreterError 操作符错误
	 */
	public static int intUnaryOperation(@NotNull Integer I, @NotNull String kind) {

		int operand = I.intValue();

		if (PLUS.equals(kind)) {
			return operand;
		} else if (MINUS.equals(kind)) {
			return -operand;
		} else {
			throw new InterpreterError("bad integer unaryOperation:\n" + kind + I);
		}
	}

	/**
	 * 对{@code Long}值使用一元操作符.
	 *
	 * <pre>
	 * OperationUtils.longUnaryOperation(123L, "+")	= 123L
	 * OperationUtils.longUnaryOperation(123L, "-")	= -123L
	 * </pre>
	 *
	 * @param L {@code Long}值
	 * @param kind 一元操作符，支持"+"和"-"
	 * @return 结果
	 * @throws InterpreterError 操作符错误
	 */
	@NotNull
	public static long longUnaryOperation(@NotNull Long L, @NotNull String kind) {

		long operand = L.longValue();

		if (PLUS.equals(kind)) {
			return operand;
		} else if (MINUS.equals(kind)) {
			return -operand;
		} else {
			throw new InterpreterError("bad long unaryOperation:\n" + kind + L);
		}
	}

	/**
	 * 对{@code Float}值使用一元操作符.
	 *
	 * <pre>
	 * OperationUtils.floatUnaryOperation(123f, "+")	= 123f
	 * OperationUtils.floatUnaryOperation(123f, "-")	= -123f
	 * </pre>
	 *
	 * @param F {@code Float}值
	 * @param kind 一元操作符，支持"+"和"-"
	 * @return 结果
	 * @throws InterpreterError 操作符错误
	 */
	@NotNull
	public static float floatUnaryOperation(@NotNull Float F, @NotNull String kind) {

		float operand = F.floatValue();

		if (PLUS.equals(kind)) {
			return operand;
		} else if (MINUS.equals(kind)) {
			return -operand;
		} else {
			throw new InterpreterError("bad float unaryOperation:\n" + kind + F);
		}
	}

	/**
	 * 对{@code Double}值使用一元操作符.
	 *
	 * <pre>
	 * OperationUtils.doubleUnaryOperation(123.456d, "+")	= 123.456d
	 * OperationUtils.doubleUnaryOperation(123.456d, "-")	= -123.456d
	 * </pre>
	 *
	 * @param D {@code Double}值
	 * @param kind 一元操作符，支持"+"和"-"
	 * @return 结果
	 * @throws InterpreterError 操作符错误
	 */
	@NotNull
	public static double doubleUnaryOperation(@NotNull Double D, @NotNull String kind) {

		double operand = D.doubleValue();

		if (PLUS.equals(kind)) {
			return operand;
		} else if (MINUS.equals(kind)) {
			return -operand;
		} else {
			throw new InterpreterError("bad double unaryOperation:\n" + kind + D);
		}
	}

	/**
	 * 对{@code BigInteger}值使用一元操作符.
	 *
	 * @param B {@code BigInteger}值
	 * @param kind 一元操作符，支持"+"和"-"
	 * @return 结果
	 * @throws InterpreterError 操作符错误
	 */
	@NotNull
	public static BigInteger bigIntegerUnaryOperation(@NotNull BigInteger B, @NotNull String kind) {

		if (PLUS.equals(kind)) {
			return B;
		} else if (MINUS.equals(kind)) {
			return B.negate();
		} else {
			throw new InterpreterError("bad bigInteger unaryOperation:\n" + kind + B);
		}
	}

	/**
	 * 对{@code BigDecimal}值使用一元操作符.
	 *
	 * @param B {@code BigDecimal}值
	 * @param kind 一元操作符，支持"+"和"-"
	 * @return 结果
	 * @throws InterpreterError 操作符错误
	 */
	@NotNull
	public static BigDecimal bigDecimalUnaryOperation(@NotNull BigDecimal B, @NotNull String kind) {
		if (PLUS.equals(kind)) {
			return B;
		} else if (MINUS.equals(kind)) {
			return B.negate();
		} else {
			throw new InterpreterError("bad bigDecimal unaryOperation:\n" + kind + B);
		}
	}

	/**
	 * 将{@code char}, {@code byte}和{@code short}转换为{@code String}或{@code Integer}.
	 *
	 * <p>
	 * 传入{@code char}，转换为对应{@code String}<br>
	 * 传入{@code byte}，转换为对应{@code Integer}<br>
	 * 传入{@code short}，转换为对应{@code Integer}
	 * </p>
	 *
	 * @param wrapper 传入值
	 * @return 转换值
	 */
	@NotNull
	public static Object promoteCharByteShort(@NotNull Object wrapper) {

		if (wrapper instanceof Character)
			return wrapper.toString();
		else if ((wrapper instanceof Byte) || (wrapper instanceof Short))
			return new Integer(((Number) wrapper).intValue());

		return wrapper;
	}
}