package com.fr.stable;

import com.fr.log.FineLoggerFactory;
import org.jetbrains.annotations.Nullable;

/**
 * 过滤器工具类.
 *
 * @author ju
 * created on 2018-03-07
 * @see Filter
 * @IncludeIntoJavadoc
 */
public abstract class Filters {

    /**
     * 接受一切的过滤器.
     */
    private static final Filter ALL = new Filter() {
        
        @Override
        public boolean accept(Object o) {
            
            return true;
        }
    };

    /**
     * 什么都不接受的过滤器.
     */
    private static final Filter NONE = new Filter() {
        
        @Override
        public boolean accept(Object o) {
            
            return false;
        }
    };

    /**
     * 过滤器是否接受特定对象.
     *
     * <p>
     * {@code null}对象不被任何过滤器接受.<br>
     * {@code null}过滤器接受任何对象.<br>
     * {@code accept()}过程抛出异常返回不接受.<br>
     * </p>
     *
     * @param filter 过滤器
     * @param t 对象
     * @param <T> 过滤器过滤类型（对象类型）
     * @return 是否接受
     */
    public static <T> boolean accept(@Nullable Filter<T> filter, @Nullable T t) {
        
        if (t == null) {
            return false;
        }
        
        if (filter == null) {
            return true;
        }
        
        boolean accept;
        try {
            accept = filter.accept(t);
        } catch (Throwable e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
            accept = false;
        }
        return accept;
    }

    /**
     * 接受该类型一切对象的过滤器.
     *
     * @param <T> 过滤类型
     * @return 过滤器
     */
    @SuppressWarnings("unchecked")
    public static <T> Filter<T> all() {
        
        return ALL;
    }

    /**
     * 不接受该类型任意对象的过滤器.
     *
     * @param <T> 过滤类型
     * @return 过滤器
     */
    @SuppressWarnings("unchecked")
    public static <T> Filter<T> none() {
        
        return NONE;
    }

    /**
     * 返回两个过滤器的交集过滤器.
     *
     * @param filter1 过滤器1
     * @param filter2 过滤器2
     * @param <T> 过滤类型
     * @return 并集过滤器
     */
    public static <T> Filter<T> and(final Filter<T> filter1, final Filter<T> filter2) {
        
        return new Filter<T>() {
            
            @Override
            public boolean accept(T t) {
                
                return Filters.accept(filter1, t) && Filters.accept(filter2, t);
            }
        };
    }

    /**
     * 返回两个过滤器的并集过滤器.
     *
     * @param filter1 过滤器1
     * @param filter2 过滤器2
     * @param <T> 过滤类型
     * @return 并集过滤器
     */
    public static <T> Filter<T> or(final Filter<T> filter1, final Filter<T> filter2) {
        
        return new Filter<T>() {
            
            @Override
            public boolean accept(T t) {
                
                return Filters.accept(filter1, t) || Filters.accept(filter2, t);
            }
        };
    }
}
