package com.fr.stable;

import com.fr.log.FineLoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 *
 */
public enum FileTypeInspector {

    DEFAULT(0) {
        @Override
        public boolean checkFileSignatureSuffix(String fileCode, String suffix) {
            String lowerCaseSuffix = suffix.toLowerCase();
            //后缀不在白名单，放行
            if (!FILE_HEADER_MAGIC_NUMBER.keySet().contains(lowerCaseSuffix)) {
                return true;
            }
            for (String magicNumber : FILE_HEADER_MAGIC_NUMBER.get(lowerCaseSuffix)) {
                if (fileCode.toUpperCase().startsWith(magicNumber)) {
                    return true;
                }
            }
            return false;
        }
    }, WHITE_LIST(1) {
        @Override
        public boolean checkFileSignatureSuffix(String fileCode, String suffix) {
            String[] magicNumbers = FILE_HEADER_MAGIC_NUMBER.get(suffix.toLowerCase());
            if (ArrayUtils.isNotEmpty(magicNumbers)) {
                for (String magicNumber : magicNumbers) {
                    if (fileCode.toUpperCase().startsWith(magicNumber)) {
                        return true;
                    }
                }
            }
            return false;
        }
    }, BLACKLIST(2) {
        //后缀黑名单校验，安全性较差
        private final String[] SUFFIX_BLACK_LIST = new String[]{"asp", "jsp", "php", "exe"};

        @Override
        public boolean checkFileSignatureSuffix(String fileCode, String suffix) {
            HashSet<String> blackSet = new HashSet<String>(Arrays.asList(SUFFIX_BLACK_LIST));
            return !blackSet.contains(suffix.toLowerCase());
        }
    };

    private int type;

    FileTypeInspector(int type) {
        this.type = type;
    }

    public int getType() {
        return type;
    }

    public static FileTypeInspector parse(int type) {
        for (FileTypeInspector value : FileTypeInspector.values()) {
            if (value.type == type) {
                return value;
            }
        }
        return DEFAULT;
    }

    /**
     * 判断传入的二进制流是否与后缀相匹配
     *
     * @param is 文件流
     * @param suffix 文件后缀
     * @return 是否匹配
     */
    public boolean checkFileType(InputStream is, String suffix) {
        byte[] bytes = readBytes(is, MAGIC_HEADER_LENGTH);
        return checkFileSignatureSuffix(encodeHexStr(bytes), suffix);
    }

    /**
     * 读取指定长度的byte数组
     *
     * @param in     {@link InputStream}，为null返回null
     * @param length 长度，小于等于0返回空byte数组
     * @return bytes
     */
    public static byte[] readBytes(InputStream in, int length) {

        if (null == in) {
            return null;
        }
        if (length <= 0) {
            return new byte[0];
        }

        byte[] b = new byte[length];
        int readLength = 0;
        try {
            readLength = in.read(b);
        } catch (IOException e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
        if (readLength > 0 && readLength < length) {
            byte[] b2 = new byte[length];
            System.arraycopy(b, 0, b2, 0, readLength);
            return b2;
        } else {
            return b;
        }
    }

    private static String encodeHexStr(byte[] bytes) {
        StringBuilder fileCodeBuilder = new StringBuilder();
        for (byte b : bytes) {
            String hv = Integer.toHexString(b & 0xFF);
            if (hv.length() < 2) {
                fileCodeBuilder.append(0);
            }
            fileCodeBuilder.append(hv);
        }
        return fileCodeBuilder.toString();
    }

    public abstract boolean checkFileSignatureSuffix(String fileCode, String suffix);

    /**
     * 暂时只检测这些类型，其他类型不检测
     * https://www.garykessler.net/library/file_sigs.html
     * https://asecuritysite.com/forensics/magic
     */
    private static final Map<String, String[]> FILE_HEADER_MAGIC_NUMBER = new ConcurrentHashMap<String, String[]>();
    private static final String[] INNER_FILE_TYPE = {"png", "jpg", "gif", "bmp", "tif", "pdf", "zip", "rar", "dwg", "gz", "ttf", "ttc"};
    private static final int MAGIC_HEADER_LENGTH = 10;

    static {
        FILE_HEADER_MAGIC_NUMBER.put("jpg", new String[]{"FFD8"});
        FILE_HEADER_MAGIC_NUMBER.put("jpeg", new String[]{"FFD8"});
        FILE_HEADER_MAGIC_NUMBER.put("gif", new String[]{"47494638"});
        FILE_HEADER_MAGIC_NUMBER.put("bmp", new String[]{"424D"});
        FILE_HEADER_MAGIC_NUMBER.put("png", new String[]{"89504E470D0A1A0A"});
        FILE_HEADER_MAGIC_NUMBER.put("pdf", new String[]{"25504446"});
        FILE_HEADER_MAGIC_NUMBER.put("doc", new String[]{"D0CF11E0A1B11AE1", "7B5C72746631"}); // 设计器导出doc实际是RTF格式:7B5C72746631
        FILE_HEADER_MAGIC_NUMBER.put("xls", new String[]{"D0CF11E0A1B11AE1"});
        FILE_HEADER_MAGIC_NUMBER.put("ppt", new String[]{"D0CF11E0A1B11AE1"});
        FILE_HEADER_MAGIC_NUMBER.put("docx", new String[]{"504B0304"});
        FILE_HEADER_MAGIC_NUMBER.put("xlsx", new String[]{"504B0304"});
        FILE_HEADER_MAGIC_NUMBER.put("pptx", new String[]{"504B0304"});
        FILE_HEADER_MAGIC_NUMBER.put("zip", new String[]{"504B0304"});

        FILE_HEADER_MAGIC_NUMBER.put("rar", new String[]{"52617221"});
        FILE_HEADER_MAGIC_NUMBER.put("tif", new String[]{"49492A00"});
        FILE_HEADER_MAGIC_NUMBER.put("dwg", new String[]{"41433130"});
        FILE_HEADER_MAGIC_NUMBER.put("gz", new String[]{"1F8B08"});
        FILE_HEADER_MAGIC_NUMBER.put("jar", new String[]{"504B0304"});
        FILE_HEADER_MAGIC_NUMBER.put("ttf", new String[]{"000100"});
        FILE_HEADER_MAGIC_NUMBER.put("ttc", new String[]{"747463"});
    }

    /**
     * 解析传入的byte[]数组, 获取其文件二进制头对应的后缀名
     * 解析失败则返回空
     * @param bytes 文件内容
     *
     * @return 后缀名
     */
    public static String resolveFileType(byte[] bytes) {
        String checkHeader = encodeHexStr(ArrayUtils.subarray(bytes, 0, MAGIC_HEADER_LENGTH));
        for (String type : INNER_FILE_TYPE) {
            if (FileTypeInspector.WHITE_LIST.checkFileSignatureSuffix(checkHeader, type)) {
                return type;
            }
        }
        return StringUtils.EMPTY;
    }

    /**
     * 解析传入的流, 获取其文件二进制头对应的后缀名
     * 解析失败则返回空
     * @param inputStream 文件流
     *
     * @return 后缀名
     */
    public static String resolveFileType(InputStream inputStream) {
        byte[] bytes = readBytes(inputStream, MAGIC_HEADER_LENGTH);
        return resolveFileType(bytes);
    }

}
