package com.fr.stable;

import com.fr.invoke.Reflect;
import com.fr.log.FineLoggerFactory;
import com.fr.value.AtomicNotNullLazyValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.awt.Color;
import java.awt.Image;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URLDecoder;
import java.text.CharacterIterator;
import java.text.StringCharacterIterator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Stack;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.fr.stable.Constants.KEY_LENGTH;

/**
 * 常用方法工具类.
 *
 * @author fanruan
 * created on 2020-08-04
 * @IncludeIntoJavadoc
 */
public final class CommonUtils {

    /** 保存在jar里的js, HTML文件，统一使用UTF-8编码. **/
    public static final String RESOURCE_ENCODER = EncodeConstants.ENCODING_UTF_8;

    /** 默认记录待删除文件的文件名. **/
    public static final String DEFAULT_RECORD_FILE = "removing_files.log";
    /** 自定义记录待删除文件的文件名的系统属性. **/
    public static final String RECORD_FILE_PARAM_KEY = "record.removing.file";

    private static final String LOCALE_MARK = "_";
    private static final int DECIMAL_TEN = 10;
    private static final int MAX_LONG_LEN = 17;
    private static final int HEX_STRING_MIN_LEN = 4;
    private static final double DELTA = 0.0000000001;
    private static final char[] DIGITS_CASE_SENSITIVE = {'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'};
    private static final char NA = "-".charAt(0);
    private static final char PL = "+".charAt(0);
    private static final char DOT = ".".charAt(0);
    private static final char UE = "E".charAt(0);
    private static final char LE = "e".charAt(0);
    private static final String NAN = "NaN";
    private static final String INFINITY = "Infinity";
    private static final char NAN_START = NAN.charAt(0);
    private static final char INFINITY_START = INFINITY.charAt(0);
    private static final int MAJOR_JAVA_VERSION = getJavaVersion();
    private static final String WEB_INF = "WEB-INF";
    private static final String RECORD_PARENT_DIR = "assist";

    private static int[] charPosition = new int[]{5, 15};
    private static Random random = new Random();
    private static char ZERO = "0".charAt(0);
    private static char NINE = "9".charAt(0);
    private static AtomicNotNullLazyValue<Boolean> isDebugValue = new AtomicNotNullLazyValue<Boolean>() {
        @NotNull
        @Override
        protected Boolean compute() {
            return "true".equals(System.getProperty("debug"));
        }
    };

    private CommonUtils() {
    }

    /**
     * 将国际化对象{@code Locale}转换为字符串.
     *
     * @param locale 国际化对象{@code Locale}
     * @return 用于表示国际化信息的字符串
     */
    @NotNull
    public static String localeToString(@NotNull Locale locale) {
        return locale.getLanguage() + LOCALE_MARK + locale.getCountry();
    }

    /**
     * 将字符串转换为国国际化对象{@code Locale}.
     *
     * @param localeStr 表示国际化信息的字符串
     * @return 国际化对象{@code Locale}
     */
    @NotNull
    public static Locale stringToLocale(@Nullable String localeStr) {

        if (StringUtils.isBlank(localeStr)) {
            return Locale.CHINA;
        }
        StringTokenizer tempStringTokenizer = new StringTokenizer(localeStr, LOCALE_MARK);
        // TODO 不合预期，输入enUS会返回enUS_CN
        String language = "zh", country = "CN";
        if (tempStringTokenizer.hasMoreTokens()) {
            language = (String) tempStringTokenizer.nextElement();
        }
        if (tempStringTokenizer.hasMoreTokens()) {
            country = (String) tempStringTokenizer.nextElement();
        }
        return new Locale(language, country);
    }


    /**
     * 检查设计器是否已经被激活.
     *
     * @param key 激活码
     * @return 设计器是否已激活
     */
    public static boolean checkDesignerActive(@Nullable String key) {
        return key != null && key.length() == KEY_LENGTH && key.charAt(charPosition[0]) == 'A' && key.charAt(charPosition[1]) == 'F';
    }

    /**
     * 使用/和\分割字符串.
     *
     * <p>
     * 通常用来分割路径.
     * </p>
     *
     * @param path 字符串（路径）
     * @return 分割结果
     */
    @NotNull
    public static String[] pathSplit(@NotNull String path) {
        return path.split("[/\\\\]");
    }

    /**
     * 将集合中的元素按照给定的连接符拼接成字符串.
     *
     * <p>
     * 实际调用元素的{@code toString()}进行拼接.
     * {@code null}会被拼接为"null".
     * 拼接成的字符串结尾不会出现单独连接符.
     * </p>
     *
     * @param collection 集合
     * @param delimit 连接符
     * @return 拼接成的字符串
     */
    @NotNull
    public static String join(@NotNull Collection collection, String delimit) {
        StringBuffer sb = new StringBuffer();
        java.util.Iterator it = collection.iterator();
        while (it.hasNext()) {
            // 误用class
            Object o = it.next();
            sb.append(o);
            if (it.hasNext()) {
                sb.append(delimit);
            }
        }
        return sb.toString();
    }

    /**
     * 将数组中的元素按照给定的连接符拼接成字符串.
     *
     * <p>
     * 将数组使用{@code Arrays.asList()}转换为{@code List}后，调用{@link CommonUtils#join(Collection, String)}进行拼接.
     * </p>
     *
     * @param array 数组
     * @param delimit 连接符
     * @return 拼接成的字符串
     * @see CommonUtils#join(Collection, String)
     */
    @NotNull
    public static String join(@NotNull Object[] array, String delimit) {
        return CommonUtils.join(Arrays.asList(array), delimit);
    }

    /**
     * 将数组中的元素拼接成字符串.
     *
     * <p>
     * 将数组使用{@code Arrays.asList()}转换为{@code List}后，调用{@link CommonUtils#join(Collection, String)}进行拼接.
     * 连接符为空.
     * </p>
     *
     * @param array 数组
     * @return 拼接成的字符串
     * @see CommonUtils#join(Collection, String)
     */
    @NotNull
    public static String join(@NotNull Object[] array) {
        return join(array, StringUtils.EMPTY);
    }

    /**
     * 将{@code double}转换为字符串.
     *
     * <p>
     * 处理了{@code Java}对{@code double}类型进行运算时产生的精度问题.
     * </p>
     *
     * <pre>
     * CommonUtils.convertNumberStringToString(Double.POSITIVE_INFINITY)    == "∞"
     * CommonUtils.convertNumberStringToString(Double.NEGATIVE_INFINITY)    == "-∞"
     * CommonUtils.convertNumberStringToString(100d)                        == "100"
     * CommonUtils.convertNumberStringToString(-0.25d)                      == "-0.25"
     * CommonUtils.convertNumberStringToString(-3.2E-7d)                    == "-0.00000032"
     * CommonUtils.convertNumberStringToString(0.05d + 0.01d)               == "0.06"
     * CommonUtils.convertNumberStringToString(123.3d / 100d)               == "1.233"
     * </pre>
     *
     * @param number {@code double}
     * @return 字符串
     */
    @Nullable
    public static String convertNumberStringToString(double number) {

        String checkInfinity = CommonUtils.checkDoubleInfinity(number);
        if (checkInfinity != null) {
            return checkInfinity;
        }
        String numberString = Double.toString(number);
        // 这个方法主要是用来增强java中Number.toString()的方法，java语言在这个地方的处理有Bug. 老是产生一些特别长的java数字。
        int dotIndex = numberString.indexOf('.');
        if (dotIndex < 0) {// 只有小数才需要处理.
            return numberString;
        }

        // 检查一下是否是以'E'来表示的数字.
        int eIndex = numberString.indexOf('E');
        numberString = stringAfterCheckE(numberString, dotIndex, eIndex);
        dotIndex = numberString.indexOf(CoreConstants.DOT);

        // 这个checkE也感觉怪怪的, 理解上是为了规避0000&9999检测.
        if (dotIndex == -1) {
            if (numberString.endsWith(".0")) {// 去掉结尾的.0
                numberString = numberString.substring(0, numberString.length() - 2);
            }
            // 可能原始数据就是这样的，不用消除
            return numberString;
        }

        // 其他的普通数字
        // 这边处理的000和999是为了解决java浮点计算精度问题, 不好规避, 如果实在需要保留, 则传入checkE=true
        // eg, 0.05 + 0.01 = 0.060000000000000005, 123.3 / 100 = 1.2329999999999999
        String decimalString = numberString.substring(dotIndex + 1);
        int cIndex;
        if ((cIndex = decimalString.indexOf("9999")) >= 0) {
            return processCalPrecision(number, numberString, dotIndex, cIndex);
        } else if ((cIndex = decimalString.indexOf("0000")) >= 0
                && eIndex == -1) {//这边处理逻辑真是翔一样啊~
            numberString = numberString.substring(0,
                    dotIndex + Math.max(cIndex, 1) + 1);
        }

        if (numberString.endsWith(".0")) {// 去掉结尾的.0
            numberString = numberString.substring(0, numberString.length() - 2);
        }

        return numberString;
    }

    /**
     * 将{@code double}转换为字符串.
     *
     * <p>
     * 见{@link CommonUtils#convertNumberStringToString(double)}
     * </p>
     *
     * @param d {@code double}
     * @return 字符串
     * @see CommonUtils#convertNumberStringToString(double)
     */
    @Nullable
    public static String doubleToString(double d) {
        return convertNumberStringToString(d);
    }

    /**
     * 将{@code Number}转换为字符串.
     *
     * @param number {@code Number}
     * @param checkE 暂无用，默认传{@code false}
     * @return 字符串
     * @deprecated 换用 {@link CommonUtils#convertNumberStringToString(Number)}
     */
    @Deprecated
    @NotNull
    public static String convertNumberStringToString(@Nullable Number number, boolean checkE) {
        //marro:原来的这个checkE一直都是false啊，去掉了。现在有个问题，如果不想忽略".000001"这样的小数怎么办。所以，把参数概念换下

        String checkInfinity = CommonUtils.checkInfinity(number);
        if (StringUtils.isNotEmpty(checkInfinity)) {
            return checkInfinity;
        }

        String numberString = number.toString();

        // 这个方法主要是用来增强java中Number.toString()的方法，java语言在这个地方的处理有Bug.老是产生一些特别长的java数字。
        int dotIndex = numberString.indexOf('.');
        if (dotIndex < 0) {// 只有小数才需要处理.
            return numberString;
        }

        // 检查一下是否是以'E'来表示的数字.
        int eIndex = numberString.indexOf('E');
        numberString = stringAfterCheckE(numberString, dotIndex, eIndex);
        dotIndex = numberString.indexOf('.');

        //这个checkE也感觉怪怪的, 理解上是为了规避0000&9999检测.
        if (checkE || dotIndex == -1 || number instanceof BigDecimal) {
            if (numberString.endsWith(".0")) {// 去掉结尾的.0
                numberString = numberString.substring(0, numberString.length() - 2);
            }

            //可能原始数据就是这样的，不用消除
            return numberString;
        }

        // 其他的普通数字
        // 这边处理的000和999是为了解决java浮点计算精度问题, 不好规避, 如果实在需要保留, 则传入checkE=true
        // eg, 0.05 + 0.01 = 0.060000000000000005, 123.3 / 100 = 1.2329999999999999
        String decimalString = numberString.substring(dotIndex + 1);
        int cIndex;
        if ((cIndex = decimalString.indexOf("9999")) >= 0) {
            return processCalPrecisionCheckE(number, numberString, cIndex);
        } else if ((cIndex = decimalString.indexOf("0000")) >= 0
                && eIndex == -1) {//这边处理逻辑真是翔一样啊~
            numberString = numberString.substring(0,
                    dotIndex + Math.max(cIndex, 1) + 1);
        }

        if (numberString.endsWith(".0")) {// 去掉结尾的.0
            numberString = numberString.substring(0, numberString.length() - 2);
        }

        return numberString;
    }

    /**
     * 将{@code Number}转换为字符串.
     *
     * @param number {@code Number}
     * @return 字符串
     */
    @NotNull
    public static String convertNumberStringToString(@Nullable Number number) {
        return convertNumberStringToString(number, false);
    }

    /**
     * 检查{@code Object}是否为正负无穷，返回对应字符串表示.
     *
     * <pre>
     * CommonUtils.checkInfinity((Object) Double.POSITIVE_INFINITY) == "∞"
     * CommonUtils.checkInfinity((Object) Double.NEGATIVE_INFINITY) == "-∞"
     * CommonUtils.checkInfinity("abc")                             == {@link StringUtils#EMPTY}
     * </pre>
     *
     * @param obj 要检查和转换的{@code Object}
     * @return 转换结果
     */
    @NotNull
    public static String checkInfinity(@Nullable Object obj) {

        if (OperationUtils.POSITIVE_INFINITY.equals(obj)) {
            return "∞";
        } else if (OperationUtils.NEGATIVE_INFINITY.equals(obj)) {
            return "-∞";
        }

        return StringUtils.EMPTY;
    }

    /**
     * 检查{@code double}是否为正负无穷，返回对应字符串表示.
     *
     * <pre>
     * CommonUtils.checkInfinity(Double.POSITIVE_INFINITY)  == "∞"
     * CommonUtils.checkInfinity(Double.NEGATIVE_INFINITY)  == "-∞"
     * CommonUtils.checkInfinity(123d)                      == {@link StringUtils#EMPTY}
     * </pre>
     *
     * @param d 要检查和转换的{@code double}
     * @return 转换结果
     */
    @NotNull
    public static String checkInfinity(double d) {

        if (Double.POSITIVE_INFINITY == d) {
            return "∞";
        } else if (Double.NEGATIVE_INFINITY == d) {
            return "-∞";
        }
        return StringUtils.EMPTY;
    }

    /**
     * 检查{@code double}是否为正负无穷，返回对应字符串表示.
     *
     * <p>
     * 如非正负无穷，返回{@code null}.
     * </p>
     *
     * <pre>
     * CommonUtils.checkInfinity(Double.POSITIVE_INFINITY)  == "∞"
     * CommonUtils.checkInfinity(Double.NEGATIVE_INFINITY)  == "-∞"
     * CommonUtils.checkInfinity(123d)                      == {@code null}
     * </pre>
     *
     * @param d 要检查和转换的{@code double}
     * @return 转换结果
     */
    @Nullable
    public static String checkDoubleInfinity(double d) {

        if (Double.POSITIVE_INFINITY == d) {
            return "∞";
        } else if (Double.NEGATIVE_INFINITY == d) {
            return "-∞";
        }
        return null;
    }

    /**
     * 判断{@code char}是否是数字.
     *
     * @param ch {@code char}
     * @return 是否是数字
     */
    public static boolean isNum(char ch) {
        return ch >= ZERO && ch <= NINE;
    }

    /**
     * 获取两个正序的int数组的交集.
     *
     * <pre>
     * CommonUtils#intersectSortedIntArray(new int[]{1, 2, 3}, new int[]{2, 3, 4, 5}) == {2, 3}
     * </pre>
     *
     * @param arr1 数组1
     * @param arr2 数组2
     * @return 交集数组
     */
    @NotNull
    public static int[] intersectSortedIntArray(@NotNull int[] arr1, @NotNull int[] arr2) {

        if (arr1 == arr2) {
            return arr1;
        }
        boolean b = arr1.length > arr2.length;
        int[] a1 = b ? arr1 : arr2;
        int[] a2 = b ? arr2 : arr1;
        int[] value = new int[a2.length];
        int len = 0;
        int i1 = 0, i2 = 0;
        while (i1 < a1.length && i2 < a2.length) {
            if (a1[i1] == a2[i2]) {
                value[len++] = a1[i1];
                i2++;
                i1++;
            } else if (a1[i1] > a2[i2]) {
                i2++;
            } else {
                i1++;
            }
        }
        if (len == value.length) {
            return value;
        } else {
            int[] res = new int[len];
            System.arraycopy(value, 0, res, 0, len);
            return res;
        }
    }

    /**
     * 判断字符串是否代表一个数字.
     *
     * <pre>
     * CommonUtils.isNumber(null) == false
     * CommonUtils.isNumber("") == false
     * CommonUtils.isNumber(" ") == false
     * CommonUtils.isNumber("123") == true
     * CommonUtils.isNumber("+NaN") == true
     * CommonUtils.isNumber("-Infinity") == true
     * CommonUtils.isNumber("-123.123e-2") == true
     * CommonUtils.isNumber("-123.123E-2") == true
     * </pre>
     *
     * @param str 字符串
     * @return 字符串是否代表数字
     */
    public static boolean isNumber(@Nullable String str) {
        return StringUtils.isEmpty(str) ? false : isNumCheck(str);
    }

    /**
     * 将字符串转换成{@code Number}.
     *
     * <pre>
     * CommonUtils.string2Number(null)          == null
     * CommonUtils.string2Number("")            == null
     * CommonUtils.string2Number(" ")           == null
     * CommonUtils.string2Number("+NaN")        == Double.NaN
     * CommonUtils.string2Number("-Infinity")   == Double.NEGATIVE_INFINITY
     * CommonUtils.string2Number("+001")        == 1.0d
     * CommonUtils.string2Number("+123.123e-7") == "0.0000123123"
     * </pre>
     *
     * @param str 字符串
     * @return {@code Number}
     */
    public static Number string2Number(@Nullable String str) {

        if (StringUtils.isBlank(str)) {
            return null;
        }
        str = getString(str);
        if (isStartWithZeroAndCannotbeDecimal(str)) { // alex:如果是以0||-0为始,且不是以0.||-0.为始的,那么就不是数字
            try {//0001, 0002, 这些是可以被转化为数字的, 也是在预期内的.
                return Double.parseDouble(str);
            } catch (NumberFormatException e) {
                return null;
            }
        }
        int len = str.length();
        if (isEmpty(str, len)) {
            return null;
        }
        Double v = string2SpecialNumber(str);
        if (v != null) {
            return v;
        }
        NumberDescribe describe = parseNumber(str, len);
        if (describe == null) {
            return null;
        }

        return getNumber(str, describe);
    }

    /**
     * 将字符串转换成{@code null}, {@code Integer}. {@code Double}或{@code Long}.
     *
     * <pre>
     * CommonUtils.string2IntegerOrDouble("")           == null
     * CommonUtils.string2IntegerOrDouble(" ")          == null
     * CommonUtils.string2IntegerOrDouble("123")        == new Integer("123")
     * CommonUtils.string2IntegerOrDouble("2147483648") == new Long("2147483648")
     * CommonUtils.string2IntegerOrDouble("1.0")        == new Double("1.0")
     * CommonUtils.string2Number("+123.123e-7")         == new Double("1.23123E-5")
     * CommonUtils.string2Number("-123.123E-7")         == new Double("-1.23123E-5")
     * </pre>
     *
     * @param str 字符串
     * @return {@code null}, {@code Integer}. {@code Double}或{@code Long}
     */
    @Nullable
    public static Number string2IntegerOrDouble(@NotNull String str) {

        int len = str.length();
        if (isEmpty(str, len)) {
            return null;
        }
        NumberDescribe describe = parseNumber(str, len);
        if (describe == null) {
            return null;
        }

        try {
            if (describe.isDoubleOrDecimal || describe.indexOfDot) {
                return Double.parseDouble(str);
            }

            boolean isIntegerLen = describe.len < DECIMAL_TEN || (describe.len == DECIMAL_TEN && describe.hasSymbol);
            if (isIntegerLen) {
                return Integer.valueOf(str);
            } else if (describe.len > MAX_LONG_LEN) {
                return Double.parseDouble(str);
            } else {
                return Long.parseLong(str);
            }
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 去除路径中的{@code ..}和{@code .}.
     *
     * <pre>
     * CommonUtils.getCanonicalPath(null)           == null
     * CommonUtils.getCanonicalPath("")             == ""
     * CommonUtils.getCanonicalPath(" ")            == " "
     * CommonUtils.getCanonicalPath("a/../b/./c")   == "b/c"
     * CommonUtils.getCanonicalPath("a/../../")     == IOException
     * </pre>
     *
     * @param path 包含{@code ..}和{@code .}的路径
     * @return 去除{@code ..}和{@code .}的路径
     * @throws IOException 路径中{@code ..}的数量超过了其前面目录的层级数，即化简得到的路径仍以{@code ..}开头
     */
    @Nullable
    public static String getCanonicalPath(@Nullable String path) throws IOException {

        if (StringUtils.isBlank(path)) {
            return path;
        }
        String[] paths = path.split(CoreConstants.SEPARATOR);
        Stack<String> stack = new Stack<String>();
        for (String unit : paths) {
            //双点出栈
            if (unit.equals(CoreConstants.DOUBLE_DOT)) {
                if (stack.isEmpty()) {
                    throw new IOException("Invalid path : " + path);
                }
                stack.pop();
            } else if (!unit.equals(CoreConstants.DOT)) {
                //非.的入栈
                stack.push(unit);
            }
        }
        return pathJoin(stack.toArray(new String[stack.size()]));
    }

    /**
     * 替换字符串中的特定字符串.
     *
     * <pre>
     * CommonUtils.replaceAllString(null, "a", "d")     == null
     * CommonUtils.replaceAllString("abac", "a", "d")   == "dbdc"
     * </pre>
     *
     * @param str 字符串
     * @param oldStr 被替换子串
     * @param newStr 替换子串
     * @return 替换结果
     */
    @Nullable
    public static String replaceAllString(@Nullable String str, @NotNull String oldStr, @NotNull String newStr) {

        // kunsnat: 只有null 才能返回null, 如"", " " 等判断为空的不能返回null 不然读取XML就变化了.
        if (str == null) {// james：allString也可能是NULL的
            return null;
        }
        return str.replaceAll("\\Q" + oldStr + "\\E", filterString(newStr));
    }

    /**
     * 批量替换字符串中的特定字符串.
     *
     * <pre>
     * CommonUtils.replaceAllString(null, new String[]{"a", "b"}, new String[]{"1", "2"})   = null
     * CommonUtils.replaceAllString("abab", new String[]{"a", "b"}, new String[]{"1", "2"}) = "1212"
     * </pre>
     *
     * @param str 字符串
     * @param oldStrArr 被替换子串数组
     * @param newStrArr 替换子串数组
     * @return 替换结果
     */
    @Nullable
    public static String replaceAllString(@Nullable String str, @NotNull String[] oldStrArr, @NotNull String[] newStrArr) {

        for (int i = 0; i < oldStrArr.length; i++) {
            str = replaceAllString(str, oldStrArr[i], newStrArr[i]);
        }
        return str;
    }

    /**
     * 将不带透明度的{@code Color}转换成{@code CSS}的颜色表示.
     *
     * <pre>
     * CommonUtils.javaColorToCSSColor(null)                = ""
     * CommonUtils.javaColorToCSSColor(new Color(1, 2, 3))  = "rgb(1,2,3)"
     * </pre>
     *
     * @param color {@code Color}
     * @return {@code CSS}颜色表示
     */
    @NotNull
    public static String javaColorToCSSColor(@Nullable Color color) {

        if (color == null) {
            return StringUtils.EMPTY;
        }

        StringBuilder cssBuf = new StringBuilder("rgb(");
        cssBuf.append(color.getRed());
        cssBuf.append(',');
        cssBuf.append(color.getGreen());
        cssBuf.append(',');
        cssBuf.append(color.getBlue());
        cssBuf.append(')');

        return cssBuf.toString();
    }

    /**
     * 将带透明度的{@code Color}转换成{@code CSS}的颜色表示.
     *
     * <pre>
     * CommonUtils.javaColorToCSSColor(null)                    = ""
     * CommonUtils.javaColorToCSSColor(new Color(1, 2, 3, 51))  = "rgba(1,2,3,0.2)"
     * </pre>
     *
     * @param color {@code Color}
     * @return {@code CSS}颜色表示
     */
    @Nullable
    public static String javaColor2JSColorWithAlpha(@Nullable Color color) {

        if (color == null) {
            return StringUtils.EMPTY;
        }

        StringBuffer cssBuf = new StringBuffer("rgba(");
        cssBuf.append(color.getRed());
        cssBuf.append(',');
        cssBuf.append(color.getGreen());
        cssBuf.append(',');
        cssBuf.append(color.getBlue());
        cssBuf.append(',');
        cssBuf.append(color.getAlpha() / 255.0);
        cssBuf.append(')');

        return cssBuf.toString();

    }

    /**
     * 将带透明度的{@code Color}转换成字符串.
     *
     * <pre>
     * CommonUtils.javaColor2String(new Color(51, 102, 204, 255))) = "0.2,0.4,0.8,1.0,"
     * </pre>
     *
     * @param color {@code Color}
     * @return 字符串
     */
    @NotNull
    public static String javaColor2String(@NotNull Color color) {

        StringBuffer buffer = new StringBuffer();
        buffer.append(color.getRed() / 255.0);
        buffer.append(Constants.DELIMITER_NORMAL);
        buffer.append(color.getGreen() / 255.0);
        buffer.append(Constants.DELIMITER_NORMAL);
        buffer.append(color.getBlue() / 255.0);
        buffer.append(Constants.DELIMITER_NORMAL);
        buffer.append(color.getAlpha() / 255.0);
        buffer.append(Constants.DELIMITER_NORMAL);
        return buffer.toString();
    }

    /**
     * 根据图像，现有布局和背景宽高返回新布局，方便展示.
     *
     * <p>
     * 当图像非{@code null}且现有布局为居中布局{@link Constants#IMAGE_CENTER}时
     * 若图像宽或高大于背景宽或高，返回平铺布局{@link Constants#IMAGE_TILED}，否则返回原布局
     * </p>
     *
     * @param image 图像
     * @param imageLayout 图像布局
     * @param width 背景宽度
     * @param height 背景高度
     * @return 新布局
     */
    public static int changeImageLayout4Draw(@Nullable Image image, int imageLayout, int width, int height) {

        //这边改变的layout只是为了方便展示,不会保存进xml的
        if (image != null && imageLayout == Constants.IMAGE_CENTER) {
            if (image.getWidth(null) > width || image.getHeight(null) > height) {
                imageLayout = Constants.IMAGE_TILED;
            }
        }
        return imageLayout;
    }

    /**
     * 把字符串中的转义字符转换为实际字符.
     *
     * <p>
     * CommonUtils.readSpecialString(null)      == ""
     * CommonUtils.readSpecialString("\\n")     == String.valueOf((char) 10)
     * CommonUtils.readSpecialString("\\r")     == String.valueOf((char) 13)
     * CommonUtils.readSpecialString("\\t")     == String.valueOf((char) 9)
     * CommonUtils.readSpecialString("\\b")     == String.valueOf((char) 8)
     * CommonUtils.readSpecialString("\\f")     == String.valueOf((char) 12)
     * CommonUtils.readSpecialString("\\\'")    == String.valueOf((char) 39)
     * CommonUtils.readSpecialString("\\\"")    == String.valueOf((char) 34)
     * CommonUtils.readSpecialString("\\\\")    == String.valueOf((char) 92)
     * CommonUtils.readSpecialString("\\1")     == String.valueOf((char) 1)
     * CommonUtils.readSpecialString("\\12")    == String.valueOf((char) 10)
     * CommonUtils.readSpecialString("\\123")   == String.valueOf((char) 83)
     * </p>
     *
     * @param str 字符串
     * @return 去转义后字符串
     * @throws Exception 无法解析转义字符
     */
    @NotNull
    public static String readSpecialString(@Nullable String str) throws Exception {

        if (str == null) {
            return StringUtils.EMPTY;
        }

        /*
         * alex:解决BUG0001421,原来就是简单地用string.split("\\")再每一块进行解析
         * 会导致连续的\\\\\\\无法解析
         */

        Pattern pattern = Pattern.compile("\\\\"); // alex:单斜杠的Pattern要写四个斜杠...

        Matcher matcher = pattern.matcher(str);
        int start = 0, end = 0;
        StringBuffer sb = new StringBuffer();
        while (matcher.find(start)) {
            start = matcher.start();
            sb.append(str.substring(end, start)); //把start前面一段写出去

            end = matcher.end();
            String tailString = str.substring(end);

            if (tailString.length() == 0) {
                throw new Exception('\\' + " can't be parsed.");
            }

            end = endOffsetByFirstChar(str, end, sb, tailString);

            start = end;
        }

        sb.append(str.substring(end));

        return sb.toString();
    }

    /**
     * 将字符串中的未转义字符转换为转义字符.
     *
     * <pre>
     * CommonUtils.writeSpecialString(String.valueOf(null)          == ""
     * CommonUtils.writeSpecialString(String.valueOf((char) 10))    == "\\n"
     * CommonUtils.writeSpecialString(String.valueOf((char) 13))    == "\\r"
     * CommonUtils.writeSpecialString(String.valueOf((char)  9))    == "\\t"
     * CommonUtils.writeSpecialString(String.valueOf((char)  8))    == "\\b"
     * CommonUtils.writeSpecialString(String.valueOf((char) 10))    == "\\n"
     * CommonUtils.writeSpecialString(String.valueOf((char) 12))    == "\\f"
     * CommonUtils.writeSpecialString(String.valueOf((char) 39))    == "\\\'"
     * CommonUtils.writeSpecialString(String.valueOf((char) 34))    == "\\\""
     * CommonUtils.writeSpecialString(String.valueOf((char) 92))    == "\\\\"
     * CommonUtils.writeSpecialString("a")                          == "a"
     * </pre>
     *
     * @param str 字符串
     * @return 转义后字符串
     */
    @NotNull
    public static String writeSpecialString(@Nullable String str) {

        if (str == null) {
            return StringUtils.EMPTY;
        }

        StringBuffer sb = new StringBuffer();
        char c;
        for (int i = 0, len = str.length(); i < len; i++) {
            switch (c = str.charAt(i)) {
                case '\n':
                    sb.append("\\n");
                    break;
                case '\\':
                    sb.append("\\\\");
                    break;
                case '\t':
                    sb.append("\\t");
                    break;
                case '\r':
                    sb.append("\\r");
                    break;
                case '\b':
                    sb.append("\\b");
                    break;
                case '\f':
                    sb.append("\\f");
                    break;
                case '\'':
                    sb.append("\\\'");
                    break;
                case '\"':
                    sb.append("\\\"");
                    break;
                default:
                    sb.append(c);
            }
        }

        return sb.toString();
    }

    /**
     * 获取类及其父类的特定方法.
     *
     * <pre>
     * CommonUtils.getDeclaredMethod(String.class, "get", new Class[]{})                                    == null
     * CommonUtils.getDeclaredMethod(String.class, "toString", new Class[]{})                               == String#toString()
     * CommonUtils.getDeclaredMethod(LinkedHashMap.class, "put", new Class[]{Object.class, Object.class})   == HashMap#put(java.lang.Object, java.lang.Object)
     * </pre>
     *
     * @param cls 类
     * @param name 方法名称
     * @param parameterTypes 方法参数类型列表
     * @return {@code Method} 或{@code null}
     */
    @Nullable
    public static Method getDeclaredMethod(@NotNull Class cls, @NotNull String name, @NotNull Class[] parameterTypes) {

        try {
            return cls.getDeclaredMethod(name, parameterTypes);
        } catch (NoSuchMethodException noSuchMethodException) {
            cls = cls.getSuperclass();
            if (cls != null) {
                return getDeclaredMethod(cls, name, parameterTypes);
            }
        }
        return null;
    }

    /**
     * 将路径节点数组用路径分隔符"/"连接起来.
     *
     * @param nodes 路径节点数组
     * @return 连接后的路径
     */
    public static String pathJoin(String... nodes) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0, len = nodes == null ? 0 : nodes.length; i < len; i++) {
            String node = nodes[i];

            if (node == null) {
                node = StringUtils.EMPTY;
            }
            if (CoreConstants.SEPARATOR.equals(node) || "\\".equals(node)) {
                continue;
            }
            // 如果i > 0,那么sb的最后一位必定是File.separatorChar,所以如果node以分隔符为起始,这个分隔符不要加进去
            if (i > 0) {
                if (node.startsWith(CoreConstants.SEPARATOR) || node.startsWith("\\")) {
                    node = node.substring(1);
                }
            }
            sb.append(node);

            if (i + 1 < len) {
                if (!(node.endsWith(CoreConstants.SEPARATOR) || node.endsWith("\\"))) {
                    //p:这个地方按照alex改进的，因为有的地方返回"\"会出错.
                    //劳驾这边不要再改成File.SEPARATOR了, 会出一系列问题, 已知的有:超链失效, 子报表, 文件夹路径url访问, weblogic部署
                    sb.append(CoreConstants.SEPARATOR);
                }
            }
        }

        return sb.toString();
    }

    /**
     * 按照指定的分隔符分割字符串.
     *
     * @param string 字符串
     * @param delimiter 分隔符
     * @return 分割结果
     */
    @NotNull
    public static String[] splitString(@Nullable String string, @NotNull String delimiter) {

        if (string == null || string.length() == 0) {
            return new String[]{};
        }

        return string.split("\\Q" + delimiter + "\\E");
    }

    /**
     * 按照指定的分隔符分割字符串.
     *
     * @param string 字符串
     * @param delimiter 分隔符
     * @return 分割结果
     */
    @NotNull
    public static String[] splitString(@Nullable String string, char delimiter) {
        return splitString(string, StringUtils.EMPTY + delimiter);
    }

    /**
     * 将除部分ASCII字符外的字符替换为&amp;#格式的Unicode表示，主要为了解决XSS注入问题.
     *
     * <p>
     * 以下字符不会被替换：
     * 小写字母，大写字母，空格，阿拉伯数字，英文句号，英文逗号，减号，下划线。
     * </p>
     *
     * <pre>
     * CommonUtils.replaceScript4Xss(null)                      == ""
     * CommonUtils.replaceScript4Xss("")                        == ""
     * CommonUtils.replaceScript4Xss("abcABC 123,.-_&lt;/帆软&gt;")     == "abcABC 123,.-_&#60;&#47;&amp;#24070;&amp;#36719;&#62;"
     * </pre>
     *
     * @param message 要替换的字符串
     * @return 替换结果
     */
    public static String replaceScript4Xss(String message) {

        if (StringUtils.isEmpty(message)) {
            return StringUtils.EMPTY;
        }

        StringBuffer builder = new StringBuffer(message.length() * 2);
        CharacterIterator it = new StringCharacterIterator(message);
        for (char ch = it.first(); ch != CharacterIterator.DONE; ch = it.next()) {
            // 小写字母
            if ((((ch > '`') && (ch < '{'))
                    // 大写字母
                    || ((ch > '@') && (ch < '[')))
                    // 空格
                    || (((ch == ' ')
                    // 阿拉伯数字
                    || ((ch > '/') && (ch < ':')))
                    // 英文句号
                    || (((ch == '.')
                    // 英文逗号
                    || (ch == ','))
                    // 减号
                    || ((ch == '-')
                    // 下划线
                    || (ch == '_'))))) {
                builder.append(ch);
            } else {
                builder.append("&#").append((int) ch).append(";");
            }
        }
        return builder.toString();
    }

    /**
     * 当前是否为DEBUG模式.
     *
     * <p>
     * 检查{@code System Property}中是否有{@code debug}这一项且为{@code true}.
     * 恒返回第一次检查结果.
     * </p>
     *
     * @return 是否为DEBUG模式
     */
    public static boolean isDebug() {
        return isDebugValue.getValue();
    }

    /**
     * 通过对应类的{@code getInstance()}方法获取其单例.
     *
     * <p>
     * 若对应类不存在{@code getInstance()}则返回{@code null}.
     * </p>
     *
     * @param clazz 类
     * @param <T>   类类型
     * @return 单例
     */
    @SuppressWarnings("unchecked")
    @Nullable
    public static <T> T getInstance(@NotNull Class<? extends T> clazz) {

        try {
            Method method = clazz.getDeclaredMethod("getInstance");
            method.setAccessible(true);
            return (T) method.invoke(clazz);
        } catch (Throwable e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 执行对应类的{@code static clear()}方法.
     *
     * <p>
     * 忽略执行过程中的所有{@code Throwable}.
     * </p>
     *
     * @param clazz 类
     */
    @SuppressWarnings("unchecked")
    public static void clearInstance(@NotNull Class clazz) {
        try {
            Method method = clazz.getDeclaredMethod("clear");
            method.setAccessible(true);
            method.invoke(clazz);
        } catch (Throwable ignored) {
        }
    }

    /**
     * 根据参数列表，调用对应类的特定构造方法.
     *
     * <p>
     * 不支持传入的参数列表为{@code null}，将返回{@code null}.
     * 若不存在与参数列表对应的构造方法，返回{@code null}.
     * 构造方法中抛出的异常会被忽略，并返回{@code null}.
     * </p>
     *
     * <pre>
     * Integer i = CommonUtils.construct(Integer.class, 1);
     * </pre>
     *
     * @param clazz 类
     * @param parameters 构造方法参数
     * @param <T> 类类型
     * @return 类的实例
     */
    @SuppressWarnings("unchecked")
    @Nullable
    public static <T> T construct(@NotNull Class<? extends T> clazz, Object... parameters) {

        try {
            if (parameters == null) {
                //暂不支持
                return null;
            }
            Class<?>[] c = new Class[parameters.length];
            for (int i = 0; i < parameters.length; i++) {
                c[i] = parameters[i].getClass();
            }
            Constructor[] constructors = clazz.getDeclaredConstructors();
            for (int i = 0; i < constructors.length; i++) {
                Constructor constructor = constructors[i];
                Class<?>[] pt = constructor.getParameterTypes();
                if (classesTypeEquals(pt, c)) {
                    constructor.setAccessible(true);
                    return (T) constructor.newInstance(parameters);
                }
            }
        } catch (Throwable ignored) {
        }
        return null;
    }


    /**
     * 删除文件，支持删除目录.
     *
     * <p>
     * 若文件不存在或为{@code null}，返回true.
     * 若文件为目录，一并递归删除其所有子目录/文件.
     * 若删除失败会在应用退出时尝试删除且将失败的文件路径记录下来.
     * 记录文件位置说明:
     * 若添加了启动参数{@link CommonUtils#RECORD_FILE_PARAM_KEY}，则使用启动参数表明的文件路径，否则使用默认路径.
     * 存在WEB-INF目录时，默认路径为WEB-INF/assist/，否则使用临时目录做父目录.
     * 默认记录文件名为{@link CommonUtils#DEFAULT_RECORD_FILE}.
     * </p>
     *
     * @param file 要删除的文件
     * @return 是否删除成功
     */
    public static boolean deleteFile(@Nullable File file) {

        if ((file == null) || (!file.exists())) {
            return true;
        }

        boolean delete_success = true;

        if (file.isDirectory()) {
            File[] files = file.listFiles();

            if (files != null) {
                for (File value : files) {
                    if (!deleteFile(value) && delete_success) {
                        delete_success = false;
                    }
                }
            }
        }
        boolean result = file.delete();
        if (!result) {
            //删除失败则退出时删除，并且记录下次启动时再尝试删除
            file.deleteOnExit();
            registerFileToDelete(file.getAbsolutePath());
        }
        delete_success = result && delete_success;

        return delete_success;
    }

    /**
     * 记录要删除的文件路径.
     *
     * @param path 文件路径
     */
    public static void registerFileToDelete(String path) {
        if (StringUtils.isEmpty(path)) {
            return;
        }
        String recordFile = getRecordFile();
        try {
            makeSureFileExist(new File(recordFile));
        } catch (IOException e) {
            FineLoggerFactory.getLogger().warn("Creat record file {} failed, please check the project permissions!", recordFile);
            FineLoggerFactory.getLogger().warn(e.getMessage(), e);
        }
        try (BufferedWriter filesToDelete = new BufferedWriter(new FileWriter(recordFile, true))) {
            filesToDelete.write(path);
            filesToDelete.newLine();
            filesToDelete.flush();
        } catch (IOException e) {
            FineLoggerFactory.getLogger().warn("file {} add to the delete list exception, please delete it manually!", path);
            FineLoggerFactory.getLogger().warn(e.getMessage(), e);
        }
    }

    /**
     * 获取记录文件路径.
     *
     * <p>
     * 若添加了启动参数，则使用启动参数表明的文件路径，否则使用默认路径.
     * 存在WEB-INF目录时，默认父目录为WEB-INF/assist/，否则使用临时目录做父目录
     * </p>
     *
     * @return 记录文件路径
     */
    private static String getRecordFile() {
        String currentPath = null;
        try {
            currentPath = URLDecoder.decode(CommonUtils.class.getProtectionDomain().getCodeSource().getLocation().getPath(), RESOURCE_ENCODER);
        } catch (UnsupportedEncodingException ignore) {
        }
        String filePath;
        if (StringUtils.isNotEmpty(currentPath) && currentPath.contains(WEB_INF)) {
            filePath = currentPath.substring(0, currentPath.indexOf(WEB_INF)) + WEB_INF + File.separator + RECORD_PARENT_DIR + File.separator + DEFAULT_RECORD_FILE;
        } else {
            filePath = System.getProperty("java.io.tmpdir") + File.separator + DEFAULT_RECORD_FILE;
        }
        String fileDeletePath = System.getProperty(RECORD_FILE_PARAM_KEY);
        return StringUtils.isEmpty(fileDeletePath) ? filePath : fileDeletePath;
    }

    /**
     * 启动时调用，尝试删除记录的上次删除失败的文件.
     *
     * <p>
     * 记录文件位置说明:
     * 若添加了启动参数{@link CommonUtils#RECORD_FILE_PARAM_KEY}，则使用启动参数表明的文件路径，否则使用默认路径.
     * 存在WEB-INF目录时，默认路径为WEB-INF/assist/，否则使用临时目录做父目录.
     * 默认记录文件名为{@link CommonUtils#DEFAULT_RECORD_FILE}.
     * </p>
     *
     */
    public static void deleteRegisteredFiles() {
        File file = new File(getRecordFile());
        if (file.exists()) {
            boolean readSuccess = true;
            ArrayList<String> list = new ArrayList<>();
            try (BufferedReader filesToDelete = new BufferedReader(new FileReader(file))){
                String path;
                while ((path = filesToDelete.readLine()) != null) {
                    list.add(path);
                }
            } catch (IOException ignore) {
                readSuccess = false;
            }
            if (readSuccess) {
                //删除记录的文件本身
                deleteFile(file);
            }
            list.forEach(filePath -> deleteFile(new File(filePath)));
        }
    }

    /**
     * 创建目录.
     *
     * <p>
     * 若传入的文件为{@code null}，返回{@code false}.
     * 若传入的文件非{@code null}，且文件已经存在或创建成功，返回{@code true}.
     * </p>
     *
     * @param file 要创建的文件
     * @return 是否创建成功
     */
    public static boolean mkdirs(@Nullable File file) {
        return file != null && (file.exists() || file.mkdirs());
    }

    /**
     * 确保文件存在.
     *
     * <p>
     * 若传入的文件为{@code null}，返回{@code false}.
     * 若传入的文件非{@code null}且存在，返回{@code true}.
     * 若传入的文件非{@code null}且不存在，创建该文件并返回{@code true}.
     * </p>
     *
     * @param file 文件
     * @return 文件是否存在
     * @throws IOException 创建文件失败
     */
    public static boolean makeSureFileExist(@Nullable File file) throws IOException {

        if (file == null) {
            return false;
        }

        if (file.exists()) {
            return true;
        }

        mkdirs(file.getParentFile());
        file.createNewFile();

        return true;
    }

    /**
     * 创建文件，若文件已存在，使用不同文件名创建.
     *
     * <p>
     * 若文件不存在，直接创建.
     * 若文件已存在，则创建"文件名(1).文件后缀".若也已存在，递增序号直至创建成功.
     * </p>
     *
     * @param path 目录
     * @param name 文件名
     * @return 创建完毕的文件
     * @throws IOException 创建文件失败
     */
    @NotNull
    public static File createDistinctFile(@Nullable String path, @NotNull String name) throws IOException {

        int i = 0;
        File file = new File(path, name);
        while (file.exists()) {
            i++;
            file = new File(path, getFileNameWithOutPostfix(name) + "(" + i + ")" + getFileNamePostfix(name));
        }
        mkdirs(file.getParentFile());
        file.createNewFile();
        return file;
    }

    /**
     * 获取文件名（去掉后缀）.
     *
     * <pre>
     * CommonUtils.getFileNameWithOutPostfix(null)  == null
     * CommonUtils.getFileNameWithOutPostfix("")    == ""
     * CommonUtils.getFileNameWithOutPostfix("a")   == "a"
     * CommonUtils.getFileNameWithOutPostfix("a.b") == "a"
     * </pre>
     *
     * @param name 文件名
     * @return 文件名（不含后缀）
     */
    @Nullable
    public static String getFileNameWithOutPostfix(@Nullable String name) {

        if (StringUtils.isNotEmpty(name)) {
            int index = name.lastIndexOf(".");
            if (index != -1) {
                return name.substring(0, index);
            }
        }
        return name;
    }

    /**
     * 获取文件名的后缀.
     *
     * <pre>
     * CommonUtils.getFileNamePostfix(null)     == ""
     * CommonUtils.getFileNamePostfix("")       == ""
     * CommonUtils.getFileNamePostfix("a")      == ""
     * CommonUtils.getFileNamePostfix("a.b")    == ".b"
     * </pre>
     *
     * @param name 文件名
     * @return 文件名（不含后缀）
     */
    @NotNull
    public static String getFileNamePostfix(@Nullable String name) {

        if (StringUtils.isNotEmpty(name)) {
            int index = name.lastIndexOf(".");
            if (index != -1) {
                return name.substring(index);
            }
        }
        return StringUtils.EMPTY;
    }

    /**
     * 将{@code List}映射为新的{@code List}.
     *
     * <p>
     * 如将{@code List<String>}映射为所有{@code String}长度的{@code List}.
     * </p>
     *
     * @param list 列表
     * @param mapper 映射关系
     * @return 新列表
     * @see Mapper
     * @deprecated 换用 {@code Stream#map(java.util.function.Function)}
     */
    @Deprecated
    @NotNull
    public static java.util.List map(@NotNull java.util.List list, @NotNull Mapper mapper) {

        int len = list.size();
        Object[] res = new Object[len];

        for (int i = 0; i < len; i++) {
            res[i] = mapper.map(i, list.get(i), list);
        }

        return Arrays.asList(res);
    }

    /**
     * 判断字符串是否可能为公式.
     *
     * <p>
     * 若字符串非空白字符串且以{@code =}开头则返回{@code true}.
     * </p>
     *
     * @param str 要判断的字符串
     * @return 是否可能为公式
     */
    public static boolean maybeFormula(String str) {
        return StringUtils.isNotBlank(str) && str.startsWith("=");
    }

    /**
     * 获取当前JAVA版本.
     *
     * <p>
     * 返回6, 7, 8...
     * </p>
     *
     * @return 当前JAVA版本
     */
    public static int getMajorJavaVersion() {

        return MAJOR_JAVA_VERSION;
    }

    /**
     * 反射调用类上的方法.
     *
     * <p>
     * 类加载器为{@code null}，使用{@code Class.forName()}.
     * 类名或方法名为{@code null}或{@code ""}，返回{@code null}.
     * 方法不存在，返回{@code null}.
     * </p>
     *
     * <pre>
     * CommonUtils.invokeMethod(null, "java.lang.String", "valueOf", new Class[]{int.class}, new Object[]{1}) == "1"
     * </pre>
     *
     * @param loader 类加载器
     * @param className 类名
     * @param methodName 方法名
     * @param argsType 参数列表参数类型
     * @param args 参数列表
     * @return 方法返回值
     */
    @SuppressWarnings("unchecked")
    @Nullable
    public static Object invokeMethod(@Nullable ClassLoader loader, @Nullable String className, @Nullable String methodName, Class[] argsType, Object[] args) {

        if (StringUtils.isEmpty(className) || StringUtils.isEmpty(methodName)) {
            return null;
        }

        try {
            Class c = loadClass(loader, className);
            Method m = c.getMethod(methodName, argsType);
            m.setAccessible(true);
            return m.invoke(c, args);
        } catch (Exception ignored) {
        }

        return null;
    }

    /**
     * 反射调用类上的方法.
     *
     * <p>
     * 内部使用{@code Class.forName()}加载类.
     * 类名或方法名为{@code null}或{@code ""}，返回{@code null}.
     * 方法不存在，返回{@code null}.
     * </p>
     *
     * <pre>
     * CommonUtils.invokeMethod("java.lang.String", "valueOf", new Class[]{int.class}, new Object[]{1}) == "1"
     * </pre>
     *
     * @param className 类名
     * @param methodName 方法名
     * @param argsType 参数列表参数类型
     * @param args 参数列表
     * @return 方法返回值
     */
    @Nullable
    public static Object invokeMethod(@Nullable String className, @Nullable String methodName, Class[] argsType, Object[] args) {
        return invokeMethod(null, className, methodName, argsType, args);
    }

    /**
     * 反射调用类上的无参方法.
     *
     * <p>
     * 类加载器为{@code null}，使用{@code Class.forName()}.
     * 类名或方法名为{@code null}或{@code ""}，返回{@code null}.
     * 方法不存在，返回{@code null}.
     * </p>
     *
     * <pre>
     * CommonUtils.invokeMethod(null, "java.util.Collections", "emptyMap")
     * </pre>
     *
     * @param loader 类加载器
     * @param className 类名
     * @param methodName 方法名
     * @return 方法返回值
     */
    @Nullable
    public static Object invokeMethod(@Nullable ClassLoader loader, @Nullable String className, @Nullable String methodName) {
        return CommonUtils.invokeMethod(loader, className, methodName, new Class[0], new Object[0]);
    }

    /**
     * 反射调用类上的无参方法.
     *
     * <p>
     * 内部使用{@code Class.forName()}加载类.
     * 类名或方法名为{@code null}或{@code ""}，返回{@code null}.
     * 方法不存在，返回{@code null}.
     * </p>
     *
     * <pre>
     * CommonUtils.invokeMethod("java.util.Collections", "emptyMap")
     * </pre>
     *
     * @param className 类名
     * @param methodName 方法名
     * @return 方法返回值
     */
    @Nullable
    public static Object invokeMethod(@Nullable String className, @Nullable String methodName) {
        return CommonUtils.invokeMethod(className, methodName, new Class[0], new Object[0]);
    }

    /**
     * 反射调用对象上的方法.
     *
     * <p>
     * 可调用私有方法.
     * 方法返回值为{@code void}，返回{@code null}.
     * </p>
     *
     * <pre>
     * CommonUtils.invokeMethod("abc", "toString")
     * </pre>
     *
     * @param object 对象
     * @param methodName 方法名
     * @param parameters 参数
     * @return 方法返回值
     * @throws NoSuchMethodException NoSuchMethodException
     * @throws InvocationTargetException InvocationTargetException
     * @throws IllegalAccessException IllegalAccessException
     */
    @Nullable
    public static Object invokeMethod(@NotNull Object object, @NotNull String methodName, Object... parameters) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {

        Class<?>[] argsType = new Class<?>[parameters.length];
        for (int i = 0; i < parameters.length; i++) {
            argsType[i] = parameters[i].getClass();
        }
        //private方法也能调，多用于单元测试
        Method m = object.getClass().getDeclaredMethod(methodName, argsType);
        m.setAccessible(true);
        return m.invoke(object, parameters);
    }

    /**
     * 获取文件类型扩展名
     *
     * @param bytes 字节数组
     * @return 扩展名
     * @deprecated  使用FileTypeInspector.resolveFileType
     */
    @Deprecated
    public static String getFileTypeFromBytes(byte[] bytes) {
        String suffix = FileTypeInspector.resolveFileType(bytes);
        return StringUtils.isEmpty(suffix) ? ".txt" : DOT + suffix;
    }

    /**
     * 使用特定编码读取流中的字符串.
     *
     * <p>
     * 会在末尾添加{@code \n}.
     * 若输入流为{@code null}或流中无内容，返回{@code ""}.
     * </p>
     *
     * @param in 输入流
     * @param charset 编码
     * @return 流中的字符串
     * @throws IOException 发生I/O错误
     */
    @NotNull
    public static String inputStream2String(@Nullable InputStream in, @NotNull String charset) throws IOException {

        if (in == null) {
            return StringUtils.EMPTY;
        }

        BufferedReader br = new BufferedReader(new InputStreamReader(in, charset));
        StringBuffer buffer = new StringBuffer();
        String line;
        while ((line = br.readLine()) != null) {
            buffer.append(line);
            buffer.append("\n");
        }
        return buffer.toString();
    }

    /**
     * 克隆指定对象.
     *
     * <p>
     * 若对象是{@link FCloneable}，调用{@link FCloneable#clone()}.
     * 若对象是{@code Cloneable}，反射调用{@code clone()}.
     * 若反射调用过程中抛出异常，或对象不是{@code Cloneable}，返回对象本身.
     * </p>
     *
     * @param object 要被克隆的对象
     * @return 克隆产生的对象
     * @throws CloneNotSupportedException 克隆失败
     * @see #strictClone(java.lang.Object)
     */
    @NotNull
    @Deprecated
    public static Object cloneObject(@NotNull Object object) throws CloneNotSupportedException {

        if (object instanceof FCloneable) {
            return ((FCloneable) object).clone();
        } else if (object instanceof Cloneable) {
            //james：反射方式
            Method clipMethod = CommonUtils.getDeclaredMethod(object.getClass(),
                    "clone", new Class[]{});
            clipMethod.setAccessible(true);

            try {
                return clipMethod.invoke(object);
            } catch (Exception e) {
            }
        }

        return object;
    }

    /**
     * 克隆指定对象.
     *
     * <p>
     * 若对象是{@link FCloneable}，调用{@link FCloneable#clone()}.
     * 若对象是{@code Cloneable}，反射调用{@code clone()}.
     * 若反射调用过程中抛出异常，或对象不是{@code Cloneable}，抛出异常.
     * 是深拷贝还是浅拷贝由对象本身克隆方法的实现决定.
     * </p>
     *
     * @param object 要被克隆的对象
     * @return 克隆产生的对象
     * @throws CloneNotSupportedException 克隆失败
     */
    @NotNull
    public static Object strictClone(@NotNull Object object) throws CloneNotSupportedException {

        if (object instanceof FCloneable) {
            return ((FCloneable) object).clone();
        } else if (object instanceof Cloneable) {
            //james：反射方式
            Method clipMethod = CommonUtils.getDeclaredMethod(object.getClass(),
                    "clone", new Class[]{});
            clipMethod.setAccessible(true);

            try {
                return clipMethod.invoke(object);
            } catch (Exception e) {
            }
        }

        throw new CloneNotSupportedException();
    }


    /**
     * 克隆{@code HashSet<String>}.
     *
     * <p>
     * 传入{@code null}，返回{@code null}.
     * 直接将原{@code HashSet}中的{@code String}添加到了新{@code HashSet}中.
     * </p>
     *
     * @param oldSet 被克隆的{@code HashSet<String>}
     * @return 克隆出来的 {@code HashSet<String>}
     */
    public static HashSet<String> cloneHashSet(HashSet<String> oldSet) {
        if (oldSet == null) {
            return null;
        }
        HashSet<String> newInvisibleSet = new HashSet<String>();
        for (String anOldSet : oldSet) {
            newInvisibleSet.add(anOldSet);
        }
        return newInvisibleSet;
    }

    /**
     * 克隆{@code HashMap<String>}.
     *
     * <p>
     * 传入{@code null}，返回{@code null}.
     * 直接将原{@code HashMap}中的{@code key}和{@code value}添加到了新{@code HashMap}中.
     * </p>
     *
     * @param oldMap 被克隆的{@code HashMap<String>}
     * @return 克隆出来的 {@code HashMap<String>}
     */
    public static HashMap<String, Object> cloneHashMap(HashMap<String, Object> oldMap) {

        if (oldMap == null) {
            return null;
        }

        HashMap<String, Object> newMap = new HashMap<String, Object>();
        for (Entry<String, Object> entry : oldMap.entrySet()) {
            newMap.put(entry.getKey(), entry.getValue());
        }

        return newMap;
    }

    /**
     * 判断对象是否是数组.
     *
     * <p>
     * 若传入{@code null}，返回false;
     * </p>
     *
     * @param obj 要判断的对象
     * @return 是否是数组
     */
    public static boolean isArray(Object obj) {
        return obj != null && obj.getClass().isArray();
    }

    /**
     * 判断对象是否为{@code null}.
     *
     * <p>
     * 对象为{@code null}或{@link Primitive#NULL}时返回{@code true}.
     * </p>
     *
     * @param value 要判断的对象
     * @return 是否为 {@code null}
     */
    public static boolean isNull(@Nullable Object value) {
        return value == null || isOtherNull(value);
    }

    /**
     * 设置属性的值，包括私有属性和父类中的属性.
     *
     * <p>
     * 向上追溯直到{@code Object}.
     * 若无此属性，抛出{@code NoSuchFieldException}.
     * </p>
     *
     * @param obj 目标对象
     * @param fieldName 属性名
     * @param value 属性值
     * @throws Exception NoSuchFieldException
     */
    public static void setPrivateFieldValue(@NotNull Object obj, @NotNull String fieldName, @NotNull Object value) throws Exception {

        Class clazz = obj.getClass();

        while (!clazz.equals(Object.class)) {
            try {
                Field field = clazz.getDeclaredField(fieldName);
                field.setAccessible(true);
                field.set(obj, value);
            } catch (NoSuchFieldException e) {
                clazz = clazz.getSuperclass();
                if (clazz.equals(Object.class)) {
                    throw e;
                }
                continue;
            }
            break;
        }
    }

    /**
     * 获取属性的值，包括私有属性和父类中的属性.
     *
     * <p>
     * 向上追溯直到{@code Object}.
     * 若无此属性，抛出{@code NoSuchFieldException}.
     * </p>
     *
     * @param obj 目标对象
     * @param fieldName 属性名
     * @return 值
     * @throws Exception NoSuchFieldException
     */
    @Nullable
    public static Object getPrivateFieldValue(@NotNull Object obj, @NotNull String fieldName) throws Exception {

        Class clazz = obj.getClass();

        Object value = null;
        while (!clazz.equals(Object.class)) {
            try {
                Field field = clazz.getDeclaredField(fieldName);
                field.setAccessible(true);
                value = field.get(obj);
            } catch (NoSuchFieldException e) {
                clazz = clazz.getSuperclass();
                if (clazz.equals(Object.class)) {
                    throw e;
                }
                continue;
            }
            break;
        }
        return value;
    }


    /**
     * 判断一个对象是否可以转化为公式.
     *
     * <p>
     * 当对象非{@code null}，且{@code toString}并{@code trim}后由{@code =}开头时返回{@code true}
     * </p>
     *
     * @param object 要判断的字符串
     * @return 是否可以转化为公式
     */
    public static boolean canBeFormula(@Nullable Object object) {
        return object != null && object.toString().trim().startsWith("=");
    }

    /**
     * 获取指定长度的随机字符串, 区分大小写.
     *
     * @param length 长度
     * @return 字符串
     */
    public static String getRandomStr(int length) {

        char[] chr = new char[length];
        for (int i = 0; i < length; i++) {
            int num = random.nextInt(52);
            chr[i] = DIGITS_CASE_SENSITIVE[num];
        }
        return new String(chr);
    }

    /**
     * 根据字符串生成指定的{@code Locale}对象.
     *
     * <p>
     * 传入{@code null}，{@code ""}或{@code " "}，返回{@code null}
     * </p>
     *
     * <pre>
     * CommonUtils.createLocale("zh")       == new Locale("zh")
     * CommonUtils.createLocale("zh_CN")    == Locale.CHINA
     * </pre>
     *
     * @param localeStr 表示语言和国家的字符串
     * @return {@code Locale}
     */
    public static Locale createLocale(@Nullable String localeStr) {

        Locale locale = null;
        if (StringUtils.isNotBlank(localeStr)) {
            String[] arr = localeStr.split("_");
            if (arr.length > 1) {
                locale = new Locale(arr[0], arr[1]);
            } else {
                locale = new Locale(arr[0], StringUtils.EMPTY);
            }
        }
        return locale;
    }

    /**
     * 判读指定的类是否是另一个类的子类.
     *
     * <p>
     * 若传入的子类或父类有{@code null}.
     * </p>
     *
     * @param childClass 子类
     * @param parentClass 父类
     * @return 是否是继承关系
     */
    public static boolean classInstanceOf(@Nullable Class childClass, @Nullable Class parentClass) {

        if (childClass == null || parentClass == null) {
            return false;
        }

        return parentClass.isAssignableFrom(childClass);
    }


    /**
     * 判读指定对象的类是否是特定类的子类.
     *
     * <p>
     * 传入的对象和类有一个为{@code null}则返回{@code null}.
     * </p>
     *
     * @param object 对象
     * @param clazz 类
     * @return 如果指定类是另一个类的子类则返回true，否则返回false
     */
    public static boolean objectInstanceOf(Object object, Class clazz) {

        if (object == null || clazz == null) {
            return false;
        }
        return clazz.isInstance(object);
    }


    /**
     * 窄化接口，用于权限控制.
     *
     * <p>
     * 传入的只允许访问的接口为{@code null}会导致{@code AssertionError}.
     * 原对象为{@code null}会返回{@code null}.
     * </p>
     *
     * @param originalObject 实现了多个接口的对象
     * @param narrowing 只允许访问的接口
     * @param <T> 只允许访问的接口类型
     * @param <K> 对象类型，需要实现接口{@code T}
     * @return 代理对象，这个对象只实现了接口{@code T}
     */
    @SuppressWarnings("unchecked")
    @Nullable
    public static <T, K extends T> T narrowInterface(@Nullable final K originalObject, @Nullable Class<T> narrowing) {

        assert narrowing != null;

        if (originalObject == null) {
            return null;
        }

        return (T) Proxy.newProxyInstance(originalObject.getClass().getClassLoader(),
            new Class[]{narrowing}, new InvocationHandler() {
                private Object object = originalObject;
                @Override
                public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
                    if (object == null) {
                        return null;
                    }
                    return method.invoke(object, args);
                }
            });
    }

    /**
     * 获取类名.
     *
     * <p>
     * 传入{@code null}会导致{@code AssertionError}
     * </p>
     *
     * @param clazz 类
     * @return 类名
     */
    public static String classNameAsMark(Class<?> clazz) {

        assert clazz != null;
        String name = clazz.getSimpleName();
        if (StringUtils.isNotBlank(name)) {
            return name;
        }
        return clazz.getName();
    }

    /**
     * 将路径中的{@code \\}替换成{@code "/"}.
     *
     * <p>
     * 若传入的路径为{@code ""}或{@code null}则返回路径本身.
     * </p>
     *
     * @param path 路径
     * @return 替换结果
     */
    @Nullable
    public static String pathTransSlash(@Nullable String path) {

        if (StringUtils.isEmpty(path)) {
            return path;
        }
        return path.replaceAll("\\\\", "/");
    }

    /**
     * 获取签名.
     *
     * @return 签名
     */
    public static long signature() {
        return Reflect.on("com.fr.protect.SignatureGenerator").call("signature").get();
    }

    /**
     * 判断两个{@code double}是否相等.
     *
     * <p>
     * 精度为{@code 1e-10}.
     * </p>
     *
     * @param a {@code double}
     * @param b {@code double}
     * @return 是否相等
     */
    public static boolean equals(double a, double b) {
        return Math.abs(a - b) < DELTA;
    }

    private static String processCalPrecision(double number, String numberString, int dotIndex, int cIndex) {
        double value = number;
        if (cIndex == 0) {
            return Long.toString(Math.round(value));
        }
        if (Math.abs(value) > Long.MAX_VALUE) {// 处理最大的数字.
            String valueString = numberString;
            int dotPos = dotIndex;
            if (dotPos > 0 && (valueString.length() - dotPos - 1) > cIndex) {
                valueString = valueString.substring(0, dotPos + cIndex + 1);
            }
            if (valueString.endsWith(".0")) {
                return valueString.substring(0, valueString.length() - 2);
            } else {
                return valueString;
            }
        }

        return getLongNumberString(numberString, cIndex, value);
    }

    private static String processCalPrecisionCheckE(Number number, String numberString, int cIndex) {
        double value = number.doubleValue();

        if (cIndex == 0) {
            return Long.toString(Math.round(value));
        }

        if (Math.abs(value) > Long.MAX_VALUE) {// 处理最大的数字.
            String valueString = Double.toString(value);
            int dotPos = valueString.indexOf(CoreConstants.DOT);

            if (dotPos > 0 && (valueString.length() - dotPos - 1) > cIndex) {
                valueString = valueString.substring(0, dotPos + cIndex + 1);
            }

            if (valueString.endsWith(".0")) {
                return valueString.substring(0, valueString.length() - 2);
            } else {
                return valueString;
            }
        }

        return getLongNumberString(numberString, cIndex, value);
    }

    private static String getLongNumberString(String numberString, int cIndex, double value) {
        String negativeString = (value < 0) ? "-" : StringUtils.EMPTY;
        long a1 = (long) Math.pow(DECIMAL_TEN, (double) cIndex - 1);
        long a = a1 * DECIMAL_TEN;

        //如果溢出了, 还是先return原值吧
        if (a1 == Long.MAX_VALUE || a == Long.MAX_VALUE) {
            return numberString;
        }

        value = Math.abs(value);// 取绝对数值.
        long longValue = Math.round(value * a);
        String longString = negativeString + Long.toString(longValue / a);
        long rLong = (a == 0) ? 0 : longValue % a;

        /*
         * CPU bug convertNumberStringToString long整型计算越界10^23 * 10 = -10
         * long a1 = 2^23; a1 * 10 = -10
         */
        if (rLong < 0) { // 说明越界
            return numberString;
        }

        if (rLong == 0) {
            StringBuffer rString = new StringBuffer(StringUtils.EMPTY);
            int i = cIndex;
            while (i > 0) {
                rString.insert(0, "0");
                i--;
            }

            return longString + "." + rString;
        } else {
            StringBuffer rString = new StringBuffer(Long.toString(rLong));

            while (rLong < a1) {
                rLong *= DECIMAL_TEN;
                rString.insert(0, "0");
            }

            return longString + "." + rString;
        }
    }

    private static String stringAfterCheckE(String numberString, int dotIndex, int eIndex) {
        if (needConvert(dotIndex, eIndex, numberString)) {
            String ePreString = numberString.substring(0, eIndex);
            StringBuilder dotStringBuf = new StringBuilder(
                    ePreString.substring(dotIndex + 1)); // 删除".".

            dotStringBuf.insert(0, ePreString.substring(0, dotIndex));

            // alex:1.24125E+20,需要取出20作为幂,+20无法正确Integer.parseInt,要把+去掉
            String eString = numberString.substring(eIndex + 1);
            if (eString.startsWith("+")) {
                eString = eString.substring(1);
            }
            int eInteger = Integer.parseInt(eString);

            if (eInteger <= 0) {
                while (dotStringBuf.charAt(dotStringBuf.length() - 1) == '0') {
                    dotStringBuf.deleteCharAt(dotStringBuf.length() - 1);
                }

                for (int i = 0; i < -eInteger; i++) {
                    dotStringBuf.insert(dotIndex - 1, "0");
                }

                dotStringBuf.insert(dotIndex, ".");
            } else {
                if (dotStringBuf.length() - dotIndex > eInteger) {
                    dotStringBuf.insert(eInteger + dotIndex, '.');
                } else {
                    while (dotStringBuf.length() - dotIndex < eInteger) {
                        dotStringBuf.append('0');
                    }
                }
            }

            //0000 && 9999 都放到下面去处理
            numberString = dotStringBuf.toString();
        }
        return numberString;
    }

    private static boolean needConvert(int dotIndex, int eIndex, String numberString) {
        return (dotIndex == 1 && eIndex > 0) || (dotIndex == 2 && eIndex > 0 && (numberString
                .startsWith("-") || numberString.startsWith("+")));
    }

    private static boolean isNumCheck(String statement) {
        int len = statement.length();
        if (len == 1) {
            return isNum(statement.charAt(0));
        }
        boolean indexOfDot = false, isDoubleOrDecimal = false, hasSymbol = false;
        char fc = statement.charAt(0);
        if (fc == NAN_START) {
            return NAN.equals(statement);
        } else if (fc == INFINITY_START) {
            return INFINITY.equals(statement);
        } else if (fc == NA || fc == PL) {
            char sc = statement.charAt(1);
            if (sc == NAN_START) {
                return NAN.equals(statement.substring(1));
            } else if (sc == INFINITY_START) {
                return INFINITY.equals(statement.substring(1));
            }
        }
        for (int i = 0; i < len; i++) {
            char c = statement.charAt(i);
            if (c == DOT) {
                if (indexOfDot) {
                    return false;
                }
                indexOfDot = true;
            }
            if (c == UE || c == LE) {
                if (isDoubleOrDecimal || i == len - 1) {
                    return false;
                }
                isDoubleOrDecimal = true;
                //E的情况下重置+-判断
                hasSymbol = false;
            }
            if (c == PL || c == NA) {
                if (hasSymbol) {
                    return false;
                }
                if (i != 0 && !isDoubleOrDecimal) {
                    return false;
                }
                hasSymbol = true;
            }
            if (otherCheck(c)) {
                return false;
            }
        }
        return true;
    }

    private static boolean otherCheck(char c) {
        return !isNum(c) && c != DOT && c != UE && c != LE && c != NA && c != PL;
    }

    private static Double string2SpecialNumber(String statement) {

        char fc = statement.charAt(0);
        if (fc == NAN_START && NAN.equals(statement)) {
            return Double.NaN;
        } else if (fc == INFINITY_START && INFINITY.equals(statement)) {
            return Double.POSITIVE_INFINITY;
        } else if (fc == PL || fc == NA) {
            //isEmpty判断了这种情况长度肯定大于1
            char sc = statement.charAt(1);
            if (sc == NAN_START && NAN.equals(statement.substring(1))) {
                return Double.NaN;
            } else if (sc == INFINITY_START && INFINITY.equals(statement.substring(1))) {
                return fc == PL ? Double.POSITIVE_INFINITY : Double.NEGATIVE_INFINITY;
            }
        }
        return null;
    }

    private static NumberDescribe parseNumber(String statement, int len) {
        boolean indexOfDot = false, isDoubleOrDecimal = false, hasSymbol = false;
        for (int i = 0; i < len; i++) {
            char c = statement.charAt(i);
            if (c == DOT) {
                if (indexOfDot || (i == len - 1)) {//昧着良心把2016.这样的字符串不识别成数字
                    return null;
                }
                indexOfDot = true;
            }
            if (c == UE || c == LE) {
                if (isDoubleOrDecimal || i == len - 1) {
                    return null;
                }
                isDoubleOrDecimal = true;
                hasSymbol = false;
            }
            if (c == PL || c == NA) {
                if (hasSymbol) {
                    return null;
                }
                if (i != 0 && !isDoubleOrDecimal) {
                    return null;
                }
                hasSymbol = true;
            }
            if (otherCheck(c)) {
                return null;
            }
        }
        NumberDescribe describe = new NumberDescribe(indexOfDot, isDoubleOrDecimal, hasSymbol, len);
        return describe;
    }

    /**
     * 字符串parse后的一个封装.
     */
    private static final class NumberDescribe {
        // 是否带点
        boolean indexOfDot;
        // 是否double decimal
        boolean isDoubleOrDecimal;
        // 是否有特殊符号 + -
        boolean hasSymbol;
        // 字符串长度
        int len;

        public NumberDescribe(boolean indexOfDot, boolean isDoubleOrDecimal, boolean hasSymbol, int len) {
            this.indexOfDot = indexOfDot;
            this.isDoubleOrDecimal = isDoubleOrDecimal;
            this.hasSymbol = hasSymbol;
            this.len = len;
        }
    }

    private static boolean isEmpty(String statement, int len) {
        return len == 0 || (len == 1 && !isNum(statement.charAt(0)));
    }

    private static String getString(String statement) {
        if (statement.startsWith("+")) {
            statement = statement.substring(1, statement.length());
        }
        return statement;
    }

    private static Number getNumber(String statement, NumberDescribe describe) {
        try {
            if (describe.isDoubleOrDecimal) {
                return new BigDecimal(statement);
            } else if (describe.indexOfDot) {
                //这toString效率这么低妈的不转了 全部BigDecimal
                // carl:当转成double后，显示字符串同原来一致，则转成double
                return new BigDecimal(statement);
            } else {
                // 过长就用BigInteger
                boolean isIntegerLen = describe.len < DECIMAL_TEN || (describe.len == DECIMAL_TEN && describe.hasSymbol);
                if (isIntegerLen) {
                    try {
                        return Integer.valueOf(statement);
                    } catch (NumberFormatException e1) {
                        return new BigInteger(statement);
                    }
                }
                return new BigInteger(statement);
            }
        } catch (NumberFormatException e1) {
            return null;
        }
    }

    private static boolean isStartWithZeroAndCannotbeDecimal(String statement) {
        return (statement.startsWith("0") && statement.length() > 1 && !statement.startsWith("0E") && !statement
                .startsWith("0."))
                || (statement.startsWith("-0") && statement.length() > 2 && !statement.startsWith("-0E") && !statement
                .startsWith("-0."));
    }

    private static String filterString(String s) {
        if (s.indexOf('$') == -1) {
            return s;
        }
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            if (c == '$') {
                sb.append("\\$");
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    private static int endOffsetByDefaultFirstChar(String str, int end, StringBuffer sb, String tailString) throws Exception {
        int octal = 0; // octal的长度
        if (tailString.matches("[0-3][0-7][0-7].*")) {
            octal = 3;
        } else if (tailString.matches("[0-7][0-7].*")) {
            octal = 2;
        } else if (tailString.matches("[0-7].*")) {
            octal = 1;
        }

        if (octal > 0) {
            sb.append(parseOctal(str.substring(end, end + octal)));
            end = end + octal;
        } else {
            throw new Exception("\\" + tailString + " can't be parsed.");
        }
        return end;
    }

    private static int endOffsetByFirstChar(String str, int end, StringBuffer sb, String tailString) throws Exception {
        char firstChar = tailString.charAt(0);
        switch (firstChar) {
            case 'n':
                sb.append('\n');
                end++;
                break;
            case 'r':
                sb.append('\r');
                end++;
                break;
            case 't':
                sb.append('\t');
                end++;
                break;
            case 'b':
                sb.append('\b');
                end++;
                break;
            case 'f':
                sb.append('\f');
                end++;
                break;
            case '"':
                sb.append('\"');
                end++;
                break;
            case '\'':
                sb.append('\'');
                end++;
                break;
            case '\\':
                sb.append('\\');
                end++;
                break;
            case 'u': {// hex -> unicode
                String hexString = str.substring(end);
                if (hexString.length() < HEX_STRING_MIN_LEN) {
                    throw new Exception("\\u" + hexString + " can't be parsed.");
                }
                sb.append(parseHex(hexString.substring(0, HEX_STRING_MIN_LEN)));
                end = end + HEX_STRING_MIN_LEN + 1;
                break;
            }
            default: {// octal -> unicode
                end = endOffsetByDefaultFirstChar(str, end, sb, tailString);
            }
        }
        return end;
    }

    private static char parseHex(String hex_str) {
        return (char) Integer.decode("0x" + hex_str).intValue();
    }

    private static char parseOctal(String octal_str) {
        return (char) Integer.decode('0' + octal_str).intValue();
    }

    private static boolean classesTypeEquals(Class<?>[] pt, Class<?>[] pa) {
        if (pt == pa) {
            return true;
        }
        if (pt == null || pa == null) {
            return false;
        }
        if (pt.length != pa.length) {
            return false;
        }
        for (int i = 0; i < pt.length; i++) {
            if (!classTypeEquals(pt[i], pa[i])) {
                return false;
            }
        }
        return true;
    }

    private static boolean classTypeEquals(Class<?> pt, Class<?> pa) {
        if (pt == pa) {
            return true;
        }
        return dealWithBaseType(pt) == dealWithBaseType(pa);
    }

    private static Class<?> dealWithBaseType(Class<?> pt) {
        if (pt == Integer.class) {
            return int.class;
        }
        if (pt == Long.class) {
            return long.class;
        }
        if (pt == Float.class) {
            return float.class;
        }
        if (pt == Boolean.class) {
            return boolean.class;
        }

        if (pt == Double.class) {
            return double.class;
        }
        if (pt == Character.class) {
            return char.class;
        }
        if (pt == Short.class) {
            return short.class;
        }
        if (pt == Byte.class) {
            return byte.class;
        }

        return pt;
    }

    private static Class loadClass(ClassLoader loader, String className) throws ClassNotFoundException {
        return loader == null ? Class.forName(className) : loader.loadClass(className);
    }

    private static boolean isOtherNull(Object value) {
        return value == Primitive.NULL;
    }
    
    /**
     * 获取java版本号 数字
     * 9.0版本之前 java版本号形如 1.7.0_80, 1.8.0_211, 1.x.y_z
     * 9.0版本之后 java版本号形如 9.0.1, 11.0.4, x.y.z
     * @return
     */
    private static int getJavaVersion() {
        String version = System.getProperty("java.version");
        if(version.startsWith("1.")) {
            version = version.substring(2, 3);
        } else {
            int dot = version.indexOf(".");
            if (dot != -1) {
                version = version.substring(0, dot);
            }
        }
        int versionNumber = 1;
        try {
            versionNumber = Integer.parseInt(version);
        } catch (Exception e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
        return versionNumber;
    }
}
