package com.fr.stable;

import com.fr.log.FineLoggerFactory;
import com.fr.third.javax.annotation.Nullable;
import org.jetbrains.annotations.NotNull;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 编码工具类.
 *
 * @author fanruan
 * created on 2020-08-25
 * @IncludeIntoJavadoc
 */
public final class CommonCodeUtils {

    /**
     * 编码长度.
     */
    public static final int ENCODE_LEN = 4;
    /**
     * 十六进制进制数.
     */
    public static final int HEX = 16;
    /**
     * 创建{@code StringBuilder}时的长度系数.
     */
    public static final double COEFFICIENT = 1.5;
    /**
     * 大写字母A的ASCII码值.
     */
    public static final int ASC_CODE_A = 65;
    /**
     * 大写字母F的ASCII码值.
     */
    public static final int ASC_CODE_F = 70;
    /**
     * 大写字母Z的ASCII码值.
     */
    public static final int ASC_CODE_Z = 90;
    /**
     * 小写字母a的ASCII码值.
     */
    public static final int ASC_CODEA = 97;
    /**
     * 小写字母f的ASCII码值.
     */
    public static final int ASC_CODEF = 102;
    /**
     * 小写字母z的ASCII码值.
     */
    public static final int ASC_CODEZ = 122;
    /**
     * 数字0的ASCII码值.
     */
    public static final int ASC_CODE_0 = 48;
    /**
     * 数字9的ASCII码值.
     */
    public static final int ASC_CODE_9 = 57;
    /**
     * 删除（DEL）的ASCII码值.
     */
    public static final int ASC_CODE_DEL = 127;
    /**
     * 左方括号（[）的ASCII码值.
     */
    public static final int ASC_CODE_LEFT_BRACKET = 91;
    /**
     * 右方括号（]）的ASCII码值.
     */
    public static final int ASC_CODE_RIGHT_BRACKET = 93;
    /**
     * 128.
     */
    public static final int HEX_ARR_LENGTH = 128;
    /**
     * 2^8.
     */
    public static final int EIGHTTIMES2 = 256;

    private final static String[] HEXDIGITS = {
            "0", "1", "2", "3", "4", "5",
            "6", "7", "8", "9", "a", "b", "c", "d", "e", "f"};

    private CommonCodeUtils() {
    }
    // 用于判断某个文本是不是被cjkEncode过的,这两个Pattern用到的频率比较高,所以static化
    private static Pattern bracketPattern = Pattern.compile("\\[[^\\]]*\\]");
    // alex:一些特殊字符在cjkEncode后是2个字符或是3个字符(α),汉字在cjkEncode后是4个字符
    private static Pattern textPattern = Pattern.compile("[0-9a-f]{2,4}", Pattern.CASE_INSENSITIVE);
    // 用于替换空白字符
    private static Pattern blankPattern = Pattern.compile("\\s+");
    private static int JUDGE_HELPER_LEN = 100;
    private static final boolean[] JUDGE_HELPER = new boolean[JUDGE_HELPER_LEN];
    private static final boolean[] HEX_ARR = new boolean[HEX_ARR_LENGTH];
    private static final char[] APPEND_HELPER = new char[JUDGE_HELPER_LEN];
    private static final int COPYRIGHT_HELPER_LEN = 9000;
    private static final boolean[] COPYRIGHT_HELPER = new boolean[COPYRIGHT_HELPER_LEN];
    private static final int[] PASSWORD_MASK_ARRAY = {19, 78, 10, 15, 100, 213, 43, 23};

    static {
        for (int i = 0; i < ' '; i++) {
            COPYRIGHT_HELPER[i] = true;
        }
        for (int i = '\u0080'; i < '\u00a0'; i++) {
            COPYRIGHT_HELPER[i] = true;
        }
        for (int i = '\u2000'; i < '\u2100'; i++) {
            COPYRIGHT_HELPER[i] = true;
        }
        COPYRIGHT_HELPER['\u00a9'] = true;
        COPYRIGHT_HELPER['\u00ae'] = true;
    }

    static {
        JUDGE_HELPER['\\'] = true;
        APPEND_HELPER['\\'] = '\\';
        JUDGE_HELPER['"'] = true;
        APPEND_HELPER['"'] = '"';
        JUDGE_HELPER['\t'] = true;
        APPEND_HELPER['\t'] = 't';
        JUDGE_HELPER['\n'] = true;
        APPEND_HELPER['\n'] = 'n';
        JUDGE_HELPER['\r'] = true;
        APPEND_HELPER['\r'] = 'r';
        JUDGE_HELPER['\f'] = true;
        APPEND_HELPER['\f'] = 'f';
        JUDGE_HELPER['\b'] = true;
        APPEND_HELPER['\b'] = 'b';
        // 数字0-9
        for (int i = ASC_CODE_0; i < ASC_CODE_9 + 1; i++) {
            HEX_ARR[i] = true;
        }
        // 大写字母
        for (int i = ASC_CODE_A; i < ASC_CODE_F + 1; i++) {
            HEX_ARR[i] = true;
        }
        // 小写字母
        for (int i = ASC_CODEA; i < ASC_CODEF + 1; i++) {
            HEX_ARR[i] = true;
        }
    }

    /**
     * 将字符串编码为{@code JavaScript}字符串.
     *
     * <pre>
     * CommonCodeUtils.javascriptEncode(null)                   = ""
     * CommonCodeUtils.javascriptEncode("")                     = ""
     * CommonCodeUtils.javascriptEncode(" ")                    = " "
     * CommonCodeUtils.javascriptEncode("eval(\"hello\")/\\")   = "eval(\\\"hello\\\")/\\\\"
     * </pre>
     *
     * @param str 原字符串
     * @return 编码字符串
     */
    @NotNull
    public static String javascriptEncode(@Nullable String str) {
        StringBuilderHelper sb = new StringBuilderHelper(str);
        if (StringUtils.isNotEmpty(str)) {
            char b, c = 0;
            int i, len = str.length();
            for (i = 0; i < len; i++) {
                b = c;
                c = str.charAt(i);
                if (c < JUDGE_HELPER_LEN) {
                    if (JUDGE_HELPER[c]) {
                        sb.append('\\', i);
                        sb.append(APPEND_HELPER[c], i);
                        sb.move();
                        continue;
                    } else if (c == '/') {
                        if (b == '<') {
                            sb.append('\\', i);
                        }
                        sb.append(c, i);
                        sb.move();
                        continue;
                    }
                }
                if (isCopyRightSymbol(c)) {
                    String t = "000" + Integer.toHexString(c);
                    sb.append('\\', i);
                    sb.append('u', i).append(t.substring(t.length() - ENCODE_LEN), i);
                    sb.move();
                }
            }
        } else {
            return StringUtils.EMPTY;
        }
        return sb.toString();
    }

    /**
     * 将{@code JavaScript}字符串解码为普通字符串.
     *
     * @param str 编码字符串
     * @return 解码字符串
     * @see CommonCodeUtils#javascriptEncode(String)
     */
    @NotNull
    public static String javascriptDecode(@Nullable String str) {
        return CommonCodeUtils.encodeString(str, new String[][]{
                {"\\", "'", "\""}, {"\\\\", "\\'", "\\\""}});
    }

    /**
     * 判断字符是否是版权标记符号.
     *
     * @param c 字符
     * @return 是否是版权标记符号
     */
    public static final boolean isCopyRightSymbol(char c) {
        return c < COPYRIGHT_HELPER_LEN && COPYRIGHT_HELPER[c];
    }

    /**
     * 利用编码关系编码字符串.
     *
     * <p>
     * 传入{@code null}，返回"".
     * 若传入的编码数组为{@code String[][] = {{"a", "b"}, {"1", "2"}}}，则将原字符串中的1替换为a，2替换为b，以此类推.
     * </p>
     *
     * @param string 原字符串
     * @param encodeArray 编码数组
     * @return 编码后的字符串
     */
    @NotNull
    public static String encodeString(@Nullable String string, @NotNull String[][] encodeArray) {
        if (string == null) {
            return StringUtils.EMPTY;
        }

        StringBuilder stringBuf = new StringBuilder();

        loop:
        for (int i = 0; i < string.length(); i++) {
            char ch = string.charAt(i);
            for (int j = 0; j < encodeArray[1].length; j++) {
                if (ch == encodeArray[1][j].charAt(0)) {
                    stringBuf.append(encodeArray[0][j]);
                    continue loop;
                }
            }
            stringBuf.append(ch);
        }
        return stringBuf.toString();
    }

    /**
     * 对URI进行UTF-8编码，对应JavaScript的{@code encodeURIComponent()}.
     *
     * <p>
     * 与{@code encodeURIComponent()}不同的是，此方法不进行CJK编码.
     * </p>
     *
     * @param uriCmp URI字符串
     * @return 编码后的字符串
     */
    @NotNull
    public static String encodeURIComponent(@NotNull String uriCmp) {
        try {
            return URLEncoder.encode(uriCmp, EncodeConstants.ENCODING_UTF_8);
        } catch (UnsupportedEncodingException e) {
            // 实际不可能抛UnsupportedEncodingException，因为UTF-8是一定存在的
            return CommonCodeUtils.encodeString(uriCmp, new String[][]{
                    {"%20", "%23", "%24", "%26", "%2B", "%2C", "%2F", "%3A", "%3B", "%3D", "%3F", "%40", "%25"},
                    {" ", "#", "$", "&", "+", ",", "/", ":", ";", "=", "?", "@", "%"}});
        }
    }

    /**
     * 编码HTML标签的属性.
     *
     * <p>
     * 输入为{@code null}，返回"".
     * 输入中的大小写字母、数字保持原样，其他字符使用{@code &#}和{@code ;}对{@code char}值进行包裹.
     * </p>
     *
     * <pre>
     * CommonCodeUtils.attributeHtmlEncode(null)            = ""
     * CommonCodeUtils.attributeHtmlEncode("abcABC123+= ")  = "abcABC123&#43;&#61;&#32;"
     * </pre>
     *
     * @param rawValue HTML标签属性值
     * @return 编码过的属性值
     */
    @NotNull
    public static String attributeHtmlEncode(@Nullable CharSequence rawValue) {

        if (rawValue == null) {
            return StringUtils.EMPTY;
        }

        int len = rawValue.length();
        StringBuilder out = new StringBuilder((int) (len * COEFFICIENT));

        // Allow: a-z A-Z 0-9
        // Allow (dec): 97-122 65-90 48-57

        for (int cnt = 0; cnt < len; cnt++) {
            char c = rawValue.charAt(cnt);
            if (isLetterOrNumber(c)) {
                out.append(c);
            } else {
                out.append("&#").append((int) c).append(';');
            }
        }

        return out.toString();
    }

    /**
     * 解码HTML标签的属性.
     *
     * <p>
     * 输入为{@code null}，返回"".
     * </p>
     *
     * <pre>
     * CommonCodeUtils.attributeHtmlEncode(null)            = ""
     * CommonCodeUtils.attributeHtmlEncode("abcABC123+= ")  = "abcABC123&#43;&#61;&#32;"
     * </pre>
     *
     * @param rawValue HTML标签属性值
     * @return 解码后的属性值
     * @see CommonCodeUtils#attributeHtmlEncode(CharSequence)
     */
    @NotNull
    public static String attributeHtmlDecode(@Nullable CharSequence rawValue) {

        if (rawValue == null) {
            return StringUtils.EMPTY;
        }
        int len = rawValue.length();
        StringBuilder out = new StringBuilder(len);
        for (int i = 0; i < len; i++) {
            char c = rawValue.charAt(i);
            boolean match = false;
            if (c == '&') {
                if (i + 1 < len && rawValue.charAt(i + 1) == '#') {
                    int end = rawValue.toString().indexOf(";", i + 1);
                    if (end != -1 && end < i + 3 + 3) {  // 编码一般不超过三位
                        String code = rawValue.subSequence(i + 2, end).toString();
                        char decode = (char) Integer.parseInt(code);
                        out.append(decode);
                        i = end;
                        match = true;
                    }
                }
            }
            if (!match) {
                out.append(c);
            }
        }

        return out.toString();
    }

    /**
     * 编码HTML字符，主要针对一些特殊的符号.
     *
     * <pre>
     * CommonCodeUtils.htmlEncode(null)                                     = ""
     * CommonCodeUtils.htmlEncode(" abc  {@literal <b>abc</b>&}\"\r\\\n")   = "{@literal  abc &nbsp; &lt;b&gt;abc&lt;/b&gt;&amp;\\&quot;\\r\\<br/>}"
     * </pre>
     *
     * @param rawValue 要编码的HTML字符串
     * @return 编码后的字符串
     */
    @NotNull
    public static String htmlEncode(@Nullable CharSequence  rawValue) {

        if (rawValue == null) {
            return StringUtils.EMPTY;
        }

        int len = rawValue.length();
        StringBuilder sb = new StringBuilder(len);

        for (int i = 0; i < len; i++) {
            char c = rawValue.charAt(i);
            i = dealWithChar(c, sb, len, i, rawValue);
        }

        return replaceBlankToHtmlBlank(sb.toString());
    }

    /**
     * 中日韩文处理（CJK编码）.
     *
     * <p>
     * "用"转换为[7528]，"软"转换为[8f6f].
     * </p>
     *
     * @param text 待处理的中日韩文字
     * @return 处理过的字符串
     * @see CommonCodeUtils#cjkDecode(String)
     */
    @NotNull
    public static String cjkEncode(@Nullable String text) {

        if (text == null) {
            return StringUtils.EMPTY;
        }

        StringBuilder newTextBuf = new StringBuilder();
        for (int i = 0, len = text.length(); i < len; i++) {
            char ch = text.charAt(i);
            if (needToEncode(ch)) {//alex:91 is '[', 93 is ']'.转换非ascii + '[' + ']'
                newTextBuf.append('[');
                newTextBuf.append(Integer.toString(ch, 16));
                newTextBuf.append(']');
            } else {
                newTextBuf.append(ch);
            }
        }

        return newTextBuf.toString();
    }

    /**
     * 中日韩文处理（CJK解码）.
     *
     * <p>
     * [7528]转换为"用"，[8f6f]转换为"软".
     * </p>
     *
     * @param text CJK编码后字符串
     * @return CJK解码后字符串
     * @see CommonCodeUtils#cjkEncode(String)
     */
    @NotNull
    public static String cjkDecode(@Nullable String text) {

        if (text == null) {
            return StringUtils.EMPTY;
        }

        //先做判断
        if (!isCJKEncoded(text)) {
            return text;
        }

        StringBuilder newTextBuf = new StringBuilder();
        for (int i = 0; i < text.length(); i++) {
            char ch = text.charAt(i);
            if (ch == ASC_CODE_LEFT_BRACKET) {
                int rightIdx = text.indexOf(ASC_CODE_RIGHT_BRACKET, i + 1);
                if (rightIdx > i + 1) {
                    String subText = text.substring(i + 1, rightIdx);
                    //james：主要是考虑[CDATA[]]这样的值的出现
                    if (subText.length() > 0) {
                        ch = (char) Integer.parseInt(subText, 16);
                    }

                    i = rightIdx;
                }
            }
            newTextBuf.append(ch);
        }
        return newTextBuf.toString();
    }

    /**
     * 判断字符串是否经CJK编码.
     *
     * @param text 字符串
     * @return 是否经过CJK编码
     */
    public static boolean isCJKEncoded(@Nullable String text) {

        /*
         * 判断是否是cjkEncode过的字符串
         * 如果符合[]中间包含非]的字符,那么这几个字符必须是5b或是5d或是[0-9a-fA-F]{4},否则就不是被cjkEncode过的
         * 算法:
         * 1.滑动窗口确定范围；
         * 2.判断窗口内的内容：
         * a.范围在2-4个字符
         * b.每个字符都需满足16进制字符（使用预设的boolean数组来判断）
         * <p>
         * 要点：长字符串减少或不使用正则，不直接操作字符串（比如截取），部分操作尽量使用左右移完成
         * 性能较之前在同一段200多字中文段落下大概提升10-15倍
         */

        if (text == null) {
            return false;
        }

        int leftPos = -1, rightPos = -1;
        boolean hasCJK = false;
        for (int i = 0; i < text.length(); i++) {
            char c = text.charAt(i);

            if (c == ASC_CODE_LEFT_BRACKET) {
                leftPos = i + 1;
                continue;
            } else if (leftPos == -1) {
                continue;
            }

            if (c == ASC_CODE_RIGHT_BRACKET) {
                rightPos = i;
                hasCJK = true;
                // 判断窗口大小，内有 2-4 个 16 进制字符，这里减一方便运算
                int cjkLength = rightPos - leftPos - 1;
                // 检测相对大小1-3
                if (cjkLength == 0 || cjkLength >>> 2 != 0) {
                    // 窗口太大或太小，不符合，直接失败
                    return false;
                }
                // 判断窗口内的内容
                if (isHex(text, leftPos, rightPos)) {
                    // 内容正确，重置游标，继续检测下一个窗口
                    leftPos = -1;
                    rightPos = -1;
                } else {
                    return false;
                }
            }
        }
        return hasCJK;
    }

    /**
     * 字符串加密.
     *
     * @param text 字符串
     * @return 加密结果
     */
    @NotNull
    public static String passwordEncode(@Nullable String text) {

        //这个方法支持CJK。
        StringBuilder passwordTextBuf = new StringBuilder();
        passwordTextBuf.append("___");
        if (text == null) {
            return passwordTextBuf.toString();
        }

        int pmIndex = 0;
        for (int i = 0; i < text.length(); i++) {
            if (pmIndex == PASSWORD_MASK_ARRAY.length) {
                pmIndex = 0;
            }

            int intValue = ((int) text.charAt(i)) ^ PASSWORD_MASK_ARRAY[pmIndex];
            StringBuilder passwordText = new StringBuilder(Integer.toHexString(intValue));

            //确保用四位数字来保存字符串.
            int pLength = passwordText.length();
            for (int j = 0; j < ENCODE_LEN - pLength; j++) {
                passwordText.insert(0, "0");
            }

            passwordTextBuf.append(passwordText);
            pmIndex++;
        }

        return passwordTextBuf.toString();
    }

    /**
     * 字符串解密.
     *
     * @param text 字符串
     * @return 解密结果
     */
    @Nullable
    public static String passwordDecode(@Nullable String text) {

        if (text != null && text.startsWith("___")) {
            text = text.substring(3); //去掉前面的"___"。

            StringBuilder textBuf = new StringBuilder();

            int pmIndex = 0;
            for (int i = 0; i <= (text.length() - ENCODE_LEN); i += ENCODE_LEN) {
                if (pmIndex == PASSWORD_MASK_ARRAY.length) {
                    pmIndex = 0;
                }

                String hexText = text.substring(i, i + ENCODE_LEN);
                int hexInt = Integer.parseInt(hexText, HEX) ^ PASSWORD_MASK_ARRAY[pmIndex];

                textBuf.append((char) hexInt);
                pmIndex++;
            }

            text = textBuf.toString();
        }

        return text;
    }

    /**
     * CJK解码字符串.
     *
     * <p>
     * 传入{@code null}，返回{@code null}.
     * 若解码过程发生异常，返回字符串本身.
     * </p>
     *
     * @param text 需要解码的字符串
     * @return 解码后的字符串
     */
    @Nullable
    public static String decodeText(@Nullable String text) {

        if (text == null) {
            return null;
        }

        /*
         * alex:用isCJKEncoded方法判断这个text是不是被cjkEncode过的
         */
        try {
            return cjkDecode(text);
        } catch (Exception e) {
            // do nothing
        }

        // 不编码，返回原始的字符串.
        return text;
    }

    /**
     * MD5加密.
     *
     * <p>
     * 要加密的字符串传入{@code null}等价于传入"".
     * </p>
     *
     * <pre>
     * CommonCodeUtils.md5Encode(null, "", "MD5")   = "d41d8cd98f00b204e9800998ecf8427e"
     * CommonCodeUtils.md5Encode("", "", "MD5")     = "d41d8cd98f00b204e9800998ecf8427e"
     * CommonCodeUtils.md5Encode("abc", "", "MD5")  = "900150983cd24fb0d6963f7d28e17f72"
     * </pre>
     *
     * @param rawPass   需要加密的字符串
     * @param salt      加盐值
     * @param algorithm 加密算法
     * @return 加密后的字符串
     */
    @Nullable
    public static String md5Encode(@Nullable String rawPass, @Nullable Object salt, @NotNull String algorithm) {
        try {
            MessageDigest md = MessageDigest.getInstance(algorithm);
            //加密后的字符串
            String pass = rawPass == null ? "" : rawPass;
            if (salt != null && StringUtils.isNotEmpty(salt.toString())) {
                pass = pass + "{" + salt.toString() + "}";
            }
            byte[] digBytes = md.digest(pass.getBytes(StandardCharsets.UTF_8));
            //转换字节数组为16进制字串
            StringBuilder resultSb = new StringBuilder();
            for (byte digByte : digBytes) {
                resultSb.append(byteToHexString(digByte));
            }
            return resultSb.toString();
        } catch (Exception ignored) {
        }
        return null;
    }

    /**
     * 转换字节为十六进制字符串.
     *
     * <pre>
     * CommonCodeUtils.byteToHexString((byte) 23)   = "17"
     * CommonCodeUtils.byteToHexString((byte) 199)  = "c7"
     * </pre>
     *
     * @param b 字节
     * @return 十六进制字符串
     */
    @NotNull
    public static String byteToHexString(byte b) {
        int n = b;
        if (n < 0) {
            n = EIGHTTIMES2 + n;
        }
        int d1 = n / HEX;
        int d2 = n % HEX;
        return HEXDIGITS[d1] + HEXDIGITS[d2];
    }

    /**
     * 转换字符串的字符集.
     *
     * <p>
     * 使用特定字符集读取字符串为字节数组，再使用特定字符集转换字节数组为字符串.
     * </p>
     *
     * @param charset 读取字符集
     * @param targetCharset 转换字符集
     * @param str 字符串
     * @return 结果字符串
     */
    @Nullable
    public static String getNewCharSetString(@Nullable String charset, @Nullable String targetCharset, @Nullable String str) {

        if (StringUtils.isEmpty(str)) {
            return str;
        }
        boolean oriCharsetBlank = StringUtils.isBlank(charset);
        boolean newCharsetBlank = StringUtils.isBlank(targetCharset);
        if (oriCharsetBlank && newCharsetBlank) {
            return str;
        }
        try {
            byte[] bytes = oriCharsetBlank ? str.getBytes() : str.getBytes(charset);
            return newCharsetBlank ? new String(bytes) : new String(bytes, targetCharset);
        } catch (UnsupportedEncodingException e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
        return str;
    }

    private final static class StringBuilderHelper {
        private StringBuilder sb;
        private String baseString;
        private int start = 0;

        private StringBuilderHelper(String baseString) {
            this.baseString = baseString;
        }

        private void initSB(int i) {
            if (sb == null) {
                sb = new StringBuilder(baseString.length());
            }
            if (start != i) {
                sb.append(baseString, start, i);
                start = i;
            }
        }

        private StringBuilderHelper append(char c, int i) {
            initSB(i);
            sb.append(c);
            return this;
        }

        private void move() {
            start++;
        }

        private StringBuilderHelper append(String c, int i) {
            initSB(i);
            sb.append(c);
            return this;
        }

        @Override
        public String toString() {
            if (sb == null) {
                return baseString;
            } else {
                initSB(baseString.length());
                return sb.toString();
            }
        }
    }

    private static boolean isLetterOrNumber(char c) {
        return (c >= ASC_CODEA && c <= ASC_CODEZ) ||
                (c >= ASC_CODE_A && c <= ASC_CODE_Z) ||
                (c >= ASC_CODE_0 && c <= ASC_CODE_9);
    }

    /**
     * 将空白转换为HTML空白.
     *
     * 把空白转换成&nbsp;同时保证break-word有效
     * 起始和结束必须是" "以满足break-word
     * 中间的空白每两个变成一个&nbsp;因为&nbsp;实际长度比一个空白大
     * 不知道什么地方会用到word-break属性？先把这个属性去掉，不然会让使用了大量空格的单元格自动换行，没办法解决
     */
    private static String replaceBlankToHtmlBlank(@NotNull String str) {
        Matcher matcher = blankPattern.matcher(str);
        int start = 0, end = 0;
        StringBuilder sb = new StringBuilder(str.length());
        while (matcher.find()) {
            end = matcher.start();
            sb.append(str, start, end);
            start = matcher.end();

            String group = matcher.group();

            //neil: 这边空格去掉会导致所有数据集里面数据的空格都被去掉.
            sb.append(" ");
            int psize = group.length();
            if (psize > 1) {
                //daniel: 这里一个空格就会 增加一个&nbsp; 并不是注释中的两个空格一个&nbsp;并且这个逻辑还不能改....会导致已经对其的模板不对其
                //IE下的效果和chrome下还不一样 IE下
                for (int pi = psize; pi > 1; pi -= 1) {
                    sb.append("&nbsp;");
                }
                sb.append(" ");
            }
        }

        if (start == 0) {    // 呵呵，如果没有空字符串， 直接返回
            return str;
        }

        sb.append(str.substring(start));

        return sb.toString();
    }

    private static int dealWithChar(char c, StringBuilder sb, int len, int i, CharSequence rawValue) {
        switch (c) {
            case '<': {
                sb.append("&lt;");
                break;
            }
            case '>': {
                sb.append("&gt;");
                break;
            }
            case '&': {
                sb.append("&amp;");
                break;
            }
            case '"': {
                sb.append("&quot;");
                break;
            }
            case '\r': {
                if (i + 1 < len && rawValue.charAt(i + 1) == '\n') {
                    i++;
                }
                sb.append("<br/>");
                break;
            }
            case '\\': {
                //  将\n换成<br/>  在格子里的\n是两个char,不是一个char
                //  相当于Utils.replaceAllString(str, "\\n", "<br/>")  这里还要将\\解析成\，与格子保持一致。
                if (i + 1 < len && rawValue.charAt(i + 1) == 'n') {
                    i++;
                    sb.append("<br/>");
                } else {
                    if (i + 1 < len && rawValue.charAt(i + 1) == '\\') {
                        i++;
                    }
                    sb.append(c);
                }
                break;
            }
            case '\n': {
                sb.append("<br/>");
                break;
            }
            default: {
                sb.append(c);
                break;
            }
        }

        return i;
    }

    private static boolean needToEncode(char ch) {
        return ch > ASC_CODE_DEL || ch == ASC_CODE_LEFT_BRACKET || ch == ASC_CODE_RIGHT_BRACKET;
    }

    /**
     * 判断一个字符串符合16进制。
     *
     * @param hex      字符串
     * @param startPos 起始游标（包含自身）
     * @param endPos   终点游标（不包含自身）
     * @return 是否为16进制字符
     */
    private static boolean isHex(String hex, int startPos, int endPos) {
        for (int i = startPos; i < endPos; i++) {
            if (!isHex(hex.charAt(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * 判断一个字符符合16进制。
     *
     * @param c 字符
     * @return 是否为16进制字符
     */
    private static boolean isHex(char c) {
        return ((int) c) >>> 7 == 0 && HEX_ARR[c];
    }
}