package com.fr.stable;

import com.fr.third.org.apache.commons.lang3.SystemUtils;
import com.fr.third.org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import com.fr.third.org.apache.commons.lang3.builder.ToStringBuilder;
import com.fr.third.org.apache.commons.lang3.builder.ToStringStyle;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;

/**
 * 协助处理代码质量的类.
 *
 * @author  fanruan
 * created on 2020/07/23
 * @IncludeIntoJavadoc
 */
public class AssistUtils {

    private static final float FLOAT_DELTA = 1e-6f;
    private static final double DOUBLE_DELTA = 1e-10;
    private static final ToStringStyle EMB_TO_STRING_STYLE = new FineToStringStyle();

    static {
        ReflectionToStringBuilder.setDefaultStyle(EMB_TO_STRING_STYLE);
    }

    /**
     * 判断 double 类型是否相等.
     *
     * <p>
     * 浮点数采用“尾数+阶码”的编码方式，类似于科学计数法的“有效数字+指数”的表示方式
     * 二进制无法精确表示大部分的十进制小数
     * 这里首先比较该数二进制是否相等，若不相等则采用近视精度比较，精度可自定义
     * </p>
     *
     * <pre>
     * AssistUtils.equalsDouble(1.0 - 0.9, 0.9 - 0.8, 1e-10)    =true
     * AssistUtils.equalsDouble(2.1, 1.2, 1e-10)    = false
     * </pre>
     *
     * @param d1    双精度浮点
     * @param d2    双精度浮点
     * @param delta 判断精度
     * @return 是否相等
     * @see #equals(double, double)
     */
    public static boolean equalsDouble(double d1, double d2, double delta) {
        if (Double.compare(d1, d2) == 0) {
            return true;
        }
        return Math.abs(d1 - d2) <= delta;
    }

    /**
     * 判断 float 类型是否相等.
     *
     * <p>
     * 浮点数采用“尾数+阶码”的编码方式，类似于科学计数法的“有效数字+指数”的表示方式
     * 二进制无法精确表示大部分的十进制小数
     * 这里首先比较该数二进制是否相等，若不相等则采用近视精度比较，精度可自定义
     * </p>
     *
     * <pre>
     * AssistUtils.equalsFloat(1.0f - 0.9f, 0.9f - 0.8f, 1e-6f)    = true
     * AssistUtils.equalsFloat(1.1f, 0.2f, 1e-6f)   = false
     * </pre>
     *
     * @param f1    单精度浮点
     * @param f2    单精度浮点
     * @param delta 判断精度
     * @return 是否相等
     * @see #equals(float, float)
     */
    public static boolean equalsFloat(float f1, float f2, float delta) {
        if (Float.compare(f1, f2) == 0) {
            return true;
        }
        return Math.abs((double) f1 - (double) f2) <= delta;
    }

    /**
     * 判断 double 类型是否相等，精度默认为 1e-10.
     *
     * <p>
     * 参数必须都为double，调用带有精度的对应的equals方法，精度为默认值
     * </p>
     *
     * <pre>
     *AssistUtils.equals(1.0 - 0.9, 0.9 - 0.8)    = true
     * </pre>
     *
     * @param d1 双精度浮点
     * @param d2 双精度浮点
     * @return 是否相等
     * @see #equalsDouble(double, double, double)
     */
    public static boolean equals(double d1, double d2) {
        return equalsDouble(d1, d2, DOUBLE_DELTA);
    }

    /**
     * 判断 float 类型是否相等，精度默认为 1e-6.
     *
     * <p>
     * 参数必须都为float，调用带有精度的对应的equals方法，精度为默认值
     * </p>
     *
     * <pre>
     * AssistUtils.equals(1.0f - 0.9f, 0.9f - 0.8f)    = true
     * </pre>
     *
     * @param f1 单精度浮点
     * @param f2 单精度浮点
     * @return 是否相等
     * @see #equalsFloat(float, float, float)
     */
    public static boolean equals(float f1, float f2) {
        return equalsFloat(f1, f2, FLOAT_DELTA);
    }

    /**
     * 判断 long 类型是否相等.
     *
     * <p>
     * 参数必须为 long, int 等可自动转换为 long.
     * </p>
     *
     * <pre>
     * AssistUtils.equals(2147483000, 2147483000L)    = true
     * </pre>
     *
     * @param l1    长整型
     * @param l2    长整型
     * @return      l1 == l2
     */
    public static boolean equals(long l1, long l2) {
        return l1 == l2;
    }

    /**
     * 判断两个对象是否相等.
     *
     * <p>
     * 两个对象有一个不为{@code null}，一个为{@code null}，返回false。同为{@code null}返回true
     * </p>
     *
     * <pre>
     * AssistUtils.equals("abc", "abc")    = true
     * AssistUtils.equals({@code null}, "abc")    = false
     * AssistUtils.equals({@code null}, {@code null})    = true
     * AssistUtils.equals(StringUtils.EMPTY, "")    = true
     * </pre>
     *
     * @param source 对象1
     * @param des    对象2
     * @return 如果两个对象相等，则返回true，否则返回false
     */
    public static boolean equals(@Nullable Object source, @Nullable Object des) {

        if (source == null && des == null) {
            return true;
        }
        if (source == null) {
            return false;
        }
        if (des == null) {
            return false;
        }
        return source.equals(des);
    }

    /**
     * 比较基本类型中int型的大小，处理溢出情况，与Comparator配合.
     *
     * <p>
     * 参数类型必须都为int，比较它们的大小。两个相等返回0，一大一小返回1，一小一大返回-1
     * </p>
     *
     * <pre>
     * AssistUtils.compare(0, 0)    = 0
     * AssistUtils.compare(Integer.MAX_VALUE, Integer.MAX_VALUE)    = 0
     * AssistUtils.compare(1, 0)    = 1
     * AssistUtils.compare(Integer.MAX_VALUE, Integer.MIN_VALUE)    = 1
     * AssistUtils.compare(0, 1)    = -1
     * AssistUtils.compare(Integer.MIN_VALUE, Integer.MAX_VALUE)    = -1
     * </pre>
     *
     * @param i 第一个数
     * @param j 第二个数
     * @return 是否 i &gt; j
     */
    public static int compare(int i, int j) {

        if (i == j) {
            return 0;
        }
        return i > j ? 1 : -1;
    }

    /**
     * 比较基本类型中long型的大小，处理溢出情况，与Comparator配合.
     *
     * <p>
     * 参数必须都为long型，比较它们的大小。两个相等返回0，一大一小返回1，一小一大返回-1
     * </p>
     *
     * <pre>
     * AssistUtils.compare(0L, 0L)    = 0
     * AssistUtils.compare(Long.MIN_VALUE, Long.MIN_VALUE)    = 0
     * AssistUtils.compare(1L, 0L)    = 1
     * AssistUtils.compare(Long.MAX_VALUE, Long.MIN_VALUE)    = 1
     * AssistUtils.compare(0L, 1L)    = -1
     * AssistUtils.compare(Long.MIN_VALUE, Long.MAX_VALUE)    = -1
     * </pre>
     *
     * @param i 第一个数
     * @param j 第二个数
     * @return 是否 i &gt; j
     */
    public static int compare(long i, long j) {

        if (i == j) {
            return 0;
        }
        return i > j ? 1 : -1;
    }

    /**
     * 返回多个属性合并计算的哈希值.
     *
     * <p>
     * 借助guava中Objects的hashCode方法计算参数的哈希值
     * </p>
     *
     * <pre>
     * AssistUtils.hashCode("abc", "dec") == AssistUtils.hashCode("abc", "dec")
     * AssistUtils.hashCode("abc", "dec")  != AssistUtils.hashCode("dec", "abc")
     * </pre>
     *
     * @param args 对象列表
     * @return 哈希值
     */
    public static int hashCode(@NotNull Object... args) {
        return com.fr.third.guava.base.Objects.hashCode(args);
    }
    
    /**
     * 深度计算哈希值。当数组嵌套数组时进行深度对比.
     *
     * <p>
     * 参数为嵌套数组进行深度哈希计算
     * </p>
     *
     * @param args 对象列表
     * @return 哈希值
     */
    public static int deepHashCode(@NotNull Object...args){
        return Arrays.deepHashCode(args);
    }
    
    /**
     * 不允许为null的判断.
     *
     * <p>
     * 一个任意类型的对象，不为{@code null}则返回自身，若为{@code null}则抛出异常
     * </p>
     *
     * @param obj 对象
     * @param <T> 类型
     * @return 如果对象为 {@code null}，则抛出异常，否则就返回对象本身
     * @throws NullPointerException 空指针异常
     */
    public static <T> T requireNonNull(T obj) {

        if (obj == null) {
            throw new NullPointerException();
        }
        return obj;
    }

    /**
     * 对象的的toString方法.
     *
     * <p>
     * 按照FineToStringStyle的格式把对象转换为String，该类为私有静态内部类，其中包含构造方法和get方法获取该类的一个对象。
     * 该类中规定了把对象转换成的字符串的格式。字符串以[]包裹内容，参数对象的每个字段前有一个空格符。
     * 每个元素以字段名=字段值的形式转换为字符串
     * </p>
     *
     * <pre>
     * TestObject test = new TestObject("lily", 20);
     * AssistUtils.equals(
     *                          "AssistUtilsTest.TestObject[\n" +
     *                         "  name=lily\n" +
     *                         "  age=20\n" +
     *                         "]",
     *                          AssistUtils.toString(test).replaceAll("\r\n", "\n"));
     * </pre>
     *
     * @param target 对象
     * @return 文本
     */
    public static String toString(@NotNull Object target) {
        return ToStringBuilder.reflectionToString(target, EMB_TO_STRING_STYLE);
    }

    /**
     * 对象的的toString方法.
     *
     * <P>
     * 按照FineToStringStyle的格式把对象转换为String，同时可以设置忽略对象中的某些字段
     * </P>
     *
     * <pre>
     * TestObject test = new TestObject("lily", 20);
     * AssistUtils.equals(
     *                          "AssistUtilsTest.TestObject[\n" +
     *                         "  name=lily\n" +
     *                         "]",
     *                           AssistUtils.toString(test, "age").replaceAll("\r\n", "\n"));
     * </pre>
     *
     * @param target   对象
     * @param excludes 要排除掉的属性名
     * @return 文本
     * @see AssistUtils#toString(Object) 其中有关于FineToStringStyle的叙述
     */
    public static String toString(@NotNull Object target, @NotNull String... excludes) {
        return ReflectionToStringBuilder.toStringExclude(target, excludes);
    }

    private static final class FineToStringStyle extends ToStringStyle {

        private static final long serialVersionUID = 1L;

        FineToStringStyle() {

            super();
            this.setContentStart("[");
            this.setFieldSeparator(SystemUtils.LINE_SEPARATOR + "  ");
            this.setFieldSeparatorAtStart(true);
            this.setContentEnd(SystemUtils.LINE_SEPARATOR + "]");
            this.setUseShortClassName(true);
            this.setUseIdentityHashCode(false);
        }

        private Object readResolve() {
            return AssistUtils.EMB_TO_STRING_STYLE;
        }
    }
}