package com.fr.stable;

import com.fr.third.org.apache.commons.lang3.BooleanUtils;
import com.fr.third.org.apache.commons.lang3.CharUtils;
import com.fr.third.org.apache.commons.lang3.ClassUtils;
import com.fr.third.org.apache.commons.lang3.builder.EqualsBuilder;
import com.fr.third.org.apache.commons.lang3.builder.HashCodeBuilder;
import com.fr.third.org.apache.commons.lang3.builder.ToStringBuilder;
import com.fr.third.org.apache.commons.lang3.builder.ToStringStyle;
import com.fr.third.org.apache.commons.lang3.math.NumberUtils;
import com.fr.third.org.apache.commons.lang3.mutable.MutableInt;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

/**
 * 数组工具类.
 *
 * <p>
 * 提供对数组的操作, 基础类型数组 (比如 {@code int[]}) 以及
 * 基础类型的包装类的数组 (比如 {@code Integer[]})
 * </p>
 *
 * <p>
 * 本类尝试优雅地处理 {@code null} 输入
 * 当输入一个 {@code null}数组时会抛出一个异常
 * 如果一个对象数组中包含一个{@code null}元素也有可能会抛出一个异常
 * 每个方法都记录了自身的功能和逻辑
 * </p>
 *
 * <p>
 * #线程安全#
 * </p>
 *
 * @IncludeIntoJavadoc
 */
public class ArrayUtils {

	/**
	 * 一个不可变的空数组，类型为{@code Object} .
	 */
	public static final Object[] EMPTY_OBJECT_ARRAY = new Object[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Class} .
	 */
	public static final Class<?>[] EMPTY_CLASS_ARRAY = new Class[0];
	/**
	 * 一个不可变的空数组，类型为 {@code String} .
	 */
	public static final String[] EMPTY_STRING_ARRAY = new String[0];
	/**
	 * 一个不可变的空数组，类型为 {@code long} .
	 */
	public static final long[] EMPTY_LONG_ARRAY = new long[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Long} .
	 */
	public static final Long[] EMPTY_LONG_OBJECT_ARRAY = new Long[0];
	/**
	 * 一个不可变的空数组，类型为 {@code int} .
	 */
	public static final int[] EMPTY_INT_ARRAY = new int[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Integer} .
	 */
	public static final Integer[] EMPTY_INTEGER_OBJECT_ARRAY = new Integer[0];
	/**
	 * 一个不可变的空数组，类型为 {@code short} .
	 */
	public static final short[] EMPTY_SHORT_ARRAY = new short[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Short} .
	 */
	public static final Short[] EMPTY_SHORT_OBJECT_ARRAY = new Short[0];
	/**
	 * 一个不可变的空数组，类型为 {@code byte} .
	 */
	public static final byte[] EMPTY_BYTE_ARRAY = new byte[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Byte} .
	 */
	public static final Byte[] EMPTY_BYTE_OBJECT_ARRAY = new Byte[0];
	/**
	 * 一个不可变的空数组，类型为 {@code double} .
	 */
	public static final double[] EMPTY_DOUBLE_ARRAY = new double[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Double} .
	 */
	public static final Double[] EMPTY_DOUBLE_OBJECT_ARRAY = new Double[0];
	/**
	 * 一个不可变的空数组，类型为 {@code float} .
	 */
	public static final float[] EMPTY_FLOAT_ARRAY = new float[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Float} .
	 */
	public static final Float[] EMPTY_FLOAT_OBJECT_ARRAY = new Float[0];
	/**
	 * 一个不可变的空数组，类型为 {@code boolean} .
	 */
	public static final boolean[] EMPTY_BOOLEAN_ARRAY = new boolean[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Boolean} .
	 */
	public static final Boolean[] EMPTY_BOOLEAN_OBJECT_ARRAY = new Boolean[0];
	/**
	 * 一个不可变的空数组，类型为 {@code char} .
	 */
	public static final char[] EMPTY_CHAR_ARRAY = new char[0];
	/**
	 * 一个不可变的空数组，类型为 {@code Character} .
	 */
	public static final Character[] EMPTY_CHARACTER_OBJECT_ARRAY = new Character[0];

	/**
	 * 一个下标值，表示在list或者array中找不到一个元素： {@code -1}，
	 * 这个值会被本类中的方法返回 ，也可用于和{@link List}中的方法的返回值
     * 进行比较.
	 */
	public static final int INDEX_NOT_FOUND = -1;

	/**
     * 构造方法.
     *
	 * <p>
     * ArrayUtils类的实例在实际编程开发中不能被构造处来
	 * 然而，可以像这样来使用它ArrayUtils.clone(new int[] {2})
	 *</p>
     *
	 * <p>
     * 公共的构造方法允许那些需要JavaBean实例的工具类去操作它
     * </p>
	 */
	public ArrayUtils() {
		super();
	}


	// NOTE: Cannot use {@code} to enclose text which includes {}, but  is OK


	// Basic methods handling multi-dimensional arrays
	//-----------------------------------------------------------------------
	/**
	 * 把一个Object类型的数组转换为字符串输出.
	 *
	 * <p>
	 * 多维数组也能被正确处理，包括多维的基础类型的数组，同时处理 {@code null}
	 * </p>
	 *
	 * <pre>
	 * String[][] strings = {
	 *                 {"1", "2"},
	 *                 {"3", "4"}
	 *         };
	 * ArrayUtils.toString(strings) -&gt; {{1,2},{3,4}}
	 * ArrayUtils.toString(null) -&gt; {}
	 * </pre>
	 *
	 * @param array  一个要转换为字符串的数组，可能为 {@code null}
	 * @return 一个表示该数组的字符串, 如果输入一个null数组则结果为'{}'
	 * @see ArrayUtils#toString(Object, String)
	 */
	public static String toString(@Nullable final Object array) {
		return toString(array, "{}");
	}

	/**
     * 把一个Object类型的数组转换为字符串输出.
	 *
     * <p>
     * 多维数组也能被正确处理，包括多维的基础类型的数组，同时处理 {@code null}
     * </p>
	 *
	 * <pre>
	 * String[][] strings = {
	 *                 {"1", "2"},
	 *                 {"3", "4"}
	 *         };
	 * ArrayUtils.toString(Strings, "is null") -&gt; {{1,2},{3,4}}
	 * ArrayUtils.toString(null, "is null") -&gt; "is null"
	 * </pre>
	 *
	 * @param array  一个要转换为字符串的数组，可能为 {@code null}
	 * @param stringIfNull  如果输入的数组为 {@code null} 规定的返回的字符串
	 * @return 一个表示该数组的字符串
	 */
	public static String toString(@Nullable final Object array, final String stringIfNull) {

		if (array == null) {
			return stringIfNull;
		}
		return new ToStringBuilder(array, ToStringStyle.SIMPLE_STYLE).append(array).toString();
	}

	/**
	 * 获取一个数组的hashCode值.
	 *
	 * <p>
	 * 调用HashCodeBuilder的toHashCode方法获取数组的哈希值，该方法同样能处理多维基础类型的数组
	 * </p>
	 *
	 * <pre>
	 * i		nt[] ints = {1, 2, 3};
	 *         int[] ints1= {2, 3 , 4};
	 *         int[] ints2= {2, 3 , 4};
	 *         String[][] strings = {
	 *                 {"1", "2"},
	 *                 {"3", "4"}
	 *         };
	 *         String[][] strings1 = {
	 *                 {"3", "4"},
	 *                 {"5", "6"}
	 *         };
	 *         String[][] strings2 = {
	 *                 {"3", "4"},
	 *                 {"5", "6"}
	 *         };
	 * ArrayUtils.hashCode(ints1) == ArrayUtils.hashCode(ints2)
	 * ArrayUtils.hashCode(ints) != ArrayUtils.hashCode(ints1)
	 * ArrayUtils.hashCode(strings1) == ArrayUtils.hashCode(strings2)
	 * ArrayUtils.hashCode(strings) != ArrayUtils.hashCode(strings1)
	 * ArrayUtils.hashCode(ints) != ArrayUtils.hashCode(strings)
	 * </pre>
	 *
	 * @param array  一个要获取hashCode值的数组, 输入参数可为{@code null}
	 * @return 一个数组的hashCode值
	 */
	public static int hashCode(@Nullable final Object array) {
		return new HashCodeBuilder().append(array).toHashCode();
	}

	/**
	 * 比较两个数组是否相等.
	 *
	 * <p>
	 * 使用EqualisBuilder的isEquals方法比较两个数组, 该方法同样能处理多维基础类型的数组
	 * </p>
	 *
	 * <pre>
	 *     int[] ints = {1, 2, 3};
	 *         int[] ints1= {2, 3 , 4};
	 *         int[] ints2= {2, 3 , 4};
	 *         String[][] strings = {
	 *                 {"1", "2"},
	 *                 {"3", "4"}
	 *         };
	 *         String[][] strings1 = {
	 *                 {"3", "4"},
	 *                 {"5", "6"}
	 *         };
	 *         String[][] strings2 = {
	 *                 {"3", "4"},
	 *                 {"5", "6"}
	 *         };
	 * ArrayUtils.isEquals(null, null) -&gt; true
	 * ArrayUtils.isEquals(ints1, ints2) -&gt; true
	 * ArrayUtils.isEquals(strings1, strings2) -&gt; true
	 * ArrayUtils.isEquals(ints, ints1) -&gt; false
	 * ArrayUtils.isEquals(strings, strings1) -&gt; false
	 * ArrayUtils.isEquals(ints, strings) -&gt; false
	 * </pre>
	 *
	 * @param array1  左半边要比较的数组, 可能为 {@code null}
	 * @param array2  右半边要比较的数组, 可能为 {@code null}
	 * @return {@code true} 如果两个数组相等
	 * @deprecated 该方法已经被 {@code java.util.Objects.deepEquals(Object, Object)}代替 ，在
	 * 未来的发布版本中可能剔除
	 */
	@Deprecated
	public static boolean isEquals(@Nullable final Object array1, @Nullable final Object array2) {
		return new EqualsBuilder().append(array1, array2).isEquals();
	}

	// To map
	//-----------------------------------------------------------------------
	/**
	 * 把给定的数组转换为 {@link Map}.
	 *
	 * <p>
	 * 数组中的每一个元素必须为{@link Map.Entry}
	 * 或者是一个数组。该数组包括至少两个元素，第一个是键，第二个是值.
	 * 这个方法返回 {@code null} 如果输入是一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * 创建一个颜色相关的map
	 * Map colorMap = ArrayUtils.toMap(new String[][] {
	 *     {"RED", "#FF0000"},
	 *     {"GREEN", "#00FF00"},
	 *     {"BLUE", "#0000FF"}});
	 * </pre>
	 *
	 * @param array  一个元素为 {@link Map.Entry} 或者数组的数组
	 *  内部数组至少有两个元素, 并且可以为 {@code null}
	 * @return 一个从输入数组转换而来的 {@code Map}
	 * @throws IllegalArgumentException  如果数组的一个元素是一个含有少于两个元素的
	 *  数组则抛出该异常
	 * @throws IllegalArgumentException  如果输入数组包含不止
	 * {@link Map.Entry} 和数组类型的元素则抛出该异常
	 */
	public static Map<Object, Object> toMap(@Nullable final Object[] array) {

		if (array == null) {
			return null;
		}
		final Map<Object, Object> map = new HashMap<Object, Object>((int) (array.length * 1.5));
		for (int i = 0; i < array.length; i++) {
			final Object object = array[i];
			if (object instanceof Map.Entry<?, ?>) {
				final Map.Entry<?,?> entry = (Map.Entry<?,?>) object;
				map.put(entry.getKey(), entry.getValue());
			} else if (object instanceof Object[]) {
				final Object[] entry = (Object[]) object;
				if (entry.length < 2) {
					throw new IllegalArgumentException("Array element " + i + ", '"
							+ object
							+ "', has a length less than 2");
				}
				map.put(entry[0], entry[1]);
			} else {
				throw new IllegalArgumentException("Array element " + i + ", '"
						+ object
						+ "', is neither of type Map.Entry nor an Array");
			}
		}
		return map;
	}

	// Generic array
	//-----------------------------------------------------------------------
	/**
	 * 创建类型安全的泛型数组.
	 *
	 * <p>
	 * Java语言不允许从泛型创建一个数组
	 * </p>
	 *
	 * <pre>
	 *public static &lt;T&gt; T[] createAnArray(int size) {
	 *return new T[size]; // compiler error here
	 *}
	 *public static &lt;T&gt; T[] createAnArray(int size) {
	 *return (T[])new Object[size]; // ClassCastException at runtime
	 *}
	 * </pre>
	 *
	 * <p>
	 * 该方法可以创建泛型新数组
	 * 比如一个String类型的数组可以被创建：
	 * </p>
	 *
	 * <pre>
	 *String[] array = ArrayUtils.toArray("1", "2");
	 *String[] emptyArray = ArrayUtils.&lt;String&gt;toArray();
	 * </pre>
	 *
	 * <p>
	 * 该方法通常用于调用者自己使用了泛型，不得不组合成一个数组的情景
	 * </p>
	 *
	 * <p>
	 * 注意 该方法只对相同类型的参数有效 以便于编译器能够推断出数组的类型
	 * 此外也可以显示的选择类型：
	 * Number[] array = ArrayUtils.&lt;Number&gt;toArray(Integer.valueOf(42), Double.valueOf(Math.PI)),
	 * 这和下面的用法相比较其实没有区别：
	 * new Number[] {Integer.valueOf(42), Double.valueOf(Math.PI)}.
	 *</p>
	 *
	 * @param  <T>   数组元素的类型
	 * @param  items  多参数的数组项, 可以为{@code}null
	 * @return 一个不为 {@code}null的数组, 除非一个{@code}null数组传入
	 */
	public static <T> T[] toArray(final T... items) {
		return items;
	}

	// Clone
	//-----------------------------------------------------------------------
	/**
	 * 克隆一个T类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组，没有对
	 * 多维数组的特殊处理
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone(["1"]) -&gt; ["1"]
	 * </pre>
	 *
	 * @param <T> 数组元素的类型
	 * @param array  一个要克隆的数组, 可能为{@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static <T> T[] clone(@Nullable final T[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个long类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone([1]) -&gt; [1]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static long[] clone(@Nullable final long[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个int类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone([1]) -&gt; [1]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static int[] clone(@Nullable final int[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个short类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone([1]) -&gt; [1]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static short[] clone(@Nullable final short[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个char类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone([1]) -&gt; [1]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static char[] clone(@Nullable final char[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个byte类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone([1]) -&gt; [1]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static byte[] clone(@Nullable final byte[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个double类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone(1.0) -&gt; [1.0]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static double[] clone(@Nullable final double[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个float类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone([1.0f]) -&gt; [1.0f]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static float[] clone(@Nullable final float[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	/**
	 * 克隆一个boolean类型的数组.
	 *
	 * <p>
	 * 该方法返回 {@code null} 如果输入一个 {@code null} 数组
	 * 否则调用clone方法返回一个克隆出来的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.clone(null) -&gt; null
	 * ArrayUtils.clone([true]) -&gt; [true]
	 * </pre>
	 *
	 * @param array  要克隆的数组, 可能为 {@code null}
	 * @return 克隆出来的数组 ，其可能为{@code null} 如果输入{@code null}
	 */
	public static boolean[] clone(@Nullable final boolean[] array) {

		if (array == null) {
			return null;
		}
		return array.clone();
	}

	// nullToEmpty
	//-----------------------------------------------------------------------
	/**
	 * 把一个为 {@code null}的T类型数组引用转变成一个空的T数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的数组
	 * @param type   想要获得的数组的类型
	 * @param <T>  类型
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 * @throws IllegalArgumentException 如果期望数组类型为{@code null}则抛出该异常
	 */
	public static <T> T[] nullToEmpty(@Nullable final T[] array, final Class<T[]> type) {

		if (type == null) {
			throw new IllegalArgumentException("The type must not be null");
		}

		if (array == null) {
			return type.cast(Array.newInstance(type.getComponentType(), 0));
		}
		return array;
	}


	/**
	 * 把一个为 {@code null}的Object类型数组引用转变成一个空的Object数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new TestObject()]) -&gt; [testObject]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Object类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Object[] nullToEmpty(@Nullable final Object[] array) {

		if (isEmpty(array)) {
			return EMPTY_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Class类型数组引用转变成一个空的Class数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([String.class]) -&gt; [String.class]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Class类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Class<?>[] nullToEmpty(@Nullable final Class<?>[] array) {

		if (isEmpty(array)) {
			return EMPTY_CLASS_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的String类型数组引用转变成一个空的String数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty(["1"]) -&gt; ["1"]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的String类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static String[] nullToEmpty(@Nullable final String[] array) {

		if (isEmpty(array)) {
			return EMPTY_STRING_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的long类型数组引用转变成一个空的long数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([1]) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的long类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static long[] nullToEmpty(@Nullable final long[] array) {

		if (isEmpty(array)) {
			return EMPTY_LONG_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的int类型数组引用转变成一个空的int数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([1]) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的int类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static int[] nullToEmpty(@Nullable final int[] array) {

		if (isEmpty(array)) {
			return EMPTY_INT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的short类型数组引用转变成一个空的short数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([1]) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的short类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static short[] nullToEmpty(@Nullable final short[] array) {

		if (isEmpty(array)) {
			return EMPTY_SHORT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的char类型数组引用转变成一个空的char数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([1]) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的char类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static char[] nullToEmpty(@Nullable final char[] array) {

		if (isEmpty(array)) {
			return EMPTY_CHAR_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的byte类型数组引用转变成一个空的byte数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([1]) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的byte类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static byte[] nullToEmpty(@Nullable final byte[] array) {

		if (isEmpty(array)) {
			return EMPTY_BYTE_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的double类型数组引用转变成一个空的double数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([1.0]) -&gt; [1.0]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的double类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static double[] nullToEmpty(@Nullable final double[] array) {

		if (isEmpty(array)) {
			return EMPTY_DOUBLE_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的float类型数组引用转变成一个空的float数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([1.0f]) -&gt; [1.0f]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的float类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static float[] nullToEmpty(@Nullable final float[] array) {

		if (isEmpty(array)) {
			return EMPTY_FLOAT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的boolean类型数组引用转变成一个空的boolean数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([false]) -&gt; [false]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的boolean类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static boolean[] nullToEmpty(@Nullable final boolean[] array) {

		if (isEmpty(array)) {
			return EMPTY_BOOLEAN_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Long类型数组引用转变成一个空的Long数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Long(1)]) -&gt; [(Long)1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Long类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Long[] nullToEmpty(@Nullable final Long[] array) {

		if (isEmpty(array)) {
			return EMPTY_LONG_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Integer类型数组引用转变成一个空的Integer数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Integer(1)]) -&gt; [(Integer)1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Integer类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Integer[] nullToEmpty(@Nullable final Integer[] array) {

		if (isEmpty(array)) {
			return EMPTY_INTEGER_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Short类型数组引用转变成一个空的Short数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Short(1)]) -&gt; [(Short)1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Short类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Short[] nullToEmpty(@Nullable final Short[] array) {

		if (isEmpty(array)) {
			return EMPTY_SHORT_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Character类型数组引用转变成一个空的Character数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Character(1)]) -&gt; [(Character)1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Character类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Character[] nullToEmpty(@Nullable final Character[] array) {

		if (isEmpty(array)) {
			return EMPTY_CHARACTER_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Byte类型数组引用转变成一个空的Byte数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Byte(1)]) -&gt; [(Byte)1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Byte类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Byte[] nullToEmpty(@Nullable final Byte[] array) {

		if (isEmpty(array)) {
			return EMPTY_BYTE_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Double类型数组引用转变成一个空的Double数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Double(1)]) -&gt; [(Double)1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Double类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Double[] nullToEmpty(@Nullable final Double[] array) {

		if (isEmpty(array)) {
			return EMPTY_DOUBLE_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Float类型数组引用转变成一个空的Float数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Float(1)]) -&gt; [(Float)1]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Float类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Float[] nullToEmpty(@Nullable final Float[] array) {

		if (isEmpty(array)) {
			return EMPTY_FLOAT_OBJECT_ARRAY;
		}
		return array;
	}

	/**
	 * 把一个为 {@code null}的Boolean类型数组引用转变成一个空的Boolean数组的引用.
	 *
	 * <p>
	 * 该方法会为一个 {@code null} 输入数组返回一个空的数组
	 * </p>
	 *
	 *<pre>
	 * ArrayUtils.nullToEmpty(null) -&gt; []
	 * ArrayUtils.nullToEmpty([new Boolean(true)]) -&gt; [(Boolean)true]
	 *</pre>
	 *
	 * @param array  一个要检查是否为{@code null} 或空的Boolean类型的数组
	 * @return 数组自身, 当输入为 {@code null}或者空时返回类型为{@code public static}的空数组
	 */
	public static Boolean[] nullToEmpty(@Nullable final Boolean[] array) {

		if (isEmpty(array)) {
			return EMPTY_BOOLEAN_OBJECT_ARRAY;
		}
		return array;
	}

	// Subarrays
	//-----------------------------------------------------------------------
	/**
	 * 生成一个包含开始和结束指数之间的元素的新Object数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含
	 * {@code null}数组会生成{@code null}
	 * 子数组元素的类型总是和输入数组的一致，因此如果输入的数组的类型是
	 * {@code Date}, 那么会有如下设想的用法：
	 * </p>
	 *
	 * <pre>
	 * Date[] someDates = (Date[])ArrayUtils.subarray(allDates, 2, 5);
	 * </pre>
	 *
	 * @param <T> 数组元素的类型
	 * @param array  要切割的目标数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(Object[], int, int)
	 */
	public static <T> T[] subarray(@Nullable final T[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		final Class<?> type = array.getClass().getComponentType();
		if (newSize <= 0) {
			@SuppressWarnings("unchecked") // OK, because array is of type T
			final T[] emptyArray = (T[]) Array.newInstance(type, 0);
			return emptyArray;
		}
		@SuppressWarnings("unchecked") // OK, because array is of type T
		final T[] subarray = (T[]) Array.newInstance(type, newSize);
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新long数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([1,2], -1, 3) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], 0, 2) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], -1, 1) -&gt; [1]
	 * ArrayUtils.subarray([1,2], -1, 0) -&gt; []
	 * ArrayUtils.subarray([1,2], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的long类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(long[], int, int)
	 */
	public static long[] subarray(@Nullable final long[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_LONG_ARRAY;
		}

		final long[] subarray = new long[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新int数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([1,2], -1, 3) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], 0, 2) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], -1, 1) -&gt; [1]
	 * ArrayUtils.subarray([1,2], -1, 0) -&gt; []
	 * ArrayUtils.subarray([1,2], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的int类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(int[], int, int)
	 */
	public static int[] subarray(@Nullable final int[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_INT_ARRAY;
		}

		final int[] subarray = new int[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新short数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([1,2], -1, 3) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], 0, 2) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], -1, 1) -&gt; [1]
	 * ArrayUtils.subarray([1,2], -1, 0) -&gt; []
	 * ArrayUtils.subarray([1,2], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的short类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(short[], int, int)
	 */
	public static short[] subarray(@Nullable final short[] array, int startIndexInclusive, int endIndexExclusive) {
		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_SHORT_ARRAY;
		}

		final short[] subarray = new short[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新char数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([1,2], -1, 3) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], 0, 2) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], -1, 1) -&gt; [1]
	 * ArrayUtils.subarray([1,2], -1, 0) -&gt; []
	 * ArrayUtils.subarray([1,2], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的char类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(char[], int, int)
	 */
	public static char[] subarray(@Nullable final char[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_CHAR_ARRAY;
		}

		final char[] subarray = new char[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新byte数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([1,2], -1, 3) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], 0, 2) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], -1, 1) -&gt; [1]
	 * ArrayUtils.subarray([1,2], -1, 0) -&gt; []
	 * ArrayUtils.subarray([1,2], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的byte类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(byte[], int, int)
	 */
	public static byte[] subarray(@Nullable final byte[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_BYTE_ARRAY;
		}

		final byte[] subarray = new byte[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新double数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([1,2], -1, 3) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], 0, 2) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], -1, 1) -&gt; [1]
	 * ArrayUtils.subarray([1,2], -1, 0) -&gt; []
	 * ArrayUtils.subarray([1,2], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的double类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(double[], int, int)
	 */
	public static double[] subarray(@Nullable final double[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_DOUBLE_ARRAY;
		}

		final double[] subarray = new double[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新float数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([1,2], -1, 3) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], 0, 2) -&gt; [1,2]
	 * ArrayUtils.subarray([1,2], -1, 1) -&gt; [1]
	 * ArrayUtils.subarray([1,2], -1, 0) -&gt; []
	 * ArrayUtils.subarray([1,2], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的float类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(float[], int, int)
	 */
	public static float[] subarray(@Nullable final float[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_FLOAT_ARRAY;
		}

		final float[] subarray = new float[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	/**
	 * 生成一个包含开始和结束指数之间的元素的新boolean数组.
	 *
	 * <p>
	 * 开始下标元素被包含，结束下标元素不包含，
	 * 传入{@code null}数组会生成{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.subarray([true,false], -1, 3) -&gt; [true,false]
	 * ArrayUtils.subarray([true,false], 0, 2) -&gt; [true,false]
	 * ArrayUtils.subarray([true,false], -1, 1) -&gt; [true]
	 * ArrayUtils.subarray([true,false], -1, 0) -&gt; []
	 * ArrayUtils.subarray([true,false], 0, 0) -&gt; []
	 * </pre>
	 *
	 * @param array  要切割的boolean类型的数组
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示
	 *      该值小于开始下标值的 (&lt; startIndex) 生成一个空数组,
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 * @return 一个包含从开始指数到结束指数之间元素的新数组
	 * @see Arrays#copyOfRange(boolean[], int, int)
	 */
	public static boolean[] subarray(@Nullable final boolean[] array, int startIndexInclusive, int endIndexExclusive) {

		if (array == null) {
			return null;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive > array.length) {
			endIndexExclusive = array.length;
		}
		final int newSize = endIndexExclusive - startIndexInclusive;
		if (newSize <= 0) {
			return EMPTY_BOOLEAN_ARRAY;
		}

		final boolean[] subarray = new boolean[newSize];
		System.arraycopy(array, startIndexInclusive, subarray, 0, newSize);
		return subarray;
	}

	// Is same length
	//-----------------------------------------------------------------------
	/**
	 * 判断两个Object类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0},
	 * 数组的任何多维方面都被忽略
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([“1”], [“1”,“2”]) -&gt; false
	 * ArrayUtils.isSameLength([“1”,“2”], [“3”,“4”]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个Object类型的数组, 可能为 {@code null}
	 * @param array2 第二个Object类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 * @see ArrayUtils#getLength(Object)
	 */
	public static boolean isSameLength(@Nullable final Object[] array1, @Nullable final Object[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个long类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([1], [1,2]) -&gt; false
	 * ArrayUtils.isSameLength([1,2], [3,4]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个long类型的数组, 可能为 {@code null}
	 * @param array2 第二个long类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final long[] array1, @Nullable final long[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个int类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([1], [1,2]) -&gt; false
	 * ArrayUtils.isSameLength([1,2], [3,4]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个int类型的数组, 可能为 {@code null}
	 * @param array2 第二个int类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final int[] array1, @Nullable final int[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个short类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([1], [1,2]) -&gt; false
	 * ArrayUtils.isSameLength([1,2], [3,4]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个short类型的数组, 可能为 {@code null}
	 * @param array2 第二个short类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final short[] array1, @Nullable final short[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个char类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([1], [1,2]) -&gt; false
	 * ArrayUtils.isSameLength([1,2], [3,4]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个char类型的数组, 可能为 {@code null}
	 * @param array2 第二个char类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final char[] array1, @Nullable final char[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个byte类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([1], [1,2]) -&gt; false
	 * ArrayUtils.isSameLength([1,2], [3,4]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个byte类型的数组, 可能为 {@code null}
	 * @param array2 第二个byte类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final byte[] array1, @Nullable final byte[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个double类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([1], [1,2]) -&gt; false
	 * ArrayUtils.isSameLength([1,2], [3,4]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个double类型的数组, 可能为 {@code null}
	 * @param array2 第二个double类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final double[] array1, @Nullable final double[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个float类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([1], [1,2]) -&gt; false
	 * ArrayUtils.isSameLength([1,2], [3,4]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个float类型的数组, 可能为 {@code null}
	 * @param array2 第二个float类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final float[] array1, @Nullable final float[] array2) {
		return getLength(array1) == getLength(array2);
	}

	/**
	 * 判断两个boolean类型数组长度是否相同.
	 *
	 * <p>
	 * 通过getLength方法获取数组长度，把{@code null} 数组的长度看作 {@code 0}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSameLength([true], [true,false]) -&gt; false
	 * ArrayUtils.isSameLength([false,true], [true,false]) -&gt; true
	 * </pre>
	 *
	 * @param array1 第一个boolean类型的数组, 可能为 {@code null}
	 * @param array2 第二个boolean类型的数组, 可能为 {@code null}
	 * @return  如果两个数组长度相同返回{@code true} ，把{@code null} 看作空数组
	 */
	public static boolean isSameLength(@Nullable final boolean[] array1, @Nullable final boolean[] array2) {
		return getLength(array1) == getLength(array2);
	}

	//-----------------------------------------------------------------------
	/**
	 * 返回明确规定的数组的长度.
	 *
	 * <p>
	 * 如果输入的数组为 {@code null}, 则返回{@code 0}，该方法可以
	 * 处理类型为 {@code Object} 的数组和基础类型的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.getLength(null)            = 0
	 * ArrayUtils.getLength([])              = 0
	 * ArrayUtils.getLength([null])          = 1
	 * ArrayUtils.getLength([true, false])   = 2
	 * ArrayUtils.getLength([1, 2, 3])       = 3
	 * ArrayUtils.getLength(["a", "b", "c"])	= 3
	 * </pre>
	 *
	 * @param array  要检索长度的数组，可能为{@code null}
	 * @return 数组的长度,  如果数组为{@code null}则返回{@code 0}
	 * @throws IllegalArgumentException 如果参数不为数组则抛出该异常
	 */
	public static int getLength(@Nullable final Object array) {

		if (array == null) {
			return 0;
		}
		return Array.getLength(array);
	}

	/**
	 * 检查两个数组是否为同一类型并且考虑多维数组.
     *
     * <p>
     * 比较两个数组的类型的名字是否相同来判断
     * </p>
     *
     * <pre>
     *     String[] strings = {"1", "2"};
     *         String[][] stringss = {
     *                 {"1", "2"},
     *                 {"3", "4"}
     *         };
     *         String[][] stringss1 = {
     *                 {"5", "6"},
     *                 {"3", "4"}
     *         };
     *         String[][] stringss2 = {
     *                 {"5", "6", "7"},
     *                 {"3", "4"}
     *         };
     * ArrayUtils.isSameType(strings, stringss) -&gt; false
	 * ArrayUtils.isSameType(stringss, stringss1) -&gt; true
	 * ArrayUtils.isSameType(stringss1, stringss2) -&gt; true
     * </pre>
	 *
	 * @param array1 第一个Object类型的数组, 必须不为 {@code null}
	 * @param array2 第二个Object类型的数组，必须不为 {@code null}
	 * @return 如果的数组类型匹配则返回{@code true}
	 * @throws IllegalArgumentException 如果有一个数组为 {@code null}则抛出该异常
	 */
	public static boolean isSameType(final Object array1, final Object array2) {

		if (array1 == null || array2 == null) {
			throw new IllegalArgumentException("The Array must not be null");
		}
		return array1.getClass().getName().equals(array2.getClass().getName());
	}

	// Reverse
	//-----------------------------------------------------------------------
	/**
     * 翻转给定Object类型数组的顺序.
     *
     * <p>
     * 该方法对输入为{@code null}的数组不做任何处理，没有对多维数组的特殊处理
     * </p>
     *
     * <pre>
     *     ArrayUtils.reverse(["1","2"] -&gt; ["2","1"]
     * </pre>
	 *
	 * @param array  要翻转Object类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final Object[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
     * 翻转给定long类型数组的顺序.
     *
     * <p>
     * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
     * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2] -&gt; [2,1]
     * </pre>
	 *
	 * @param array  要翻转long类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final long[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
     * 翻转给定int类型数组的顺序.
     *
     * <p>
     * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
     * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2] -&gt; [2,1]
     * </pre>
	 *
	 * @param array  要翻转int类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final int[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
     * 翻转给定short类型数组的顺序.
     *
     * <p>
     * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
     * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2] -&gt; [2,1]
     * </pre>
	 *
	 * @param array  要翻转short类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final short[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
     * 翻转给定char类型数组的顺序.
     *
     * <p>
     * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
     * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2] -&gt; [2,1]
     * </pre>
	 *
	 * @param array  要翻转char类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final char[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
     * 翻转给定byte类型数组的顺序.
     *
     * <p>
     * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
     * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2] -&gt; [2,1]
     * </pre>
	 *
	 * @param array  要翻转byte类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final byte[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
     * 翻转给定double类型数组的顺序.
     *
     * <p>
     * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
     * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2] -&gt; [2,1]
     * </pre>
	 *
	 * @param array  要翻转double类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final double[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
	 * 翻转给定float类型数组的顺序.
	 *
	 * <p>
	 * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2] -&gt; [2,1]
     * </pre>
     *
	 * @param array  要翻转float类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final float[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
	 * 翻转给定boolean类型的数组的顺序.
	 *
	 * <p>
	 * 数组倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([true,false] -&gt; [false,true]
     * </pre>
     *
	 * @param array  要翻转boolean类型的数组 ，可能为 {@code null}
	 */
	public static void reverse(@Nullable final boolean[] array) {

		if (array == null) {
			return;
		}
		reverse(array, 0, array.length);
	}

	/**
	 * 翻转给定boolean类型数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse([true,true,false,false], -1, 5) -&gt; [false,false,true,true]
     *     ArrayUtils.reverse([true,true,false,false], 0, 5) -&gt; [false,false,true,true]
     *     ArrayUtils.reverse([true,true,false,false], -1, 4) -&gt; [false,false,true,true]
     *     ArrayUtils.reverse([true,true,false,false], 1, 2) -&gt; [true,false,true,false]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final boolean[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		boolean tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * 翻转给定byte数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse([1,2,3,4], -1, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 0, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], -1, 4) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 1, 2) -&gt; [1,3,2,4]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final byte[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		byte tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * 翻转给定char数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse([1,2,3,4], -1, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 0, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], -1, 4) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 1, 2) -&gt; [1,3,2,4]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final char[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		char tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * 翻转给定double数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse([1,2,3,4], -1, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 0, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], -1, 4) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 1, 2) -&gt; [1,3,2,4]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final double[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		double tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * 翻转给定float数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse([1,2,3,4], -1, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 0, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], -1, 4) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 1, 2) -&gt; [1,3,2,4]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final float[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		float tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * 翻转给定int数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse([1,2,3,4], -1, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 0, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], -1, 4) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 1, 2) -&gt; [1,3,2,4]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final int[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		int tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * 翻转给定long数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse([1,2,3,4], -1, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 0, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], -1, 4) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 1, 2) -&gt; [1,3,2,4]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final long[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		long tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * <p>
	 * 翻转给定Object数组的给定区间的顺序.
	 * </p>
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
	 *
     * <pre>
     *     ArrayUtils.reverse(["1","2","3","4"], -1, 5) -&gt; ["4","3","2","1"]
     *     ArrayUtils.reverse(["1","2","3","4"], 0, 5) -&gt; ["4","3","2","1"]
     *     ArrayUtils.reverse(["1","2","3","4"], -1, 4) -&gt; ["4","3","2","1"]
     *     ArrayUtils.reverse(["1","2","3","4"], 1, 2) -&gt; ["1","3","2","4"]
     * </pre>
     *
     * @param array 要翻转short类型的数组 ，可能为 {@code null}
     * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
     *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
     * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
     *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
     *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final Object[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		Object tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	/**
	 * 翻转给定short数组的给定区间的顺序.
	 *
	 * <p>
	 * 指定区间倒序，该方法对输入为{@code null}的数组不做任何处理
	 * </p>
     *
     * <pre>
     *     ArrayUtils.reverse([1,2,3,4], -1, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 0, 5) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], -1, 4) -&gt; [4,3,2,1]
     *     ArrayUtils.reverse([1,2,3,4], 1, 2) -&gt; [1,3,2,4]
     * </pre>
	 *
	 * @param array 要翻转short类型的数组 ，可能为 {@code null}
	 * @param startIndexInclusive  开始的下标. 小于0的 (&lt;0)
	 *      将被提升至0, 超出数组长度的 (&gt;array.length) 会生成一个空数组
	 * @param endIndexExclusive  到endIndex-1为止的元素会在返回的子数组中显示，
	 *      该值若小于开始下标值的 (&lt; startIndex) 生成一个空数组，
	 *      超出数组长度值的 (&gt;array.length) 会被下压至数组长度
	 */
	public static void reverse(@Nullable final short[] array, final int startIndexInclusive, final int endIndexExclusive) {

		if (array == null) {
			return;
		}
		int i = startIndexInclusive < 0 ? 0 : startIndexInclusive;
		int j = Math.min(array.length, endIndexExclusive) - 1;
		short tmp;
		while (j > i) {
			tmp = array[j];
			array[j] = array[i];
			array[i] = tmp;
			j--;
			i++;
		}
	}

	// Swap
	//-----------------------------------------------------------------------
	/**
	 * 交换给定Object数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap(["1", "2", "3"], 0, 2) -&gt; ["3", "2", "1"]
	 *     ArrayUtils.swap(["1", "2", "3"], 0, 0) -&gt; ["1", "2", "3"]
	 *     ArrayUtils.swap(["1", "2", "3"], 1, 0) -&gt; ["2", "1", "3"]
	 *     ArrayUtils.swap(["1", "2", "3"], 0, 5) -&gt; ["1", "2", "3"]
	 *     ArrayUtils.swap(["1", "2", "3"], -1, 1) -&gt; ["2", "1", "3"]
	 * </pre>
	 *
	 * @param array 要交换元素的Object类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 * @see ArrayUtils#swap(Object[], int, int, int)
	 */
	public static void swap(@Nullable final Object[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定long数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 * <pre>
	 * Examples:
     *     ArrayUtils.swap([1, 2, 3], 0, 2) -&gt; [3, 2, 1]
     *     ArrayUtils.swap([1, 2, 3], 0, 0) -&gt; [1, 2, 3]
     *     ArrayUtils.swap([1, 2, 3], 1, 0) -&gt; [2, 1, 3]
     *     ArrayUtils.swap([1, 2, 3], 0, 5) -&gt; [1, 2, 3]
     *     ArrayUtils.swap([1, 2, 3], -1, 1) -&gt; [2, 1, 3]
	 *</pre>
	 *
	 * @param array  要交换元素的long类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final long[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定int数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3], 0, 2) -&gt; [3, 2, 1]
	 *     ArrayUtils.swap([1, 2, 3], 0, 0) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], 1, 0) -&gt; [2, 1, 3]
	 *     ArrayUtils.swap([1, 2, 3], 0, 5) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], -1, 1) -&gt; [2, 1, 3]
	 * </pre>
	 *
	 * @param array  要交换元素的int类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final int[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定short数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 *<pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3], 0, 2) -&gt; [3, 2, 1]
	 *     ArrayUtils.swap([1, 2, 3], 0, 0) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], 1, 0) -&gt; [2, 1, 3]
	 *     ArrayUtils.swap([1, 2, 3], 0, 5) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], -1, 1) -&gt; [2, 1, 3]
	 * </pre>
	 *
	 * @param array  要交换元素的short类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final short[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定char数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 *<pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3], 0, 2) -&gt; [3, 2, 1]
	 *     ArrayUtils.swap([1, 2, 3], 0, 0) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], 1, 0) -&gt; [2, 1, 3]
	 *     ArrayUtils.swap([1, 2, 3], 0, 5) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], -1, 1) -&gt; [2, 1, 3]
	 * </pre>
	 *
	 * @param array  要交换元素的char类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final char[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定byte数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3], 0, 2) -&gt; [3, 2, 1]
	 *     ArrayUtils.swap([1, 2, 3], 0, 0) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], 1, 0) -&gt; [2, 1, 3]
	 *     ArrayUtils.swap([1, 2, 3], 0, 5) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], -1, 1) -&gt; [2, 1, 3]
	 * </pre>
	 *
	 * @param array  要交换元素的byte类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final byte[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定double数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3], 0, 2) -&gt; [3, 2, 1]
	 *     ArrayUtils.swap([1, 2, 3], 0, 0) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], 1, 0) -&gt; [2, 1, 3]
	 *     ArrayUtils.swap([1, 2, 3], 0, 5) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], -1, 1) -&gt; [2, 1, 3]
	 * </pre>
	 *
	 * @param array  要交换元素的double类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final double[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定float数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3], 0, 2) -&gt; [3, 2, 1]
	 *     ArrayUtils.swap([1, 2, 3], 0, 0) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], 1, 0) -&gt; [2, 1, 3]
	 *     ArrayUtils.swap([1, 2, 3], 0, 5) -&gt; [1, 2, 3]
	 *     ArrayUtils.swap([1, 2, 3], -1, 1) -&gt; [2, 1, 3]
	 * </pre>
	 *
	 * @param array  要交换元素的float类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final float[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定boolean数组中的两个元素.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null}输入或者
	 * 空数组输入或超出范围的下标指数不做任何操作.
	 * 负的下标指数会被提升至0
	 * </p>
	 *
	 * <pre>
	 * Examples:
     *     ArrayUtils.swap([true, false, true], 0, 2) -&gt; [true, false, true]
     *     ArrayUtils.swap([true, false, true], 0, 0) -&gt; [true, false, true]
     *     ArrayUtils.swap([true, false, true], 1, 0) -&gt; [false, true, true]
     *     ArrayUtils.swap([true, false, true], 0, 5) -&gt; [true, false, true]
     *     ArrayUtils.swap([true, false, true], -1, 1) -&gt; [false, true, true]
	 * </pre>
	 *
	 * @param array  要交换元素的boolean类型的数组 可能为 {@code null}
	 * @param offset1 第一个要交换元素的下标
	 * @param offset2 第二个要交换元素的下标
	 */
	public static void swap(@Nullable final boolean[] array, final int offset1, final int offset2) {

		if (array == null || array.length == 0) {
			return;
		}
		swap(array, offset1, offset2, 1);
	}

	/**
	 * 交换给定boolean数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([true, false, true, false], 0, 2, 1) -&gt; [true, false, true, false]
	 *     ArrayUtils.swap([true, false, true, false], 0, 0, 1) -&gt; [true, false, true, false]
	 *     ArrayUtils.swap([true, false, true, false], 0, 2, 2) -&gt; [true, false, true, false]
	 *     ArrayUtils.swap([true, false, true, false], -3, 2, 2) -&gt; [true, false, true, false]
	 *     ArrayUtils.swap([true, false, true, false], 0, 3, 3) -&gt; [false, false, true, true]
	 * </pre>
	 *
	 * @param array  要交换元素的boolean类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final boolean[] array, int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final boolean aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	/**
	 * 交换给定byte数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 *<pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 2, 1) -&gt; [3, 2, 1, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 0, 1) -&gt; [1, 2, 3, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 2, 0, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], -3, 2, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 3, 3) -&gt; [4, 2, 3, 1]
	 * </pre>
	 *
	 * @param array  要交换元素的byte类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final byte[] array, int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final byte aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	/**
	 * 交换给定char数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 2, 1) -&gt; [3, 2, 1, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 0, 1) -&gt; [1, 2, 3, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 2, 0, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], -3, 2, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 3, 3) -&gt; [4, 2, 3, 1]
	 * </pre>
	 *
	 * @param array  要交换元素的char类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final char[] array, int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final char aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	/**
	 * 交换给定char数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 2, 1) -&gt; [3, 2, 1, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 0, 1) -&gt; [1, 2, 3, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 2, 0, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], -3, 2, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 3, 3) -&gt; [4, 2, 3, 1]
	 * </pre>
	 *
	 * @param array  要交换元素的double类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final double[] array,  int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final double aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	/**
	 * 交换给定float数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 2, 1) -&gt; [3, 2, 1, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 0, 1) -&gt; [1, 2, 3, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 2, 0, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], -3, 2, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 3, 3) -&gt; [4, 2, 3, 1]
	 * </pre>
	 *
	 * @param array  要交换元素的float类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final float[] array, int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final float aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}

	}

	/**
	 * 交换给定int数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 2, 1) -&gt; [3, 2, 1, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 0, 1) -&gt; [1, 2, 3, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 2, 0, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], -3, 2, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 3, 3) -&gt; [4, 2, 3, 1]
	 * </pre>
	 *
	 * @param array  要交换元素的int类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final int[] array,  int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final int aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	/**
	 * 交换给定long数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 2, 1) -&gt; [3, 2, 1, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 0, 1) -&gt; [1, 2, 3, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 2, 0, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], -3, 2, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 3, 3) -&gt; [4, 2, 3, 1]
	 * </pre>
	 *
	 * @param array  要交换元素的long类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final long[] array,  int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final long aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	/**
	 * 交换给定Object数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap(["1", "2", "3", "4"], 0, 2, 1) -&gt; ["3", "2", "1", "4"]
	 *     ArrayUtils.swap(["1", "2", "3", "4"], 0, 0, 1) -&gt; ["1", "2", "3", "4"]
	 *     ArrayUtils.swap(["1", "2", "3", "4"], 2, 0, 2) -&gt; ["3", "4", "1", "2"]
	 *     ArrayUtils.swap(["1", "2", "3", "4"], -3, 2, 2) -&gt; ["3", "4", "1", "2"]
	 *     ArrayUtils.swap(["1", "2", "3", "4"], 0, 3, 3) -&gt; ["4", "2", "3", "1"]
	 * </pre>
	 *
	 * @param array  要交换元素的Object类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final Object[] array,  int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final Object aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	/**
	 * 交换给定short数组中的一系列元素.
	 *
	 * <p>
	 * 该方法对于{@code null}输入或者空数组输入或超出范围的下标指数不做任何操作。
	 * 负的下标指数会被提升至0, 如果在子数组中有任一要交换的元素下标超出范围，那么
	 * 会在数组长度的范围内交换尽可能多的元素。
     * 任一下标小于0都很被提升至0，下标依次自增1len次，实行两两交换。
	 * </p>
	 *
	 * <pre>
	 * Examples:
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 2, 1) -&gt; [3, 2, 1, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 0, 1) -&gt; [1, 2, 3, 4]
	 *     ArrayUtils.swap([1, 2, 3, 4], 2, 0, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], -3, 2, 2) -&gt; [3, 4, 1, 2]
	 *     ArrayUtils.swap([1, 2, 3, 4], 0, 3, 3) -&gt; [4, 2, 3, 1]
	 * </pre>
	 *
	 * @param array  要交换元素的short类型的数组 可能为 {@code null}
	 * @param offset1 要交换的系列元素中的第一个元素的下标
	 * @param offset2 要交换的系列元素中的第二个元素的下标
	 * @param len 从给定索引开始要交换的元素个数
	 */
	public static void swap(@Nullable final short[] array,  int offset1, int offset2, int len) {

		if (array == null || array.length == 0 || offset1 >= array.length || offset2 >= array.length) {
			return;
		}
		if (offset1 < 0) {
			offset1 = 0;
		}
		if (offset2 < 0) {
			offset2 = 0;
		}
		if (offset1 == offset2) {
			return;
		}
		len = Math.min(Math.min(len, array.length - offset1), array.length - offset2);
		for (int i = 0; i < len; i++, offset1++, offset2++) {
			final short aux = array[offset1];
			array[offset1] = array[offset2];
			array[offset2] = aux;
		}
	}

	// Shift
	//-----------------------------------------------------------------------
	/**
	 * 改变给定Object数组的顺序.
	 *
	 * <p>
     * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift(["1","2","3"], 1) -&gt; ["2","3","1"]
	 * ArrayUtils.shift(["1","2","3"], 2) -&gt; ["3","2","1"]
	 * ArrayUtils.shift(["1","2","3"], 4) -&gt; ["2","3","1"]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final Object[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定long数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3], 1) -&gt; [2,3,1]
	 * ArrayUtils.shift([1,2,3], 2) -&gt; [3,2,1]
	 * ArrayUtils.shift([1,2,3], 4) -&gt; [2,3,1]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final long[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定lnt数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3], 1) -&gt; [2,3,1]
	 * ArrayUtils.shift([1,2,3], 2) -&gt; [3,2,1]
	 * ArrayUtils.shift([1,2,3], 4) -&gt; [2,3,1]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final int[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定short数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3], 1) -&gt; [2,3,1]
	 * ArrayUtils.shift([1,2,3], 2) -&gt; [3,2,1]
	 * ArrayUtils.shift([1,2,3], 4) -&gt; [2,3,1]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final short[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定char数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3], 1) -&gt; [2,3,1]
	 * ArrayUtils.shift([1,2,3], 2) -&gt; [3,2,1]
	 * ArrayUtils.shift([1,2,3], 4) -&gt; [2,3,1]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final char[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定byte数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3], 1) -&gt; [2,3,1]
	 * ArrayUtils.shift([1,2,3], 2) -&gt; [3,2,1]
	 * ArrayUtils.shift([1,2,3], 4) -&gt; [2,3,1]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final byte[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定double数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1.0,2.0,3.0], 1) -&gt; [2.0,3.0,1.0]
	 * ArrayUtils.shift([1.0,2.0,3.0], 2) -&gt; [3.0,2.0,1.0]
	 * ArrayUtils.shift([1.0,2.0,3.0], 4) -&gt; [2.0,3.0,1.0]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final double[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定short数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1.0f,2.0f,3.0f], 1) -&gt; [2.0f,3.0f,1.0f]
	 * ArrayUtils.shift([1.0f,2.0f,3.0f], 2) -&gt; [3.0f,2.0f,1.0f]
	 * ArrayUtils.shift([1.0f,2.0f,3.0f], 4) -&gt; [2.0f,3.0f,1.0f]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final float[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 改变给定boolean数组的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理。该方法对于{@code null} 或者空数组输入不做任何操作
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([true,false,true], 1) -&gt; [true,true,false]
	 * ArrayUtils.shift([true,false,true], 2) -&gt; [false,true,true]
	 * ArrayUtils.shift([true,false,true], 4) -&gt; [true,true,false]
	 * </pre>
	 *
	 * @param array  要移动的数组, 可能为 {@code null}
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final boolean[] array, final int offset) {

		if (array == null) {
			return;
		}
		shift(array, 0, array.length, offset);
	}

	/**
	 * 移动给定boolean数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([true, false, true, false], -1, 5, 1) -&gt; [false, true, false, true]
	 * ArrayUtils.shift([true, false, true, false], 0, 4, 1) -&gt; [false, true, false, true]
	 * ArrayUtils.shift([true, false, true, false], 1, 3, 1) -&gt; [true, true, false, false]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final boolean[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定byte数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3,4], -1, 5, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 0, 4, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 1, 3, 1) -&gt; [1,3,2,4]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final byte[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定char数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3,4], -1, 5, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 0, 4, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 1, 3, 1) -&gt; [1,3,2,4]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final char[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定double数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3,4], -1, 5, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 0, 4, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 1, 3, 1) -&gt; [1,3,2,4]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final double[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定float数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3,4], -1, 5, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 0, 4, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 1, 3, 1) -&gt; [1,3,2,4]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final float[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定int数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3,4], -1, 5, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 0, 4, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 1, 3, 1) -&gt; [1,3,2,4]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final int[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定long数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3,4], -1, 5, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 0, 4, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 1, 3, 1) -&gt; [1,3,2,4]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final long[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定Object数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift(["1","2","3","4"], -1, 5, 1) -&gt; ["4","1","2","3"]
	 * ArrayUtils.shift(["1","2","3","4"], 0, 4, 1) -&gt; ["4","1","2","3"]
	 * ArrayUtils.shift(["1","2","3","4"], 1, 3, 1) -&gt; ["1","3","2","4"]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final Object[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	/**
	 * 移动给定short数组的一系列元素的顺序.
	 *
	 * <p>
	 * 没有对多维数组的特殊处理. 该方法对于{@code null} 或空数组输入不做任何操作。
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.shift([1,2,3,4], -1, 5, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 0, 4, 1) -&gt; [4,1,2,3]
	 * ArrayUtils.shift([1,2,3,4], 1, 3, 1) -&gt; [1,3,2,4]
	 * </pre>
	 *
	 * @param array
	 *            要移动的数组, 可能为 {@code null}
	 * @param startIndexInclusive
	 *            开始的下标. 过小的值(&lt;0)会被提升至0, 过大的值 (&gt;array.length) 会造成没有改变
	 * @param endIndexExclusive
	 *            到 endIndex-1为止的数组元素将被移动 。过小的值 (&lt; 开始下标) 会导致无改变。
	 *            过大的值 (&gt;数组长度) 会被下压至数组长度大小
	 * @param offset
	 *          数组元素移动的次数 ，如果该值大于数组的长度，则有效的移动次数是该值对数组长度求余的结果
	 */
	public static void shift(@Nullable final short[] array, int startIndexInclusive, int endIndexExclusive, int offset) {

		if (array == null) {
			return;
		}
		if (startIndexInclusive >= array.length - 1 || endIndexExclusive <= 0) {
			return;
		}
		if (startIndexInclusive < 0) {
			startIndexInclusive = 0;
		}
		if (endIndexExclusive >= array.length) {
			endIndexExclusive = array.length;
		}
		int n = endIndexExclusive - startIndexInclusive;
		if (n <= 1) {
			return;
		}
		offset %= n;
		if (offset < 0) {
			offset += n;
		}
		// For algorithm explanations and proof of O(n) time complexity and O(1) space complexity
		// see https://beradrian.wordpress.com/2015/04/07/shift-an-array-in-on-in-place/
		while (n > 1 && offset > 0) {
			final int n_offset = n - offset;

			if (offset > n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n - n_offset,  n_offset);
				n = offset;
				offset -= n_offset;
			} else if (offset < n_offset) {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset,  offset);
				startIndexInclusive += offset;
				n = n_offset;
			} else {
				swap(array, startIndexInclusive, startIndexInclusive + n_offset, offset);
				break;
			}
		}
	}

	// IndexOf search
	// ----------------------------------------------------------------------

	// Object IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在Object数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * String[] strings = {"123", "321", "132", "213"};
	 * ArrayUtils.indexOf(strings, "111") -&gt; -1
	 * ArrayUtils.indexOf(null, "111") -&gt; -1
	 * ArrayUtils.indexOf(strings, "123") -&gt; 0
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param objectToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final Object[] array, final Object objectToFind) {
		return indexOf(array, objectToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在Object数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * String[] strings = {"123", "321", "132", "213","123"};
	 * ArrayUtils.indexOf(strings, "111",-1) -&gt; -1
	 * ArrayUtils.indexOf(strings, null,1) -&gt; -1
	 * ArrayUtils.indexOf(null, "111", 1) -&gt; -1
	 * ArrayUtils.indexOf(strings, "123", 2) -&gt; 4
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param objectToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final Object[] array, final Object objectToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		if (objectToFind == null) {
			for (int i = startIndex; i < array.length; i++) {
				if (array[i] == null) {
					return i;
				}
			}
		} else {
			for (int i = startIndex; i < array.length; i++) {
				if (objectToFind.equals(array[i])) {
					return i;
				}
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到Object数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * String[] strings = {"123", "321", "132", "213","123"}
	 * ArrayUtils.lastIndexOf(strings, "123") -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param objectToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final Object[] array, final Object objectToFind) {
		return lastIndexOf(array, objectToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找Object数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 *String[] strings = {"123", "321", "132", "213","123"}
	 * ArrayUtils.lastIndexOf(null, "123", 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(strings,  "123", -2) -&gt; -1
	 * ArrayUtils.lastIndexOf(strings, null, 5) -&gt; -1
	 * ArrayUtils.lastIndexOf(strings, "123", 5) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param objectToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final Object[] array, final Object objectToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		if (objectToFind == null) {
			for (int i = startIndex; i >= 0; i--) {
				if (array[i] == null) {
					return i;
				}
			}
		} else if (array.getClass().getComponentType().isInstance(objectToFind)) {
			for (int i = startIndex; i >= 0; i--) {
				if (objectToFind.equals(array[i])) {
					return i;
				}
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定Object数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * String[] strings = {"123", "321", "132", "213","123"};
	 * ArrayUtils.contains(strings, "321") -&gt; true
	 * ArrayUtils.contains(strings, "111") -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param objectToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(@Nullable final Object[] array, final Object objectToFind) {
		return indexOf(array, objectToFind) != INDEX_NOT_FOUND;
	}

	// long IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在long数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * long[] longs = new long[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(longs, 5) -&gt; -1
	 * ArrayUtils.indexOf(null, 5) -&gt; -1
	 * ArrayUtils.indexOf(longs, 3) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final long[] array, final long valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在long数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * long[] longs = new long[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(longs, 5, -2) -&gt; -1
	 * ArrayUtils.indexOf(longs, 3, -2) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final long[] array, final long valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到long数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * long[] longs = new long[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(longs, 1) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final long[] array, final long valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找long数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * long[] longs = new long[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(longs, 1, -1)-&gt; -1
	 * ArrayUtils.lastIndexOf(longs, 8, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(longs, 1, 6)-&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final long[] array, final long valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定long数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * long[] longs = new long[]{1,2,3,4,1};
	 * ArrayUtils.contains(longs, 2) -&gt; true
	 * ArrayUtils.contains(longs, 6) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(final long[] array, final long valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	// int IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在int数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * int[] ints = new int[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(ints, 5) -&gt; -1
	 * ArrayUtils.indexOf(null, 5) -&gt; -1
	 * ArrayUtils.indexOf(ints, 3) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final int[] array, final int valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在int数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * int[] ints = new int[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(ints, 5, -2) -&gt; -1
	 * ArrayUtils.indexOf(ints, 3, -2) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final int[] array, final int valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到int数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * int[] ints = new int[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(ints, 1) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(final int[] array, final int valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找int数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * int[] ints = new int[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(ints, 1, -1)-&gt; -1
	 * ArrayUtils.lastIndexOf(ints, 8, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(ints, 1, 6)-&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final int[] array, final int valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定int数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * int[] ints = new int[]{1,2,3,4,1};
	 * ArrayUtils.contains(ints, 2) -&gt; true
	 * ArrayUtils.contains(ints, 6) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(final int[] array, final int valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	// short IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在short数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * short[] shorts = new short[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(shorts, 5) -&gt; -1
	 * ArrayUtils.indexOf(null, 5) -&gt; -1
	 * ArrayUtils.indexOf(shorts, 3) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final short[] array, final short valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在short数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * short[] shorts = new short[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(shorts, 5, -2) -&gt; -1
	 * ArrayUtils.indexOf(shorts, 3, -2) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final short[] array, final short valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到short数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * short[] shorts = new short[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(shorts, 1) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final short[] array, final short valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找short数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * short[] shorts = new short[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(shorts, 1, -1)-&gt; -1
	 * ArrayUtils.lastIndexOf(shorts, 8, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(shorts, 1, 6)-&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final short[] array, final short valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定short数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * short[] shorts = new short[]{1,2,3,4,1};
	 * ArrayUtils.contains(shorts, 2) -&gt; true
	 * ArrayUtils.contains(shorts, 6) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(final short[] array, final short valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	// char IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在char数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * char[] chars = new char[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(chars, 5) -&gt; -1
	 * ArrayUtils.indexOf(null, 5) -&gt; -1
	 * ArrayUtils.indexOf(chars, 3) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final char[] array, final char valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在char数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * char[] chars = new char[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(chars, 5, -2) -&gt; -1
	 * ArrayUtils.indexOf(chars, 3, -2) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final char[] array, final char valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到char数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * char[] chars = new char[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(chars, 1) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final char[] array, final char valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找char数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * char[] chars = new char[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(chars, 1, -1)-&gt; -1
	 * ArrayUtils.lastIndexOf(chars, 8, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(chars, 1, 6)-&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final char[] array, final char valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定char数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * char[] chars = new char[]{1,2,3,4,1};
	 * ArrayUtils.contains(chars, 2) -&gt; true
	 * ArrayUtils.contains(chars, 6) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(@Nullable final char[] array, final char valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	// byte IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在byte数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * byte[] bytes = new byte[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(bytes, 5) -&gt; -1
	 * ArrayUtils.indexOf(null, 5) -&gt; -1
	 * ArrayUtils.indexOf(bytes, 3) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final byte[] array, final byte valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在byte数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * byte[] bytes = new byte[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(bytes, 5, -2) &gt; -1
	 * ArrayUtils.indexOf(bytes, 3, -2) &gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final byte[] array, final byte valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到byte数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * byte[] bytes = new byte[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(bytes, 1) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final byte[] array, final byte valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找byte数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * byte[] bytes = new byte[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(bytes, 1, -1)-&gt; -1
	 * ArrayUtils.lastIndexOf(bytes, 8, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(bytes, 1, 6)-&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final byte[] array, final byte valueToFind, int startIndex) {

		if (array == null) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定byte数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * byte[] bytes = new byte[]{1,2,3,4,1};
	 * ArrayUtils.contains(bytes, 2) -&gt; true
	 * ArrayUtils.contains(bytes, 6) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(@Nullable final byte[] array, final byte valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	// double IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在double数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * double[] doubles = new double[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(doubles, 5) -&gt; -1
	 * ArrayUtils.indexOf(null, 5) -&gt; -1
	 * ArrayUtils.indexOf(doubles, 3) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final double[] array, final double valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 找到目标对象近似值在double数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 该方法会返回数组中第一个在目标值+-阈值范围内的元素的下标
	 *</p>
	 *
	 * <pre>
	 * doubles = new double[]{1,2,3,4,1,5.01};
	 * ArrayUtils.indexOf(doubles, 5.0,0.000001 ) -&gt; -1
	 * ArrayUtils.indexOf(doubles, 5.0, 0.1) -&gt; 5
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param tolerance 搜索的值的活动范围
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final double[] array, final double valueToFind, final double tolerance) {
		return indexOf(array, valueToFind, 0, tolerance);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在double数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * double[] doubles = new double[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(doubles, 5, -2) -&gt; -1
	 * ArrayUtils.indexOf(doubles, 3, -2) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final double[] array, final double valueToFind, int startIndex) {

		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 从给定的索引值开始寻找目标对象近似值在double数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * 从给定的起点开始搜索数组中第一个在目标值+-阈值范围内的元素
	 * 返回其在数组中的下标
	 * </p>
	 *
	 * <pre>
	 * doubles = new double[]{1,2,3,4,1,5.01};
	 * ArrayUtils.indexOf(doubles, 5.0, -1,0.000001 ) -&gt; -1
	 * ArrayUtils.indexOf(doubles, 5.0, -1,0.1) -&gt; 5
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @param tolerance 搜索的值的活动范围
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final double[] array, final double valueToFind, int startIndex, final double tolerance) {

		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		final double min = valueToFind - tolerance;
		final double max = valueToFind + tolerance;
		for (int i = startIndex; i < array.length; i++) {
			if (array[i] >= min && array[i] <= max) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到double数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * double[] doubles = new double[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(doubles, 1) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final double[] array, final double valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 找到double数组中最后一个给定对象的近似值的下标值.
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 搜索数组中最后一个在目标值+-阈值范围内的元素返回其在数组中的下标
	 *</p>
	 *
	 * <pre>
	 * doubles = new double[]{1,2,3,4,1,5.01};
	 * ArrayUtils.lastIndexOf(doubles,5.0, 0.1) -&gt; 5
	 * ArrayUtils.indexOf(doubles, 5.0, 0.000001 ) -&gt; -1
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param tolerance 搜索的值的活动范围
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final double[] array, final double valueToFind, final double tolerance) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE, tolerance);
	}

	/**
	 * 从给定的起点寻找double数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * double[] doubles = new double[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(doubles, 1, -1)-&gt; -1
	 * ArrayUtils.lastIndexOf(doubles, 8, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(doubles, 1, 6)-&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final double[] array, final double valueToFind, int startIndex) {

		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 从给定的起点寻找double数组中最后一个给定对象近似值的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象。返回最后一个数组中在目标值+-阈值范围内的元素的下标
	 * </p>
	 *
	 * <pre>
	 * doubles = new double[]{1,2,3,4,1,5.01};
	 * ArrayUtils.lastIndexOf(doubles, 5.0, -1,0.000001) -&gt; -1
	 * ArrayUtils.lastIndexOf(doubles, 5.0, 5,0.1) -&gt; 5
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @param tolerance  搜索的目标值的活动范围
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final double[] array, final double valueToFind, int startIndex, final double tolerance) {

		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		final double min = valueToFind - tolerance;
		final double max = valueToFind + tolerance;
		for (int i = startIndex; i >= 0; i--) {
			if (array[i] >= min && array[i] <= max) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定double数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * double[] doubles = new double[]{1,2,3,4,1};
	 * ArrayUtils.contains(doubles, 2) -&gt; true
	 * ArrayUtils.contains(doubles, 6) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(@Nullable final double[] array, final double valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象近似值是否在给定double数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * 检查目标对象+-阈值范围内是否有值在数组中
	 * </p>
	 *
	 * <pre>
	 * doubles = new double[]{1,2,3,4,1,5.01};
	 * ArrayUtils.contains(doubles, 5.0, 0.1) -&gt; true
	 * ArrayUtils.contains(doubles, 5.0,0.000001) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @param tolerance  搜索值的活动范围
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(@Nullable final double[] array, final double valueToFind, final double tolerance) {
		return indexOf(array, valueToFind, 0, tolerance) != INDEX_NOT_FOUND;
	}

	// float IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在float数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * float[] floats = new float[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(floats, 5) -&gt; -1
	 * ArrayUtils.indexOf(null, 5) -&gt; -1
	 * ArrayUtils.indexOf(floats, 3) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final float[] array, final float valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在float数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * float[] floats = new float[]{1,2,3,4,1};
	 * ArrayUtils.indexOf(floats, 5, -2) -&gt; -1
	 * ArrayUtils.indexOf(floats, 3, -2) -&gt; 2
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final float[] array, final float valueToFind, int startIndex) {

		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到float数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * float[] floats = new float[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(floats, 1) -&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final float[] array, final float valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找float数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * float[] floats = new float[]{1,2,3,4,1};
	 * ArrayUtils.lastIndexOf(floats, 1, -1)-&gt; -1
	 * ArrayUtils.lastIndexOf(floats, 8, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(floats, 1, 6)-&gt; 4
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final float[] array, final float valueToFind, int startIndex) {

		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定float数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * float[] floats = new float[]{1,2,3,4,1};
	 * ArrayUtils.contains(floats, 2) -&gt; true
	 * ArrayUtils.contains(floats, 6) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(@Nullable final float[] array, final float valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	// boolean IndexOf
	//-----------------------------------------------------------------------
	/**
	 * 找到目标对象在boolean数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 *  booleans = new boolean[]{true, false, false, true};
	 * ArrayUtils.indexOf(booleans, true) -&gt; 0
	 * ArrayUtils.indexOf(null, true) -&gt; -1
	 * ArrayUtils.indexOf(booleans, false) -&gt; 1
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @return 目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final boolean[] array, final boolean valueToFind) {
		return indexOf(array, valueToFind, 0);
	}

	/**
	 * 从给定的索引值开始寻找目标对象在boolean数组中的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组。
	 * 负的开始索引会被当做0，如果开始索引大于数组长度则返回{@link #INDEX_NOT_FOUND} ({@code -1})。
	 * </p>
	 *
	 * <pre>
	 * boolean[] booleans = new boolean[]{true, true, true, true};
	 * ArrayUtils.indexOf(booleans, true, -2) -&gt; 0
	 * ArrayUtils.indexOf(booleans, false, 0) -&gt; -1
	 * </pre>
	 *
	 * @param array  要搜索对象的数组, 可能为 {@code null}
	 * @param valueToFind  要搜索的对象, 可能为 {@code null}
	 * @param startIndex  开始寻找的起点的索引
	 * @return 从给定起点开始寻找到的目标对象在数组中的下标值
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int indexOf(@Nullable final boolean[] array, final boolean valueToFind, int startIndex) {

		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			startIndex = 0;
		}
		for (int i = startIndex; i < array.length; i++) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 找到boolean数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * boolean[] booleans = new boolean[]{true, true, true, true};
	 * ArrayUtils.lastIndexOf(booleans,true)-&gt; 3
	 * ArrayUtils.lastIndexOf(booleans,false)-&gt; -1
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(@Nullable final boolean[] array, final boolean valueToFind) {
		return lastIndexOf(array, valueToFind, Integer.MAX_VALUE);
	}

	/**
	 * 从给定的起点寻找boolean数组中最后一个给定对象的下标值.
	 *
	 * <p>
	 * 该方法会返回{@link #INDEX_NOT_FOUND} ({@code -1}) 如果输入一个 {@code null} 数组
	 * 一个负的起点值会返回{@link #INDEX_NOT_FOUND} ({@code -1}). 一个超过数组长度的起点值会
	 * 从数组结尾开始搜索目标对象
	 * </p>
	 *
	 * <pre>
	 * boolean[] booleans = new boolean[]{true, true, true, true};
	 * ArrayUtils.lastIndexOf(booleans, true, -1) -&gt; -1
	 * ArrayUtils.lastIndexOf(booleans, false, 2) -&gt; -1
	 * ArrayUtils.lastIndexOf(booleans, true, 6) -&gt; 3
	 * </pre>
	 *
	 * @param array  需要查找对象的数组, 可能为 {@code null}
	 * @param valueToFind  目标对象, 可能为 {@code null}
	 * @param startIndex  向后遍历的起点值
	 * @return 数组中最后一个目标对象的下标
	 * {@link #INDEX_NOT_FOUND} ({@code -1}) 如果没找到或者输入{@code null} 数组会返回它
	 */
	public static int lastIndexOf(final boolean[] array, final boolean valueToFind, int startIndex) {
		if (isEmpty(array)) {
			return INDEX_NOT_FOUND;
		}
		if (startIndex < 0) {
			return INDEX_NOT_FOUND;
		} else if (startIndex >= array.length) {
			startIndex = array.length - 1;
		}
		for (int i = startIndex; i >= 0; i--) {
			if (valueToFind == array[i]) {
				return i;
			}
		}
		return INDEX_NOT_FOUND;
	}

	/**
	 * 检查目标对象是否在给定boolean数组中.
	 *
	 * <p>
	 * 该方法返回{@code false} 如果传入一个{@code null} 数组
	 * </p>
	 *
	 * <pre>
	 * boolean[] booleans = new boolean[]{true, true, true, true};
	 * ArrayUtils.contains(booleans, true) -&gt; true
	 * ArrayUtils.contains(booleans, false) -&gt; false
	 * </pre>
	 *
	 * @param array  要搜索的数组
	 * @param valueToFind  要查找的对象
	 * @return {@code true} 如果数组中有目标对象则返回该值
	 */
	public static boolean contains(final boolean[] array, final boolean valueToFind) {
		return indexOf(array, valueToFind) != INDEX_NOT_FOUND;
	}

	// Primitive/Object array converters
	// ----------------------------------------------------------------------

	// Character array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Character对象数组为基本类型char数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Character[] characters = new Character[]{new Character('a'), new Character('b')};
	 * ArrayUtils.toPrimitive(characters) -&gt; ['a', 'b']
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Character}数组, 可能为 {@code null}
	 * @return 一个{@code char} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static char[] toPrimitive(@Nullable final Character[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_CHAR_ARRAY;
		}
		final char[] result = new char[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].charValue();
		}
		return result;
	}

	/**
	 * 转换Character对象数组为基本类型char数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Character[] characters = new Character[]{null};
	 * ArrayUtils.toPrimitive(characters, 'n') -&gt; ['n']
	 *</pre>
	 *
	 * @param array  一个 {@code Character}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code char} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static char[] toPrimitive(@Nullable final Character[] array, final char valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_CHAR_ARRAY;
		}
		final char[] result = new char[array.length];
		for (int i = 0; i < array.length; i++) {
			final Character b = array[i];
			result[i] = (b == null ? valueForNull : b.charValue());
		}
		return result;
	}

	/**
	 * 转换char对象数组为Character数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * char[] chars = new char[]{'a'};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(chars) -&gt; [new Character('a')]
	 * </pre>
	 *
	 * @param array 一个 {@code char}数组
	 * @return  一个{@code Character} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Character[] toObject(@Nullable final char[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_CHARACTER_OBJECT_ARRAY;
		}
		final Character[] result = new Character[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = Character.valueOf(array[i]);
		}
		return result;
	}

	// Long array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Long对象数组为基本类型long数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Long[] longs = new Long[]{new Long(1), new Long(2)};
	 * ArrayUtils.toPrimitive(longs) -&gt; [1,2]
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Long}数组, 可能为 {@code null}
	 * @return 一个{@code long} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static long[] toPrimitive(@Nullable final Long[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_LONG_ARRAY;
		}
		final long[] result = new long[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].longValue();
		}
		return result;
	}

	/**
	 * 转换Long对象数组为基本类型long数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Long[] longs = new Long[]{null};
	 * ArrayUtils.toPrimitive(longs, 1) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个 {@code Long}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code long} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static long[] toPrimitive(@Nullable final Long[] array, final long valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_LONG_ARRAY;
		}
		final long[] result = new long[array.length];
		for (int i = 0; i < array.length; i++) {
			final Long b = array[i];
			result[i] = (b == null ? valueForNull : b.longValue());
		}
		return result;
	}

	/**
	 * 转换long对象数组为Long数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * long[] longs = new long[]{1};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(longs) -&gt; [new Long(1)]
	 * </pre>
	 *
	 * @param array 一个 {@code long}数组
	 * @return  一个{@code Long} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Long[] toObject(@Nullable final long[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_LONG_OBJECT_ARRAY;
		}
		final Long[] result = new Long[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = Long.valueOf(array[i]);
		}
		return result;
	}

	// Int array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Integer对象数组为基本类型int数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Integer[] ints = new Integer[]{new Integer(1), new Integer(2)};
	 * ArrayUtils.toPrimitive(ints) -&gt; [1,2]
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Integer}数组, 可能为 {@code null}
	 * @return 一个{@code int} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static int[] toPrimitive(@Nullable final Integer[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_INT_ARRAY;
		}
		final int[] result = new int[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].intValue();
		}
		return result;
	}

	/**
	 * 转换Integer对象数组为基本类型int数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Integer[] ints = new Integer[]{null};
	 * ArrayUtils.toPrimitive(ints, 1) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个 {@code Integer}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code int} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static int[] toPrimitive(@Nullable final Integer[] array, final int valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_INT_ARRAY;
		}
		final int[] result = new int[array.length];
		for (int i = 0; i < array.length; i++) {
			final Integer b = array[i];
			result[i] = (b == null ? valueForNull : b.intValue());
		}
		return result;
	}

	/**
	 * 转换int对象数组为Integer数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * int[] ints = new int[]{1};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(ints) -&gt; [new Integer(1)]
	 * </pre>
	 *
	 * @param array 一个 {@code int}数组
	 * @return  一个{@code Integer} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Integer[] toObject(final int[] array) {
		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_INTEGER_OBJECT_ARRAY;
		}
		final Integer[] result = new Integer[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = Integer.valueOf(array[i]);
		}
		return result;
	}

	// Short array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Short对象数组为基本类型short数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Short[] shorts = new Short[]{new Short("1"), new Short("2")};
	 * ArrayUtils.toPrimitive(shorts) -&gt; [1,2]
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Short}数组, 可能为 {@code null}
	 * @return 一个{@code short} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static short[] toPrimitive(@Nullable final Short[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_SHORT_ARRAY;
		}
		final short[] result = new short[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].shortValue();
		}
		return result;
	}

	/**
	 * 转换Short对象数组为基本类型short数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Short[] shorts = new Short[]{null};
	 * ArrayUtils.toPrimitive(shorts, 1) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个 {@code Short}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code short} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static short[] toPrimitive(@Nullable final Short[] array, final short valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_SHORT_ARRAY;
		}
		final short[] result = new short[array.length];
		for (int i = 0; i < array.length; i++) {
			final Short b = array[i];
			result[i] = (b == null ? valueForNull : b.shortValue());
		}
		return result;
	}

	/**
	 * 转换short对象数组为Short数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * short[] shorts = new short[]{1};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(shorts) -&gt; [new Short(1)]
	 * </pre>
	 *
	 * @param array 一个 {@code short}数组
	 * @return  一个{@code Short} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Short[] toObject(@Nullable final short[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_SHORT_OBJECT_ARRAY;
		}
		final Short[] result = new Short[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = Short.valueOf(array[i]);
		}
		return result;
	}

	// Byte array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Byte对象数组为基本类型byte数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Byte[] bytes = new Byte[]{new Byte("1"), new Byte("2")};
	 * ArrayUtils.toPrimitive(bytes) -&gt; [1,2]
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Byte}数组, 可能为 {@code null}
	 * @return 一个{@code byte} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static byte[] toPrimitive(@Nullable final Byte[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_BYTE_ARRAY;
		}
		final byte[] result = new byte[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].byteValue();
		}
		return result;
	}

	/**
	 * 转换Byte对象数组为基本类型byte数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Byte[] bytes = new Byte[]{null};
	 * ArrayUtils.toPrimitive(bytes, 1) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个 {@code Byte}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code byte} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static byte[] toPrimitive(@Nullable final Byte[] array, final byte valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_BYTE_ARRAY;
		}
		final byte[] result = new byte[array.length];
		for (int i = 0; i < array.length; i++) {
			final Byte b = array[i];
			result[i] = (b == null ? valueForNull : b.byteValue());
		}
		return result;
	}

	/**
	 * 转换byte对象数组为Byte数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * byte[] bytes = new byte[]{1};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(bytes) -&gt; [new Byte(1)]
	 * </pre>
	 *
	 * @param array 一个 {@code byte}数组
	 * @return  一个{@code Byte} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Byte[] toObject(@Nullable final byte[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_BYTE_OBJECT_ARRAY;
		}
		final Byte[] result = new Byte[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = Byte.valueOf(array[i]);
		}
		return result;
	}

	// Double array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Double对象数组为基本类型double数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Double[] doubles = new Double[]{new Double("1"), new Double("2")};
	 * ArrayUtils.toPrimitive(doubles) -&gt; [1,2]
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Double}数组, 可能为 {@code null}
	 * @return 一个{@code double} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static double[] toPrimitive(@Nullable final Double[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_DOUBLE_ARRAY;
		}
		final double[] result = new double[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].doubleValue();
		}
		return result;
	}

	/**
	 * 转换Double对象数组为基本类型double数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Double[] doubles = new Double[]{null};
	 * ArrayUtils.toPrimitive(doubles, 1) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个 {@code Double}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code double} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static double[] toPrimitive(@Nullable final Double[] array, final double valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_DOUBLE_ARRAY;
		}
		final double[] result = new double[array.length];
		for (int i = 0; i < array.length; i++) {
			final Double b = array[i];
			result[i] = (b == null ? valueForNull : b.doubleValue());
		}
		return result;
	}

	/**
	 * 转换double对象数组为Double数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * double[] doubles = new double[]{1};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(doubles) -&gt; [new Double(1)]
	 * </pre>
	 *
	 * @param array 一个 {@code double}数组
	 * @return  一个{@code Double} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Double[] toObject(@Nullable final double[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_DOUBLE_OBJECT_ARRAY;
		}
		final Double[] result = new Double[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = Double.valueOf(array[i]);
		}
		return result;
	}

	//   Float array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Float对象数组为基本类型float数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Float[] floats = new Float[]{new Float("1"), new Float("2")};
	 * ArrayUtils.toPrimitive(floats) -&gt; [1,2]
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Float}数组, 可能为 {@code null}
	 * @return 一个{@code float} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static float[] toPrimitive(@Nullable final Float[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_FLOAT_ARRAY;
		}
		final float[] result = new float[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].floatValue();
		}
		return result;
	}

	/**
	 * 转换Float对象数组为基本类型float数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Float[] floats = new Float[]{null};
	 * ArrayUtils.toPrimitive(floats, 1) -&gt; [1]
	 *</pre>
	 *
	 * @param array  一个 {@code Float}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code float} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static float[] toPrimitive(@Nullable final Float[] array, final float valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_FLOAT_ARRAY;
		}
		final float[] result = new float[array.length];
		for (int i = 0; i < array.length; i++) {
			final Float b = array[i];
			result[i] = (b == null ? valueForNull : b.floatValue());
		}
		return result;
	}

	/**
	 * 转换float对象数组为Float数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * float[] floats = new float[]{1};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(floats) -&gt; [new Float(1)]
	 * </pre>
	 *
	 * @param array 一个 {@code float}数组
	 * @return  一个{@code Float} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Float[] toObject(@Nullable final float[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_FLOAT_OBJECT_ARRAY;
		}
		final Float[] result = new Float[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = Float.valueOf(array[i]);
		}
		return result;
	}

	/**
	 *从包装器类型的数组创建基元类型的数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * @param array  一个包装器类型对象的数组
	 * @return 对应基元类型的数组，或原始数组
	 */
	public static Object toPrimitive(@Nullable final Object array) {

		if (array == null) {
			return null;
		}
		final Class<?> ct = array.getClass().getComponentType();
		final Class<?> pt = ClassUtils.wrapperToPrimitive(ct);
		if(Integer.TYPE.equals(pt)) {
			return toPrimitive((Integer[]) array);
		}
		if(Long.TYPE.equals(pt)) {
			return toPrimitive((Long[]) array);
		}
		if(Short.TYPE.equals(pt)) {
			return toPrimitive((Short[]) array);
		}
		if(Double.TYPE.equals(pt)) {
			return toPrimitive((Double[]) array);
		}
		if(Float.TYPE.equals(pt)) {
			return toPrimitive((Float[]) array);
		}
		return array;
	}

	// Boolean array converters
	// ----------------------------------------------------------------------
	/**
	 * 转换Boolean对象数组为基本类型boolean数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * Boolean[] booleans = new Boolean[]{new Boolean(true), new Boolean(false)};
	 * ArrayUtils.toPrimitive(booleans) -&gt; [true,false]
	 * ArrayUtils.toPrimitive({@code null}) -&gt; {@code null}
	 * ArrayUtils.toPrimitive([]) -&gt; []
	 * </pre>
	 *
	 * @param array  一个 {@code Boolean}数组, 可能为 {@code null}
	 * @return 一个{@code boolean} 数组, 如果输入为{@code null}则其为{@code null}
	 * @throws NullPointerException 如果数组内容为{@code null}则抛出该异常
	 */
	public static boolean[] toPrimitive(@Nullable final Boolean[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_BOOLEAN_ARRAY;
		}
		final boolean[] result = new boolean[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].booleanValue();
		}
		return result;
	}

	/**
	 * 转换Boolean对象数组为基本类型boolean数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 *<pre>
	 * Boolean[] booleans = new Boolean[]{null};
	 * ArrayUtils.toPrimitive(booleans, true) -&gt; [true]
	 *</pre>
	 *
	 * @param array  一个 {@code Boolean}数组, 可能为 {@code null}
	 * @param valueForNull  如果元素为 {@code null} 用该值替换
	 * @return 一个{@code boolean} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static boolean[] toPrimitive(@Nullable final Boolean[] array, final boolean valueForNull) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_BOOLEAN_ARRAY;
		}
		final boolean[] result = new boolean[array.length];
		for (int i = 0; i < array.length; i++) {
			final Boolean b = array[i];
			result[i] = (b == null ? valueForNull : b.booleanValue());
		}
		return result;
	}

	/**
	 * 转换boolean对象数组为Boolean数组.
	 *
	 * <p>
	 * 该方法为{@code null}输入返回{@code null},同时可以处理{@code null}
	 * </p>
	 *
	 * <pre>
	 * boolean[] booleans = new boolean[]{true};
	 * ArrayUtils.toObject({@code null}) -&gt; {@code null}
	 * ArrayUtils.toObject([]) -&gt; []
	 * ArrayUtils.toObject(booleans) -&gt; [new Boolean(true)]
	 * </pre>
	 *
	 * @param array 一个 {@code boolean}数组
	 * @return  一个{@code Boolean} 数组, 如果输入为{@code null}则其为{@code null}
	 */
	public static Boolean[] toObject(@Nullable final boolean[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_BOOLEAN_OBJECT_ARRAY;
		}
		final Boolean[] result = new Boolean[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = (array[i] ? Boolean.TRUE : Boolean.FALSE);
		}
		return result;
	}

	// ----------------------------------------------------------------------
	/**
	 * 检查一个Object数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new String[]{"1"}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 * @see ArrayUtils#getLength(Object)
	 */
	public static boolean isEmpty(@Nullable final Object[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个long数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new long[]{1}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final long[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个int数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new int[]{1}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final int[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个short数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new short[]{1}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final short[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个char数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new char[]{1}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final char[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个byte数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new byte[]{1}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final byte[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个double数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new double[]{1}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final double[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个float数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new float[]{1}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final float[] array) {
		return getLength(array) == 0;
	}

	/**
	 *检查一个boolean数组是否为空或{@code null}.
	 *
	 * <p>
	 * 通过长度是否为0来判断，借助getLength方法
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isEmpty({@code null}) -&gt; true
	 * ArrayUtils.isEmpty([]) -&gt; true
	 * ArrayUtils.isEmpty(new boolean[]{true}) -&gt; false
	 * </pre>
	 *
	 * @param array  要检查的数组
	 * @return {@code true} 如果数组为空或者{@code null}返回该值
	 */
	public static boolean isEmpty(@Nullable final boolean[] array) {
		return getLength(array) == 0;
	}

	// ----------------------------------------------------------------------
	/**
	 * 检查一个Object数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new String[]{"1"}) -&gt; true
	 * </pre>
	 *
	 * @param <T> 数组元素的类型
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static <T> boolean isNotEmpty(@Nullable final T[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个long数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new long[]{1}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final long[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个int数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new int[]{1}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final int[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个short数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new short[]{1}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final short[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个char数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new char[]{1}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final char[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个byte数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new byte[]{1}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final byte[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个double数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new double[]{1}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final double[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个float数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new float[]{1}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final float[] array) {
		return !isEmpty(array);
	}

	/**
	 * 检查一个boolean数组是否不为空或 {@code null}.
	 *
	 * <p>
	 * 借助isEmpty结果的相反值来判断是否不为空
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isNotEmpty({@code null}) -&gt; false
	 * ArrayUtils.isNotEmpty([]) -&gt; false
	 * ArrayUtils.isEmpty(new boolean[]{true}) -&gt; true
	 * </pre>
	 *
	 * @param array  要检测的数组
	 * @return {@code true} 如果数组不为空和 {@code null}则返回该值
	 */
	public static boolean isNotEmpty(@Nullable final boolean[] array) {
		return !isEmpty(array);
	}

	/**
	 * 把所给的Object数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll({@code null}, {@code null})     = {@code null}
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * ArrayUtils.addAll([{@code null}], [{@code null}]) = [{@code null}, {@code null}]
	 * ArrayUtils.addAll(["a", "b", "c"], ["1", "2", "3"]) = ["a", "b", "c", "1", "2", "3"]
	 * </pre>
	 *
	 * @param <T> 数组元素的类型
	 * @param array1  第一个要把元素添加进新数组的数组, 可能为 {@code null}
	 * @param array2  第二个要把元素添加进新数组的数组, 可能为 {@code null}
	 * @return 新数组, 其为{@code null} 如果输入的数组都为 {@code null}.
	 *      新数组的类型为第一个数组的类型，除非第一个数组为@code null}，
	 *      那么新数组的类型就为第二个数组的类型
	 * @throws IllegalArgumentException 如果输入的数组的类型不相容则抛出该异常
	 */
	public static <T> T[] addAll(@Nullable final T[] array1, @Nullable final T... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final Class<?> type1 = array1.getClass().getComponentType();
		@SuppressWarnings("unchecked") // OK, because array is of type T
		final T[] joinedArray = (T[]) Array.newInstance(type1, array1.length + array2.length);
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		try {
			System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		} catch (final ArrayStoreException ase) {
			// Check if problem was due to incompatible types
			/*
			 * We do this here, rather than before the copy because:
			 * - it would be a wasted check most of the time
			 * - safer, in case check turns out to be too strict
			 */
			final Class<?> type2 = array2.getClass().getComponentType();
			if (!type1.isAssignableFrom(type2)) {
				throw new IllegalArgumentException("Cannot store " + type2.getName() + " in an array of "
						+ type1.getName(), ase);
			}
			throw ase; // No, so rethrow original
		}
		return joinedArray;
	}

	/**
	 * 把所给的boolean数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素添加进新数组的数组
	 * @param array2  第二个要把元素添加进新数组的数组
	 * @return 新的boolean类型的数组.
	 */
	public static boolean[] addAll(@Nullable final boolean[] array1, @Nullable final boolean... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final boolean[] joinedArray = new boolean[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 把所给的char数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素添加进新数组的数组
	 * @param array2  第二个要把元素添加进新数组的数组
	 * @return 新的char类型的数组.
	 */
	public static char[] addAll(@Nullable final char[] array1, @Nullable final char... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final char[] joinedArray = new char[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 把所给的byte数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素添加进新数组的数组
	 * @param array2  第二个要把元素添加进新数组的数组
	 * @return 新的byte类型的数组.
	 */
	public static byte[] addAll(@Nullable final byte[] array1, @Nullable final byte... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final byte[] joinedArray = new byte[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 把所给的short数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素添加进新数组的数组
	 * @param array2  第二个要把元素添加进新数组的数组
	 * @return 新的short类型的数组.
	 */
	public static short[] addAll(@Nullable final short[] array1, @Nullable final short... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final short[] joinedArray = new short[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 把所给的int数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素添加进新数组的数组
	 * @param array2  第二个要把元素添加进新数组的数组
	 * @return 新的int类型的数组.
	 */
	public static int[] addAll(@Nullable final int[] array1, @Nullable final int... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final int[] joinedArray = new int[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 把所给的long数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素添加进新数组的数组
	 * @param array2  第二个要把元素添加进新数组的数组
	 * @return 新的long类型的数组.
	 */
	public static long[] addAll(@Nullable final long[] array1, @Nullable final long... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final long[] joinedArray = new long[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 把所给的float数组的所有元素添加到新数组中.
	 *
	 * <p>
	 * 新数组包含{@code array1}的所有元素，其后跟着 {@code array2}的元素
	 * 返回的数组往往是一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素添加进新数组的数组
	 * @param array2  第二个要把元素添加进新数组的数组
	 * @return 新的float类型的数组.
	 */
	public static float[] addAll(@Nullable final float[] array1, @Nullable final float... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final float[] joinedArray = new float[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 把所给double数组的所有元素添加到新的数组中.
	 *
	 * <p>
	 * 新数组包含第一个数组的所有元素，其后跟着第二个数组的元素
	 * 并且往往返回一个新的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.addAll(array1, {@code null})   = array1的复制
	 * ArrayUtils.addAll({@code null}, array2)   = array2的复制
	 * ArrayUtils.addAll([], [])         = []
	 * </pre>
	 *
	 * @param array1  第一个要把元素加入新数组的数组
	 * @param array2  第二个要把元素加入新数组的数组
	 * @return 新的double数组
	 */
	public static double[] addAll(@Nullable final double[] array1, @Nullable final double... array2) {

		if (array1 == null) {
			return clone(array2);
		} else if (array2 == null) {
			return clone(array1);
		}
		final double[] joinedArray = new double[array1.length + array2.length];
		System.arraycopy(array1, 0, joinedArray, 0, array1.length);
		System.arraycopy(array2, 0, joinedArray, array1.length, array2.length);
		return joinedArray;
	}

	/**
	 * 复制给定的Object数组并在新数组的末尾添加给定的Object元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, {@code null})      = IllegalArgumentException
	 * ArrayUtils.add({@code null}, "a")       = ["a"]
	 * ArrayUtils.add(["a"], {@code null})     = ["a", {@code null}]
	 * ArrayUtils.add(["a"], "b")      = ["a", "b"]
	 * ArrayUtils.add(["a", "b"], "c") = ["a", "b", "c"]
	 * </pre>
	 *
	 * @param <T> 输入数组的元素的类型
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素, 可能为 {@code null}
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 * 返回的数组的类型为输入数组的类型 (除非其为 {@code null}),
	 * 此时类型为添加的元素的类型，如果两个参数都为{@code null}
	 * 会抛出一个IllegalArgumentException异常
	 * @throws IllegalArgumentException 如果两个参数都为{@code null}则抛出该异常
	 */
	public static <T> T[] add(@Nullable final T[] array, @Nullable final T element) {

		Class<?> type;
		if (array != null) {
			type = array.getClass().getComponentType();
		} else if (element != null) {
			type = element.getClass();
		} else {
			throw new IllegalArgumentException("Arguments cannot both be null");
		}
		@SuppressWarnings("unchecked") // type must be T
		final
		T[] newArray = (T[]) copyArrayGrow1(array, type);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的boolean数组并在新数组的末尾添加给定的boolean元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, true)          = [true]
	 * ArrayUtils.add([true], false)       = [true, false]
	 * ArrayUtils.add([true, false], true) = [true, false, true]
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static boolean[] add(@Nullable final boolean[] array, final boolean element) {

		final boolean[] newArray = (boolean[])copyArrayGrow1(array, Boolean.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的byte数组并在新数组的末尾添加给定的byte元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0)   = [0]
	 * ArrayUtils.add([1], 0)    = [1, 0]
	 * ArrayUtils.add([1, 0], 1) = [1, 0, 1]
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static byte[] add(@Nullable final byte[] array, final byte element) {

		final byte[] newArray = (byte[])copyArrayGrow1(array, Byte.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的char数组并在新数组的末尾添加给定的char元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, '0')       = ['0']
	 * ArrayUtils.add(['1'], '0')      = ['1', '0']
	 * ArrayUtils.add(['1', '0'], '1') = ['1', '0', '1']
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static char[] add(@Nullable final char[] array, final char element) {

		final char[] newArray = (char[])copyArrayGrow1(array, Character.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的double数组并在新数组的末尾添加给定的double元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0)   = [0]
	 * ArrayUtils.add([1], 0)    = [1, 0]
	 * ArrayUtils.add([1, 0], 1) = [1, 0, 1]
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static double[] add(@Nullable final double[] array, final double element) {

		final double[] newArray = (double[])copyArrayGrow1(array, Double.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的float数组并在新数组的末尾添加给定的float元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0)   = [0]
	 * ArrayUtils.add([1], 0)    = [1, 0]
	 * ArrayUtils.add([1, 0], 1) = [1, 0, 1]
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static float[] add(@Nullable final float[] array, final float element) {

		final float[] newArray = (float[])copyArrayGrow1(array, Float.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的int数组并在新数组的末尾添加给定的int元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0)   = [0]
	 * ArrayUtils.add([1], 0)    = [1, 0]
	 * ArrayUtils.add([1, 0], 1) = [1, 0, 1]
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static int[] add(@Nullable final int[] array, final int element) {

		final int[] newArray = (int[])copyArrayGrow1(array, Integer.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的long数组并在新数组的末尾添加给定的long元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0)   = [0]
	 * ArrayUtils.add([1], 0)    = [1, 0]
	 * ArrayUtils.add([1, 0], 1) = [1, 0, 1]
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static long[] add(@Nullable final long[] array, final long element) {

		final long[] newArray = (long[])copyArrayGrow1(array, Long.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 复制给定的short数组并在新数组的末尾添加给定的short元素.
	 *
	 * <p>
	 * 新数组包含和输入的数组相同的元素并且在末尾加上了给定的元素
	 * 新数组元素的类型和输入的数组的元素类型一致
	 * 如果新数组是 {@code null}，会返回一个单元素的新数组，该数组的
	 * 类型就是这个元素的类型
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0)   = [0]
	 * ArrayUtils.add([1], 0)    = [1, 0]
	 * ArrayUtils.add([1, 0], 1) = [1, 0, 1]
	 * </pre>
	 *
	 * @param array  要添加元素的数组, 可能为 {@code null}
	 * @param element  要添加到末尾的元素
	 * @return 一个包含现存元素和指定要添加的元素的新数组
	 */
	public static short[] add(@Nullable final short[] array, final short element) {

		final short[] newArray = (short[]) copyArrayGrow1(array, Short.TYPE);
		newArray[newArray.length - 1] = element;
		return newArray;
	}

	/**
	 * 返回一个比参数数组长度大一的新数组.
	 *
	 * <p>
	 * 数组最后一位的值保留为默认值
	 * </p>
	 *
	 *
	 * @param array 要复制的数组, 必须不为 {@code null}.
	 * @param newArrayComponentType 如果 {@code array} 为 {@code null}, 创建一个
	 * 			长度为1的该类型的数组
	 * @return 一个比参数数组长度大一的复制的新数组.
	 */
	private static Object copyArrayGrow1(final Object array, final Class<?> newArrayComponentType) {

		if (array != null) {
			final int arrayLength = Array.getLength(array);
			final Object newArray = Array.newInstance(array.getClass().getComponentType(), arrayLength + 1);
			System.arraycopy(array, 0, newArray, 0, arrayLength);
			return newArray;
		}
		return Array.newInstance(newArrayComponentType, 1);
	}

	/**
	 * 在Object数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0, {@code null})      = IllegalArgumentException
	 * ArrayUtils.add({@code null} 0, "a")       = ["a"]
	 * ArrayUtils.add(["a"], 1, {@code null})     = ["a", {@code null}]
	 * ArrayUtils.add(["a"], 1, "b")      = ["a", "b"]
	 * ArrayUtils.add(["a", "b"], 3, "c") = ["a", "b", "c"]
	 * </pre>
	 *
	 * @param <T> 数组元素的类型
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @throws IllegalArgumentException 如果输入的数组和要添加的元素都为{@code null}则抛出该异常
	 * @deprecated 该方法已经被{@link #insert(int, Object[], Object...) insert(int, T[], T...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static <T> T[] add(@Nullable final T[] array, final int index, @Nullable final T element) {

		Class<?> clss = null;
		if (array != null) {
			clss = array.getClass().getComponentType();
		} else if (element != null) {
			clss = element.getClass();
		} else {
			throw new IllegalArgumentException("Array and element cannot both be null");
		}
		@SuppressWarnings("unchecked") // the add method creates an array of type clss, which is type T
		final T[] newArray = (T[]) add(array, index, element, clss);
		return newArray;
	}

	/**
     * 在boolean数组的指定位置插入指定元素.
     *
     * <p>
     * 如果指定位置上有元素，则移动它及其之后的所有元素，
     * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
     * 返回的数组的元素类型和输入的数组的保持一致
     * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
     * </p>
	 *
	 * <pre>
	 * ArrayUtils.add({@code null}, 0, true)          = [true]
	 * ArrayUtils.add([true], 0, false)       = [false, true]
	 * ArrayUtils.add([false], 1, true)       = [false, true]
	 * ArrayUtils.add([true, false], 1, true) = [true, true, false]
	 * </pre>
	 *
     * @param array  要加元素的数组，可能为{@code null}
     * @param index  新元素的位置
     * @param element  要添加的元素
     * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, boolean[], boolean...) insert(int, boolean[], boolean...)} 取代并且有可能
     * 在以后发布的版本中移除
	 */
	@Deprecated
	public static boolean[] add(@Nullable final boolean[] array, final int index, @Nullable final boolean element) {
		return (boolean[]) add(array, index, Boolean.valueOf(element), Boolean.TYPE);
	}

	/**
	 * 在char数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add(null, 0, 'a')            = ['a']
	 * ArrayUtils.add(['a'], 0, 'b')           = ['b', 'a']
	 * ArrayUtils.add(['a', 'b'], 0, 'c')      = ['c', 'a', 'b']
	 * ArrayUtils.add(['a', 'b'], 1, 'k')      = ['a', 'k', 'b']
	 * ArrayUtils.add(['a', 'b', 'c'], 1, 't') = ['a', 't', 'b', 'c']
	 * </pre>
	 *
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, char[], char...) insert(int, char[], char...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static char[] add(@Nullable final char[] array, final int index, final char element) {
		return (char[]) add(array, index, Character.valueOf(element), Character.TYPE);
	}

	/**
	 * 在byte数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add([1], 0, 2)         = [2, 1]
	 * ArrayUtils.add([2, 6], 2, 3)      = [2, 6, 3]
	 * ArrayUtils.add([2, 6], 0, 1)      = [1, 2, 6]
	 * ArrayUtils.add([2, 6, 3], 2, 1)   = [2, 6, 1, 3]
	 * </pre>
	 *
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, byte[], byte...) insert(int, byte[], byte...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static byte[] add(@Nullable final byte[] array, final int index, final byte element) {
		return (byte[]) add(array, index, Byte.valueOf(element), Byte.TYPE);
	}

	/**
	 * 在short数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add([1], 0, 2)         = [2, 1]
	 * ArrayUtils.add([2, 6], 2, 10)     = [2, 6, 10]
	 * ArrayUtils.add([2, 6], 0, -4)     = [-4, 2, 6]
	 * ArrayUtils.add([2, 6, 3], 2, 1)   = [2, 6, 1, 3]
	 * </pre>
	 *
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, short[], short...) insert(int, short[], short...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static short[] add(@Nullable final short[] array, final int index, final short element) {
		return (short[]) add(array, index, Short.valueOf(element), Short.TYPE);
	}

	/**
	 * 在int数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add([1], 0, 2)         = [2, 1]
	 * ArrayUtils.add([2, 6], 2, 10)     = [2, 6, 10]
	 * ArrayUtils.add([2, 6], 0, -4)     = [-4, 2, 6]
	 * ArrayUtils.add([2, 6, 3], 2, 1)   = [2, 6, 1, 3]
	 * </pre>
	 *
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, int[], int...) insert(int, int[], int...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static int[] add(@Nullable final int[] array, final int index, final int element) {
		return (int[]) add(array, index, Integer.valueOf(element), Integer.TYPE);
	}

	/**
	 * 在long数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add([1L], 0, 2L)           = [2L, 1L]
	 * ArrayUtils.add([2L, 6L], 2, 10L)      = [2L, 6L, 10L]
	 * ArrayUtils.add([2L, 6L], 0, -4L)      = [-4L, 2L, 6L]
	 * ArrayUtils.add([2L, 6L, 3L], 2, 1L)   = [2L, 6L, 1L, 3L]
	 * </pre>
	 *
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, long[], long...) insert(int, long[], long...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static long[] add(@Nullable final long[] array, final int index, final long element) {
		return (long[]) add(array, index, Long.valueOf(element), Long.TYPE);
	}

	/**
	 * 在float数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add([1.1f], 0, 2.2f)               = [2.2f, 1.1f]
	 * ArrayUtils.add([2.3f, 6.4f], 2, 10.5f)        = [2.3f, 6.4f, 10.5f]
	 * ArrayUtils.add([2.6f, 6.7f], 0, -4.8f)        = [-4.8f, 2.6f, 6.7f]
	 * ArrayUtils.add([2.9f, 6.0f, 0.3f], 2, 1.0f)   = [2.9f, 6.0f, 1.0f, 0.3f]
	 * </pre>
	 *
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, float[], float...) insert(int, float[], float...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static float[] add(@Nullable final float[] array, final int index, final float element) {
		return (float[]) add(array, index, Float.valueOf(element), Float.TYPE);
	}

	/**
	 * 在double数组的指定位置插入指定元素.
	 *
	 * <p>
	 * 如果指定位置上有元素，则移动它及其之后的所有元素，
	 * 该方法返回一个包含所有已有元素以及指定位置上有指定元素的新数组
	 * 返回的数组的元素类型和输入的数组的保持一致
	 * 如果输入数组为{@code null},则返回一个和指定元素类型相同的数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.add([1.1], 0, 2.2)              = [2.2, 1.1]
	 * ArrayUtils.add([2.3, 6.4], 2, 10.5)        = [2.3, 6.4, 10.5]
	 * ArrayUtils.add([2.6, 6.7], 0, -4.8)        = [-4.8, 2.6, 6.7]
	 * ArrayUtils.add([2.9, 6.0, 0.3], 2, 1.0)    = [2.9, 6.0, 1.0, 0.3]
	 * </pre>
	 *
	 * @param array  要加元素的数组，可能为{@code null}
	 * @param index  新元素的位置
	 * @param element  要添加的元素
	 * @return 一个包含现存元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围 (index &lt; 0 || index &gt; array.length)则抛出该异常
	 * @deprecated t该方法已经被{@link #insert(int, double[], double...) insert(int, double[], double...)} 取代并且有可能
	 * 在以后发布的版本中移除
	 */
	@Deprecated
	public static double[] add(@Nullable final double[] array, final int index, final double element) {
		return (double[]) add(array, index, Double.valueOf(element), Double.TYPE);
	}

	/**
	 * add（数组、索引、元素）方法的底层实现.
	 *
	 * <p>
	 * 最后一个参数是类，它可能元素.getClass不相等
	 * </p>
	 *
	 * @param array  要添加元素的数组 可能为 {@code null}
	 * @param index  要添加的元素的位置
	 * @param element  要添加的元素
	 * @param clss 添加的元素的类型
	 * @return 一个包含现存元素和新元素的新数组
	 */
	private static Object add(@Nullable final Object array, final int index, final Object element, final Class<?> clss) {

		if (array == null) {
			if (index != 0) {
				throw new IndexOutOfBoundsException("Index: " + index + ", Length: 0");
			}
			final Object joinedArray = Array.newInstance(clss, 1);
			Array.set(joinedArray, 0, element);
			return joinedArray;
		}
		final int length = Array.getLength(array);
		if (index > length || index < 0) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + length);
		}
		final Object result = Array.newInstance(clss, length + 1);
		System.arraycopy(array, 0, result, 0, index);
		Array.set(result, index, element);
		if (index < length) {
			System.arraycopy(array, index, result, index + 1, length - index);
		}
		return result;
	}

	/**
	 * 从指定Object数组的指定位置移除一个元素.
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove(["a"], 0)           = []
	 * ArrayUtils.remove(["a", "b"], 0)      = ["b"]
	 * ArrayUtils.remove(["a", "b"], 1)      = ["a"]
	 * ArrayUtils.remove(["a", "b", "c"], 1) = ["a", "c"]
	 * </pre>
	 *
	 * @param <T> 输入数组的元素类型
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	@SuppressWarnings("unchecked") // remove() always creates an array of the same type as its input
	public static <T> T[] remove(final T[] array, final int index) {
		return (T[]) remove((Object) array, index);
	}

	/**
	 * 从指定Object数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, "a")            = {@code null}
	 * ArrayUtils.removeElement([], "a")              = []
	 * ArrayUtils.removeElement(["a"], "b")           = ["a"]
	 * ArrayUtils.removeElement(["a", "b"], "a")      = ["b"]
	 * ArrayUtils.removeElement(["a", "b", "a"], "a") = ["b", "a"]
	 * </pre>
	 *
	 * @param <T> 数组元素的类型
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static <T> T[] removeElement(@Nullable final T[] array, final Object element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定boolean数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove([true], 0)              = []
	 * ArrayUtils.remove([true, false], 0)       = [false]
	 * ArrayUtils.remove([true, false], 1)       = [true]
	 * ArrayUtils.remove([true, true, false], 1) = [true, false]
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static boolean[] remove(final boolean[] array, final int index) {
		return (boolean[]) remove((Object) array, index);
	}

	/**
	 * 从指定boolean数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, true)                = {@code null}
	 * ArrayUtils.removeElement([], true)                  = []
	 * ArrayUtils.removeElement([true], false)             = [true]
	 * ArrayUtils.removeElement([true, false], false)      = [true]
	 * ArrayUtils.removeElement([true, false, true], true) = [false, true]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static boolean[] removeElement(@Nullable final boolean[] array, final boolean element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定byte数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove([1], 0)          = []
	 * ArrayUtils.remove([1, 0], 0)       = [0]
	 * ArrayUtils.remove([1, 0], 1)       = [1]
	 * ArrayUtils.remove([1, 0, 1], 1)    = [1, 1]
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static byte[] remove(final byte[] array, final int index) {
		return (byte[]) remove((Object) array, index);
	}

	/**
	 * 从指定byte数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, 1)        = {@code null}
	 * ArrayUtils.removeElement([], 1)          = []
	 * ArrayUtils.removeElement([1], 0)         = [1]
	 * ArrayUtils.removeElement([1, 0], 0)      = [1]
	 * ArrayUtils.removeElement([1, 0, 1], 1)   = [0, 1]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static byte[] removeElement(@Nullable final byte[] array, final byte element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定char数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove(['a'], 0)           = []
	 * ArrayUtils.remove(['a', 'b'], 0)      = ['b']
	 * ArrayUtils.remove(['a', 'b'], 1)      = ['a']
	 * ArrayUtils.remove(['a', 'b', 'c'], 1) = ['a', 'c']
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static char[] remove(final char[] array, final int index) {
		return (char[]) remove((Object) array, index);
	}

	/**
	 * 从指定char数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, 'a')            = {@code null}
	 * ArrayUtils.removeElement([], 'a')              = []
	 * ArrayUtils.removeElement(['a'], 'b')           = ['a']
	 * ArrayUtils.removeElement(['a', 'b'], 'a')      = ['b']
	 * ArrayUtils.removeElement(['a', 'b', 'a'], 'a') = ['b', 'a']
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static char[] removeElement(@Nullable final char[] array, final char element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定double数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove([1.1], 0)           = []
	 * ArrayUtils.remove([2.5, 6.0], 0)      = [6.0]
	 * ArrayUtils.remove([2.5, 6.0], 1)      = [2.5]
	 * ArrayUtils.remove([2.5, 6.0, 3.8], 1) = [2.5, 3.8]
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static double[] remove(final double[] array, final int index) {
		return (double[]) remove((Object) array, index);
	}

	/**
	 * 从指定double数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, 1.1)            = {@code null}
	 * ArrayUtils.removeElement([], 1.1)              = []
	 * ArrayUtils.removeElement([1.1], 1.2)           = [1.1]
	 * ArrayUtils.removeElement([1.1, 2.3], 1.1)      = [2.3]
	 * ArrayUtils.removeElement([1.1, 2.3, 1.1], 1.1) = [2.3, 1.1]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static double[] removeElement(@Nullable final double[] array, final double element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定float数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove([1.1], 0)           = []
	 * ArrayUtils.remove([2.5, 6.0], 0)      = [6.0]
	 * ArrayUtils.remove([2.5, 6.0], 1)      = [2.5]
	 * ArrayUtils.remove([2.5, 6.0, 3.8], 1) = [2.5, 3.8]
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static float[] remove(final float[] array, final int index) {
		return (float[]) remove((Object) array, index);
	}

	/**
	 * 从指定float数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, 1.1)            = {@code null}
	 * ArrayUtils.removeElement([], 1.1)              = []
	 * ArrayUtils.removeElement([1.1], 1.2)           = [1.1]
	 * ArrayUtils.removeElement([1.1, 2.3], 1.1)      = [2.3]
	 * ArrayUtils.removeElement([1.1, 2.3, 1.1], 1.1) = [2.3, 1.1]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static float[] removeElement(@Nullable final float[] array, final float element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定int数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove([1], 0)         = []
	 * ArrayUtils.remove([2, 6], 0)      = [6]
	 * ArrayUtils.remove([2, 6], 1)      = [2]
	 * ArrayUtils.remove([2, 6, 3], 1)   = [2, 3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static int[] remove(final int[] array, final int index) {
		return (int[]) remove((Object) array, index);
	}

	/**
	 * 从指定int数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, 1)      = {@code null}
	 * ArrayUtils.removeElement([], 1)        = []
	 * ArrayUtils.removeElement([1], 2)       = [1]
	 * ArrayUtils.removeElement([1, 3], 1)    = [3]
	 * ArrayUtils.removeElement([1, 3, 1], 1) = [3, 1]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static int[] removeElement(@Nullable final int[] array, final int element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定long数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove([1], 0)         = []
	 * ArrayUtils.remove([2, 6], 0)      = [6]
	 * ArrayUtils.remove([2, 6], 1)      = [2]
	 * ArrayUtils.remove([2, 6, 3], 1)   = [2, 3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static long[] remove(final long[] array, final int index) {
		return (long[]) remove((Object) array, index);
	}

	/**
	 * 从指定long数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, 1)      = {@code null}
	 * ArrayUtils.removeElement([], 1)        = []
	 * ArrayUtils.removeElement([1], 2)       = [1]
	 * ArrayUtils.removeElement([1, 3], 1)    = [3]
	 * ArrayUtils.removeElement([1, 3, 1], 1) = [3, 1]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static long[] removeElement(@Nullable final long[] array, final long element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 *从指定short数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.remove([1], 0)         = []
	 * ArrayUtils.remove([2, 6], 0)      = [6]
	 * ArrayUtils.remove([2, 6], 1)      = [2]
	 * ArrayUtils.remove([2, 6, 3], 1)   = [2, 3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static short[] remove(final short[] array, final int index) {
		return (short[]) remove((Object) array, index);
	}

	/**
	 * 从指定short数组中删除第一个目标元素.
	 *
	 * <p>
	 * 该方法返回一个除了第一个出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElement({@code null}, 1)      = {@code null}
	 * ArrayUtils.removeElement([], 1)        = []
	 * ArrayUtils.removeElement([1], 2)       = [1]
	 * ArrayUtils.removeElement([1, 3], 1)    = [3]
	 * ArrayUtils.removeElement([1, 3, 1], 1) = [3, 1]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param element  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static short[] removeElement(@Nullable final short[] array, final short element) {

		final int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}
		return remove(array, index);
	}

	/**
	 * 从指定的数组的指定位置移除一个元素..
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * @param array  要移除元素的数组
	 * @param index  要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	private static Object remove(final Object array, final int index) {

		final int length = getLength(array);
		if (index < 0 || index >= length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + length);
		}

		final Object result = Array.newInstance(array.getClass().getComponentType(), length - 1);
		System.arraycopy(array, 0, result, 0, index);
		if (index < length - 1) {
			System.arraycopy(array, index + 1, result, index, length - index - 1);
		}

		return result;
	}

	/**
	 * 从给定Object数组中移除多个指定位置上的元素.
	 *
	 * <p>
	 * 该方法返回包含除指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll(["a", "b", "c"], 0, 2) = ["b"]
	 * ArrayUtils.removeAll(["a", "b", "c"], 1, 2) = ["a"]
	 * </pre>
	 *
	 * @param <T> 输入数组的元素类型
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	@SuppressWarnings("unchecked") // removeAll() always creates an array of the same type as its input
	public static <T> T[] removeAll(final T[] array, final int... indices) {
		return (T[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定Object数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, "a", "b")            = {@code null}
	 * ArrayUtils.removeElements([], "a", "b")              = []
	 * ArrayUtils.removeElements(["a"], "b", "c")           = ["a"]
	 * ArrayUtils.removeElements(["a", "b"], "a", "c")      = ["b"]
	 * ArrayUtils.removeElements(["a", "b", "a"], "a")      = ["b", "a"]
	 * ArrayUtils.removeElements(["a", "b", "a"], "a", "a") = ["b"]
	 * </pre>
	 *
	 * @param <T> 输入数组的元素类型
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static <T> T[] removeElements(@Nullable final T[] array, final T... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<T, MutableInt> occurrences = new HashMap<T, MutableInt>(values.length);
		for (final T v : values) {
			final MutableInt count = occurrences.get(v);
			if (count == null) {
				occurrences.put(v, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final T key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		@SuppressWarnings("unchecked") // removeAll() always creates an array of the same type as its input
		final T[] result = (T[]) removeAll(array, toRemove);
		return result;
	}

	/**
	 * 从给定byte数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([1], 0)             = []
	 * ArrayUtils.removeAll([2, 6], 0)          = [6]
	 * ArrayUtils.removeAll([2, 6], 0, 1)       = []
	 * ArrayUtils.removeAll([2, 6, 3], 1, 2)    = [2]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 2)    = [6]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 1, 2) = []
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static byte[] removeAll(final byte[] array, final int... indices) {
		return (byte[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定byte数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, 1, 2)      = {@code null}
	 * ArrayUtils.removeElements([], 1, 2)        = []
	 * ArrayUtils.removeElements([1], 2, 3)       = [1]
	 * ArrayUtils.removeElements([1, 3], 1, 2)    = [3]
	 * ArrayUtils.removeElements([1, 3, 1], 1)    = [3, 1]
	 * ArrayUtils.removeElements([1, 3, 1], 1, 1) = [3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static byte[] removeElements(@Nullable final byte[] array, final byte... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final Map<Byte, MutableInt> occurrences = new HashMap<Byte, MutableInt>(values.length);
		for (final byte v : values) {
			final Byte boxed = Byte.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final byte key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (byte[]) removeAll(array, toRemove);
	}

	/**
	 * 从给定short数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([1], 0)             = []
	 * ArrayUtils.removeAll([2, 6], 0)          = [6]
	 * ArrayUtils.removeAll([2, 6], 0, 1)       = []
	 * ArrayUtils.removeAll([2, 6, 3], 1, 2)    = [2]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 2)    = [6]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 1, 2) = []
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static short[] removeAll(final short[] array, final int... indices) {
		return (short[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定short数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, 1, 2)      = {@code null}
	 * ArrayUtils.removeElements([], 1, 2)        = []
	 * ArrayUtils.removeElements([1], 2, 3)       = [1]
	 * ArrayUtils.removeElements([1, 3], 1, 2)    = [3]
	 * ArrayUtils.removeElements([1, 3, 1], 1)    = [3, 1]
	 * ArrayUtils.removeElements([1, 3, 1], 1, 1) = [3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static short[] removeElements(@Nullable final short[] array, final short... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<Short, MutableInt> occurrences = new HashMap<Short, MutableInt>(values.length);
		for (final short v : values) {
			final Short boxed = Short.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final short key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (short[]) removeAll(array, toRemove);
	}

	/**
	 * 从给定int数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([1], 0)             = []
	 * ArrayUtils.removeAll([2, 6], 0)          = [6]
	 * ArrayUtils.removeAll([2, 6], 0, 1)       = []
	 * ArrayUtils.removeAll([2, 6, 3], 1, 2)    = [2]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 2)    = [6]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 1, 2) = []
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static int[] removeAll(final int[] array, final int... indices) {
		return (int[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定int数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, 1, 2)      = {@code null}
	 * ArrayUtils.removeElements([], 1, 2)        = []
	 * ArrayUtils.removeElements([1], 2, 3)       = [1]
	 * ArrayUtils.removeElements([1, 3], 1, 2)    = [3]
	 * ArrayUtils.removeElements([1, 3, 1], 1)    = [3, 1]
	 * ArrayUtils.removeElements([1, 3, 1], 1, 1) = [3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static int[] removeElements(@Nullable final int[] array, final int... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<Integer, MutableInt> occurrences = new HashMap<Integer, MutableInt>(values.length);
		for (final int v : values) {
			final Integer boxed = Integer.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final int key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (int[]) removeAll(array, toRemove);
	}

	/**
	 * 从给定char数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([1], 0)             = []
	 * ArrayUtils.removeAll([2, 6], 0)          = [6]
	 * ArrayUtils.removeAll([2, 6], 0, 1)       = []
	 * ArrayUtils.removeAll([2, 6, 3], 1, 2)    = [2]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 2)    = [6]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 1, 2) = []
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static char[] removeAll(final char[] array, final int... indices) {
		return (char[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定char数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, 1, 2)      = {@code null}
	 * ArrayUtils.removeElements([], 1, 2)        = []
	 * ArrayUtils.removeElements([1], 2, 3)       = [1]
	 * ArrayUtils.removeElements([1, 3], 1, 2)    = [3]
	 * ArrayUtils.removeElements([1, 3, 1], 1)    = [3, 1]
	 * ArrayUtils.removeElements([1, 3, 1], 1, 1) = [3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static char[] removeElements(@Nullable final char[] array, final char... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<Character, MutableInt> occurrences = new HashMap<Character, MutableInt>(values.length);
		for (final char v : values) {
			final Character boxed = Character.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final char key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (char[]) removeAll(array, toRemove);
	}

	/**
	 * 从给定long数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([1], 0)             = []
	 * ArrayUtils.removeAll([2, 6], 0)          = [6]
	 * ArrayUtils.removeAll([2, 6], 0, 1)       = []
	 * ArrayUtils.removeAll([2, 6, 3], 1, 2)    = [2]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 2)    = [6]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 1, 2) = []
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static long[] removeAll(final long[] array, final int... indices) {
		return (long[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定long数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, 1, 2)      = {@code null}
	 * ArrayUtils.removeElements([], 1, 2)        = []
	 * ArrayUtils.removeElements([1], 2, 3)       = [1]
	 * ArrayUtils.removeElements([1, 3], 1, 2)    = [3]
	 * ArrayUtils.removeElements([1, 3, 1], 1)    = [3, 1]
	 * ArrayUtils.removeElements([1, 3, 1], 1, 1) = [3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static long[] removeElements(@Nullable final long[] array, final long... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<Long, MutableInt> occurrences = new HashMap<Long, MutableInt>(values.length);
		for (final long v : values) {
			final Long boxed = Long.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final long key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (long[]) removeAll(array, toRemove);
	}

	/**
	 * 从给定float数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([1], 0)             = []
	 * ArrayUtils.removeAll([2, 6], 0)          = [6]
	 * ArrayUtils.removeAll([2, 6], 0, 1)       = []
	 * ArrayUtils.removeAll([2, 6, 3], 1, 2)    = [2]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 2)    = [6]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 1, 2) = []
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static float[] removeAll(final float[] array, final int... indices) {
		return (float[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定float数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, 1, 2)      = {@code null}
	 * ArrayUtils.removeElements([], 1, 2)        = []
	 * ArrayUtils.removeElements([1], 2, 3)       = [1]
	 * ArrayUtils.removeElements([1, 3], 1, 2)    = [3]
	 * ArrayUtils.removeElements([1, 3, 1], 1)    = [3, 1]
	 * ArrayUtils.removeElements([1, 3, 1], 1, 1) = [3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static float[] removeElements(@Nullable final float[] array, final float... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<Float, MutableInt> occurrences = new HashMap<Float, MutableInt>(values.length);
		for (final float v : values) {
			final Float boxed = Float.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final float key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (float[]) removeAll(array, toRemove);
	}

	/**
	 * 从给定double数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([1], 0)             = []
	 * ArrayUtils.removeAll([2, 6], 0)          = [6]
	 * ArrayUtils.removeAll([2, 6], 0, 1)       = []
	 * ArrayUtils.removeAll([2, 6, 3], 1, 2)    = [2]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 2)    = [6]
	 * ArrayUtils.removeAll([2, 6, 3], 0, 1, 2) = []
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static double[] removeAll(final double[] array, final int... indices) {
		return (double[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定double数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, 1, 2)      = {@code null}
	 * ArrayUtils.removeElements([], 1, 2)        = []
	 * ArrayUtils.removeElements([1], 2, 3)       = [1]
	 * ArrayUtils.removeElements([1, 3], 1, 2)    = [3]
	 * ArrayUtils.removeElements([1, 3, 1], 1)    = [3, 1]
	 * ArrayUtils.removeElements([1, 3, 1], 1, 1) = [3]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static double[] removeElements(@Nullable final double[] array, final double... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<Double, MutableInt> occurrences = new HashMap<Double, MutableInt>(values.length);
		for (final double v : values) {
			final Double boxed = Double.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final double key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (double[]) removeAll(array, toRemove);
	}

	/**
	 * 从给定boolean数组中移除多个指定位置上的元素.
	 *
	 * <p>该方法返回包含除那些指定位置元素外的其他和输入数组相同的元素的数组
	 * 数组元素的类型和输入数组的保持一致
	 * 如果输入的数组为{@code null},一个IndexOutOfBoundsException异常被抛出
	 * 因为此时没有有效元素可以移除
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAll([true, false, true], 0, 2) = [false]
	 * ArrayUtils.removeAll([true, false, true], 1, 2) = [true]
	 * </pre>
	 *
	 * @param array   要移除元素的数组
	 * @param indices 若干个要移除的元素的位置
	 * @return 一个包含所有已存在的元素但除去那些特定位置元素的新数组
	 * @throws IndexOutOfBoundsException 如果 index超出范围
	 * (index &lt; 0 || index &gt;= array.length), 或者输入数组为{@code null}抛出该异常
	 */
	public static boolean[] removeAll(final boolean[] array, final int... indices) {
		return (boolean[]) removeAll((Object) array, indices);
	}

	/**
	 *删除给定double数组中指定数量的指定元素的引用.
	 *
	 * <p>
	 * 该方法返回一个除了那些先出现的指定元素的包含其他所有元素的新数组
	 * 返回数组元素的类型和输入数组保持一致，如果数组中没有指定元素
	 * 那么不会从数组中移除任何元素。
	 * 如果指定的元素数量大于数组长度，只删除所有匹配项
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeElements({@code null}, true, false)               = {@code null}
	 * ArrayUtils.removeElements([], true, false)                 = []
	 * ArrayUtils.removeElements([true], false, false)            = [true]
	 * ArrayUtils.removeElements([true, false], true, true)       = [false]
	 * ArrayUtils.removeElements([true, false, true], true)       = [false, true]
	 * ArrayUtils.removeElements([true, false, true], true, true) = [false]
	 * </pre>
	 *
	 * @param array  要移除元素的数组, 可能为 {@code null}
	 * @param values  要移除的元素
	 * @return 一个包含所有已存在的元素但除去第一个出现的特定元素的新数组
	 */
	public static boolean[] removeElements(@Nullable final boolean[] array, final boolean... values) {

		if (isEmpty(array) || isEmpty(values)) {
			return clone(array);
		}
		final HashMap<Boolean, MutableInt> occurrences = new HashMap<Boolean, MutableInt>(2); // only two possible values here
		for (final boolean v : values) {
			final Boolean boxed = Boolean.valueOf(v);
			final MutableInt count = occurrences.get(boxed);
			if (count == null) {
				occurrences.put(boxed, new MutableInt(1));
			} else {
				count.increment();
			}
		}
		final BitSet toRemove = new BitSet();
		for (int i = 0; i < array.length; i++) {
			final boolean key = array[i];
			final MutableInt count = occurrences.get(key);
			if (count != null) {
				if (count.decrementAndGet() == 0) {
					occurrences.remove(key);
				}
				toRemove.set(i);
			}
		}
		return (boolean[]) removeAll(array, toRemove);
	}

	/**
	 * 删除由索引指定的多个数组元素.
	 *
	 * <pre>
	 * 对indices排序，从末尾开始判断有无indice超出范围
	 * 如果有则抛出一个IndexOutOfBoundsException异常
	 * 如果没有则继续向下执行生成结果数组
	 * </pre>
	 * @param array 要移除元素的数组
	 * @param indices 要移除的元素的位置
	 * @return 相同类型的新数组除去由{@code indices}的唯一值指定的元素
	 */
	// package protected for access by unit tests
	static Object removeAll(final Object array, final int... indices) {
		final int length = getLength(array);
		int diff = 0; // number of distinct indexes, i.e. number of entries that will be removed
		final int[] clonedIndices = clone(indices);
		Arrays.sort(clonedIndices);

		// identify length of result array
		if (isNotEmpty(clonedIndices)) {
			int i = clonedIndices.length;
			int prevIndex = length;
			while (--i >= 0) {
				final int index = clonedIndices[i];
				if (index < 0 || index >= length) {
					throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + length);
				}
				if (index >= prevIndex) {
					continue;
				}
				diff++;
				prevIndex = index;
			}
		}

		// create result array
		final Object result = Array.newInstance(array.getClass().getComponentType(), length - diff);
		if (diff < length) {
			int end = length; // index just after last copy
			int dest = length - diff; // number of entries so far not copied
			for (int i = clonedIndices.length - 1; i >= 0; i--) {
				final int index = clonedIndices[i];
				if (end - index > 1) { // same as (cp > 0)
					final int cp = end - index - 1;
					dest -= cp;
					System.arraycopy(array, index + 1, result, dest, cp);
					// Afer this copy, we still have room for dest items.
				}
				end = index;
			}
			if (end > 0) {
				System.arraycopy(array, 0, result, 0, end);
			}
		}
		return result;
	}

	/**
	 * 删除由索引指定的多个数组元素.
	 *
	 * <p>
	 * removeElements方法中调用该方法，传入的indices确定是数组中存在的元素的下标
	 * 也就是保证了没有超出范围的indice，所以直接生成结果数组
	 * </p>
	 *
	 * @param array 要移除元素的数组
	 * @param indices 要移除的元素的位置
	 * @return 相同类型的新数组除去由{@code indices}的唯一值指定的元素
	 */
	// package protected for access by unit tests
	static Object removeAll(final Object array, final BitSet indices) {
		final int srcLength = getLength(array);
		// No need to check maxIndex here, because method only currently called from removeElements()
		// which guarantee to generate on;y valid bit entries.
//        final int maxIndex = indices.length();
//        if (maxIndex > srcLength) {
//            throw new IndexOutOfBoundsException("Index: " + (maxIndex-1) + ", Length: " + srcLength);
//        }
		final int removals = indices.cardinality(); // true bits are items to remove
		final Object result = Array.newInstance(array.getClass().getComponentType(), srcLength - removals);
		int srcIndex = 0;
		int destIndex = 0;
		int count;
		int set;
		while ((set = indices.nextSetBit(srcIndex)) != -1) {
			count = set - srcIndex;
			if (count > 0) {
				System.arraycopy(array, srcIndex, result, destIndex, count);
				destIndex += count;
			}
			srcIndex = indices.nextClearBit(set);
		}
		count = srcLength - srcIndex;
		if (count > 0) {
			System.arraycopy(array, srcIndex, result, destIndex, count);
		}
		return result;
	}

	/**
	 * 检查给定Object数组是否排序.
	 *
	 * <p>
	 *  检查数组是否按照给定的 {@code compareTo}方法排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted(["1","2","3","4"])		-&gt;	true
	 * ArrayUtils.isSorted(["1","3","2","4"])		-&gt;	false
	 * ArrayUtils.isSorted(["1"])		-&gt;	true
	 * ArrayUtils.isSorted(["4"])		-&gt;	true
	 * ArrayUtils.isSorted([])		-&gt;	true
	 * ArrayUtils.isSorted({@code null})		-&gt;	true
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @param <T> 要检查的数组的类型, 必须实现 {@code Comparable}
	 * @return  返回数组是否排序
	 */
	public static <T extends Comparable<? super T>> boolean isSorted(final T[] array) {

		return isSorted(array, new Comparator<T>() {
			@Override
			public int compare(final T o1, final T o2) {
				return o1.compareTo(o2);
			}
		});
	}


	/**
	 * 检查给定Object数组是否排序.
	 *
	 * <p>
	 * 检查提供的数组是否根据提供的{@code Comparator}进行排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted({@code null}, comparator) 		-&gt;	true
	 * ArrayUtils.isSorted([], comparator) 		-&gt;	true
	 * ArrayUtils.isSorted(["1"], comparator) 		-&gt;	true;
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @param comparator 要比较的{@code Comparator}
	 * @param <T> 输入数组的元素类型
	 * @return 返回数组是否排序
	 * @throws IllegalArgumentException 当comparartor为{@code null}时抛出该异常
	 */
	public static <T> boolean isSorted(@Nullable final T[] array, final Comparator<T> comparator) {

		if (comparator == null) {
			throw new IllegalArgumentException("Comparator should not be null.");
		}

		if (array == null || array.length < 2) {
			return true;
		}

		T previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final T current = array[i];
			if (comparator.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定int数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([1,2,3,4])		-&gt; 	true
	 * ArrayUtils.isSorted([1,3,2,4])		-&gt; 	false
	 * ArrayUtils.isSorted([4,3,2,1])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @return 返回数组是否按照自然顺序排序
	 */
	public static boolean isSorted(@Nullable final int[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		int previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final int current = array[i];
			if (NumberUtils.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定long数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([1,2,3,4])		-&gt; 	true
	 * ArrayUtils.isSorted([1,3,2,4])		-&gt; 	false
	 * ArrayUtils.isSorted([4,3,2,1])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @return 返回数组是否按照自然顺序排序
	 */
	public static boolean isSorted(@Nullable final long[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		long previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final long current = array[i];
			if (NumberUtils.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定short数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([1,2,3,4])		-&gt; 	true
	 * ArrayUtils.isSorted([1,3,2,4])		-&gt; 	false
	 * ArrayUtils.isSorted([4,3,2,1])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @return 返回数组是否按照自然顺序排序
	 */
	public static boolean isSorted(@Nullable final short[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		short previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final short current = array[i];
			if (NumberUtils.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定double数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([1,2,3,4])		-&gt; 	true
	 * ArrayUtils.isSorted([1,3,2,4])		-&gt; 	false
	 * ArrayUtils.isSorted([4,3,2,1])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @return 返回数组是否按照自然顺序排序
	 */
	public static boolean isSorted(@Nullable final double[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		double previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final double current = array[i];
			if (Double.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定float数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([1,2,3,4])		-&gt; 	true
	 * ArrayUtils.isSorted([1,3,2,4])		-&gt; 	false
	 * ArrayUtils.isSorted([4,3,2,1])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @return 返回数组是否按照自然顺序排序
	 */
	public static boolean isSorted(@Nullable final float[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		float previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final float current = array[i];
			if (Float.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定byte数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([1,2,3,4])		-&gt; 	true
	 * ArrayUtils.isSorted([1,3,2,4])		-&gt; 	false
	 * ArrayUtils.isSorted([4,3,2,1])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @return 返回数组是否按照自然顺序排序
	 */
	public static boolean isSorted(@Nullable final byte[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		byte previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final byte current = array[i];
			if (NumberUtils.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定char数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([1,2,3,4])		-&gt; 	true
	 * ArrayUtils.isSorted([1,3,2,4])		-&gt; 	false
	 * ArrayUtils.isSorted([4,3,2,1])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 要检查的数组
	 * @return 返回数组是否按照字符顺序排序
	 */
	public static boolean isSorted(@Nullable final char[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		char previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final char current = array[i];
			if (CharUtils.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 检查给定char数组是否排序.
	 *
	 * <p>
	 * 此方法检查提供的数组是否按自然顺序排序，即是否从小到大排序
	 * {@code false}在{@code true}之前
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isSorted([1])		-&gt; 	true
	 * ArrayUtils.isSorted([])		-&gt; 	true
	 * ArrayUtils.isSorted({@code null})		-&gt; 	true
	 * ArrayUtils.isSorted([false,true,true])		-&gt; 	true
	 * ArrayUtils.isSorted([true, false])		-&gt; 	false
	 * ArrayUtils.isSorted([false, true, false])		-&gt; 	false
	 * </pre>
	 *
	 * @param array 数组
	 * @return 是否有序
	 */
	public static boolean isSorted(@Nullable final boolean[] array) {

		if (array == null || array.length < 2) {
			return true;
		}

		boolean previous = array[0];
		final int n = array.length;
		for (int i = 1; i < n; i++) {
			final boolean current = array[i];
			if (BooleanUtils.compare(previous, current) > 0) {
				return false;
			}

			previous = current;
		}
		return true;
	}

	/**
	 * 从指定的boolean数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, true)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], true)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([false, false], true)		-&gt;	[false, false]
	 * ArrayUtils.removeAllOccurences([false, true, false], true)		-&gt;	[false, false]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static boolean[] removeAllOccurences(@Nullable final boolean[] array, final boolean element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的char数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static char[] removeAllOccurences(@Nullable final char[] array, final char element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的byte数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static byte[] removeAllOccurences(@Nullable final byte[] array, final byte element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的short数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static short[] removeAllOccurences(@Nullable final short[] array, final short element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的int数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static int[] removeAllOccurences(@Nullable final int[] array, final int element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的long数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static long[] removeAllOccurences(@Nullable final long[] array, final long element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的float数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static float[] removeAllOccurences(@Nullable final float[] array, final float element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的double数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static double[] removeAllOccurences(@Nullable final double[] array, final double element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 从指定的Object数组中移除指定元素的引用.
	 *
	 * <p>
	 * 所有后续元素都向左移动
	 * 如果数组不包含这样的元素，则不会从数组中移除任何元素
	 * 如果输入数组是null，则返回null
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.removeAllOccurences({@code null}, 1)		-&gt;	{@code null}
	 * ArrayUtils.removeAllOccurences([], 1)		-&gt;	[]
	 * ArrayUtils.removeAllOccurences([1,2], 3)		-&gt;	[1, 2]
	 * ArrayUtils.removeAllOccurences([1, 2, 1], 2)		-&gt;	[1, 1]
	 * </pre>
	 *
	 * @param <T> 输入数组的元素类型
	 * @param element 要移除的元素
	 * @param array 输入的数组
	 * @return 一个包含所有现存元素但除去指定的元素的新数组
	 */
	public static <T> T[] removeAllOccurences(@Nullable final T[] array, final T element) {

		int index = indexOf(array, element);
		if (index == INDEX_NOT_FOUND) {
			return clone(array);
		}

		final int[] indices = new int[array.length - index];
		indices[0] = index;
		int count = 1;

		while ((index = indexOf(array, element, indices[count - 1] + 1)) != INDEX_NOT_FOUND) {
			indices[count++] = index;
		}

		return removeAll(array, Arrays.copyOf(indices, count));
	}

	/**
	 * 把Object数组转换为字符串数组.
	 *
	 * <p>
	 * 返回一个数组，该数组包含参数数组中每个元素的字符串表示形式
	 *输入数组为{@code null}时此方法返回{@code null}
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.toStringArray({@code null})		-&gt;	{@code null}
	 * ArrayUtils.toStringArray([])		-&gt;	[]
	 * ArrayUtils.toStringArray([1, 2])		-&gt;	["1","2"]
	 * </pre>
	 *
	 * @param array  要处理的{@code Object[]}数组 , 可能为{@code null}
	 * @return 和参数数组一样大小的，包含所有参数数组元素的{@code String}形式的数组
	 * 如果输入的数组为 {@code null}则返回 {@code null}
	 * @throws NullPointerException 如果参数数组包含 {@code null}则抛出该异常
	 */
	public static String[] toStringArray(@Nullable final Object[] array) {

		if (array == null) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_STRING_ARRAY;
		}

		final String[] result = new String[array.length];
		for (int i = 0; i < array.length; i++) {
			result[i] = array[i].toString();
		}

		return result;
	}

	/**
	 * 把Object数组转换为字符串数组.
	 *
	 * <p>
	 * 返回一个数组，该数组包含参数数组中每个元素的字符串表示形式
	 * 并且可以处理参数数组中的{@code null}
	 *输入数组为{@code null}时此方法返回{@code null}
	 * </p>
	 *
	 * @param array 要处理的{@code Object[]}数组 , 可能为{@code null}
	 * @param valueForNullElements 如果数组中有{@code null}要替换的值
	 * @return 一个 {@code String}数组, 如果输入的数组为 {@code null}则返回 {@code null}
	 */
	public static String[] toStringArray(@Nullable final Object[] array, final String valueForNullElements) {

		if (null == array) {
			return null;
		} else if (array.length == 0) {
			return EMPTY_STRING_ARRAY;
		}

		final String[] result = new String[array.length];
		for (int i = 0; i < array.length; i++) {
			final Object object = array[i];
			result[i] = (object == null ? valueForNullElements : object.toString());
		}

		return result;
	}

	/**
	 * 向给定boolean数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static boolean[] insert(final int index, @Nullable final boolean[] array, @Nullable final boolean... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final boolean[] result = new boolean[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定byte数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static byte[] insert(final int index, @Nullable final byte[] array, @Nullable final byte... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final byte[] result = new byte[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定char数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static char[] insert(final int index, @Nullable final char[] array, @Nullable final char... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final char[] result = new char[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定double数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static double[] insert(final int index, @Nullable final double[] array, @Nullable final double... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final double[] result = new double[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定float数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static float[] insert(final int index, @Nullable final float[] array, @Nullable final float... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final float[] result = new float[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定int数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static int[] insert(final int index, @Nullable final int[] array, @Nullable final int... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final int[] result = new int[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定long数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static long[] insert(final int index, @Nullable final long[] array, @Nullable final long... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final long[] result = new long[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定short数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static short[] insert(final int index, @Nullable final short[] array, @Nullable final short... values) {

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final short[] result = new short[array.length + values.length];

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 * 向给定Object数组插入给定元素.
	 *
	 * <p>
	 * 将元素插入给定索引处的数组中，往往返回一个新数组
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.insert(index, {@code null}, {@code null})      = {@code null}
	 * ArrayUtils.insert(index, array, {@code null})     = array的复制
	 * ArrayUtils.insert(index, {@code null}, values)    = {@code null}
	 * </pre>
	 *
	 * @param <T>  {@code array} 和{@code values}中元素的类型
	 * @param index {@code array}中要插入新元素的位置
	 * @param array 要插入元素的数组, 可能为 {@code null}
	 * @param values 要插入的值, 可能为 {@code null}
	 * @return 一个包含现有元素和新元素的新数组
	 * @throws IndexOutOfBoundsException 如果index超出范围{@code index < 0}
	 * 或{@code index > 数组长度}则抛出该异常
	 */
	public static <T> T[] insert(final int index, @Nullable final T[] array, @Nullable final T... values) {
		/*
		 * Note on use of @SafeVarargs:
		 *
		 * By returning null when 'array' is null, we avoid returning the vararg
		 * array to the caller. We also avoid relying on the type of the vararg
		 * array, by inspecting the component type of 'array'.
		 */

		if (array == null) {
			return null;
		}
		if (values == null || values.length == 0) {
			return clone(array);
		}
		if (index < 0 || index > array.length) {
			throw new IndexOutOfBoundsException("Index: " + index + ", Length: " + array.length);
		}

		final Class<?> type = array.getClass().getComponentType();
		@SuppressWarnings("unchecked") // OK, because array and values are of type T
		final
		T[] result = (T[]) Array.newInstance(type, array.length + values.length);

		System.arraycopy(values, 0, result, index, values.length);
		if (index > 0) {
			System.arraycopy(array, 0, result, 0, index);
		}
		if (index < array.length) {
			System.arraycopy(array, index, result, index + values.length, array.length - index);
		}
		return result;
	}

	/**
	 *随机的改变指定Object数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final Object[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定Object数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final Object[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定boolean数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final boolean[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定boolean数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final boolean[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定byte数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final byte[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定byte数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final byte[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定char数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final char[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定char数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final char[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定short数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final short[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定short数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final short[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定int数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final int[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定int数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final int[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定long数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final long[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定long数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final long[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定float数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final float[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定float数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final float[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 *随机的改变指定double数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final double[] array) {
		shuffle(array, new Random());
	}

	/**
	 * 随机的改变指定double数组的元素顺序.
	 *
	 * <p>
	 * 使用Fisher-Yates 算法随机的改变数组中元素的顺序
	 * </p>
	 *
	 * @param array   要打乱的数组
	 * @param random  用于置换元素的随机性来源
	 * @see <a href="https://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle">Fisher-Yates shuffle algorithm</a>
	 */
	public static void shuffle(@Nullable final double[] array, final Random random) {

		for (int i = array.length; i > 1; i--) {
			swap(array, i - 1, random.nextInt(i), 1);
		}
	}

	/**
	 * 判断在给定索引处是否可以安全地访问给定数组.
	 *
	 * <p>
	 * 通过index和数组长度的大小关系来判断index处是否可以安全的访问
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.isArrayIndexValid({@code null}, 0)	-&gt;	false
	 * ArrayUtils.isArrayIndexValid([1,2,3], 5)	-&gt;	false
	 * ArrayUtils.isArrayIndexValid([1,2,3], 1)	-&gt;	true
	 * </pre>
	 *
	 * @param <T> 数组元素的类型
	 * @param array 要判断的数组, 可能为{@code null}
	 * @param index 要检查的数组的索引
	 * @return 给定的索引在给定数组中是否可以安全访问
	 */
	public static <T> boolean isArrayIndexValid(@Nullable T[] array, int index){

		if(getLength(array) == 0 || array.length <= index){
			return false;
		}

		return index >= 0;
	}

	/**
	 * 判断两个整数数组是否相同.
	 *
	 * <p>
	 * 首先判断长度是否相同,再逐个元素比较是否相同
	 * </p>
	 *
	 * <pre>
	 *ArrayUtils.isSameArray([1,2], {@code null})	-&gt;	false
	 * ArrayUtils.isSameArray([1,2], [1,1])	-&gt;	false
	 * ArrayUtils.isSameArray([1,2], [1,2])	-&gt;	true
	 * </pre>
	 *
	 * @param array1 数组1
	 * @param array2 数组2
	 * @return 相同则返回true，否则返回false
	 */
	public static boolean isSameArray(@Nullable int[] array1, @Nullable int[] array2) {

		int len1 = array1 == null ? 0 : array1.length;
		int len2 = array2 == null ? 0 : array2.length;
		if (len1 != len2) {
			return false;
		}

		for (int i = 0; i < len1; i++) {
			if (array1[i] != array2[i]) {
				return false;
			}
		}

		return true;
	}

	/**
	 * 判断两个字节数组是否相同.
	 *
	 * <p>
	 * 首先判断长度是否相同,再逐个元素比较是否相同
	 * </p>
	 *
	 * <pre>
	 *ArrayUtils.isSameArray([1,2], {@code null})	-&gt;	false
	 * ArrayUtils.isSameArray([1,2], [1,1])	-&gt;	false
	 * ArrayUtils.isSameArray([1,2], [1,2])	-&gt;	true
	 * </pre>
	 *
	 * @param array1 数组1
	 * @param array2 数组2
	 * @return 相同则返回true，否则返回false
	 */
	public static boolean isSameArray(@Nullable byte[] array1, @Nullable byte[] array2) {

		int len1 = array1 == null ? 0 : array1.length;
		int len2 = array2 == null ? 0 : array2.length;
		if (len1 != len2) {
			return false;
		}

		for (int i = 0; i < len1; i++) {
			if (array1[i] != array2[i]) {
				return false;
			}
		}

		return true;
	}

	/**
	 * 把数组转换为一个ArrayList.
	 *
	 * <p>
	 * java.util.Arrays.toList 方法返回的是不可变的list，而且List类型是一个内部私有的类
	 * 此方法返回一个ArrayList
	 * </p>
	 *
	 * <pre>
	 * ArrayUtils.toList({@code null})	-&gt;	[]
	 * ArrayUtils.toList([])	-&gt;	[]
	 * ArrayUtils.toList(["1","2"])	-&gt;	["1","2"]
	 * </pre>
	 *
	 * @param array 要转换的数组
	 * @param <T> 数组类型
	 * @return 转换得到的ArrayList
	 */
	public static <T> List<T> toList(@Nullable T[] array) {

		List<T> list = new ArrayList<T>();
		if (array == null || array.length == 0) {
			return list;
		}
		Collections.addAll(list, array);
		return list;
	}
}