package com.fr.scheduler.tool;

import com.fr.scheduler.bean.FineTriggerProvider;
import com.fr.scheduler.enums.CronMisfireInstType;
import com.fr.scheduler.enums.FineTriggerType;
import com.fr.scheduler.enums.SimpleMisfireInstType;
import com.fr.stable.StringUtils;
import com.fr.third.v2.org.quartz.CronExpression;
import com.fr.third.v2.org.quartz.CronScheduleBuilder;
import com.fr.third.v2.org.quartz.JobKey;
import com.fr.third.v2.org.quartz.ScheduleBuilder;
import com.fr.third.v2.org.quartz.SimpleScheduleBuilder;
import com.fr.third.v2.org.quartz.Trigger;
import com.fr.third.v2.org.quartz.TriggerBuilder;
import org.jetbrains.annotations.NotNull;

import java.text.ParseException;
import java.util.Date;
import java.util.TimeZone;

/**
 * 触发器创建工具类.
 *
 * <p>
 * 用于创建添加任务所需的Trigger.
 * </p>
 *
 * @author Cloud.Liu
 * created on 2020-09-09
 * @IncludeIntoJavadoc
 */
public class FineTriggerBuilder {

    /**
     * 创建一个Trigger.
     *
     * <pre>
     * 如：
     * Trigger trigger = FineTriggerBuilder.newTrigger()
     *         .withName("触发器名称")
     *         .withGroup("触发器组名")
     *         .startNow()                      // 立即开始
     *         .neverEnd()                      // 无结束时间
     *         .simpleSchedule()                // 简单重复类型触发器
     *         .repeatWithIntervalInHours(1)    // 重复间隔1小时
     *         .repeatForever()                 // 无限重复
     *         .withSimpleDefaultMisfireInst()  // 默认Misfire策略
     *         .build();
     * </pre>
     *
     * @return Builder
     * @see SimpleMisfireInstType 简单重复类型触发器的Misfire策略
     * @see CronMisfireInstType Cron表达式类型触发器的Misfire策略
     */
    public static NameStep newTrigger() {
        return new Steps();
    }

    /**
     * 从{@link FineTriggerProvider}创建Trigger.
     *
     * @param provider provider
     * @return Trigger
     * @throws RuntimeException provider为{@link com.fr.scheduler.bean.FineCronTriggerProvider}且Cron表达式不合法
     */
    public static Trigger fromFineTriggerProvider(FineTriggerProvider<? extends Trigger> provider) {

        StopStep stopStep = newTrigger()
                .withName(provider.getName())
                .withGroup(provider.getGroup())
                .startAt(provider.getStartTime());

        TypeStep typeStep;
        if (provider.getStopTime() == null) {
            typeStep = stopStep.neverEnd();
        } else {
            typeStep = stopStep.stopAt(provider.getStopTime());
        }

        return typeStep.withSchedule(provider.createSchedule()).build();
    }

    /**
     * 指定Trigger名称.
     */
    public interface NameStep {
        /**
         * 指定Trigger名称.
         *
         * @param name 名称
         * @return Builder
         */
        GroupStep withName(@NotNull String name);
    }

    /**
     * 指定TriggerGroup.
     */
    public interface GroupStep {
        /**
         * 指定TriggerGroup.
         *
         * @param group TriggerGroup
         * @return Builder
         */
        StartStep withGroup(@NotNull String group);
    }

    /**
     * 指定开始时间.
     */
    public interface StartStep {
        /**
         * 立即开始.
         *
         * @return Builder
         */
        StopStep startNow();

        /**
         * 于特定时间开始.
         *
         * @param startTime 开始时间
         * @return Builder
         */
        StopStep startAt(@NotNull Date startTime);
    }

    /**
     * 指定结束时间.
     */
    public interface StopStep {
        /**
         * 只执行一次.
         *
         * @return Builder
         */
        SimpleMisfireStep onlyOnce();

        /**
         * 永不结束.
         *
         * @return Builder
         */
        TypeStep neverEnd();

        /**
         * 于特定时间结束.
         *
         * @param stopTime 结束时间
         * @return Builder
         */
        TypeStep stopAt(@NotNull Date stopTime);
    }

    /**
     * 指定触发器类型.
     */
    public interface TypeStep {


        BuildStep withSchedule(ScheduleBuilder<? extends Trigger> scheduleBuilder);

        /**
         * 创建简单重复类型触发器.
         *
         * @return Builder
         */
        SimpleStep simpleSchedule();

        /**
         * 创建Cron表达式类型触发器.
         *
         * @param cronExpression Cron表达式
         * @return Builder
         */
        CronMisfireStep cronSchedule(CronExpression cronExpression);

        /**
         * 创建Cron表达式类型触发器.
         *
         * @param cronExpression Cron表达式
         * @return Builder
         * @throws ParseException Cron表达式解析出错
         */
        CronMisfireStep cronSchedule(String cronExpression) throws ParseException;

        /**
         * 创建Cron表达式类型触发器，并指定时区.
         *
         * @param cronExpression Cron表达式
         * @param timeZone 时区
         * @return Builder
         */
        CronMisfireStep cronSchedule(CronExpression cronExpression, TimeZone timeZone);

        /**
         * 创建Cron表达式类型触发器，并指定时区.
         *
         * @param cronExpression Cron表达式
         * @param timeZone 时区
         * @return Builder
         * @throws ParseException Cron表达式解析出错
         */
        CronMisfireStep cronSchedule(String cronExpression, TimeZone timeZone) throws ParseException;
    }

    /**
     * 指定简单重复触发器重复间隔.
     */
    public interface SimpleStep {
        /**
         * 指定重复间隔为秒.
         *
         * @param interval 重复间隔
         * @return Builder
         */
        SimpleRepeatStep repeatWithIntervalInSeconds(int interval);

        /**
         * 指定重复间隔为分钟.
         *
         * @param interval 重复间隔
         * @return Builder
         */
        SimpleRepeatStep repeatWithIntervalInMinutes(int interval);

        /**
         * 指定重复间隔为小时.
         *
         * @param interval 重复间隔
         * @return Builder
         */
        SimpleRepeatStep repeatWithIntervalInHours(int interval);

        /**
         * 指定重复间隔为天.
         *
         * @param interval 重复间隔
         * @return Builder
         */
        SimpleRepeatStep repeatWithIntervalInDays(int interval);

        /**
         * 指定重复间隔为周.
         *
         * @param interval 重复间隔
         * @return Builder
         */
        SimpleRepeatStep repeatWithIntervalInWeeks(int interval);
    }

    /**
     * 指定简单重复触发器重复次数.
     */
    public interface SimpleRepeatStep {
        /**
         * 永远重复执行.
         *
         * @return Builder
         */
        SimpleMisfireStep repeatForever();

        /**
         * 重复执行特定次数.
         *
         * @param times 次数
         * @return Builder
         */
        SimpleMisfireStep repeatForTimes(int times);
    }

    /**
     * 指定简单重复触发器Misfire策略.
     *
     * <p>
     * Misfire指任务由于关机等原因未能在该触发时触发.<br>
     * Misfire策略指定了重新上线发现Misfire后的处理策略.
     * </p>
     */
    public interface SimpleMisfireStep {

        /**
         * 默认策略.
         *
         * <p>
         * 见{@link SimpleMisfireInstType#DEFAULT}.
         * </p>
         *
         * @return Builder
         */
        BuildStep withSimpleDefaultMisfireInst();

        /**
         * 指定策略.
         *
         * <p>
         * 见{@link SimpleMisfireInstType}.
         * </p>
         *
         * @return Builder
         */
        BuildStep withSimpleMisfireInst(SimpleMisfireInstType simpleMisfireInst);
    }

    /**
     * 指定Cron表达式触发器Misfire策略.
     *
     * <p>
     * Misfire指任务由于关机等原因未能在该触发时触发.<br>
     * Misfire策略指定了重新上线发现Misfire后的处理策略.
     * </p>
     */
    public interface CronMisfireStep {

        /**
         * 默认策略.
         *
         * <p>
         * 见{@link CronMisfireInstType#DEFAULT}.
         * </p>
         *
         * @return Builder
         */
        BuildStep withCronDefaultMisfireInst();

        /**
         * 指定策略.
         *
         * <p>
         * 见{@link CronMisfireInstType}.
         * </p>
         *
         * @return Builder
         */
        BuildStep withCronMisfireInst(CronMisfireInstType cronMisfireInst);
    }

    /**
     * 创建Trigger.
     */
    public interface BuildStep {

        /**
         * 指定节点ID.
         *
         * <p>
         * 一般不手动调用本方法.
         * </p>
         *
         * @param nodeId ID
         * @return Builder
         */
        BuildStep nodeId(String nodeId);

        /**
         * 指定对应Job.
         *
         * <p>
         * 一般不手动调用本方法.
         * </p>
         *
         * @param jobName jobName
         * @param jobGroup jobGroup
         * @return Builder
         */
        default BuildStep forJob(String jobName, String jobGroup) {
            return forJob(JobKey.jobKey(jobName, jobGroup));
        }

        /**
         * 指定对应Job.
         *
         * <p>
         * 一般不手动调用本方法.
         * </p>
         *
         * @param jobKey jobKey
         * @return Builder
         */
        BuildStep forJob(JobKey jobKey);

        /**
         * 创建Trigger.
         *
         * @return Trigger
         */
        Trigger build();
    }

    private static class Steps implements NameStep, GroupStep, StartStep, StopStep, TypeStep, SimpleStep, SimpleRepeatStep, SimpleMisfireStep, CronMisfireStep, BuildStep {

        private String name;
        private String group;
        private Date startTime;
        private Date stopTime;
        private FineTriggerType triggerType;
        private String nodeId;
        private JobKey jobKey;

        private ScheduleBuilder<? extends Trigger> scheduleBuilder;

        @Override
        public GroupStep withName(@NotNull String name) {
            if (StringUtils.isEmpty(name)) {
                throw new IllegalArgumentException("Name should not be null or empty.");
            } else {
                this.name = name;
            }
            return this;
        }

        @Override
        public StartStep withGroup(@NotNull String group) {
            if (StringUtils.isEmpty(group)) {
                throw new IllegalArgumentException("Group should not be null or empty.");
            } else {
                this.group = group;
            }
            return this;
        }

        @Override
        public StopStep startNow() {
            this.startTime = new Date();
            return this;
        }

        @Override
        public StopStep startAt(@NotNull Date startTime) {
            // TODO const
            if (startTime.getTime() < new Date().getTime() - 30000) {
                throw new IllegalArgumentException("Start time should be later than now.");
            } else {
                this.startTime = startTime;
            }
            return this;
        }

        @Override
        public SimpleMisfireStep onlyOnce() {
            this.stopTime = null;
            this.triggerType = FineTriggerType.SIMPLE;
            scheduleBuilder = SimpleScheduleBuilder.simpleSchedule();
            return this;
        }

        @Override
        public TypeStep neverEnd() {
            this.stopTime = null;
            return this;
        }

        @Override
        public TypeStep stopAt(@NotNull Date stopTime) {
            if (stopTime.getTime() < startTime.getTime()) {
                throw new IllegalArgumentException("Stop time should be later than start time.");
            } else {
                this.stopTime = stopTime;
            }
            return this;
        }

        @Override
        public BuildStep withSchedule(ScheduleBuilder<? extends Trigger> scheduleBuilder) {
            this.scheduleBuilder = scheduleBuilder;
            return this;
        }

        @Override
        public SimpleStep simpleSchedule() {
            this.triggerType = FineTriggerType.SIMPLE;
            this.scheduleBuilder = SimpleScheduleBuilder.simpleSchedule();
            return this;
        }

        @Override
        public SimpleRepeatStep repeatWithIntervalInSeconds(int interval) {
            ((SimpleScheduleBuilder) scheduleBuilder).withIntervalInSeconds(interval);
            return this;
        }

        @Override
        public SimpleRepeatStep repeatWithIntervalInMinutes(int interval) {
            ((SimpleScheduleBuilder) scheduleBuilder).withIntervalInMinutes(interval);
            return this;
        }

        @Override
        public SimpleRepeatStep repeatWithIntervalInHours(int interval) {
            ((SimpleScheduleBuilder) scheduleBuilder).withIntervalInHours(interval);
            return this;
        }

        @Override
        public SimpleRepeatStep repeatWithIntervalInDays(int interval) {
            ((SimpleScheduleBuilder) scheduleBuilder).withIntervalInHours(interval * 24);
            return this;
        }

        @Override
        public SimpleRepeatStep repeatWithIntervalInWeeks(int interval) {
            ((SimpleScheduleBuilder) scheduleBuilder).withIntervalInHours(interval * 24 * 7);
            return this;
        }

        @Override
        public SimpleMisfireStep repeatForever() {
            ((SimpleScheduleBuilder) scheduleBuilder).repeatForever();
            return this;
        }

        @Override
        public SimpleMisfireStep repeatForTimes(int times) {
            if (times < -1) {
                throw new IllegalArgumentException("Repeat count should be larger than -1.");
            } else {
                ((SimpleScheduleBuilder) scheduleBuilder).withRepeatCount(times);
                return this;
            }
        }

        @Override
        public CronMisfireStep cronSchedule(CronExpression cronExpression) {
            this.triggerType = FineTriggerType.CRON;
            this.scheduleBuilder = CronScheduleBuilder.cronSchedule(cronExpression);
            ((CronScheduleBuilder) scheduleBuilder).inTimeZone(TimeZone.getDefault());
            return this;
        }

        @Override
        public CronMisfireStep cronSchedule(String cronExpression) throws ParseException {
            this.triggerType = FineTriggerType.CRON;
            this.scheduleBuilder = CronScheduleBuilder.cronSchedule(new CronExpression(cronExpression));
            ((CronScheduleBuilder) scheduleBuilder).inTimeZone(TimeZone.getDefault());
            return this;
        }

        @Override
        public CronMisfireStep cronSchedule(CronExpression cronExpression, TimeZone timeZone) {
            this.triggerType = FineTriggerType.CRON;
            this.scheduleBuilder = CronScheduleBuilder.cronSchedule(cronExpression);
            ((CronScheduleBuilder) scheduleBuilder).inTimeZone(timeZone);
            return this;
        }

        @Override
        public CronMisfireStep cronSchedule(String cronExpression, TimeZone timeZone) throws ParseException {
            this.triggerType = FineTriggerType.CRON;
            this.scheduleBuilder = CronScheduleBuilder.cronSchedule(new CronExpression(cronExpression));
            ((CronScheduleBuilder) scheduleBuilder).inTimeZone(timeZone);
            return this;
        }

        @Override
        public BuildStep withSimpleDefaultMisfireInst() {
            return this;
        }

        @Override
        public BuildStep withSimpleMisfireInst(SimpleMisfireInstType simpleMisfireInst) {
            simpleMisfireInst.modifySimpleScheduleBuilder((SimpleScheduleBuilder) scheduleBuilder);
            return this;
        }

        @Override
        public BuildStep withCronDefaultMisfireInst() {
            return this;
        }

        @Override
        public BuildStep withCronMisfireInst(CronMisfireInstType cronMisfireInst) {
            cronMisfireInst.modifyCronScheduleBuilder((CronScheduleBuilder) scheduleBuilder);
            return this;
        }

        @Override
        public BuildStep nodeId(String nodeId) {
            this.nodeId = nodeId;
            return this;
        }

        @Override
        public BuildStep forJob(JobKey jobKey) {
            this.jobKey = jobKey;
            return this;
        }

        @Override
        public Trigger build() {

            TriggerBuilder<Trigger> triggerBuilder = TriggerBuilder.newTrigger();
            triggerBuilder.withIdentity(name, group);
            triggerBuilder.startAt(startTime);
            // null for infinite
            triggerBuilder.endAt(stopTime);
            // 指定job
            if (jobKey != null) {
                triggerBuilder.forJob(jobKey);
            }
            triggerBuilder.withSchedule(scheduleBuilder);
            // appointId 具体逻辑写在FineScheduler里
            triggerBuilder.appointId(nodeId);
            return triggerBuilder.build();
        }
    }
}
