package com.fr.scheduler.tool;

import com.fr.log.FineLoggerFactory;
import com.fr.scheduler.job.JobState;
import com.fr.third.v2.org.quartz.JobDetail;
import com.fr.third.v2.org.quartz.JobKey;
import com.fr.third.v2.org.quartz.SchedulerException;
import com.fr.third.v2.org.quartz.Trigger;
import com.fr.third.v2.org.quartz.TriggerKey;
import com.fr.third.v2.org.quartz.impl.matchers.GroupMatcher;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Set;

/**
 * 定时任务管理工具接口.
 *
 * <p>
 * 实现类为{@link FineScheduler}.
 * </p>
 *
 * @author Cloud.Liu
 * created on 2020-09-09
 * @see FineScheduler
 */
public interface FineSchedulerProvider {

    /**
     * 初始化Fine-Scheduler.
     *
     * @param properties 配置项
     * @IncludeIntoJavadoc
     */
    void initScheduler(Properties properties);

    /**
     * 判断Fine-Scheduler是否已启动.
     *
     * @return 是否已启动
     * @IncludeIntoJavadoc
     */
    boolean isStarted();

    /**
     * 关闭Fine-Scheduler.
     *
     * @IncludeIntoJavadoc
     */
    void shutdownScheduler();

    /**
     * 判断Fine-Scheduler是否已关闭.
     *
     * @return 是否已关闭
     * @IncludeIntoJavadoc
     */
    boolean isShutdown();

    /**
     * 调度一个Trigger.
     *
     * <p>
     * 该Trigger对应的JobDetail应已经存在.<br>
     * 即该Trigger已经使用{@code forJob()}与特定JobDetail绑定过.
     * </p>
     *
     * <p>
     * 发生错误则返回{@code null}.
     * </p>
     *
     * @param trigger trigger
     * @return 任务的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date scheduleTrigger(Trigger trigger);

    /**
     * 调度一个Trigger,支持在初始化之前提交..
     *
     * <p>
     * 该Trigger对应的JobDetail应已经存在.<br>
     * 即该Trigger已经使用{@code forJob()}与特定JobDetail绑定过.
     * </p>
     *
     * <p>
     * 发生错误或者scheduler尚未初始化则返回{@code null}.
     * </p>
     *
     * @param trigger trigger
     * @return 任务的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date submitTrigger(Trigger trigger);

    /**
     * 创建一个任务.
     *
     * <p>
     * 使用{@link FineJobBuilder}创建JobDetail.<br>
     * 使用{@link FineTriggerBuilder}创建Trigger.
     * </p>
     *
     * <p>
     * 这个任务会在本机（单机模式）或集群中任一节点（集群模式）执行.<br>
     * 创建失败时返回null，否则返回任务的下次执行时间.
     * </p>
     *
     * @param jobDetail 任务
     * @param trigger 触发器
     * @return 任务的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date addJob(@NotNull JobDetail jobDetail, @NotNull Trigger trigger);

    /**
     * 提交一个任务,支持在初始化之前提交.
     *
     * <p>
     * 使用{@link FineJobBuilder}创建JobDetail.<br>
     * 使用{@link FineTriggerBuilder}创建Trigger.
     * </p>
     *
     * <p>
     * 这个任务会在本机（单机模式）或集群中任一节点（集群模式）执行.<br>
     * 创建失败时返回null.<br>
     * 未初始化时返回null.<br>
     * 否则返回任务的下次执行时间.
     * </p>
     *
     * @param jobDetail 任务
     * @param trigger 触发器
     * @return 任务的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date submitJob(@NotNull JobDetail jobDetail, @NotNull Trigger trigger);

    /**
     * 创建一个在特定某个节点执行的任务.
     *
     * <p>
     * 使用{@link FineJobBuilder}创建JobDetail.<br>
     * 使用{@link FineTriggerBuilder}创建Trigger.
     * </p>
     *
     * <p>
     * 这个任务会在指定的特定节点执行.<br>
     * 创建失败时返回null，否则返回任务的下次执行时间.
     * </p>
     *
     * @param jobDetail 任务
     * @param trigger 触发器
     * @param nodeId 指定的节点ID
     * @return 任务的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date addJobRunOnSpecificNode(@NotNull JobDetail jobDetail, @NotNull Trigger trigger, @NotNull String nodeId);

    /**
     * 创建一个在任意一个节点执行的任务.
     *
     * <p>
     * 使用{@link FineJobBuilder}创建JobDetail.<br>
     * 使用{@link FineTriggerBuilder}创建Trigger.
     * </p>
     *
     * <p>
     * 这个任务会在本机（单机模式）或集群中任一节点（集群模式）执行.<br>
     * 同{@link FineSchedulerProvider#addJob(com.fr.third.v2.org.quartz.JobDetail, com.fr.third.v2.org.quartz.Trigger)}.<br>
     * 创建失败时返回null，否则返回任务的下次执行时间.
     * </p>
     *
     * @param jobDetail 任务
     * @param trigger 触发器
     * @return 任务的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date addJobRunOnRandomNode(@NotNull JobDetail jobDetail, @NotNull Trigger trigger);

    /**
     * 创建一个在指定多个节点中任意一个执行的任务.
     *
     * <p>
     * 使用{@link FineJobBuilder}创建JobDetail.<br>
     * 使用{@link FineTriggerBuilder}创建Trigger.
     * </p>
     *
     * <p>
     * 这个任务会在指定节点中的随机节点执行.<br>
     * 创建失败时返回null，否则返回任务的下次执行时间.
     * </p>
     *
     * @param jobDetail 任务
     * @param trigger 触发器
     * @param nodes 指定节点
     * @return 任务的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date addJobRunOnRandomNodeIn(@NotNull JobDetail jobDetail, @NotNull Trigger trigger, String... nodes);

    /**
     * 创建一个在特定多个节点均执行的任务.
     *
     * <p>
     * 使用{@link FineJobBuilder}创建JobDetail.<br>
     * 使用{@link FineTriggerBuilder}创建Trigger.
     * </p>
     *
     * <p>
     * 指定的多个节点均会执行这个任务.<br>
     * 创建失败时返回null，否则返回任务的下次执行时间.
     * </p>
     *
     * @param jobDetail 任务
     * @param trigger 触发器
     * @param nodes 指定节点
     * @return 任务最早的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date addJobRunOnEachNodeIn(@NotNull JobDetail jobDetail, @NotNull Trigger trigger, @NotNull String... nodes);

    /**
     * 创建一个在所有节点均执行的任务.
     *
     * <p>
     * 使用{@link FineJobBuilder}创建JobDetail.<br>
     * 使用{@link FineTriggerBuilder}创建Trigger.
     * </p>
     *
     * <p>
     * 所有节点都会执行这个任务.<br>
     * 所有节点是任务执行时实时获取的.<br>
     * 创建失败时返回null，否则返回任务的下次执行时间.
     * </p>
     *
     * @param jobDetail 任务
     * @param trigger 触发器
     * @return 任务最早的下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date addJobRunOnEveryNode(@NotNull JobDetail jobDetail, @NotNull Trigger trigger);

    /**
     * 通知本节点有任务会在特定时间后触发.
     *
     * @param time 时间
     */
    void notifyScheduler(long time);

    /**
     * 修改节点ID.
     *
     * @param oldId 旧节点ID
     * @param newId 新节点ID
     * @throws SchedulerException 修改节点ID失败
     * @IncludeIntoJavadoc
     */
    void changeAppointId(String oldId, String newId) throws SchedulerException;

    /**
     * 立即执行一次特定任务.
     *
     * @param jobKey jobKey
     * @IncludeIntoJavadoc
     */
    void runOnce(JobKey jobKey);

    /**
     * 立即执行一次特定任务.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @IncludeIntoJavadoc
     */
    default void runOnce(String jobName, String jobGroup) {
        runOnce(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 获取所有属于特定JobGroup的Job的JobKey.
     *
     * <p>
     * 使用GroupMatcher进行JobGroup匹配。
     * </p>
     *
     * <pre>
     * 如：
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupEquals("JobGroup");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupNotEquals("JobGroup");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupContains("_key_");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupNotContains("_key_");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupStartsWith("Prefix_");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupNotStartsWith("Prefix_");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupEndsWith("Prefix_");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.jobGroupNotEndsWith("Prefix_");
     * GroupMatcher&lt;JobKey&gt; matcher = GroupMatcher.anyJobGroup();
     * </pre>
     *
     * @param matcher JobKey Matcher
     * @return JobKey
     * @throws SchedulerException 获取JobKey失败
     * @IncludeIntoJavadoc
     */
    Set<JobKey> getJobKeysByJobGroup(GroupMatcher<JobKey> matcher) throws SchedulerException;

    /**
     * 获取所有属于特定TriggerGroup的Trigger的TriggerKey.
     *
     * <p>
     * 使用GroupMatcher进行TriggerGroup匹配。
     * </p>
     *
     * <pre>
     * 如：
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupEquals("JobGroup");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupNotEquals("JobGroup");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupContains("_key_");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupNotContains("_key_");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupStartsWith("Prefix_");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupNotStartsWith("Prefix_");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupEndsWith("Prefix_");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.triggerGroupNotEndsWith("Prefix_");
     * GroupMatcher&lt;TriggerKey&gt; matcher = GroupMatcher.anytriggerGroup();
     * </pre>
     *
     * @param matcher Trigger Matcher
     * @return Trigger
     * @throws SchedulerException 获取TriggerKey失败
     * @IncludeIntoJavadoc
     */
    Set<TriggerKey> getTriggerKeysByTriggerGroup(GroupMatcher<TriggerKey> matcher) throws SchedulerException;

    /**
     * 删除任务.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @IncludeIntoJavadoc
     */
    default void removeJob(String jobName, String jobGroup) {
        removeJob(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 删除任务.
     *
     * @param jobKey jobKey
     * @IncludeIntoJavadoc
     */
    void removeJob(JobKey jobKey);

    /**
     * 删除任务,支持在初始化前提交.
     *
     * @param jobKey jobKey
     * @IncludeIntoJavadoc
     */
    void revokeJob(JobKey jobKey);

    /**
     * 暂停任务.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @IncludeIntoJavadoc
     */
    default void pauseJob(String jobName, String jobGroup) {
        pauseJob(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 暂停任务.
     *
     * @param jobKey jobKey
     * @IncludeIntoJavadoc
     */
    void pauseJob(JobKey jobKey);

    /**
     * 恢复任务执行.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @IncludeIntoJavadoc
     */
    default void resumeJob(String jobName, String jobGroup) {
        resumeJob(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 恢复Job执行.
     *
     * @param jobKey jobKey
     * @IncludeIntoJavadoc
     */
    void resumeJob(JobKey jobKey);

    /**
     * 恢复<code>Job</code>执行，忽略<code>Misfire</code>.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @IncludeIntoJavadoc
     */
    default void resumeJobIgnoreMisfire(String jobName, String jobGroup) {
        resumeJobIgnoreMisfire(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 恢复<code>Job</code>执行，忽略<code>Misfire</code>.
     *
     * @param jobKey jobKey
     * @IncludeIntoJavadoc
     */
    void resumeJobIgnoreMisfire(JobKey jobKey);

    /**
     * 判断Job是否正在执行.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @return 是否正在执行
     * @IncludeIntoJavadoc
     */
    default boolean isJobExecuting(String jobName, String jobGroup) {
        return isJobExecuting(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 判断Job是否正在执行.
     *
     * @param jobKey jobKey
     * @return 是否正在执行
     * @IncludeIntoJavadoc
     */
    default boolean isJobExecuting(JobKey jobKey) {
        try {
            return getStateOfJob(jobKey).equals(JobState.EXECUTING);
        } catch (SchedulerException e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
        return false;
    }

    /**
     * 获取Job下次执行时间.
     *
     * <p>
     * 如果Job没有下次执行时间，返回null.
     * </p>
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @return 下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    default Date getNextFireTimeOfJob(String jobName, String jobGroup) {
        return getNextFireTimeOfJob(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 获取Job下次执行时间.
     *
     * <p>
     * 如果Job没有下次执行时间，返回null.
     * </p>
     *
     * @param jobKey jobKey
     * @return 下次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date getNextFireTimeOfJob(JobKey jobKey);

    /**
     * 获取Job上次执行时间.
     *
     * <p>
     * 如果Job没有上次执行时间，返回null.
     * </p>
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @return 上次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    default Date getPreFireTimeOfJob(String jobName, String jobGroup) {
        return getPreFireTimeOfJob(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 获取Job上次执行时间.
     *
     * @param jobKey jobKey
     * @return 上次执行时间
     * @IncludeIntoJavadoc
     */
    @Nullable
    Date getPreFireTimeOfJob(JobKey jobKey);

    /**
     * 判断Job是否存在.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @return 是否存在
     * @IncludeIntoJavadoc
     */
    default boolean jobExist(String jobName, String jobGroup) {
        return jobExist(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 判断Job是否存在.
     *
     * @param jobKey jobKey
     * @return 是否存在
     * @IncludeIntoJavadoc
     */
    boolean jobExist(JobKey jobKey);

    /**
     * 获取Job的所有Trigger.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @return job的所有Trigger
     * @IncludeIntoJavadoc
     */
    default List<Trigger> getTriggersOfJob(String jobName, String jobGroup) {
        return getTriggersOfJob(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 获取Job的所有Trigger.
     *
     * @param jobKey jobKey
     * @return job的所有Trigger
     * @IncludeIntoJavadoc
     */
    List<Trigger> getTriggersOfJob(JobKey jobKey);

    /**
     * 获取Job状态.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @return Job状态
     * @throws SchedulerException 获取Job状态时发生错误
     * @IncludeIntoJavadoc
     */
    default JobState getStateOfJob(String jobName, String jobGroup) throws SchedulerException{
        return getStateOfJob(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 获取Job状态.
     *
     * @param jobKey jobKey
     * @return Job状态 {@link JobState}
     * @throws SchedulerException 获取Job状态时发生错误
     * @IncludeIntoJavadoc
     */
    JobState getStateOfJob(JobKey jobKey) throws SchedulerException;

    /**
     * 获取JobDetail.
     *
     * @param jobName jobName
     * @param jobGroup jobGroup
     * @return JobDetail
     * @IncludeIntoJavadoc
     */
    @Nullable
    default JobDetail getJobDetail(String jobName, String jobGroup) {
        return getJobDetail(JobKey.jobKey(jobName, jobGroup));
    }

    /**
     * 获取JobDetail.
     *
     * @param jobKey jobKey
     * @return JobDetail
     * @IncludeIntoJavadoc
     */
    @Nullable
    JobDetail getJobDetail(JobKey jobKey);

    /**
     * 获取正在执行的Job信息.
     *
     * <p>
     *     这里的正在执行是指本机ThreadPool里正在执行的Job信息.
     * </p>
     *
     * @return Job信息
     */
    @NotNull
    String getExecutingInfo();
}
