package com.fr.scheduler.tool;

import com.fr.log.FineLoggerFactory;
import com.fr.scheduler.cluster.ClusterMode;
import com.fr.scheduler.cluster.FineSchedulerClusterProvider;
import com.fr.stable.StringUtils;
import org.jetbrains.annotations.NotNull;

import java.io.BufferedReader;
import java.io.FileReader;
import java.util.Properties;

/**
 * fine-scheduler初始化Properties创建工具类.
 *
 * <p>
 * 用于创建初始化{@link FineScheduler#initScheduler(Properties)}用的Properties.
 * </p>
 *
 * <pre>
 * 创建一个单机模式Fine-Scheduler的Properties：
 * Properties properties = FineSchedulerPropertiesBuilder.newProperties()
 *         .driver("com.mysql.jdbc.Driver")
 *         .url("jdbc:mysql://localhost:3306/scheduler?useUnicode=true{@literal &}characterEncoding=UTF-8{@literal &}serverTimezone=Asia/Shanghai")
 *         .username("username")
 *         .password("password")
 *         .standalone("nodeID")
 *         .threadCount(5)
 *         .build();
 * </pre>
 *
 * @author Cloud.Liu
 * created on 2020-08-24
 * @IncludeIntoJavadoc
 */
public class FineSchedulerPropertiesBuilder {

    /**
     * Quartz JobStore类.
     */
    public static final String PROPERTIES_FIXED_JOB_STORE_CLASS = "com.fr.third.v2.org.quartz.jobStore.class";
    /**
     * Quartz 数据源.
     */
    public static final String PROPERTIES_FIXED_DATASOURCE_NAME = "com.fr.third.v2.org.quartz.jobStore.dataSource";
    /**
     * Quartz 是否开启集群模式.
     */
    public static final String PROPERTIES_FIXED_IS_CLUSTERED = "com.fr.third.v2.org.quartz.jobStore.isClustered";
    /**
     * Quartz 是否使用Properties形式的JobDataMap.
     */
    public static final String PROPERTIES_FIXED_USE_PROPERTIES = "com.fr.third.v2.org.quartz.jobStore.useProperties";
    /**
     * Quartz 线程池类.
     */
    public static final String PROPERTIES_FIXED_THREAD_POOL_CLASS = "com.fr.third.v2.org.quartz.threadPool.class";
    /**
     * Quartz 驱动代理类.
     */
    public static final String PROPERTIES_FIXED_DRIVER_DELEGATE_CLASS = "com.fr.third.v2.org.quartz.jobStore.driverDelegateClass";

    /**
     * Quartz 数据源驱动.
     */
    public static final String PROPERTIES_REQUIRED_DRIVER = "com.fr.third.v2.org.quartz.dataSource.quartzDS.driver";
    /**
     * Quartz 数据源URL.
     */
    public static final String PROPERTIES_REQUIRED_URL = "com.fr.third.v2.org.quartz.dataSource.quartzDS.URL";
    /**
     * Quartz 数据源用户名.
     */
    public static final String PROPERTIES_REQUIRED_USERNAME = "com.fr.third.v2.org.quartz.dataSource.quartzDS.user";
    /**
     * Quartz 数据源密码.
     */
    public static final String PROPERTIES_REQUIRED_PASSWORD = "com.fr.third.v2.org.quartz.dataSource.quartzDS.password";
    /**
     * Quartz 当前节点ID.
     */
    public static final String PROPERTIES_REQUIRED_NODE_ID = "com.fr.third.v2.org.quartz.jobStore.currentId";

    /**
     * Quartz 连接池提供类.
     */
    public static final String PROPERTIES_OPTIONAL_CONNECTION_PROVIDER = "com.fr.third.v2.org.quartz.dataSource.quartzDS.connectionProvider.class";
    /**
     * Quartz 表名前缀.
     */
    public static final String PROPERTIES_OPTIONAL_TABLE_PREFIX = "com.fr.third.v2.org.quartz.jobStore.tablePrefix";
    /**
     * Quartz 线程池线程数.
     */
    public static final String PROPERTIES_OPTIONAL_THREAD_COUNT = "com.fr.third.v2.org.quartz.threadPool.threadCount";
    /**
     * Quartz 数据源最大连接数.
     */
    public static final String PROPERTIES_OPTIONAL_MAX_CONNECTIONS = "com.fr.third.v2.org.quartz.dataSource.quartzDS.maxConnections";
    /**
     * Quartz 数据连接验证.
     */
    public static final String PROPERTIES_OPTIONAL_VALIDATION_QUERY = "com.fr.third.v2.org.quartz.dataSource.quartzDS.validationQuery";
    /**
     * Quartz 是否进行数据连接验证.
     */
    public static final String PROPERTIES_OPTIONAL_VALIDATE_ON_CHECKOUT = "com.fr.third.v2.org.quartz.dataSource.quartzDS.validateOnCheckout";
    /**
     * Quartz 最大等待时间.
     */
    public static final String PROPERTIES_OPTIONAL_MAX_WAIT = "com.fr.third.v2.org.quartz.dataSource.quartzDS.maxWait";
    /**
     * Quartz misfire界定时间.
     */
    public static final String PROPERTIES_OPTIONAL_MISFIRE_THRESHOLD = "com.fr.third.v2.org.quartz.jobStore.misfireThreshold";
    /**
     * Quartz Job类载入工具类.
     */
    public static final String PROPERTIES_OPTIONAL_CLASS_LOAD_HELPER = "com.fr.third.v2.org.quartz.scheduler.classLoadHelper.class";

    /**
     * FineQuartz 集群模式.
     *
     * @see ClusterMode
     */
    public static final String PROPERTIES_CLUSTER_MODE = "clusterMode";

    /**
     * FineQuartz 集群工具提供类.
     *
     * @see FineSchedulerClusterProvider
     */
    public static final String PROPERTIES_CLUSTER_PROVIDER = "clusterProvider";

    /**
     * Quartz 默认JobStore类.
     */
    public static final String JOB_STORE_CLASS = "com.fr.third.v2.org.quartz.impl.jdbcjobstore.JobStoreTX";
    /**
     * Quartz 默认数据源名称.
     */
    public static final String DATASOURCE_NAME = "quartzDS";
    /**
     * Quartz 默认不开启集群模式.
     */
    public static final String IS_CLUSTERED = String.valueOf(false);
    /**
     * Quartz 默认不使用仅Properties的JobDataMap.
     */
    public static final String USE_PROPERTIES = String.valueOf(false);
    /**
     * Quartz 默认线程池类.
     */
    public static final String THREAD_POOL_CLASS = "com.fr.third.v2.org.quartz.simpl.SimpleThreadPool";
    /**
     * Quartz 默认驱动代理类.
     */
    public static final String DRIVER_DELEGATE_CLASS = "com.fr.third.v2.org.quartz.impl.jdbcjobstore.StdJDBCDelegate";

    /**
     * 数据源配置项前缀.
     */
    private static final String DATASOURCE_PREFIX = "com.fr.third.v2.org.quartz.dataSource.quartzDS.";
    public static final String PROP_SCHED_INSTANCE_NAME = "com.fr.third.v2.org.quartz.scheduler.instanceName";

    /**
     * 新建一个初始化{@link FineScheduler}用的{@code Properties}.
     *
     * @return Builder
     */
    public static DriverStep newProperties() {
        return new Steps();
    }

    /**
     * 新建一个初始化{@link FineScheduler}用的{@code Properties}.
     *
     * <p>
     * 直接返回一个builder，使用build方法根据传入的properties文件路径来构建properties实例对象
     * </p>
     *
     * @return Builder
     */
    public static FileStep createPropertiesByFile(){
        return new Steps();
    }

    /**
     * 指定数据库驱动.
     */
    public interface DriverStep {
        URLStep driver(@NotNull String driver);
    }

    /**
     * 指定数据库URL.
     */
    public interface URLStep {
        UserNameStep url(@NotNull String url);
    }

    /**
     * 指定数据库用户名.
     */
    public interface UserNameStep {
        PasswordStep username(@NotNull String username);
    }

    /**
     * 指定数据库密码.
     */
    public interface PasswordStep {
        ClusterStep password(@NotNull String password);
    }

    /**
     * 指定单机/集群.
     */
    public interface ClusterStep {
        BuildStep standalone(String nodeId);

        BuildStep cluster(FineSchedulerClusterProvider provider, String nodeId);
    }

    /**
     * 创建Fine-Scheduler初始化Properties.
     */
    public interface BuildStep {

        BuildStep connectionProvider(@NotNull String providerClass);

        BuildStep connectionProviderProperties(@NotNull Properties properties);

        BuildStep schema(@NotNull String schema);

        BuildStep threadCount(int count);

        BuildStep misfireThreshold(long threshold);

        BuildStep classLoadHelper(@NotNull String helperClass);

        BuildStep driverDelegate(@NotNull String driverDelegate);

        BuildStep schedulerNameProvider(@NotNull String providerName);

        @Deprecated
        BuildStep maxConnections(int count);

        @Deprecated
        BuildStep validationQuery(@NotNull String query);

        @Deprecated
        BuildStep validateOnCheckout(boolean validate);

        @Deprecated
        BuildStep maxWait(int waitTime);

        Properties build();
    }

    public interface FileStep {
        Properties build(@NotNull String filePath) throws Exception;
    }

    private static class Steps implements DriverStep, URLStep, UserNameStep, PasswordStep, ClusterStep, BuildStep, FileStep {

        private String driver;
        private String url;
        private String username;
        private String password;
        private String connectionProvider = "com.fr.scheduler.quartz.cp.HikariConnectionProvider";
        private Properties connectionProviderProperties = new Properties();
        private String tablePrefix = "QRTZ_";
        private String nodeId = StringUtils.EMPTY;
        private String threadCount = String.valueOf(100);
        private String maxConnection = String.valueOf(50);
        private String validationQuery = StringUtils.EMPTY;
        private String validateOnCheckout = String.valueOf(true);
        private String misfireThreshold = String.valueOf(60000);
        private String helperClass = StringUtils.EMPTY;
        private String driverDelegate = StringUtils.EMPTY;
        private String maxWait = String.valueOf(500000);
        private String schedulerNameProvider = "com.fr.third.v2.org.quartz.impl.SchedulerNameDefaultProvider";

        private ClusterMode clusterMode = ClusterMode.STANDALONE;
        private FineSchedulerClusterProvider provider = null;

        @Override
        public URLStep driver(@NotNull String driver) {
            this.driver = driver;
            return this;
        }

        @Override
        public UserNameStep url(@NotNull String url) {
            this.url = url;
            return this;
        }

        @Override
        public PasswordStep username(@NotNull String username) {
            this.username = username;
            return this;
        }

        @Override
        public ClusterStep password(@NotNull String password) {
            this.password = password;
            return this;
        }

        @Override
        public BuildStep standalone(String nodeId) {
            FineLoggerFactory.getLogger().info("This quartz is in standalone mode.");
            this.clusterMode = ClusterMode.STANDALONE;
            this.nodeId = nodeId;
            this.provider = FineSchedulerClusterProvider.DEFAULT;
            return this;
        }

        @Override
        public BuildStep cluster(FineSchedulerClusterProvider provider, String nodeId) {
            FineLoggerFactory.getLogger().info("This quartz is in cluster mode.");
            this.clusterMode = ClusterMode.CLUSTER;
            this.nodeId = nodeId;
            this.provider = provider;
            return this;
        }

        @Override
        public BuildStep connectionProvider(@NotNull String providerClass) {
            this.connectionProvider = providerClass;
            return this;
        }

        @Override
        public BuildStep connectionProviderProperties(@NotNull Properties properties) {
            this.connectionProviderProperties = properties;
            return this;
        }

        @Override
        public BuildStep schema(@NotNull String schema) {
            if (StringUtils.isNotEmpty(schema)) {
                tablePrefix = schema + "." + tablePrefix;
            }
            return this;
        }

        @Override
        public BuildStep threadCount(int count) {
            if (count <= 0) {
                throw new IllegalArgumentException("Thread count must be larger than 0.");
            } else {
                this.threadCount = String.valueOf(count);
            }
            return this;
        }

        @Override
        public BuildStep maxConnections(int count) {
            if (count <= 0) {
                throw new IllegalArgumentException("Max connection count must be larger than 0.");
            } else {
                this.maxConnection = String.valueOf(count);
            }
            return this;
        }

        @Override
        public BuildStep validationQuery(@NotNull String query) {
            if (StringUtils.isNotEmpty(query)) {
                validationQuery = query;
            }
            return this;
        }

        @Override
        public BuildStep validateOnCheckout(boolean validate) {
            validateOnCheckout = String.valueOf(validate);
            return this;
        }

        @Override
        public BuildStep maxWait(int waitTime) {
            maxWait = String.valueOf(waitTime);
            return this;
        }

        @Override
        public BuildStep misfireThreshold(long threshold) {
            if (threshold <= 0L) {
                throw new IllegalArgumentException("Miss fire threshold count must be larger than 0.");
            } else {
                this.misfireThreshold = String.valueOf(threshold);
            }
            return this;
        }

        @Override
        public BuildStep classLoadHelper(@NotNull String helperClass) {
            this.helperClass = helperClass;
            return this;
        }

        @Override
        public BuildStep driverDelegate(@NotNull String driverDelegate) {
            if (!StringUtils.isEmpty(driverDelegate)) {
                this.driverDelegate = driverDelegate;
            }
            return this;
        }

        @Override
        public BuildStep schedulerNameProvider(@NotNull String providerName) {
            if (!StringUtils.isEmpty(providerName)) {
                schedulerNameProvider = providerName;
            }
            return this;
        }

        @Override
        public Properties build() {

            Properties properties = new Properties();
            addFixedProperties(properties);
            addRequiredProperties(properties);
            addOptionalProperties(properties);
            addClusterProperties(properties);
            return properties;
        }

        @Override
        public Properties build(String filePath) throws Exception {

            Properties properties = new Properties();
            BufferedReader reader = new BufferedReader(new FileReader(filePath));
            properties.load(reader);
            Properties resProperties = new Properties();
            addFixedProperties(resProperties);
            addOptionalProperties(resProperties);
            if (properties.getProperty(PROPERTIES_FIXED_IS_CLUSTERED).equals("false")) {
                properties.put(PROPERTIES_CLUSTER_MODE, ClusterMode.STANDALONE);
                properties.put(PROPERTIES_CLUSTER_PROVIDER, FineSchedulerClusterProvider.DEFAULT);
            } else {
                properties.put(PROPERTIES_CLUSTER_MODE, ClusterMode.CLUSTER);
                String className = properties.getProperty(PROPERTIES_CLUSTER_PROVIDER);
                Class clazz = Class.forName(className);
                properties.put(PROPERTIES_CLUSTER_PROVIDER, clazz.newInstance());
            }
            resProperties.putAll(properties);
            return resProperties;
        }

        private void addFixedProperties(Properties properties) {

            properties.put(PROPERTIES_FIXED_JOB_STORE_CLASS, JOB_STORE_CLASS);
            properties.put(PROPERTIES_FIXED_DATASOURCE_NAME, DATASOURCE_NAME);
            properties.put(PROPERTIES_FIXED_IS_CLUSTERED, IS_CLUSTERED);
            properties.put(PROPERTIES_FIXED_USE_PROPERTIES, USE_PROPERTIES);
            properties.put(PROPERTIES_FIXED_THREAD_POOL_CLASS, THREAD_POOL_CLASS);
            properties.put(PROPERTIES_FIXED_DRIVER_DELEGATE_CLASS, DRIVER_DELEGATE_CLASS);
        }

        private void addRequiredProperties(Properties properties) {

            properties.put(PROPERTIES_REQUIRED_DRIVER, driver);
            properties.put(PROPERTIES_REQUIRED_URL, url);
            properties.put(PROPERTIES_REQUIRED_USERNAME, username);
            properties.put(PROPERTIES_REQUIRED_PASSWORD, password);
            properties.put(PROPERTIES_REQUIRED_NODE_ID, nodeId);
        }

        private void addOptionalProperties(Properties properties) {

            if (StringUtils.isNotEmpty(connectionProvider)) {
                properties.put(PROPERTIES_OPTIONAL_CONNECTION_PROVIDER, connectionProvider);
            }
            if (StringUtils.isNotEmpty(schedulerNameProvider)) {
                properties.put(PROP_SCHED_INSTANCE_NAME, schedulerNameProvider);
            }
            properties.put(PROPERTIES_OPTIONAL_TABLE_PREFIX, tablePrefix);
            properties.put(PROPERTIES_OPTIONAL_THREAD_COUNT, threadCount);
            if (connectionProviderProperties.isEmpty()) {
                properties.put(PROPERTIES_OPTIONAL_MAX_CONNECTIONS, maxConnection);
                properties.put(PROPERTIES_OPTIONAL_VALIDATION_QUERY, validationQuery);
                properties.put(PROPERTIES_OPTIONAL_VALIDATE_ON_CHECKOUT, validateOnCheckout);
                properties.put(PROPERTIES_OPTIONAL_MAX_WAIT, maxWait);
            } else {
                connectionProviderProperties.forEach((k, v) -> properties.put(DATASOURCE_PREFIX + k, v));
            }
            properties.put(PROPERTIES_OPTIONAL_MISFIRE_THRESHOLD, misfireThreshold);
            if (StringUtils.isNotEmpty(helperClass)) {
                properties.put(PROPERTIES_OPTIONAL_CLASS_LOAD_HELPER, helperClass);
            }
            // 若传入了driverDelegate，覆写默认值
            if (StringUtils.isNotEmpty(driverDelegate)) {
                properties.put(PROPERTIES_FIXED_DRIVER_DELEGATE_CLASS, driverDelegate);
            }
        }

        private void addClusterProperties(Properties properties) {

            properties.put(PROPERTIES_CLUSTER_MODE, clusterMode);
            properties.put(PROPERTIES_CLUSTER_PROVIDER, provider);
        }
    }
}
