package com.fr.scheduler.tool;

import com.fr.log.FineLoggerFactory;
import com.fr.scheduler.cluster.ClusterMode;
import com.fr.scheduler.cluster.FineQuartzClusterUtils;
import com.fr.scheduler.cluster.FineSchedulerClusterProvider;
import com.fr.scheduler.cluster.FineSchedulerClusterConstants;
import com.fr.scheduler.job.FineDispatchJob;
import com.fr.scheduler.job.FineJob;
import com.fr.scheduler.job.JobState;
import com.fr.stable.ArrayUtils;
import com.fr.stable.StringUtils;
import com.fr.third.org.apache.commons.lang3.tuple.Pair;
import com.fr.third.v2.org.quartz.JobDetail;
import com.fr.third.v2.org.quartz.JobKey;
import com.fr.third.v2.org.quartz.Scheduler;
import com.fr.third.v2.org.quartz.SchedulerException;
import com.fr.third.v2.org.quartz.Trigger;
import com.fr.third.v2.org.quartz.TriggerKey;
import com.fr.third.v2.org.quartz.impl.StdSchedulerFactory;
import com.fr.third.v2.org.quartz.impl.matchers.GroupMatcher;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.stream.Collectors;

/**
 * fine-scheduler初始化与定时任务管理工具类.
 *
 * <p>
 * 使用这个类管理任务.<br>
 * 使用{@link FineSchedulerPropertiesBuilder}构建用于初始化{@link FineScheduler#initScheduler(Properties)}的Properties.
 * </p>
 *
 * @author Cloud.Liu
 * created on 2020-09-09
 * @IncludeIntoJavadoc
 */
public class FineScheduler implements FineSchedulerProvider {

    private static final FineScheduler INSTANCE = new FineScheduler();
    private Scheduler scheduler = null;

    private final BlockingQueue<JobKey> blockingRemoveQueue = new LinkedBlockingQueue<>(128);
    private final BlockingQueue<Pair<JobDetail, Trigger>> blockingAddJobTriggerQueue = new LinkedBlockingQueue<>(128);
    private final BlockingQueue<Trigger> blockingAddTriggerQueue = new LinkedBlockingQueue<>(128);


    /**
     * 获取定时任务管理工具单例.
     * @return 定时任务管理工具
     */
    public static FineScheduler getInstance() {
        return INSTANCE;
    }

    private FineScheduler() {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void initScheduler(Properties properties) {

        initCluster(properties);

        try {
            StdSchedulerFactory factory = new StdSchedulerFactory(properties);
            scheduler = factory.getScheduler();
            scheduler.start();
            FineLoggerFactory.getLogger().info("Scheduler has been started.");
            executeRemoveQueueTask();
            executeAddJobTriggerQueueTask();
            executeAddTriggerQueueTask();
        } catch (Exception e) {
            FineLoggerFactory.getLogger().error("Scheduler start error", e);
        }
    }

    private void executeRemoveQueueTask() {
        int length = blockingRemoveQueue.size();
        for (int i = 0; i < length; i++) {
            JobKey jobKey = blockingRemoveQueue.poll();
            revokeJob(jobKey);
            FineLoggerFactory.getLogger().info("Remove job {}, {} from the queue from the database!", jobKey.getName(), jobKey.getGroup());
        }
    }

    private void executeAddJobTriggerQueueTask() {
        int length = blockingAddJobTriggerQueue.size();
        for (int i = 0; i < length; i++) {
            Pair<JobDetail, Trigger> pair = blockingAddJobTriggerQueue.poll();
            JobDetail jobDetail = pair.getLeft();
            submitJob(jobDetail, pair.getRight());
            FineLoggerFactory.getLogger().info("Add job {}, {} in the queue to the database!", jobDetail.getKey().getName(), jobDetail.getKey().getGroup());
        }
    }

    private void executeAddTriggerQueueTask() {
        int length = blockingAddTriggerQueue.size();
        for (int i = 0; i < length; i++) {
            Trigger trigger = blockingAddTriggerQueue.poll();
            submitTrigger(trigger);
            FineLoggerFactory.getLogger().info("Add trigger {}, {} in the queue to the database!", trigger.getKey().getName(), trigger.getKey().getGroup());
        }
    }

    private boolean waitForRemoveExecute(JobKey jobKey) {
        if (!isStarted()) {
            blockingRemoveQueue.add(jobKey);
            FineLoggerFactory.getLogger().info("Add job {}, {} to the removing queue!", jobKey.getName(), jobKey.getGroup());
            return true;
        }
        return false;
    }

    private boolean waitForAddJobTriggerExecute(Pair<JobDetail, Trigger> detailTriggerPair) {
        if (!isStarted()) {
            blockingAddJobTriggerQueue.add(detailTriggerPair);
            JobDetail jobDetail = detailTriggerPair.getLeft();
            FineLoggerFactory.getLogger().info("Add job {}, {} to the adding queue!", jobDetail.getKey().getName(), jobDetail.getKey().getGroup());
            return true;
        }
        return false;
    }

    private boolean waitForAddTriggerExecute(Trigger trigger) {
        if (!isStarted()) {
            blockingAddTriggerQueue.add(trigger);
            FineLoggerFactory.getLogger().info("Add trigger {}, {} to the adding queue!", trigger.getKey().getName(), trigger.getKey().getGroup());
            return true;
        }
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isStarted() {

        if (scheduler == null) {
            return false;
        }

        try {
            return scheduler.isStarted();
        } catch (Exception ignored) {
            // StdScheduler不会抛出Exception
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    public void shutdownScheduler() {

        if (scheduler == null) {
            return;
        }

        shutdownCluster();

        try {
            scheduler.shutdown();
            scheduler = null;
            FineLoggerFactory.getLogger().info("Scheduler has been shutdown.");
        } catch (SchedulerException e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isShutdown() {

        if (scheduler == null) {
            return true;
        }

        try {
            return scheduler.isShutdown();
        } catch (Exception ignored) {
            // StdScheduler不会抛出Exception
            return false;
        }
    }

    /**
     * 获取QuartzScheduler.
     *
     * @return QuartzScheduler
     * @deprecated 不推荐直接使用QuartzScheduler.
     */
    @Deprecated
    public Scheduler getScheduler() {
        return scheduler;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nullable
    public Date scheduleTrigger(Trigger trigger) {
        try {
            return scheduler.scheduleJob(trigger);
        } catch (SchedulerException e) {
            FineLoggerFactory.getLogger().error("Schedule trigger failed.", e);
            return null;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nullable
    public Date submitTrigger(Trigger trigger) {
        if (waitForAddTriggerExecute(trigger)) {
            return null;
        }
        return scheduleTrigger(trigger);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nullable
    public Date addJob(@NotNull JobDetail jobDetail, @NotNull Trigger trigger) {
        try {
            return scheduler.scheduleJob(jobDetail, trigger);
        } catch (SchedulerException e) {
            FineLoggerFactory.getLogger().error("Add job failed.", e);
            return null;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nullable
    public Date submitJob(@NotNull JobDetail jobDetail, @NotNull Trigger trigger) {
        if (waitForAddJobTriggerExecute(Pair.of(jobDetail, trigger))) {
            return null;
        }
        return addJob(jobDetail, trigger);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nullable
    public Date addJobRunOnSpecificNode(@NotNull JobDetail jobDetail, @NotNull Trigger trigger, @NotNull String nodeId) {

        if (!FineQuartzClusterUtils.getInstance().isCluster()) {
            FineLoggerFactory.getLogger().error("Unable to add job run on specific node in standalone mode.");
            return null;
        }

        trigger.setAppointId(nodeId);
        Date date = addJob(jobDetail, trigger);
        notifyOtherScheduler(date, nodeId);
        return date;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nullable
    public Date addJobRunOnRandomNode(@NotNull JobDetail jobDetail, @NotNull Trigger trigger) {
        return addJob(jobDetail, trigger);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Nullable
    public Date addJobRunOnRandomNodeIn(@NotNull JobDetail jobDetail, @NotNull Trigger trigger, String... nodes) {

        if (!FineQuartzClusterUtils.getInstance().isCluster()) {
            FineLoggerFactory.getLogger().error("Unable to add job run on random node in some nodes in standalone mode.");
            return null;
        }

        if (ArrayUtils.isEmpty(nodes)) {
            FineLoggerFactory.getLogger().error("Nodes should not be empty in [addJobRunOnRandomNodeIn].");
            return null;
        }
        trigger.setAppointId(StringUtils.join(FineSchedulerClusterConstants.SEPARATOR_NODE_ID, nodes));
        Date date = addJob(jobDetail, trigger);
        notifyOtherScheduler(date, nodes);
        return date;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    @Override
    @Nullable
    public Date addJobRunOnEachNodeIn(@NotNull JobDetail jobDetail, @NotNull Trigger trigger, @NotNull String... nodes) {

        if (!FineQuartzClusterUtils.getInstance().isCluster()) {
            FineLoggerFactory.getLogger().error("Unable to add job run on each node in some nodes in standalone mode.");
            return null;
        }

        Class<? extends FineJob> originalJobClass = (Class<? extends FineJob>) jobDetail.getJobClass();
        jobDetail.getJobDataMap().put(FineSchedulerClusterConstants.NODE_JOB_CLASS, originalJobClass);
        jobDetail.getJobDataMap().put(FineSchedulerClusterConstants.NODE_TO_DISPATCH, String.join(FineSchedulerClusterConstants.SEPARATOR_NODE_ID, nodes));

        jobDetail = jobDetail.getJobBuilder().ofType(FineDispatchJob.class).build();
        return addJobRunOnRandomNode(jobDetail, trigger);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nullable Date addJobRunOnEveryNode(@NotNull JobDetail jobDetail, @NotNull Trigger trigger) {

        if (!FineQuartzClusterUtils.getInstance().isCluster()) {
            FineLoggerFactory.getLogger().error("Unable to add job run on every node in standalone mode.");
            return null;
        }

        Class<? extends FineJob> originalJobClass = (Class<? extends FineJob>) jobDetail.getJobClass();
        jobDetail.getJobDataMap().put(FineSchedulerClusterConstants.NODE_JOB_CLASS, originalJobClass);
        jobDetail.getJobDataMap().put(FineSchedulerClusterConstants.NODE_TO_DISPATCH, FineSchedulerClusterConstants.DISPATCH_TO_ALL_NODES);

        jobDetail = jobDetail.getJobBuilder().ofType(FineDispatchJob.class).build();
        return addJobRunOnRandomNode(jobDetail, trigger);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyScheduler(long time) {
        scheduler.notify(time);
        FineLoggerFactory.getLogger().error("This node {} notified {}.", FineQuartzClusterUtils.getInstance().getCurrentNodeId(), time);
    }

    /**
     * 展示所有的任务.
     *
     * <p>
     * 打印日志.
     * </p>
     */
    public void displayAllJobs() {

        try {
            // 所有JobKeys
            Set<JobKey> jobKeys = scheduler.getJobKeys(GroupMatcher.anyGroup());

            int jobIndex = 0;
            for (JobKey jobKey : jobKeys) {
                // 对于每一个Job
                JobDetail jobDetail = scheduler.getJobDetail(jobKey);
                List<? extends Trigger> triggersOfJob = scheduler.getTriggersOfJob(jobKey);
                FineLoggerFactory.getLogger().error("Job " + jobIndex);
                FineLoggerFactory.getLogger().error("\tName {} Group {}", jobKey.getName(), jobKey.getGroup());
                int triggerIndex = 0;
                for (Trigger trigger : triggersOfJob) {
                    FineLoggerFactory.getLogger().error("\t\tTrigger " + triggerIndex);
                    String appointId = trigger.getAppointId();
                    if (!StringUtils.isEmpty(appointId)) {
                        // 只有特定节点能取到这个trigger
                        String[] nodes = appointId.split(FineSchedulerClusterConstants.SEPARATOR_NODE_ID);
                        FineLoggerFactory.getLogger().error("\t\t\twill run on random node in {}.", Arrays.toString(nodes));
                    } else {
                        // 所有节点都能取到这个trigger
                        if (jobDetail.getJobClass().equals(FineDispatchJob.class)) {
                            // 是分发任务，可能分发到特定节点或所有节点
                            String nodeToDispatch = jobDetail.getJobDataMap().getString(FineSchedulerClusterConstants.NODE_TO_DISPATCH);
                            Class<? extends FineJob> realJobClass = (Class<? extends FineJob>) jobDetail.getJobDataMap().get(FineSchedulerClusterConstants.NODE_JOB_CLASS);
                            if (StringUtils.equals(nodeToDispatch, FineSchedulerClusterConstants.DISPATCH_TO_ALL_NODES)) {
                                // 分发到所有节点
                                FineLoggerFactory.getLogger().error("\t\t\twill run on all nodes.");
                            } else {
                                // 分发到特定节点
                                String[] nodes = nodeToDispatch.split(FineSchedulerClusterConstants.SEPARATOR_NODE_ID);
                                FineLoggerFactory.getLogger().error("\t\t\twill run on all nodes in {}.", Arrays.toString(nodes));
                            }
                        } else {
                            // 不是分发任务，在任意一个节点上执行
                            FineLoggerFactory.getLogger().error("\t\t\twill run on random node.");
                        }
                    }
                }
                jobIndex++;
            }
        } catch (SchedulerException exception) {
            FineLoggerFactory.getLogger().error(exception.getMessage(), exception);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void changeAppointId(String oldId, String newId) throws SchedulerException {

        FineLoggerFactory.getLogger().info("Change appointId from {} to {}.", oldId, newId);

        FineLoggerFactory.getLogger().info("Change appointId for normal job trigger.");
        Set<TriggerKey> triggerKeys = scheduler.getTriggerKeys(GroupMatcher.triggerGroupNotStartsWith(FineSchedulerClusterConstants.NODE_TRIGGER_PREFIX));
        FineLoggerFactory.getLogger().info("Normal job trigger count {}." + triggerKeys.size());
        for (TriggerKey key : triggerKeys) {
            Trigger trigger = scheduler.getTrigger(key);
            if (trigger != null) {
                String appointId = trigger.getAppointId();
                if (appointIdContains(appointId, oldId)) {
                    String newAppointId = changeNodeIdInAppointId(appointId, oldId, newId);
                    scheduler.changeAppointId(appointId, newAppointId, key.getName(), key.getGroup());
                    FineLoggerFactory.getLogger().info("AppointID changes from {} to {} for trigger {}.", oldId, newId, trigger.getKey());
                }
            }
        }

        FineLoggerFactory.getLogger().info("Change appointId for dispatch job.");
        Set<JobKey> jobKeys = scheduler.getJobKeys(GroupMatcher.jobGroupNotStartsWith(FineSchedulerClusterConstants.NODE_JOB_PREFIX));
        FineLoggerFactory.getLogger().info("Dispatch job count {}." + jobKeys.size());
        for (JobKey key : jobKeys) {
            JobDetail job = scheduler.getJobDetail(key);
            if (job != null) {
                if (job.getJobClass().equals(FineDispatchJob.class)) {
                    // 分发Job
                    String dispatchTo = job.getJobDataMap().getString(FineSchedulerClusterConstants.NODE_TO_DISPATCH);
                    if (appointIdContains(dispatchTo, oldId)) {
                        // 需要取出所有这个job的trigger，更新Job，然后再把trigger放回去
                        List<? extends Trigger> triggersOfJob = scheduler.getTriggersOfJob(key);
                        scheduler.unscheduleJobs(triggersOfJob.stream().map(Trigger::getKey).collect(Collectors.toList()));
                        FineLoggerFactory.getLogger().info("Dispatch job {} unscheduled.", key.toString());
                        String newDispatchTo = changeNodeIdInAppointId(dispatchTo, oldId, newId);
                        job.getJobDataMap().put(FineSchedulerClusterConstants.NODE_TO_DISPATCH, newDispatchTo);
                        scheduler.deleteJob(key);
                        FineLoggerFactory.getLogger().info("Dispatch job {} deleted.", key.toString());
                        scheduler.scheduleJob(job, new HashSet<>(triggersOfJob), true);
                        FineLoggerFactory.getLogger().info("Dispatch job {} re-added after changing NODE_TO_DISPATCH from {} to {}.", key, dispatchTo, newDispatchTo);
                    }
                }
            }
        }
        FineLoggerFactory.getLogger().info("AppointID changed from {} to {}.", oldId, newId);
    }

    /**
     * {@inheritDoc}
     */
    public void runOnce(JobKey jobKey) {

        Trigger trigger = FineTriggerBuilder.newTrigger()
                .withName("onceTrigger" + jobKey.getName())
                .withGroup("onceTrigger" + jobKey.getGroup())
                .startNow()
                .onlyOnce()
                .withSimpleDefaultMisfireInst()
                .forJob(jobKey)
                .build();

        try {
            scheduler.scheduleJob(trigger);
        } catch (SchedulerException e) {
            FineLoggerFactory.getLogger().error("Run job once failed: " + e.getMessage(), e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Set<JobKey> getJobKeysByJobGroup(GroupMatcher<JobKey> matcher) throws SchedulerException {
        return scheduler.getJobKeys(matcher);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Set<TriggerKey> getTriggerKeysByTriggerGroup(GroupMatcher<TriggerKey> matcher) throws SchedulerException {
        return scheduler.getTriggerKeys(matcher);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeJob(JobKey jobKey) {

        // 如实地删除传入的Job
        try {
            List<? extends Trigger> triggers = scheduler.getTriggersOfJob(jobKey);
            for (Trigger trigger : triggers) {
                scheduler.pauseTrigger(trigger.getKey());
                scheduler.unscheduleJob(trigger.getKey());
            }
            scheduler.deleteJob(jobKey);
            FineLoggerFactory.getLogger().info("Job {} removed.", jobKey.toString());
        } catch (Exception e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void revokeJob(JobKey jobKey) {
        if (waitForRemoveExecute(jobKey) || !jobExist(jobKey)) {
            return;
        }
        removeJob(jobKey);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void pauseJob(JobKey jobKey) {

        // 如实地暂停传入的Job
        try {
            scheduler.pauseJob(jobKey);
            FineLoggerFactory.getLogger().info("Job {} paused.", jobKey.toString());
        } catch (Exception e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void resumeJob(JobKey jobKey) {

        try {
            scheduler.resumeJob(jobKey);
            FineLoggerFactory.getLogger().info("Job {} resumed.", jobKey.toString());
        } catch (Exception e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void resumeJobIgnoreMisfire(JobKey jobKey) {
        try {
            this.scheduler.resumeJobIgnoreMisfire(jobKey);
            FineLoggerFactory.getLogger().info("Job {} resumed and ignore misfire.", jobKey.toString());
        } catch (Exception e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Date getNextFireTimeOfJob(JobKey jobKey) {

        List<? extends Trigger> triggersOfJob = getTriggersOfJob(jobKey);
        if (!triggersOfJob.isEmpty()) {
            return triggersOfJob.stream().map(Trigger::getNextFireTime).filter(Objects::nonNull).min(Date::compareTo).orElse(null);
        } else {
            return null;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Date getPreFireTimeOfJob(JobKey jobKey) {

        List<? extends Trigger> triggersOfJob = getTriggersOfJob(jobKey);
        if (!triggersOfJob.isEmpty()) {
            return triggersOfJob.stream().map(Trigger::getPreviousFireTime).filter(Objects::nonNull).max(Date::compareTo).orElse(null);
        } else {
            return null;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean jobExist(JobKey jobKey) {

        try {
            return scheduler.checkExists(jobKey);
        } catch (SchedulerException e) {
            FineLoggerFactory.getLogger().error(e, e.getMessage());
        }
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Trigger> getTriggersOfJob(JobKey jobKey) {

        List<Trigger> triggers = new ArrayList<>();
        try {
            triggers = new ArrayList<>(scheduler.getTriggersOfJob(jobKey));
        } catch (SchedulerException e) {
            FineLoggerFactory.getLogger().error(e.getMessage(), e);
        }
        return triggers;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public JobState getStateOfJob(JobKey jobKey) throws SchedulerException {

        // 正在执行：有子任务
        Set<JobKey> nodeJobKeys = scheduler.getJobKeys(GroupMatcher.jobGroupStartsWith(FineSchedulerClusterConstants.NODE_JOB_PREFIX + jobKey.getGroup() + FineSchedulerClusterConstants.SEPARATOR_NODE_ID));
        for (JobKey nodeJobKey : nodeJobKeys) {
            // 多个JobGroup可能相同
            if (nodeJobKey.getName().contains(FineSchedulerClusterConstants.NODE_JOB_PREFIX + jobKey.getName() + FineSchedulerClusterConstants.SEPARATOR_NODE_ID)) {
                return JobState.EXECUTING;
            }
        }

        // 已结束，Job已经被移除了
        if (!jobExist(jobKey)) {
            return JobState.COMPLETED;
        }

        List<? extends Trigger> triggersOfJob = getTriggersOfJob(jobKey);

        // 暂停：Job的多个Trigger是同步启停的，有Trigger暂停则认为该Job暂停
        for (Trigger trigger : triggersOfJob) {
            Trigger.TriggerState triggerState = scheduler.getTriggerState(trigger.getKey());
            if (triggerState == Trigger.TriggerState.PAUSED) {
                return JobState.PAUSED;
            }
        }

        // 正在执行：qrtz_fired_triggers表中有trigger记录
        for (Trigger trigger : triggersOfJob) {
            if (scheduler.isTriggerExecuting(trigger.getKey().getName(), trigger.getKey().getGroup())) {
                return JobState.EXECUTING;
            }
        }

        // 已结束，Job已经没有触发器了
        if (triggersOfJob.isEmpty()) {
            return JobState.COMPLETED;
        }

        // 已结束，触发器处于COMPLETE状态，马上要被移除
        if (triggersOfJob.size() == 1) {
            if (scheduler.getTriggerState(triggersOfJob.get(0).getKey()) == Trigger.TriggerState.COMPLETE) {
                return JobState.COMPLETED;
            }
        }

        // 运行中
        return JobState.RUNNING;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public JobDetail getJobDetail(JobKey jobKey) {
        try {
            return scheduler.getJobDetail(jobKey);
        } catch (SchedulerException exception) {
            return null;
        }
    }

    @Override
    public @NotNull String getExecutingInfo() {
        return scheduler.getExecutingJobInfo();
    }

    /* *************************** Private Methods *************************** */

    /**
     * 初始化集群.
     *
     * @param properties 配置项
     */
    private void initCluster(Properties properties) {

        // TODO 容错
        FineQuartzClusterUtils.getInstance().setMode((ClusterMode) properties.get(FineSchedulerPropertiesBuilder.PROPERTIES_CLUSTER_MODE));
        FineQuartzClusterUtils.getInstance().setClusterProvider((FineSchedulerClusterProvider) properties.get(FineSchedulerPropertiesBuilder.PROPERTIES_CLUSTER_PROVIDER));
        FineQuartzClusterUtils.getInstance().setCurrentNodeId(properties.getProperty(FineSchedulerPropertiesBuilder.PROPERTIES_REQUIRED_NODE_ID));

        // 初始化notifier
        FineQuartzClusterUtils.getInstance().getClusterProvider().register(properties.getProperty(FineSchedulerPropertiesBuilder.PROPERTIES_REQUIRED_NODE_ID));
    }

    /**
     * 关闭集群.
     */
    private void shutdownCluster() {
        FineQuartzClusterUtils.getInstance().getClusterProvider().unregister(FineQuartzClusterUtils.getInstance().getCurrentNodeId());
        FineQuartzClusterUtils.getInstance().reset();
    }

    /**
     * 通知其他Scheduler.
     *
     * <p>
     * 会过滤掉本节点.
     * </p>
     *
     * @param time 时间
     * @param nodeIds 节点ID
     */
    private void notifyOtherScheduler(@Nullable Date time, String ... nodeIds) {

        if (time == null) {
            return;
        }

        for (String nodeId : nodeIds) {
            if (!nodeId.equals(FineQuartzClusterUtils.getInstance().getCurrentNodeId())) {
                long fireLater = time.getTime() - new Date().getTime();
                FineLoggerFactory.getLogger().info("Notify node " + nodeId + " that a job will trigger in " + fireLater + " millis.");
                try {
                    FineQuartzClusterUtils.getInstance().getClusterProvider().notify(nodeId, time.getTime());
                } catch (Exception e) {
                    FineLoggerFactory.getLogger().error("Notify failed.", e);
                }
            }
        }
    }

    /**
     * 判断Job是不是被分发的Job.
     *
     * @param jobKey jobKey
     * @return 是否是被分发的Job
     */
    private boolean isNodeJob(JobKey jobKey) {
        return jobKey.getName().startsWith(FineSchedulerClusterConstants.NODE_JOB_PREFIX);
    }

    /**
     * AppointId是否包含当前nodeId.
     *
     * <p>
     * 即这个任务是否会被nodeId对应节点取到.
     * </p>
     *
     * @param appointId appointId
     * @param nodeId nodeId
     * @return 是否包含
     */
    private boolean appointIdContains(String appointId, String nodeId) {
        return appointId != null
                && (appointId.equals(nodeId)
                || appointId.startsWith(nodeId + FineSchedulerClusterConstants.SEPARATOR_NODE_ID)
                || appointId.endsWith(FineSchedulerClusterConstants.SEPARATOR_NODE_ID + nodeId)
                || appointId.contains(FineSchedulerClusterConstants.SEPARATOR_NODE_ID + nodeId + FineSchedulerClusterConstants.SEPARATOR_NODE_ID));
    }

    /**
     * 替换AppointId中的特定nodeId.
     *
     * @param appointId appointId
     * @param oldId oldId
     * @param newId newId
     * @return appointId
     */
    private String changeNodeIdInAppointId(String appointId, String oldId, String newId) throws SchedulerException {

        String splitter = FineSchedulerClusterConstants.SEPARATOR_NODE_ID;

        if (StringUtils.equals(appointId, oldId)) {
            return newId;
        } else if (appointId.startsWith(oldId + splitter)) {
            return appointId.replace(oldId + splitter, newId + splitter);
        } else if (appointId.endsWith(splitter + oldId)) {
            return appointId.replace(splitter + oldId, splitter + newId);
        } else if (appointId.contains(splitter + oldId + splitter)) {
            return appointId.replace(splitter + oldId + splitter, splitter + newId + splitter);
        } else {
            throw new SchedulerException("Replace nodeId " + oldId + " in appointId " + appointId + " to " + newId + " failed.");
        }
    }
}
