package com.fr.scheduler.tool;

import com.fr.scheduler.bean.FineJobProvider;
import com.fr.scheduler.job.FineJob;
import com.fr.stable.StringUtils;
import com.fr.third.v2.org.quartz.JobBuilder;
import com.fr.third.v2.org.quartz.JobDataMap;
import com.fr.third.v2.org.quartz.JobDetail;
import org.jetbrains.annotations.NotNull;

import java.util.HashMap;
import java.util.Map;

/**
 * 任务创建工具类.
 *
 * <p>
 * 用于创建添加任务所需的JobDetail.
 * </p>
 *
 * @author Cloud.Liu
 * created on 2020-09-09
 * @IncludeIntoJavadoc
 */
public class FineJobBuilder {

    /**
     * 创建一个JobDetail.
     *
     * <pre>
     * 如：
     * JobDetail jobDetail = FineJobBuilder.newJob()
     *         .withName("任务名称")
     *         .withGroup("任务组名")
     *         .withoutDescription()        // 无任务描述
     *         .withJobClass(TestJob.class) // 任务类
     *         .withoutDataMap()            // 任务不需要额外提供数据
     *         .build();
     * </pre>
     *
     * @return Builder
     */
    public static NameStep newJob() {
        return new Steps();
    }

    /**
     * 从{@link FineJobProvider}创建JobDetail.
     *
     * @param provider provider
     * @return JobDetail
     */
    public static JobDetail fromFineJobProvider(FineJobProvider provider) {
        return newJob()
                .withName(provider.getName())
                .withGroup(provider.getGroup())
                .withDescription(provider.getDescription())
                .withJobClass(provider.getJobClass())
                .withDataMap(provider.getJobDataMap())
                .withCreator(provider.getCreator())
                .build();
    }

    /**
     * 指定Job名称.
     */
    public interface NameStep {
        /**
         * 指定Job名称.
         *
         * @param name 名称
         * @return Builder
         */
        GroupStep withName(@NotNull String name);
    }

    /**
     * 指定JobGroup.
     */
    public interface GroupStep {
        /**
         * 指定JobGroup.
         *
         * @param group JobGroup
         * @return Builder
         */
        DescStep withGroup(@NotNull String group);
    }

    /**
     * 指定Job描述.
     */
    public interface DescStep {
        /**
         * 不创建描述.
         *
         * @return Builder
         */
        ClassStep withoutDescription();

        /**
         * 创建描述.
         *
         * @param description 描述
         * @return Builder
         */
        ClassStep withDescription(String description);
    }

    /**
     * 指定Job类.
     */
    public interface ClassStep {
        /**
         * 指定Job类.
         *
         * @param jobClass Job类
         * @return Builder
         */
        DataStep withJobClass(@NotNull Class<? extends FineJob> jobClass);
    }

    /**
     * 指定Job携带信息.
     */
    public interface DataStep {
        /**
         * Job不携带信息.
         *
         * @return Builder
         */
        BuildStep withoutDataMap();

        /**
         * 指定Job携带信息.
         *
         * @param dataMap 携带信息
         * @return Builder
         */
        BuildStep withDataMap(@NotNull Map<String, Object> dataMap);
    }

    /**
     * 创建JobDetail.
     */
    public interface BuildStep {

        /**
         * 指定任务创建者.
         *
         * @param creator 创建者
         * @return Builder
         */
        BuildStep withCreator(String creator);
        /**
         * 创建JobDetail.
         *
         * @return JobDetail
         */
        JobDetail build();
    }

    private static class Steps implements NameStep, GroupStep, DescStep, ClassStep, DataStep, BuildStep {

        private String name;
        private String group;
        private String description;
        private Class<? extends FineJob> jobClass;
        private Map<String, Object> jobDataMap;
        private String creator;

        @Override
        public GroupStep withName(@NotNull String name) {
            if (StringUtils.isEmpty(name)) {
                throw new IllegalArgumentException("Name should not be null or empty.");
            } else {
                this.name = name;
            }
            return this;
        }

        @Override
        public DescStep withGroup(@NotNull String group) {
            if (StringUtils.isEmpty(group)) {
                throw new IllegalArgumentException("Group should not be null or empty.");
            } else {
                this.group = group;
            }
            return this;
        }

        @Override
        public ClassStep withoutDescription() {
            return this;
        }

        @Override
        public ClassStep withDescription(@NotNull String description) {
            this.description = description;
            return this;
        }

        @Override
        public DataStep withJobClass(@NotNull Class<? extends FineJob> jobClass) {
            this.jobClass = jobClass;
            return this;
        }

        @Override
        public BuildStep withoutDataMap() {
            this.jobDataMap = new HashMap<>();
            return this;
        }

        @Override
        public BuildStep withDataMap(@NotNull Map<String, Object> dataMap) {
            this.jobDataMap = dataMap;
            return this;
        }

        @Override
        public BuildStep withCreator(String creator) {
            this.creator = creator;
            return this;
        }

        @Override
        public JobDetail build() {

            return JobBuilder.newJob(jobClass)
                    .withIdentity(name, group)
                    .withDescription(description)
                    .withCreator(creator)
                    .setJobData(new JobDataMap(jobDataMap))
                    .build();
        }
    }
}
